/*
gdpc - a program for visualising molecular dynamic simulations  
Copyright (C) 1999 Jonas Frantz   

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <gtk/gtk.h>
#include <stdio.h>
#include <math.h>
#include "parameters.h"

static struct xyzstruc *coords;
static gint numatoms;
static gboolean nfframe=TRUE;
static float ic[3][3]={{1.0,0.0,0.0},{0.0,1.0,0.0},{0.0,0.0,1.0}};


/*************************************************************************/
/* Transforms relative coordinates in input file to absolute coordinates */
/* on the drawable pixmap.                                               */
/*************************************************************************/
gint transf_abs(float x,float xmin,float xmax,gint absxsize) 
{
float newx;
	newx=(x-xmin)/(xmax-xmin);
	return (gint) (newx*absxsize);
}


/*****************************************************************************/
/* This function does the actual drawing of the circles accordingly to mode. */
/*****************************************************************************/
void drawcircles(GdkPixmap *pixmap,GdkColor *colors,float xmin,float xmax,
	    float ymin,float ymax,float zmin,float zmax,gint absxsize,
	    gint absysize,gint mode,gint radius,gint vary,struct xyzstruc *coords,
	    gboolean usetypes, gint numtypes) 
{
gint x,y,c,i,rtmp;
GdkGC *gc;

    gc = gdk_gc_new (pixmap);

    for(i=0;i<numatoms;i++) {
	x=transf_abs(coords[i].xcoord,xmin,xmax,absxsize);
	y=transf_abs(coords[i].ycoord,ymin,ymax,absysize);
	if(coords[i].zcoord>=zmin && coords[i].zcoord<=zmax) {

	    if (usetypes) c=transf_abs(coords[i].atype,0,numtypes+1,NUMCOLORS);
	    else c=transf_abs(coords[i].zcoord,zmin,zmax,NUMCOLORS);

	    if (vary==1) {
		rtmp=(int)(radius*(0.5*(coords[i].zcoord-zmin)/(zmax-zmin))+0.5*radius);
	    }
	    else if (vary==2) {
		rtmp=(int)(radius*(0.5*(-coords[i].zcoord+zmax)/(zmax-zmin))+0.5*radius);
	    }
	    else rtmp=radius;

	    gdk_gc_set_foreground (gc, &colors[c+2]);
	    if(x>0 && y>0 && x<(absxsize-radius/2) && y<(absysize-radius/2)) {
		if (mode==0) {
		    gdk_draw_rectangle(pixmap,gc,TRUE,x-rtmp/2+xborder,(absysize-y)-rtmp/2+yborder,rtmp,rtmp);
		}
		else if (mode==1) {
		    gdk_draw_arc(pixmap,gc,TRUE,x-rtmp/2+xborder,(absysize-y)-rtmp/2+yborder,rtmp,rtmp,0,360*64);
		}
		else if (mode==2) {
		    gdk_gc_set_foreground (gc, &colors[c*8+2]);
		    gdk_draw_arc(pixmap,gc,TRUE,x-rtmp/2+xborder,(absysize-y)-rtmp/2+yborder,rtmp,rtmp,0,360*64);
		    gdk_gc_set_foreground (gc, &colors[c*8+3]);
gdk_draw_arc(pixmap,gc,TRUE,x-0.453125*rtmp+xborder,(absysize-y)-0.453125*rtmp+yborder,rtmp*0.875,rtmp*0.875,0,360*64);
		    gdk_gc_set_foreground (gc, &colors[c*8+4]);
gdk_draw_arc(pixmap,gc,TRUE,x-0.40625*rtmp+xborder,(absysize-y)-0.40625*rtmp+yborder,rtmp*0.75,rtmp*0.75,0,360*64);
		    gdk_gc_set_foreground (gc, &colors[c*8+5]);
gdk_draw_arc(pixmap,gc,TRUE,x-0.359375*rtmp+xborder,(absysize-y)-0.359375*rtmp+yborder,rtmp*0.625,rtmp*0.625,0,360*64);
		    gdk_gc_set_foreground (gc, &colors[c*8+6]);
gdk_draw_arc(pixmap,gc,TRUE,x-0.3125*rtmp+xborder,(absysize-y)-0.3125*rtmp+yborder,rtmp*0.5,rtmp*0.5,0,360*64);
		    gdk_gc_set_foreground (gc, &colors[c*8+7]);
gdk_draw_arc(pixmap,gc,TRUE,x-0.265625*rtmp+xborder,(absysize-y)-0.265625*rtmp+yborder,rtmp*0.375,rtmp*0.375,0,360*64);
		    gdk_gc_set_foreground (gc, &colors[c*8+8]);
gdk_draw_arc(pixmap,gc,TRUE,x-0.2175*rtmp+xborder,(absysize-y)-0.2175*rtmp+yborder,rtmp*0.25,rtmp*0.25,0,360*64);
		    gdk_gc_set_foreground (gc, &colors[c*8+9]);
gdk_draw_arc(pixmap,gc,TRUE,x-0.155*rtmp+xborder,(absysize-y)-0.155*rtmp+yborder,rtmp*0.125,rtmp*0.125,0,360*64);
		}
	    }
	}
    }
    gdk_gc_unref(gc);
}


/**********************************************************************/
/* This function rotates the coordinates of the atoms, sorts them and */
/* calls the drawcircles to draw them.                                */
/**********************************************************************/
void rotateatoms(GdkPixmap *pixmap,GdkColor *colors,struct GlobalParams *params)
{
gint i;
float isin,icos,jsin,jcos,ksin,kcos,newx,newy,newz,newx2,maxx,minx,maxy,miny,maxz,minz;
float imsin,imcos,jmsin,jmcos,newmx,newmy,newmz,icl;
struct xyzstruc *newcoords;
float ictmp[3];

    minx=0.0;
    maxx=0.0;
    miny=0.0;
    maxy=0.0;
    minz=0.0;
    maxz=0.0;
    isin=sin(params->iangle*(PI/180.0));
    icos=cos(params->iangle*(PI/180.0));
    jsin=sin(params->jangle*(PI/180.0));
    jcos=cos(params->jangle*(PI/180.0));
    ksin=sin(params->kangle*(PI/180.0));
    kcos=cos(params->kangle*(PI/180.0));
    imsin=sin(params->jmangle*(PI/180.0));
    imcos=cos(params->jmangle*(PI/180.0));
    jmsin=sin(params->imangle*(-PI/180.0));
    jmcos=cos(params->imangle*(-PI/180.0));

    newcoords = (struct xyzstruc *) g_malloc(numatoms*sizeof(struct xyzstruc));

    for (i=0;i<3;i++) { 
	newx=ic[i][0];
	newy=ic[i][1];
	newz=ic[i][2];
	if (params->iangle!=0.0) {
	    newy=ic[i][1]*icos-ic[i][2]*isin;
	    newz=ic[i][1]*isin+ic[i][2]*icos;
	}
	if (params->jangle!=0.0) {
	    newx=ic[i][0]*jcos-newz*jsin;
	    newz=ic[i][0]*jsin+newz*jcos;
	}
	if (params->kangle!=0.0) {
	    newx2=newx;
	    newx=newx*kcos-newy*ksin;
	    newy=newx2*ksin+newy*kcos;
	}

	newmx=newx;
	newmy=newy;
	newmz=newz;

	if (params->imangle!=0.0) {
	    newmy=newy*imcos-newz*imsin;
	    newmz=newy*imsin+newz*imcos;
	}
	if (params->jmangle!=0.0) {
	newmx=newx*jmcos-newmz*jmsin;
	newmz=newx*jmsin+newmz*jmcos;
	}

	icl=sqrt(newmx*newmx+newmy*newmy+newmz*newmz);

	ic[i][0]=newmx/icl;
	ic[i][1]=newmy/icl;
	ic[i][2]=newmz/icl;
    }

	for(i=0;i<numatoms;i++) {
	    newcoords[i].xcoord=ic[0][0]*coords[i].xcoord+ic[0][1]*coords[i].ycoord+ic[0][2]*coords[i].zcoord;
	    newcoords[i].ycoord=ic[1][0]*coords[i].xcoord+ic[1][1]*coords[i].ycoord+ic[1][2]*coords[i].zcoord;
	    newcoords[i].zcoord=ic[2][0]*coords[i].xcoord+ic[2][1]*coords[i].ycoord+ic[2][2]*coords[i].zcoord;
	    newcoords[i].atype=coords[i].atype;
	    newcoords[i].index=i;
            if (newcoords[i].xcoord>maxx) maxx=newcoords[i].xcoord;
            if (newcoords[i].ycoord>maxy) maxy=newcoords[i].ycoord;
            if (newcoords[i].zcoord>maxz) maxz=newcoords[i].zcoord;
            if (newcoords[i].xcoord<minx) minx=newcoords[i].xcoord;
            if (newcoords[i].ycoord<miny) miny=newcoords[i].ycoord;
            if (newcoords[i].zcoord<minz) minz=newcoords[i].zcoord;
	}
    params->iangle=0.0;
    params->jangle=0.0;
    params->kangle=0.0;
    params->imangle=0.0;
    params->jmangle=0.0;

    if (ic[0][0]!=0.0) params->zc=atan(ic[0][1]/ic[0][0])*(180.0/PI);
    else params->zc=0.0;
    if (ic[0][0]<0.0 && ic[0][1]>0.0) params->zc+=180;
    else if (ic[0][0]<0.0 && ic[0][1]<0.0) params->zc+=180;
    else if (ic[0][0]>0.0 && ic[0][1]<0.0) params->zc+=360;
    
    ictmp[0]=ic[2][0]*cos(-params->zc*(PI/180.0))-ic[2][1]*sin(-params->zc*(PI/180.0));

    if (ic[2][2]!=0.0) params->yc=atan(-ictmp[0]/ic[2][2])*(180.0/PI);
    else params->yc=0.0;
    if (ic[2][2]<0.0 && ictmp[0]>0.0) params->yc+=180;
    else if (ic[2][2]<0.0 && ictmp[0]<0.0) params->yc+=180;
    else if (ic[2][2]>0.0 && ictmp[0]<0.0) params->yc+=360;

    ictmp[0]=ic[1][0]*cos(-params->zc*(PI/180.0))-ic[1][1]*sin(-params->zc*(PI/180.0));
    ictmp[1]=ic[1][0]*sin(-params->zc*(PI/180.0))+ic[1][1]*cos(-params->zc*(PI/180.0));
    ictmp[2]=ictmp[0]*sin(-params->yc*(PI/180.0))+ic[1][2]*cos(-params->yc*(PI/180.0));

    if (ictmp[1]!=0.0) params->xc=atan(ictmp[2]/ictmp[1])*(180.0/PI);
    else params->xc=0.0;
    if (ictmp[1]<0.0 && ictmp[2]>0.0) params->xc+=180;
    else if (ictmp[1]<0.0 && ictmp[2]<0.0) params->xc+=180;
    else if (ictmp[1]>0.0 && ictmp[2]<0.0) params->xc+=360;

    if (params->xc<=0.0) params->xc+=360.0;
    if (params->xc>=360.0) params->xc-=360.0;
    if (params->yc<=0.0) params->yc+=360.0;
    if (params->yc>=360.0) params->yc-=360.0;
    if (params->zc<=0.0) params->zc+=360.0;
    if (params->zc>=360.0) params->zc-=360.0;

    if (params->sort==2) {
	sortatoms(newcoords,0,numatoms-1,FALSE);	    
    }
    else sortatoms(newcoords,0,numatoms-1,TRUE);	    

    for (i=0;i<numatoms;i++) {
	newcoords[i].zcoord=coords[newcoords[i].index].zcoord;
    }

    if(params->xmin==65535.0) {
	params->xmax2=maxx;
	params->xmin2=minx;
    }
    else {
	params->xmax2=params->xmax;
	params->xmin2=params->xmin;
    }
    if(params->ymin==65535.0) {
	params->ymax2=maxy;
	params->ymin2=miny;
    }
    else {
	params->ymax2=params->ymax;
	params->ymin2=params->ymin;
    }
    if(params->zmin==65535.0) {
	params->zmax2=maxz;
	params->zmin2=minz;
    }
    else {
	params->zmax2=params->zmax;
	params->zmin2=params->zmin;
    }
    drawcircles(pixmap,colors,params->xmin2,params->xmax2,params->ymin2,
		params->ymax2,params->zmin2,params->zmax2,params->absxsize,
		params->absysize,params->mode,params->radius,params->vary,
		newcoords,params->usetypes,params->numtypes);
    g_free(newcoords);
}


/*********************************************************************/
/* Clears the drawable area and draws the rectangle which represents */
/* border of the simulationbox.                                      */
/*********************************************************************/
void cleardrawable(GtkWidget *widget,GdkPixmap *pixmap,GdkColor *colors, struct GlobalParams *params) 
{
GdkGC *gc;
gint bgcol,fgcol;

    if (params->whitebg) {
	bgcol=1;
	fgcol=0;
    }else {
	bgcol=0;
	fgcol=1;
    }

    gc = gdk_gc_new (pixmap);
    gdk_gc_set_foreground (gc, &colors[bgcol]);
    gdk_draw_rectangle(pixmap,gc,TRUE,0,0,params->absxsize+2*xborder,params->absysize+2*yborder);
    gdk_gc_set_foreground (gc, &colors[fgcol]);
    gdk_draw_rectangle(pixmap,gc,TRUE,xborder-2,yborder-2,params->absxsize+4,params->absysize+4);
    gdk_gc_set_foreground (gc, &colors[bgcol]);
    gdk_draw_rectangle (pixmap,gc,TRUE,xborder,yborder,params->absxsize,params->absysize);
}


/***********************************************************************/
/* Reads the input file and processes it, then it calls rotateatoms to */
/* rotate the coordinates and draw them.                               */
/***********************************************************************/
gboolean drawatoms(GdkPixmap *pixmap,GdkColor *colors,GtkWidget *widget,FILE *fp,
		   float *atime,struct GlobalParams *params) 
{
static gboolean framecheck=FALSE;
gint n, i, j, numtypes;
char buf[160];
char arg[20][40];
char timestr[40]="0.0\0";
gint nreadxyz, numalloc;
float maxx, maxy, maxz, minx, miny, minz;
static struct xyzstruc lastframe;
gboolean timecheck, endframe;
gchar AType[MAXTYPES][5];
gboolean typescheck;

    numtypes=0;
    minx=0.0;
    miny=0.0;
    minz=0.0;
    maxx=0.0;
    maxy=0.0;
    maxz=0.0;

/* If file is in xyz format start reading here ! */
	if (params->fxyz) {
    		if (fgets(buf,160,fp) == NULL) return FALSE;
		n=sscanf(buf,"%d",&nreadxyz);
		if (n!=1) {
		     printf("xyz format ERROR on line 1 : %s\nToo many columns on first row of frame. 
			     Make sure the input file is in xyz format.\n",buf);
		     return FALSE;
		}
		if (fgets(buf,160,fp) == NULL) return FALSE;      /*Read in comment line */
		n=sscanf(buf,"%s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s",
			arg[0],arg[1],arg[2],arg[3],arg[4],arg[5],arg[6],arg[7],arg[8],arg[9],
			arg[10],arg[11],arg[12],arg[13],arg[14],arg[15],arg[16],arg[17],arg[18],
			arg[19]);
		timecheck=FALSE;
		for (i=1;i<n;i++) {
		    if (strcmp(arg[i],"fs")==0) {
			strcpy((char *) timestr,(char *) arg[i-1]);
			timecheck=TRUE;
		    }
		}
		if(timecheck) n=sscanf(timestr,"%f",atime);
	    if (nfframe) g_free(coords);
	    nfframe=TRUE;
	    coords = (struct xyzstruc *) g_malloc(nreadxyz*sizeof(struct xyzstruc));
	    endframe=FALSE;
	    numatoms=0;
	    numtypes=0;
	    for(i=0;i<nreadxyz;i++) {
		if(fgets(buf,160,fp) == NULL) endframe=TRUE;
		n=sscanf(buf,"%s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s",
			arg[0],arg[1],arg[2],arg[3],arg[4],arg[5],arg[6],arg[7],arg[8],arg[9],
			arg[10],arg[11],arg[12],arg[13],arg[14],arg[15],arg[16],arg[17],arg[18],
			arg[19]);
		if (n<params->xcolumn || n<params->ycolumn || n<params->zcolumn || n<params->tcolumn) {
		    printf("Error in xyz input file : %s\nExiting.\n",buf); 
		    gtk_exit(0);
		}
		if (params->scol>0) {
		    if (strcmp(params->fstring,arg[params->scol-1])) continue;
		}
		typescheck=FALSE;
		for (j=0;j<numtypes;j++) {
		    if(strcmp(AType[j],arg[0])==0) {
			typescheck=TRUE;
			break;
		    }
		}
		if (!typescheck) {
		    strcpy(AType[numtypes],arg[0]);
		    coords[numatoms].atype=numtypes;
		    numtypes++;
		}
		else coords[numatoms].atype=j;
		n=sscanf(arg[params->xcolumn-1],"%f",&coords[numatoms].xcoord);
		if (n==0) printf("There seems to be a problem with converting \'%s\' to a number.\n",arg[params->xcolumn-1]);
		n=sscanf(arg[params->ycolumn-1],"%f",&coords[numatoms].ycoord);
		if (n==0) printf("There seems to be a problem with converting \'%s\' to a number.\n",arg[params->ycolumn-1]);
		n=sscanf(arg[params->zcolumn-1],"%f",&coords[numatoms].zcoord);
		if (n==0) printf("There seems to be a problem with converting \'%s\' to a number.\n",arg[params->zcolumn-1]);
		if (coords[numatoms].xcoord>maxx) maxx=coords[numatoms].xcoord;
		if (coords[numatoms].ycoord>maxy) maxy=coords[numatoms].ycoord;
		if (coords[numatoms].zcoord>maxz) maxz=coords[numatoms].zcoord;
		if (coords[numatoms].xcoord<minx) minx=coords[numatoms].xcoord;
		if (coords[numatoms].ycoord<miny) miny=coords[numatoms].ycoord;
		if (coords[numatoms].zcoord<minz) minz=coords[numatoms].zcoord;
		numatoms++;
		if (endframe) { 
		    printf("Anomaly : End of file reached !\n");
		    break;
		}
	    }
	    if(params->xmin==65535.0) {
		params->xmax2=maxx;
		params->xmin2=minx;
	    }
	    else {
		params->xmax2=params->xmax;
		params->xmin2=params->xmin;
	    }
	    if(params->ymin==65535.0) {
		params->ymax2=maxy;
		params->ymin2=miny;
	    }
	    else {
		params->ymax2=params->ymax;
		params->ymin2=params->ymin;
	    }
	    if(params->zmin==65535.0) {
		params->zmax2=maxz;
		params->zmin2=minz;
	    }
	    else {
		params->zmax2=params->zmax;
		params->zmin2=params->zmin;
	    }
	    
	    if(params->erase) {
		cleardrawable(widget,pixmap,colors,params);
	    }

	    params->numtypes=numtypes;
	    rotateatoms(pixmap,colors,params);
	    if (endframe) {
		return FALSE;
	    }
	}

/* If not in xyz format start redaing from here ! */

    else {
	if (nfframe) g_free(coords);
	nfframe=TRUE;
	numalloc=ALLOCTHIS;
	i=0;
	coords=(struct xyzstruc *) g_malloc(numalloc*sizeof(struct xyzstruc));
	if (framecheck) {
            coords[i].xcoord=lastframe.xcoord;
            coords[i].ycoord=lastframe.ycoord;
            coords[i].zcoord=lastframe.zcoord;
            coords[i].tcoord=lastframe.tcoord;
	    i++;
	}
	framecheck=TRUE;
	endframe=TRUE;
	while(fgets(buf,160,fp) != NULL) {
	    if(i+1==numalloc) {
		numalloc+=ALLOCTHIS;
		coords=g_realloc(coords,numalloc*sizeof(struct xyzstruc));
	    }
	    n=sscanf(buf,"%s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s %s",
                     arg[0],arg[1],arg[2],arg[3],arg[4],arg[5],arg[6],arg[7],arg[8],arg[9],
                     arg[10],arg[11],arg[12],arg[13],arg[14],arg[15],arg[16],arg[17],arg[18],
                     arg[19]);
	    if (n<params->xcolumn || n<params->ycolumn || n<params->zcolumn || n<params->tcolumn) {
		printf("Error in input file : %sAre you sure the inputfile isn't in xyz format ?\nExiting.\n",buf); 
		gtk_exit(0);
	    }
	    if (params->scol>0) {
		if (strcmp(params->fstring,arg[params->scol-1])) continue;
	    }
	    n=sscanf(arg[params->xcolumn-1],"%f",&coords[i].xcoord);
	    if (n==0) printf("There seems to be a problem with converting \'%s\' to a number.\n",arg[params->xcolumn-1]);
	    n=sscanf(arg[params->ycolumn-1],"%f",&coords[i].ycoord);
	    if (n==0) printf("There seems to be a problem with converting \'%s\' to a number.\n",arg[params->ycolumn-1]);
	    n=sscanf(arg[params->zcolumn-1],"%f",&coords[i].zcoord);
	    if (n==0) printf("There seems to be a problem with converting \'%s\' to a number.\n",arg[params->zcolumn-1]);
	    n=sscanf(arg[params->tcolumn-1],"%f",&coords[i].tcoord);
	    if (n==0) printf("There seems to be a problem with converting \'%s\' to a number.\n",arg[params->tcolumn-1]);
	    if (coords[i].tcoord==coords[0].tcoord) { 
		if (coords[i].xcoord>maxx) maxx=coords[i].xcoord;
		if (coords[i].ycoord>maxy) maxy=coords[i].ycoord;
		if (coords[i].zcoord>maxz) maxz=coords[i].zcoord;
		if (coords[i].xcoord<minx) minx=coords[i].xcoord;
		if (coords[i].ycoord<miny) miny=coords[i].ycoord;
		if (coords[i].zcoord<minz) minz=coords[i].zcoord;
		i++;
	    }
	    else {
		endframe=FALSE;
		break;
	    }
	}
	*atime=coords[i-1].tcoord;
	lastframe.xcoord=coords[i].xcoord;
	lastframe.ycoord=coords[i].ycoord;
	lastframe.zcoord=coords[i].zcoord;
	lastframe.tcoord=coords[i].tcoord;
	if(params->xmin==65535.0) {
	    params->xmax2=maxx;
	    params->xmin2=minx;
	}
	else {
	    params->xmax2=params->xmax;
	    params->xmin2=params->xmin;
	}
	if(params->ymin==65535.0) {
	    params->ymax2=maxy;
	    params->ymin2=miny;
	}
	else {
	    params->ymax2=params->ymax;
	    params->ymin2=params->ymin;
	}
	if(params->zmin==65535.0) {
	    params->zmax2=maxz;
	    params->zmin2=minz;
	}
	else {
	    params->zmax2=params->zmax;
	    params->zmin2=params->zmin;
	}

	if(params->erase) {
	    cleardrawable(widget,pixmap,colors,params);
	}

	numatoms=i;
	rotateatoms(pixmap,colors,params);
	if (endframe) {
	    return FALSE;
	}
    }
return TRUE;
}


/*********************************************************/
/* This function resets the orientation of the system by */
/* reseting the vector that handles the rotation.        */
/*********************************************************/
void resetic() 
{
    ic[0][0]=1.0;ic[0][1]=0.0;ic[0][2]=0.0;
    ic[1][0]=0.0;ic[1][1]=1.0;ic[1][2]=0.0;
    ic[2][0]=0.0;ic[2][1]=0.0;ic[2][2]=1.0;
}
