/*****************************************************************************/
/* File: lzss.c                                                              */
/* Author: David Chatenay                                                    */
/* Last Modified: Wed Jan 15 1997                                            */
/*                                                                           */
/* A standard interface for Lzss encoding of file, plus low-level routines   */
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "common.h"
#include "crunch.h"
#include "buffer.h"
#include "header.h"


/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
/*                               MISCELLANOUS                                */
/*****************************************************************************/
/* Way to insert a string in the array of found */
/* strings (which is double ended in fact)...   */
#define FROM_BEGIN       0
#define FROM_END         1

/* Macro to increment with modulo */
#define INCR(i)    (i)++; if ((i) == CBUFFER_SIZE) i = 0;
/* Macro to have a correct index in circular buffer */
#define CORRECT(c) ((c) % (dword)CBUFFER_SIZE)



/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
/*                           LOW-LEVEL FUNCTIONS                             */
/*****************************************************************************/
/* Initialization of Lzss vars needed */
void LzssInitContext(LzssContext *h)
{
    int i;

    for (i=0; i<256; i++) {
	h->Access[i].first = NULL;
	h->Access[i].last = NULL;
	h->Access[i].number = 0;
    }
    for (i=0; i<CBUFFER_SIZE; i++) {
	h->Dictionnary[i].position = (word)i;
        h->Dictionnary[i].next = NULL;
    }
    h->Begin = 0;
    h->BufferLength = 0;
}


/* Find all positions of a single byte */
static word FindSingleChar(LzssContext *h, word *str, byte c, byte way)
{
    word found=0, p;
    LzssIndex *i;

    /* First index */
    i = h->Access[c].first;
    if (!i)
        /* Not an indexed byte */
        return 0;
    else {
	/* Compute the first insertion point */
	if (way == FROM_BEGIN)
	    p = 0;
	else
	    p = CBUFFER_SIZE - 1;
	while (i) {
	    /* Insert each found position into the strings */
	    str[p] = i->position;
	    if (way == FROM_BEGIN)
	        p++;
	    else
	        p--;
	    found++;
	    i = i->next;
	}
    }

    return found;
}


static word FindStringsFromByte(LzssContext *h, word *str, byte first, byte c,
				byte way)
{
    word found=0, insert, i, max, pos;
    LzssIndex *ind;

    /* Init of index */
    if (way == FROM_BEGIN)
	insert = 0;
    else
        insert = CBUFFER_SIZE - 1;
    /* For each position of the old char... */
    max = h->Access[first].number;
    ind = h->Access[first].first;
    for (i=0; i<max; i++) {
	pos = CORRECT(ind->position + 1);
	if (pos != h->Begin && pos < h->BufferLength
	    && h->CircularBuffer[pos] == c) {
	    /* It's a good string */
	    str[insert] = ind->position;
	    found++;
	    if (way == FROM_BEGIN)
	        insert++;
	    else
	        insert--;
	}
	ind = ind->next;
    }
    return found;
}


/* Find all strings composed of last strings */
/* plus the byte c, inserting with way. lf   */
/* is the number of strings found last round */
/* Returns the number of matching strings... */
static word FindStrings(LzssContext *h, word *str, word lf, word len, byte c,
			byte way)
{
    word found=0, search, insert, i, pos;

    /* Init of indexes through the strings */
    if (way == FROM_BEGIN) {
	insert = 0;
	search = CBUFFER_SIZE - 1;
    } else {
        search = 0;
	insert = CBUFFER_SIZE - 1;
    }
    /* For each string found at last round... */
    for (i=0; i<lf; i++) {
	/* Get the following char... */
	pos = CORRECT(str[search] + len);
	/* ...and compare it to the new! */
	if (pos != h->Begin && pos < h->BufferLength 
	    && h->CircularBuffer[pos] == c) {
	    /* It's a good string */
	    str[insert] = str[search];
	    found++;
	    if (way == FROM_BEGIN) {
	        insert++;
		search--;
	    } else {
		search++;
		insert--;
	    }
	} else {
	    /* It's not a good string */
	    if (way == FROM_BEGIN)
	        search--;
	    else
	        search++;
	}
    }
    return found;
}


/* Inserts in the dictionnary the byte b  */
/* at position pos in the circular buffer */
static void InsertIndex(LzssContext *h, byte b, word pos)
{
    LzssIndex *current, *new;

    /* Find the cell position*/
    new = h->Dictionnary + pos;
    /* Go to end of list */
    current = h->Access[b].last;
    /* Is there a previous index? */
    if (!current)
        /* No: we init it. */
        h->Access[b].first = h->Access[b].last = new;
    else 
        /* Yes: we modify end of list pointer */
	h->Access[b].last = current->next = new;
    /* We mark the end of list */
    new->next = NULL;
    /* We increment the number of items */
    h->Access[b].number++;
}


/* Remove the first index to the byte  */
/* b. Some checks are made, to avoid a */
/* corrupted index table... (normally  */
/* this may never happen...)           */
static void RemoveIndex(LzssContext *h, byte b)
{
    LzssIndex *i;

    i = h->Access[b].first;
    if (i->position == h->Begin) {
	if (i->next)
	    h->Access[b].first = i->next;
	else
	    h->Access[b].first = h->Access[b].last = NULL;
	h->Access[b].number--;
    } else {
	DEBUG0("[Position Fault]\n");
	DEBUG2("[%d != %d]\n", i->position, h->Begin);
	ERROR("Bug in module lzss.c: Index table is corrupted!");
	exit(-1);
    }
}


/* Inserts a char at the last position */
/* in the circular buffer and into the */
/* index table, deleting the previous  */
/* entry.                              */
static void InsertByte(LzssContext *h, byte c)
{
    if (h->BufferLength == CBUFFER_SIZE) {
	/* Buffer is full, we must free the first char...   */
	/* Find index occupied by first char, and remove it */
	RemoveIndex(h, h->CircularBuffer[h->Begin]);
	/* Insert new char */
	h->CircularBuffer[h->Begin] = c;
	/* Insert into to index table */
	InsertIndex(h, c, h->Begin);
	/* Increment the beginning of buffer */
	INCR(h->Begin);
    } else {
	/* Buffer is not full yet... */
	h->CircularBuffer[h->BufferLength] = c;
	/* Insert into to index table */
	InsertIndex(h, c, h->BufferLength);
	/* Increment buffer length */
	h->BufferLength++;
    }
}


/* Outputs the byte b, with a 0 in before */
static int OutputByte(byte b, Buffer *out)
{
    Code c;

    c.code_len = 9;
    c.code[0] = (b << 1) & 0xfe;
    c.code[1] = (b >> 7) & 0x01;

    return AddCode(&c, out);
}


/* Outputs the string s, with a 1 before */
static int OutputString(word pos, word len, Buffer *out)
{
    Code c;

    /* The _absolute_ value is output */
    /* This suppress conversion */
    c.code_len = CBUFFER_LOG + MAX_LOG + 1;
    c.code[0] = 0x01 | ((len & Mask[MAX_LOG]) << 1)
                | ((pos & Mask[7 - MAX_LOG]) << (MAX_LOG + 1));
    c.code[1] = (pos >> (7 - MAX_LOG)) & 0xff;
    c.code[2] = (pos >> (15 - MAX_LOG)) & Mask[MAX_LOG + CBUFFER_LOG - 15];

    return AddCode(&c, out);
}


/* The raw file encoding function */
static int LzssRawEncode(LzssContext *h, Buffer *in, Buffer *out)
{
#define FIND_NEXT(x) (x) = Get(in); \
                     if (in->buff_length == 0) { \
			 DEBUG0("[exit on FIND_NEXT]\n");\
			 TORI(OutputByte((x), out)); \
			 FlushBuffer(out); \
		         return 0; \
		     }
#define FIND_NEXT2(x) (x) = Get(in); \
                      if (in->buff_length == 0) { \
			  DEBUG0("[exit on FIND_NEXT2]\n");\
			  TORI(OutputByte(old, out)); \
			  TORI(OutputByte((x), out)); \
			  FlushBuffer(out); \
		          return 0; \
		      }
#define OUTPUT_BYTE(x) TORI(OutputByte((x), out)); \
		       InsertByte(h, (x));
#define INIT_STRING string[0]=old; string[1]=current; last_length=2;
    word OldFoundNumber, NewFoundNumber, last_length, i;
    byte string[MAX_LENGTH + MIN_LENGTH];
    byte current, way, *buffer, old;
    word found[CBUFFER_SIZE], str;

    /* Initialization */
    buffer = h->CircularBuffer; /* Access speed-up */
    ResetBuffer(in);
    ReadBuffer(in);
    /* Insert first chars */
    way = FROM_BEGIN;
    FIND_NEXT(old);
    FIND_NEXT2(current);
    OldFoundNumber = FindStringsFromByte(h, found, old, current, way);
    while (OldFoundNumber == 0) {
	OUTPUT_BYTE(old);
	old = current;
	FIND_NEXT2(current);
	OldFoundNumber = FindStringsFromByte(h, found, old, current, way);
    }
    /* Init the last found strings */
    INIT_STRING;
    way = FROM_END;

    /* Loop while not EOF */
    while (in->buff_length != 0) {
	/* Get a new char */
	current = Get(in);
	if (in->buff_length == 0) {
	    DEBUG0("[exit on update string]\n");
	    if (way == FROM_BEGIN)
	        str = found[CBUFFER_SIZE - 1];
	    else
	        str = found[0];
	    TORI(OutputString(str, last_length - MIN_LENGTH, out));
	    TORI(OutputByte(current, out));
	    FlushBuffer(out);
	    return 0;
	}
	/* Find the new string == old strings + new char */
	NewFoundNumber = FindStrings(h, found, OldFoundNumber, last_length,
				       current, way);
	if (NewFoundNumber != 0) {
	    /* We found some, so we iterate */
	    string[last_length++] = current;
	    if (last_length != MAX_LENGTH + MIN_LENGTH - 1) {
		/* We didn't reach the size limit, */
		/* so we simply iterate...         */
		OldFoundNumber = NewFoundNumber;
		way = !way;
	    } else {
		/* We went to the limit of size */
		/* Output the complete string */
		if (way == FROM_BEGIN)
		    str = found[0];
		else
		    str = found[CBUFFER_SIZE - 1];
		TORI(OutputString(str, MAX_LENGTH - 1, out));
		/* Insert the string into the dictionnary */
		for (i=0; i<last_length; i++)
		    InsertByte(h, string[i]);
		/* Search for next found byte */
		FIND_NEXT(old);
		FIND_NEXT2(current);
		OldFoundNumber = FindStringsFromByte(h, found, old, current,
						     way);
		while (OldFoundNumber == 0) {
		    OUTPUT_BYTE(old);
		    old = current;
		    FIND_NEXT2(current);
		    OldFoundNumber = FindStringsFromByte(h, found, old,
							 current, way);
		}
		INIT_STRING;
		way = !way;
	    }
	} else {
	    /* We found none, so we emit the latent string */
	    /* It's a string */
	    if (last_length < MIN_LENGTH) {
		/* It doesn't match the minimal length */
		/* We emit each of its chars... */
		for (i=0; i<last_length; i++) {
		    TORI(OutputByte(string[i], out));
		    InsertByte(h, string[i]);
		}
	    } else {
	        /* It's a complete string */
		if (way == FROM_BEGIN)
		    str = found[CBUFFER_SIZE - 1];
		else
		    str = found[0];
		TORI(OutputString(str, last_length - MIN_LENGTH, out));
		/* We insert the string into the dictionnary */
		for (i=0; i<last_length; i++)
		    InsertByte(h, string[i]);
	    }
	    old = current;
	    FIND_NEXT2(current);
	    OldFoundNumber = FindStringsFromByte(h, found, old, current, way);
	    while (OldFoundNumber == 0) {
		OUTPUT_BYTE(old);
		old = current;
	        FIND_NEXT2(current);
		OldFoundNumber = FindStringsFromByte(h, found, old,
						     current, way);
	    }
	    INIT_STRING;
	    way = !way;
	}
    }
    DEBUG0("[exit on normal case]\n");
    if (last_length < MIN_LENGTH) {
	/* It doesn't match the minimal length */
	/* We emit each of its chars... */
	DEBUG0("[Output bytes]\n");
	for (i=0; i<last_length; i++) {
	    TORI(OutputByte(string[i], out));
	    InsertByte(h, string[i]);
	}
    } else {
	DEBUG1("[Output string [%d]]\n", last_length);
	if (way == FROM_BEGIN)
	    str = found[CBUFFER_SIZE - 1];
	else
	    str = found[0];
	/* It's a complete string */
	TORI(OutputString(str, last_length - MIN_LENGTH, out));
    }
    FlushBuffer(out);

    return 0;
#undef FIND_NEXT
#undef FIND_NEXT2
#undef OUTPUT_BYTE
#undef INIT_STRING
}


/* Outputs the byte b, with a 0 in before, in the block out */
static int OutputByteToBlock(byte b, byte *out, dword size, dword *bytec,
			     word *bitc)
{
    Code c;

    c.code_len = 9;
    c.code[0] = (b << 1) & 0xfe;
    c.code[1] = (b >> 7) & 0x01;

    return AddCodeToBlock(&c, out, size, bytec, bitc);
}


/* Outputs the string s, with a 1 before */
static int OutputStringToBlock(word pos, word len, byte *out, dword size,
			dword *bytec, word *bitc)
{
    Code c;

    /* The _absolute_ value is output */
    /* This suppress conversion */
    c.code_len = CBUFFER_LOG + MAX_LOG + 1;
    c.code[0] = 0x01 | ((len & Mask[MAX_LOG]) << 1)
                | ((pos & Mask[7 - MAX_LOG]) << (MAX_LOG + 1));
    c.code[1] = (pos >> (7 - MAX_LOG)) & 0xff;
    c.code[2] = (pos >> (15 - MAX_LOG)) & Mask[MAX_LOG + CBUFFER_LOG - 15];

    return AddCodeToBlock(&c, out, size, bytec, bitc);
}


/* The raw block encoding function */
int LzssBlockEncode(LzssContext *h, byte *in, dword sin, byte *out,
		    dword *sout)
{
#define FIND_NEXT(x) (x) = in[in_count]; in_count++; \
                     if (in_count == in_size) { \
			 DEBUG0("[exit on FIND_NEXT]\n"); \
			 TORI(OutputByteToBlock((x), out, out_size, \
						&byte_count, &bit_count)); \
			 if (bit_count > 0) byte_count++; \
			 (*sout) = byte_count; \
		         return 0; \
		     }
#define FIND_NEXT2(x) (x) = in[in_count]; in_count++; \
                      if (in_count == in_size) { \
			  DEBUG0("[exit on FIND_NEXT2]\n"); \
			  TORI(OutputByteToBlock(old, out, out_size, \
						 &byte_count, &bit_count)); \
			  TORI(OutputByteToBlock((x), out, out_size, \
						 &byte_count, &bit_count)); \
			  if (bit_count > 0) byte_count++; \
			  (*sout) = byte_count; \
		          return 0; \
		      }
#define OUTPUT_BYTE(x) TORI(OutputByteToBlock((x), out, out_size, \
					      &byte_count, &bit_count)); \
		       InsertByte(h, (x));
#define INIT_STRING string[0]=old; string[1]=current; last_length=2;
    word OldFoundNumber, NewFoundNumber, last_length, i, bit_count=0;
    dword in_size, in_count=0, byte_count=0, out_size;
    byte string[MAX_LENGTH + MIN_LENGTH];
    word found[CBUFFER_SIZE], str, j;
    byte current, way, *buffer, old;

    /* Initialization */
    buffer = h->CircularBuffer; /* Access speed-up */
    in_size = sin;
    out_size = (*sout);
    /* Insert first chars */
    way = FROM_BEGIN;
    FIND_NEXT(old);
    FIND_NEXT2(current);
    OldFoundNumber = FindStringsFromByte(h, found, old, current, way);
    while (OldFoundNumber == 0) {
	OUTPUT_BYTE(old);
	old = current;
	FIND_NEXT2(current);
	OldFoundNumber = FindStringsFromByte(h, found, old, current, way);
    }
    /* Init the last found strings */
    INIT_STRING;
    way = FROM_END;

    /* Loop on input block size */
    while (in_count < in_size) {
	/* Get a new char */
	current = in[in_count]; in_count++;
	if (in_count == in_size) {
	    DEBUG0("[exit on first strike]\n");
	    if (way == FROM_BEGIN)
	        str = found[CBUFFER_SIZE - 1];
	    else
	        str = found[0];
	    TORI(OutputStringToBlock(str, last_length - MIN_LENGTH, out,
				     out_size, &byte_count, &bit_count));
	    TORI(OutputByteToBlock(current, out, out_size, &byte_count,
				   &bit_count));
	    if (bit_count > 0)
	        byte_count++;
	    (*sout) = byte_count;
	    return 0;
	}
	/* Find the new string == old strings + new char */
	NewFoundNumber = FindStrings(h, found, OldFoundNumber, last_length,
				       current, way);
	if (NewFoundNumber != 0) {
	    /* We found some, so we iterate */
	    string[last_length++] = current;
	    if (last_length != MAX_LENGTH + MIN_LENGTH - 1) {
		/* We didn't reach the size limit, */
		/* so we simply iterate...         */
		OldFoundNumber = NewFoundNumber;
		way = !way;
	    } else {
		/* We went to the limit of size */
		/* Output the complete string */
		if (way == FROM_BEGIN)
		    str = found[0];
		else
		    str = found[CBUFFER_SIZE - 1];
		TORI(OutputStringToBlock(str, MAX_LENGTH - 1, out, out_size,
					 &byte_count, &bit_count));
		/* Insert the string into the dictionnary */
		for (j=0; j<last_length; j++)
		    InsertByte(h, string[j]);
		/* Search for next found byte */
		FIND_NEXT(old);
		FIND_NEXT2(current);
		OldFoundNumber = FindStringsFromByte(h, found, old, current,
						     way);
		while (OldFoundNumber == 0) {
		    OUTPUT_BYTE(old);
		    old = current;
		    FIND_NEXT2(current);
		    OldFoundNumber = FindStringsFromByte(h, found, old,
							 current, way);
		}
		INIT_STRING;
		way = !way;
	    }
	} else {
	    /* We found none, so we emit the latent string */
	    /* It's a string */
	    if (last_length < MIN_LENGTH) {
		/* It doesn't match the minimal length */
		/* We emit each of its chars... */
		for (j=0; j<last_length; j++) {
		    TORI(OutputByteToBlock(string[j], out, out_size,
					   &byte_count, &bit_count));
		    InsertByte(h, string[j]);
		}
	    } else {
	        /* It's a complete string */
		if (way == FROM_BEGIN)
		    str = found[CBUFFER_SIZE - 1];
		else
		    str = found[0];
		TORI(OutputStringToBlock(str, last_length - MIN_LENGTH, out,
					 out_size, &byte_count, &bit_count));
		/* We insert the string into the dictionnary */
		for (j=0; j<last_length; j++)
		    InsertByte(h, string[j]);
	    }
	    old = current;
	    FIND_NEXT2(current);
	    OldFoundNumber = FindStringsFromByte(h, found, old, current, way);
	    while (OldFoundNumber == 0) {
		OUTPUT_BYTE(old);
		old = current;
	        FIND_NEXT2(current);
		OldFoundNumber = FindStringsFromByte(h, found, old, current,
						     way);
	    }
	    INIT_STRING;
	    way = !way;
	}
    }
    DEBUG0("[exit on normal case]\n");
    if (last_length < MIN_LENGTH) {
	/* It doesn't match the minimal length */
	/* We emit each of its chars... */
	DEBUG0("  [output bytes]\n");
	for (j=0; j<last_length; j++) {
	    TORI(OutputByteToBlock(string[j], out, out_size, &byte_count,
				   &bit_count));
	    InsertByte(h, string[j]);
	}
    } else {
	/* It's a complete string */
	DEBUG1("  [output string [%d bytes]]\n", last_length);
	if (way == FROM_BEGIN)
	    str = found[CBUFFER_SIZE - 1];
	else
	    str = found[0];
	TORI(OutputStringToBlock(str, last_length - MIN_LENGTH, out, out_size,
				 &byte_count, &bit_count));
    }
    /* Output the length of encoded block */
    (*sout) = ((bit_count == 0) ? byte_count : (byte_count + 1));

    return 0;
#undef FIND_NEXT
#undef FIND_NEXT2
#undef OUTPUT_BYTE
#undef INIT_STRING
}


static void InsertByte2(LzssContext *h, byte c)
{
    if (h->BufferLength == CBUFFER_SIZE) {
	/* Buffer is full  */
	/* Insert new char */
	h->CircularBuffer[h->Begin] = c;
	/* Increment the beginning of buffer */
	INCR(h->Begin);
    } else {
	/* Buffer is not full yet... */
	h->CircularBuffer[h->BufferLength++] = c;
    }
}


static int LzssRawDecode(LzssContext *h, Buffer *in, Buffer *out,
			 dword file_length)
{
    word length, position, i;
    dword l;
    Code c;
    byte b;

    c.code[0] = c.code[1] = c.code[2] = 0x00;
    l = 0;
    ReadBuffer(in);
    while (l != file_length && in->buff_length > 0) {
	if (in->buffer[in->byte_count] & RMask[in->bit_count]) {
	    /* It's a string */
	    /* Read it */
	    c.code_len = CBUFFER_LOG + MAX_LOG + 1;
	    TORI(ReadCode(&c, in));
	    /* Decompose length and position */
	    length = (c.code[0] >> 1) & Mask[MAX_LOG];
	    length += MIN_LENGTH;
	    position = (word)((c.code[0] >> (MAX_LOG + 1)) &
			      Mask[7 - MAX_LOG]);
	    position |= (word)c.code[1] << (7 - MAX_LOG);
	    position |= (word)c.code[2] << (15 - MAX_LOG);
	    /* Output the bytes */
	    for (i=0; i<length; i++)
		Put(out, h->CircularBuffer[CORRECT(position+i)]);
	    /* Insert them in the circular buffer */
	    for (i=0; i<length; i++)
		InsertByte2(h, h->CircularBuffer[CORRECT(position+i)]);
	    /* Increment decoded bytes */
	    l += length;
	} else {
	    /* It's a byte */
	    /* Read it */
	    c.code_len = 9;
	    TORI(ReadCode(&c, in));
	    b = ((byte)(c.code[0] & 0xfe) >> 1) |
	        ((byte)(c.code[1] & 0x01) << 7);
	    /* Output the byte */
	    Put(out, b);
	    /* Insert it in the circular buffer */
	    InsertByte2(h, b);
	    /* Increment decoded bytes */
	    l++;
	}
    }
    FlushBuffer(out);
    return 0;
}


int LzssBlockDecode(LzssContext *h, byte *in, dword sin, byte *out, 
		    dword *sout)
{
    dword byte_count=0, size, out_count=0, max_out;
    word length, position, i, bit_count=0;
    Code c;
    byte b;

    /* Initialization */
    c.code[0] = c.code[1] = c.code[2] = 0x00;
    size = sin;
    max_out = *sout;
    /* Main loop (strange stop, but should work) */
    while (byte_count < size - 1) {
	if (in[byte_count] & RMask[bit_count]) {
	    /* It's a string */
	    /* Read it */
	    c.code_len = CBUFFER_LOG + MAX_LOG + 1;
	    TORI(ReadCodeFromBlock(&c, in, size, &byte_count, &bit_count));
	    /* Decompose length and position */
	    length = (c.code[0] >> 1) & Mask[MAX_LOG];
	    length += MIN_LENGTH;
	    position = (word)((c.code[0] >> (MAX_LOG + 1)) &
			      Mask[7 - MAX_LOG]);
	    position |= (word)c.code[1] << (7 - MAX_LOG);
	    position |= (word)c.code[2] << (15 - MAX_LOG);
	    /* Output the bytes */
	    if (out_count + length > max_out) {
		DEBUG3("Fuck [%d + %d, %d]!\n", out_count, length, byte_count);
		DEBUG2("Last bytes: %2.2x %2.2x\n",
		       h->CircularBuffer[CORRECT(position)],
		       h->CircularBuffer[CORRECT(position+1)]);
		DEBUG2("Last string: %x [%x]\n", position, length);
		return -1;
	    }
	    for (i=0; i<length; i++)
	        out[out_count + i] = h->CircularBuffer[CORRECT(position+i)];
	    out_count += i;
	    /* Insert them in the circular buffer */
	    for (i=0; i<length; i++)
		InsertByte2(h, h->CircularBuffer[CORRECT(position+i)]);
	} else {
	    /* It's a byte */
	    /* Read it */
	    c.code_len = 9;
	    TORI(ReadCodeFromBlock(&c, in, size, &byte_count, &bit_count));
	    b = ((byte)(c.code[0] & 0xfe) >> 1) |
	        ((byte)(c.code[1] & 0x01) << 7);
	    /* Output the byte */
	    out[out_count++] = b;
	    if (out_count > max_out) {
		DEBUG1("Fuck2 [%d]!\n", out_count);
		return -1;
	    }
	    /* Insert it in the circular buffer */
	    InsertByte2(h, b);
	}
    }
    (*sout) = out_count;

    return 0;
}


/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
/*                          HIGH LEVEL FUNCTIONS                             */
/*****************************************************************************/
/* ENCODING */
int LzssEncode(int in, int out, float *cr)
{
    Buffer *bin, *bout;
    CrunchHeader head;
    LzssContext h;
    int r;

    /* Header write */
    TORI(ComputeHeader(in, &head, METHOD_LZSS));
    TORI(WriteHeader(out, &head));
 
    /* Allocate buffers */
    bin = NewBuffer(in);
    bout = NewBuffer(out);

    /* Encoding sequence */
    LzssInitContext(&h);
    r = LzssRawEncode(&h, bin, bout);

    /* Write compressed length */
    TORI(WriteLength(&head, bin, bout));
 
    /* Compute compression ratio */
    *cr = ((float)bin->total - (float)bout->total) / (float)bin->total;

    /* Free buffers */
    KillBuffer(bin);
    KillBuffer(bout);

    return r;
}


/* DECODING */
int LzssDecode(int in, int out)
{
    Buffer *bin, *bout;
    CrunchHeader head;
    LzssContext h;
    int r;

    /* Read header */
    TORI(ReadHeader(in, &head));
    if (CheckFileHeader(in, &head) != METHOD_LZSS)
        return -1;
    TORI(SeekRealPosition(in, &head, 0));
 
    /* Allocate buffers */
    bin = NewBuffer(in);
    bout = NewBuffer(out);

    /* Decoding sequence */
    LzssInitContext(&h);
    r = LzssRawDecode(&h, bin, bout, head.OriginalLength);

    /* Free buffers */
    KillBuffer(bin);
    KillBuffer(bout);

    return r;
}
