/*****************************************************************************/
/* File: ad_huff.c                                                           */
/* Author: David Chatenay                                                    */
/* Last Modified: Fri Oct 18 1996                                            */
/*                                                                           */
/* Implementation of the Huffman core. Long and nasty, heavy local memory use*/
/*****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <string.h>
#include "common.h"
#include "crunch.h"
#include "buffer.h"
#include "header.h"


/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
/*                               MISCELLANOUS                                */
/*****************************************************************************/
/* STRUCTURES DEFINITION */
/*** The cell (complex element of tree) */
typedef struct _adhuffcell {
    dword n;    /* Frequency */
    bool  leaf; /* Is a leaf? */
    union {
	struct {
	    Code  code;
	    byte  character;
	} leaf;  /* The leaf parameters */
	struct {
	    struct _adhuffcell *left;
	    struct _adhuffcell *right;
	} node;  /* The node parameters */
    } lon;
} AdHuffCell;
/*** Huffman header (container of used vars) */
typedef struct {
    AdHuffCell *root;       /* The root of the Huffman tree */
    AdHuffCell chars[256];  /* The 256 characters */
    AdHuffCell nodes[256];  /* The nodes (to avoid malloc) */
    AdHuffCell *index[257]; /* The sorted index   */
    int used_nodes;         /* Number of used nodes */
} AdHuffContext;




/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
/*                           LOW-LEVEL FUNCTIONS                             */
/*****************************************************************************/
static void InitHeader(AdHuffContext *h)
{
    int i;

    DEBUG0("InitHeader:\n");
    for (i=0; i<256; i++) {
	h->chars[i].n = 1;
	h->chars[i].leaf = TRUE;
	h->chars[i].lon.leaf.character = (byte)i;
	h->chars[i].lon.leaf.code.code_len = 0;
	(void)memset(h->chars[i].lon.leaf.code.code, 0, 16);
        h->index[i] = &(h->chars[i]);
    }

    h->root = NULL;
    h->index[256] = NULL;
    h->used_nodes = 0;
    DEBUG0("Done.\n\n");
}


/*****************************************************************************/
    /* Subroutine for buildtree */
    static AdHuffCell *NewCell(AdHuffContext *h, dword n, AdHuffCell *l,
			       AdHuffCell *r)
    {
	if (h->used_nodes < 256) {
	    AdHuffCell *p;
	    
	    p = &(h->nodes[h->used_nodes]);
	    p->n = n;
	    p->leaf = FALSE;
	    p->lon.node.left = l;
	    p->lon.node.right = r;
	    h->used_nodes++;
	    return p;
	} else {
	    ERROR("Bug in module ad_huff.c: No more free nodes!");
	    exit(-1);
	}
    }
/*** Build the Huffman tree with computed stats */
static void BuildTree(AdHuffContext *h)
{
    AdHuffCell  **index, *new;
    int i, j;
    dword n;

    DEBUG0("  [BuildTree]\n");

    /* Sort and compute used chars */
    DEBUG0("    [Sort]\n");
    for (i=0; i<256; i++)
        h->index[i] = &(h->chars[i]);
    for (i=1; i<256; i++) {
        j = i;
        while (j != 0 && h->index[j]->n > h->index[j-1]->n) {
            /* Inversion of the cells */
            new = h->index[j-1];
            h->index[j-1] = h->index[j];
            h->index[j] = new;
            j--;
        }
    }

    /* Build the tree */
    DEBUG0("    [Build]\n");
    index = h->index;
    h->used_nodes = 0;
    for (i=255; i>0; i--) {
	/* Concatenate the last two cells */
	n = index[i]->n + index[i-1]->n;
	new = index[i+1] = NewCell(h, n, index[i], index[i-1]);
	/* Re-sort the array (insertion) */
	j = i + 1;
	while (j != 0 && index[j]->n > index[j-1]->n) {
	    /* Swap */
	    new = index[j-1];
	    index[j-1] = index[j];
	    index[j] = new;
	    j--;
	}
    }
    h->root = index[0];
}



/*****************************************************************************/
    /* Subroutines for ComputeCodes */
    static void RecursiveCode(AdHuffCell *p, Code c)
    {
	if (p->leaf) {
	    (void)memcpy(p->lon.leaf.code.code, c.code, 16);
	    p->lon.leaf.code.code_len = c.code_len;
	} else {
	    int b, n, l;

	    l = c.code_len;
	    c.code_len++;
	    b = l / 8;
	    n = l % 8;
	    if (p->lon.node.left) {
		c.code[b] &= XMask[n];
		RecursiveCode(p->lon.node.left, c);
	    }
	    if (p->lon.node.right) {
		c.code[b] |= RMask[n];
		RecursiveCode(p->lon.node.right, c);
	    }
	}
    }
/*** Compute huffman codes */
static void ComputeCodes(AdHuffContext *h)
{
    Code c;

    DEBUG0("  [ComputeCodes]\n");
    c.code_len = 0;
    RecursiveCode(h->root, c);
}


/*****************************************************************************/
    /* Subroutines for Encode and Decode */
static int Encode(AdHuffContext *h, Buffer *in, Buffer *out)
{
    int i;

    ReadBuffer(in);

    while (in->buff_length > 0) {
	DEBUG1("  [Read %d]\n", in->buff_length);
	/* Output with current model */
	for (i=0; i<in->buff_length; i++) {
	    TORI(AddCode(&(h->chars[in->buffer[i]].lon.leaf.code), out));
	    h->chars[in->buffer[i]].n++;
	}

	/* Modify model */
	BuildTree(h);
	ComputeCodes(h);
	ReadBuffer(in);
    }
    FlushBuffer(out);

    return 0;
}


static int Decode(AdHuffContext *h, Buffer *in, Buffer *out, dword file_length)
{
    dword i, l, m;
    byte *buffer;
    AdHuffCell *p;

    buffer = in->buffer;
    ReadBuffer(in);
    l = 0; 
    while (l != file_length) {
	/* Output with current model */
	if (file_length - l < BUFF_LEN)
	    m = file_length - l;
	else
	    m = BUFF_LEN;
	for (i=0; i<m; i++) {
	    p = h->root;
	    while (!p->leaf) {
		if (buffer[in->byte_count] & RMask[in->bit_count])
		    p = p->lon.node.right;
		else
		    p = p->lon.node.left;
		in->bit_count++;
		if (in->bit_count == 8) {
		    IncrCountI(in);
		    in->bit_count = 0;
		}
	    }
	    Put(out, p->lon.leaf.character);
	    h->chars[p->lon.leaf.character].n++;
	    l++;
	}
	
	if (l == file_length) {
	    /* Modify model */
	    BuildTree(h);
	    ComputeCodes(h);
	}

	/* Modify model */
	BuildTree(h);
	ComputeCodes(h);
    }
    FlushBuffer(out);

    return 0;
}


/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
/*                          HIGH LEVEL FUNCTIONS                             */
/*****************************************************************************/
/* HUFFMAN ENCODING */
int AdaptativeHuffmanEncode(int in, int out, float *cr)
{
    Buffer *bin, *bout;
    CrunchHeader head;
    AdHuffContext hh;
    int r;

    /* Header write */
    TORI(ComputeHeader(in, &head, METHOD_AD_HUFF));
    TORI(WriteHeader(out, &head));

    /* Allocate buffers */
    bin = NewBuffer(in);
    bout = NewBuffer(out);

    /* Encoding sequence */
    InitHeader(&hh);

    /* Build first tree and compute first codes */
    BuildTree(&hh);
    ComputeCodes(&hh);
    /* Encode */
    r = Encode(&hh, bin, bout);

    /* Write compressed length */
    TORI(WriteLength(&head, bin, bout));

    /* Compute compression ratio */
    *cr = ((float)bin->total - (float)bout->total) / (float)bin->total;
 
    /* Free buffers */
    KillBuffer(bin);
    KillBuffer(bout);

    return r;
}


/* HUFFMAN DECODING */
int AdaptativeHuffmanDecode(int in, int out)
{
    Buffer *bin, *bout;
    CrunchHeader head;
    AdHuffContext hh;
    int r;

    /* Read header */
    TORI(ReadHeader(in, &head));
    if (CheckFileHeader(in, &head) != METHOD_AD_HUFF)
        return -1;
    TORI(SeekRealPosition(in, &head, 0));
 
    /* Allocate buffers */
    bin = NewBuffer(in);
    bout = NewBuffer(out);

    /* Decoding sequence */
    InitHeader(&hh);
    BuildTree(&hh);
    ComputeCodes(&hh);
    r = Decode(&hh, bin, bout, head.OriginalLength);

    /* Free buffers */
    KillBuffer(bin);
    KillBuffer(bout);

    return r;
}
