/*
    OpenGUI - Drawing & Windowing library

    Copyright (C) 1996,2000  Marian Krivos

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    nezmar@internet.alcatel.sk
    linux.cc - Linux support routines
*/

#define __USE_GNU

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <errno.h>
#include <termios.h>
#include <signal.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/vt.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/poll.h>
#include <sys/mman.h>
#include <sys/stat.h>

#include "config.h"
#include "fastgl.h"
#include "_fastgl.h"
#include "dll.h"

int svgalib=-1, nofb; // 0=ofbis; 1=svgalib
char fb_modename[32];

/* We invoke the old interrupt handler after setting text mode */
/* We catch all signals that cause an exit by default (aka almost all) */

static char sig2catch[] =
{SIGHUP, SIGINT, SIGQUIT, SIGILL,
 SIGTRAP, SIGIOT, SIGBUS, SIGFPE,
 SIGSEGV, SIGPIPE, SIGALRM, SIGTERM,
 SIGXCPU, SIGXFSZ, SIGVTALRM,
 SIGPWR};
 
static struct sigaction old_signal_handler[sizeof(sig2catch)];

extern "C" char *strsignal __P ((int __sig));
static void RestoreGPM(void);
static void RunGPM(int which);

#define zero_sa_mask(maskptr) memset(maskptr, 0, sizeof(sigset_t))

#ifdef FRAMEBUFFER

#include <linux/fb.h>
#include "linux.h"

/* If TURN_ON is TRUE, request for permission to do direct i/o on the
   port numbers in the range [FROM,FROM+NUM-1].  Otherwise, turn I/O
   permission off for that range.  This call requires root privileges.

   Portability note: not all Linux platforms support this call.  Most
   platforms based on the PC I/O architecture probably will, however.
   E.g., Linux/Alpha for Alpha PCs supports this.  */
extern "C" int ioperm(unsigned long int __from, unsigned long int __num,
			int __turn_on);

/* Set the I/O privilege level to LEVEL.  If LEVEL>3, permission to
   access any I/O port is granted.  This call requires root
   privileges. */
extern "C" int iopl(int __level);

static int msefd=-1, gpmmouse;
static int fbfd = 0;
static struct fb_var_screeninfo vinfo;
static struct fb_fix_screeninfo finfo;
static int screensize;
static FB *f;

#define GPMDATADEVICE "/dev/gpmdata"
#define DEV_TYPE_NONE 0
#define DEV_TYPE_GPMDATA 1
#define DEV_TYPE_MOUSE 2

static int devtype = 0;
static int oldx, oldy, but = 0;
static int mousex, mousey, xmax, ymax;

typedef int FBListKeyType;
typedef FB *FBListValType;

static char *ttynames[] =
{"/dev/tty0", "/dev/console", NULL};
static char *ttyfmts[] =
{"/dev/tty%d", "/dev/tty%02x", "/dev/tty%x", "/dev/tty%02d", NULL};
static int cttyname = 0;
static int originaltty;
static struct vt_mode vtm;
unsigned short switching; /* Flag to prevent reentrancy */

static void FBVTswitch(int s);

static FBList ttylist;
static unsigned short listinit = FALSE;

void FBerror(unsigned short flags, const char *fmt,...)
{
	unsigned short fatal = flags & FATAL;
	unsigned short syserr = flags & SYSERR;
	static int no_in=0;
	va_list args;

	va_start(args, fmt);

	printf("ofbis %s: ", (fatal ? "fatal error" : "warning"));
	vprintf(fmt, args);
	va_end(args);
	if (syserr)
	{
		printf(" : %s", strerror(errno));
	}
	putchar('\n');
	if (fatal && no_in==0)
	{
		no_in = 1;
		FBshutdown();
		no_in = 0;
	}
}

static void *FBalloc( unsigned long size )
{
  void    *s;

  if ( ( s = malloc( size ) ) == NULL )
  {
    FBerror( FATAL | SYSERR, "FBalloc: couldn't get %ld bytes", size );
  }
  return (void *)s;
}

static void FBfree( void *p )
{
  (void) free(p);
}

static inline void FBListDeleteNode(FBListNode n)
{
	FBfree(n);
}

static inline void FBListDelete(FBList lst)
{
	FBListNode n = (FBListNode) lst;
	FBListNode i;

	while (n != NULL)
	{
		i = n->next;
		FBListDeleteNode(n);
		n = i;
	}
}

static inline FBList FBListInit(FBList lst)
{
	if (lst != NULL)
		FBListDelete(lst);
	return NULL;
}

static inline int FBListEmpty(FBList lst)
{
	return (lst == NULL);
}

static inline FBListNode FBListNewNode(FBListKeyType k, FBListValType v)
{
	FBListNode n = (FBListNode) FBalloc(sizeof(struct fb_lnde));

	n->key = k;
	n->val = v;
	n->next = NULL;
	return n;
}

static inline FBList FBListInsertNode(FBList lst, FBListNode n)
{
	if (lst != NULL)
	{
		n->next = (FBListNode) lst;
	}
	return (FBList) n;
}

static inline FBList FBListAdd(FBList lst, FBListKeyType k, FBListValType v)
{
	return FBListInsertNode(lst, FBListNewNode(k, v));
}

static inline FBListNode FBListGetAnyNode(FBList lst)
{
	return (FBListNode) lst;
}

static inline FBListValType FBListFindKey(FBList lst, FBListKeyType k)
{
	FBListNode n = (FBListNode) lst;

	while ((n != NULL) && (n->key != k))
	{
		n = n->next;
	}
	if (n != NULL)
	{
		return n->val;
	}
	else
	{
		return (FBListValType) 0;
	}
}

static inline FBList FBListRemoveKey(FBList lst, FBListKeyType k)
{
	FBListNode n = (FBListNode) lst;
	FBListNode p = n;

	if ((n->next == NULL) && (n->key == k))
	{
		FBListDeleteNode(p);
		return NULL;
	}
	else if (n->key == k)
	{
		n = n->next;
		FBListDeleteNode(p);
		return (FBList) n;
	}
	else
	{
		while ((n->next != NULL) && (n->next->key != k))
		{
			n = n->next;
		}
		p = n->next;
		n->next = n->next->next;
		FBListDeleteNode(p);
		return lst;
	}
}

static inline FBList FBListRemoveNode(FBList lst, FBListKeyType k)
{
	FBListNode n = (FBListNode) lst;
	FBListNode p = n;

	if ((n->next == NULL) && (n->key == k))
	{
		return NULL;
	}
	else if (n->key == k)
	{
		return (FBList) n->next;
	}
	else
	{
		while ((n->next != NULL) && (n->next->key != k))
		{
			n = n->next;
		}
		p = n->next;
		n->next = n->next->next;
		return lst;
	}
}

static void FBmap(FB * f)
{
	off_t offset = 0;

	/* Map fb into memory */

	if ((f->sbuf = (unsigned short *) mmap((void *) 0, f->finf.smem_len,
		 PROT_READ | PROT_WRITE, MAP_SHARED, f->fb, offset)) == (void *) -1)
	{
		FBerror(FATAL | SYSERR, "FBopen: fb mmap failed");
	}
}

static void FBunmap(FB * f)
{
	/* Unmap framebuffer from memory */

	if ((f->visible) && (munmap((void *) f->sbuf, f->finf.smem_len) == -1))
	{
		FBerror(WARNING | SYSERR, "FBclose: munmap failed");
	}
}

/*
   ** Description
   ** Open the mouse device.
   **
   ** 1998-08-06 CG
   ** 1999-05-20 Tomas
 */
void FBmouseopen(int xm, int ym)
{
	if (msefd != -1)
	{
		return;
	}

	devtype = DEV_TYPE_GPMDATA;
	if ((msefd = open(GPMDATADEVICE, O_RDWR | O_NDELAY)) == -1)
	{
		return;
	}
	tcflush(msefd, TCIFLUSH);
	mousex = (xmax = xm) / 2;
	mousey = (ymax = ym) / 2;
}

void FBmouseclose(void)
{
	if (msefd == -1)
	{
		return;
	}

	close(msefd);
	msefd = -1;
	remove(GPMDATADEVICE);
}

int FBcheckmouse(int& type, int& key, int& x, int& y, int& buttons)
{
	signed char buf[8];
	int dx, dy;

#ifdef FRAMEBUFFER
	if (svgalib==0 && f->sbuf == f->sbak) return 0;
#endif
	if (msefd == -1) return 0;
	int size_to_read = (devtype == DEV_TYPE_GPMDATA ? 5 : 3);

	if (read(msefd, &buf, size_to_read) == -1)
	{
		return 0;
	}
	if (buf[0]==0) return 0; // ????
//fprintf(stderr,"asd %d %d %d %d %d %d\n", buf[0], buf[1], buf[2], buf[3], buf[4], buf[5]);
	but = ~buf[0] & 0x07;
	dx = (int) ((buf[3] * 2));
	dy = (int) (-((buf[4] * 2)));
	oldy = mousey;
	oldx = mousex;

	if (((mousex + dx) >= 0) && ((mousex + dx) < xmax))
		mousex += dx;
	if (((mousey + dy) >= 0) && ((mousey + dy) < ymax))
		mousey += dy;
	x = mousex;
	y = mousey;

	if (but == 4)
	{
		buttons = 1;
	}
	else if (but == 1)
	{
		buttons = 2;
//		buttons = 4;
	}
	else buttons = 0;
//printf("but %d ,%d buttons\n", but, buttons);
	type = MOUSEEVENT;
	key = 0;
	return 1;
}

// ------------------------------------------------------------------------

static int fb_setmode(char *name, int fb)
{
    FILE *fp;
    char line[80],label[32],value[16];
    int  geometry=0, timings=0;
	
    /* load current values */
    if (-1 == ioctl(fb,FBIOGET_VSCREENINFO,&vinfo)) {
		perror("ioctl FBIOGET_VSCREENINFO");
		exit(1);
    }
    
    if (NULL == name)
		return -1;
    fp = fopen("/etc/fb.modes","r");
    if (!fp) if (NULL == (fp = fopen("/etc/mode.conf","r")))
		return -1;
    while (NULL != fgets(line,79,fp))
	{
	if (1 == sscanf(line, "mode \"%31[^\"]\"",label) &&
	    0 == strcmp(label,name)) {
	    /* fill in new values */
	    vinfo.sync  = 0;
	    vinfo.vmode = 0;
	    while (NULL != fgets(line,79,fp) &&
		   NULL == strstr(line,"endmode")) {
		if (5 == sscanf(line," geometry %d %d %d %d %d",
				&vinfo.xres,&vinfo.yres,
				&vinfo.xres_virtual,&vinfo.yres_virtual,
				&vinfo.bits_per_pixel))
		    geometry = 1;
		if (7 == sscanf(line," timings %d %d %d %d %d %d %d",
				&vinfo.pixclock,
				&vinfo.left_margin,  &vinfo.right_margin,
				&vinfo.upper_margin, &vinfo.lower_margin,
				&vinfo.hsync_len,    &vinfo.vsync_len))
		    timings = 1;
		if (1 == sscanf(line, " hsync %15s",value) &&
		    0 == strcasecmp(value,"high"))
		    vinfo.sync |= FB_SYNC_HOR_HIGH_ACT;
		if (1 == sscanf(line, " vsync %15s",value) &&
		    0 == strcasecmp(value,"high"))
		    vinfo.sync |= FB_SYNC_VERT_HIGH_ACT;
		if (1 == sscanf(line, " csync %15s",value) &&
		    0 == strcasecmp(value,"high"))
		    vinfo.sync |= FB_SYNC_COMP_HIGH_ACT;
		if (1 == sscanf(line, " extsync %15s",value) &&
		    0 == strcasecmp(value,"true"))
		    vinfo.sync |= FB_SYNC_EXT;
		if (1 == sscanf(line, " laced %15s",value) &&
		    0 == strcasecmp(value,"true"))
		    vinfo.vmode |= FB_VMODE_INTERLACED;
		if (1 == sscanf(line, " double %15s",value) &&
		    0 == strcasecmp(value,"true"))
		    vinfo.vmode |= FB_VMODE_DOUBLE;
	    }
	    /* ok ? */
	    if (!geometry || !timings)
		return -1;
	    /* set */
	    vinfo.xoffset = 0;
	    vinfo.yoffset = 0;
	    if (-1 == ioctl(fb,FBIOPUT_VSCREENINFO,&vinfo))
		{
			return -1;
		}
	    /* look what we have now ... */
	    if (-1 == ioctl(fb,FBIOGET_VSCREENINFO,&vinfo)) {
			perror("ioctl FBIOGET_VSCREENINFO");
			exit(1);
	    }
	    return 0;
	}
    }
    return -1;
}
FB *FBopen(const char *fbname, unsigned short opts, int xm, int ym)
{
	/* Allocate new FB handle */

	FB *f = (FB *) FBalloc(sizeof(FB));

	/* Open framebuffer fbname */

	if (!fbname)
	{
		/* Try to set name from the environment */
		fbname = (const char *) getenv("FRAMEBUFFER");
		if (!fbname)
		{
			fbname = "/dev/fb0";
		}
	}

	if ((f->fb = open(fbname, O_RDONLY /*WR */  | O_NONBLOCK)) == -1)
	{
		FBerror(FATAL | SYSERR, "FBopen: open failed on %s", fbname);
	}

	/* Get screen info */

	FBgetfix(f);
	FBgetvar(f);

	/* Open VT */

	(void) close(f->fb);

	FBVTopen(f);

	if ((f->fb = open(fbname, O_RDWR)) == -1)
	{
		FBerror(FATAL | SYSERR, "FBopen: open failed on %s", fbname);
	}

	FBgetfix(f);

	/* Disable ywrap */
	f->vinf.xoffset = 0;
	f->vinf.yoffset = 0;
	f->vinf.vmode &= ~FB_VMODE_YWRAP;

	/* Update screen info */

	FBputvar(f);

	/* Map fb into memory */

	FBmap(f);
	f->sbak = NULL;
	if (*fb_modename)
	{
		if (!fb_setmode(fb_modename,f->fb))
		{
			if (verbose) printf("modeline -> '%s'\n", fb_modename);
		}
		else
		{
			if (verbose) printf("modeline '%s' setting error\n", fb_modename);
		}
	}
	return (f);
}

int  FBclose(FB * f)
{

	/* Unmap framebuffer from memory */
	FBunmap(f);

	/* Close VT */
	FBVTclose(f);

	/* Close framebuffer device */
	if (close(f->fb) == -1)
	{
		FBerror(WARNING | SYSERR, "FBclose: close fb failed");
	}
	FBfree(f);
	f = 0;
	return (1);
}
// ------------------------------------------------------------------------
/*
   ** void FBgetfix( FB *f )
   **
   ** Get fixed screen info into f->finf
 */
static void  FBgetfix(FB * f)
{
	if (ioctl(f->fb, FBIOGET_FSCREENINFO, &f->finf) == -1)
	{
		FBerror(FATAL | SYSERR, "FBgetfix: Get fixed screen settings failed %d", f->fb);
	}
}

/*
   ** void FBgetvar( FB *f )
   **
   ** Get variable screen info into f->var
 */
static void FBgetvar(FB * f)
{
	if (ioctl(f->fb, FBIOGET_VSCREENINFO, &f->vinf) == -1)
	{
		FBerror(FATAL | SYSERR, "FBgetvar: Get variable screen settings failed");
	}
}

/*
   ** void FBputvar( FB *f )
   **
   ** Put variable screen info into framebuffer and update f->var
 */
static void FBputvar(FB * f)
{
	if (ioctl(f->fb, FBIOPUT_VSCREENINFO, &f->vinf) == -1)
	{
		/* Don't quit, because the sparc crashes then. */
		FBerror(SYSERR, "FBputvar: Put variable screen settings failed");
	}

	if (ioctl(f->fb, FBIOGET_FSCREENINFO, &f->finf) == -1)
	{
		FBerror(FATAL | SYSERR, "FBputvar: Get fixed screen settings failed");
	}

	if (ioctl(f->fb, FBIOGET_VSCREENINFO, &f->vinf) == -1)
	{
		FBerror(FATAL | SYSERR, "FBputvar: Update variable screen settings failed");
	}
}

// ------------------------------------------------------------------------

static void FBVTopen(FB * f)
{
	char ttynam[11];
	int i = 0;
	struct vt_stat vts;

	/* Open current console */

	while ((f->tty = open(ttynames[cttyname], O_WRONLY)) == -1)
	{
		FBerror(WARNING | SYSERR, "FBVTopen: open failed on %s",
				ttynames[cttyname]);

		if (ttynames[++cttyname] == NULL)
		{
			FBerror(FATAL, "FBVTopen: failed to open a tty");
		}
		else
		{
			FBerror(WARNING, "FBVTopen: trying %s", ttynames[cttyname]);
		}
	}

	/* Get number of free VT */

	if (ioctl(f->tty, VT_OPENQRY, &f->ttyno) == -1)
	{
		FBerror(FATAL | SYSERR, "FBVTopen: no free ttys");
	}

	/* Close current console */

	if (close(f->tty) == -1)
	{
		FBerror(WARNING | SYSERR, "FBVTopen: failed to close %s",
				ttynames[cttyname]);
	}

	close(0);
	close(1);
	close(2);
	do	{
		(void) sprintf(ttynam, ttyfmts[i++], f->ttyno);
	}
	while ((ttyfmts[i] != NULL) &&
		   (f->tty = open(ttynam, /*O_RDONLY*/O_RDWR)) == -1);

	if (f->tty == -1)
	{
		FBerror(FATAL | SYSERR, "FBVTopen: failed to open %s", ttynam);
	}

	dup(f->tty);
	dup(f->tty);
	setsid();

	if ((i = open("/dev/tty", O_RDWR)) >= 0)
	{
		ioctl(i, TIOCNOTTY, 0);
		close(i);
	}
	/* Get current VT number so we can switch back to it later */

	if (ioctl(f->tty, VT_GETSTATE, &vts) == -1)
	{
		FBerror(FATAL | SYSERR, "FBVTopen: couldn't get VT state");
	}
	originaltty = vts.v_active;

	/* Switch to new VT */

	if (ioctl(f->tty, VT_ACTIVATE, f->ttyno) == -1)
	{
		FBerror(FATAL | SYSERR, "FBVTopen: couldn't switch to VT %d", f->ttyno);
	}

	/* Wait for new VT to become active */

	if (ioctl(f->tty, VT_WAITACTIVE, f->ttyno) == -1)
	{
		FBerror(FATAL | SYSERR, "FBVTopen: VT %d didn't become active", f->ttyno);
	}
	f->visible = TRUE;

	/* Get mode of new VT */

	if (ioctl(f->tty, VT_GETMODE, &vtm) == -1)
	{
		FBerror(FATAL | SYSERR, "FBVTopen: Couldn't get mode of VT %d", f->ttyno);
	}

	/* Adjust mode parameters */

	vtm.mode = VT_PROCESS;
	vtm.relsig = SIGUSR2;
	vtm.acqsig = SIGUSR2;

	/* Set signal handler for VT switches */

	(void) signal(SIGUSR2, FBVTswitch);

	/* Set new mode parameters */

	if (ioctl(f->tty, VT_SETMODE, &vtm) == -1)
	{
		FBerror(FATAL | SYSERR, "FBVTopen: Couldn't set mode of VT %d", f->ttyno);
	}

	/* Disable cursor */
	if (ioctl(f->tty, KDSETMODE, KD_GRAPHICS) == -1)
	{
		FBerror(FATAL | SYSERR, "FBVTopen: Couldn't set keyboard graphics mode on VT %d", f->ttyno);
	}

	/* Initialise tty list if not already done */

	if (listinit == FALSE)
	{
		ttylist = FBListInit(ttylist);
	}

	/* Add new FB to tty list */
	ttylist = FBListAdd(ttylist, f->ttyno, f);
}

static FB *FBfindFB(void)
{
	struct vt_stat vts;
	int curcon;
	FB *f;

	/* Open current console */

	if ((curcon = open(ttynames[cttyname], O_RDONLY)) == -1)
	{
		FBerror(FATAL | SYSERR, "FBVTswitch: open failed on %s",
				ttynames[cttyname]);
	}

	/* Get current VT number so we can find which FB struct to use */

	if (ioctl(curcon, VT_GETSTATE, &vts) == -1)
	{
		FBerror(FATAL | SYSERR, "FBVTswitch: couldn't get VT state");
	}

	f = FBListFindKey(ttylist, vts.v_active);

	/* Close current console */

	if (close(curcon) == -1)
	{
		FBerror(WARNING | SYSERR, "FBVTswitch: failed to close %s",
				ttynames[cttyname]);
	}
	return f;
}

/* Shutdown ofbis in case of a fatal error */

void FBshutdown(void)
{
	FB *f;

	while ((f = FBfindFB()) != NULL)
	{
		FBclose(f);
	}

	exit(1);
}

static void FBVTswitch(int s)
{
	static FB *f;				/* Non reentrant */

	/* Set this handler again, otherwise signal reverts to default handling */

	(void) signal(SIGUSR2, FBVTswitch);

	/* If the switch has already been acknowledged, and the user tries to */
	/* switch again, ignore until the alarm signal is raised. */

	if ((switching) && (s == SIGUSR2))
	{
		return;
	}

	if (switching == FALSE)
	{
		switching = TRUE;
		f = FBfindFB();
	}

	/* If drawing, a change in the screen buffer will cause problems: */
	/* delay switch and hope next time it isn't drawing. */
	if (f->visible)
	{
		/* Switching out, allocate new backing store */
		if (!f->sbak)
			f->sbak = (unsigned short *) FBalloc(f->finf.smem_len);
		if (f->sbak == NULL)
		{
			FBerror(FATAL | SYSERR, "FBVTswitch: failed to allocate backing store");
		}
		/* Copy framebuffer to backing store */

		(void) memcpy(f->sbak, f->sbuf, f->finf.smem_len);

		/* Unmap current framebuffer */

		FBunmap(f);

		/* Make backing store current so it can be written to */

		f->sbuf = f->sbak;
		f->visible = FALSE;

		/* Release console for switch */

		if (ioctl(f->tty, VT_RELDISP, 1) == -1)
		{
			FBerror(FATAL | SYSERR, "FBVTswitch: switch away from VT %d denied", f->tty);
		}
	}
	else
	{
#ifdef INDEX_COLORS
		_set_fgl_palette();
#endif
		if (ioctl(f->tty, VT_RELDISP, VT_ACKACQ) == -1)
		{
			FBerror(FATAL | SYSERR, "FBVTswitch: switch to VT %d denied", f->tty);
		}
		FBmap(f);
		memcpy(f->sbuf, f->sbak, f->finf.smem_len);
		f->visible = TRUE;
	}
	videobase = (FGPixel *)f->sbuf;
	switching = FALSE;
}
/*
static void  FBVTswitchoriginal(FB * f)
{
	if (ioctl(f->tty, VT_ACTIVATE, originaltty) == -1)
	{
		FBerror(WARNING | SYSERR, "FBVTswitchoriginal: VT switch failed");
	}
}

static void FBVTswitchfb(FB * f)
{
	if (ioctl(f->tty, VT_ACTIVATE, f->ttyno) == -1)
	{
		FBerror(WARNING | SYSERR, "FBVTswitchfb: VT switch failed");
	}
}
*/
static void FBVTclose(FB * f)
{
	/* If backing store allocated, free it */

	if (f->sbak)
	{
		FBfree(f->sbak);
	}

	/* Enable cursor */

	if (ioctl(f->tty, KDSETMODE, KD_TEXT) == -1)
	{
		FBerror(FATAL | SYSERR, "FBVTclose: Couldn't set keyboard graphics mode on VT %d", f->ttyno);
	}

	/* Restore mode parameters */

	vtm.mode = VT_AUTO;

	/* Set new mode parameters */

	if (ioctl(f->tty, VT_SETMODE, &vtm) == -1)
	{
		FBerror(FATAL | SYSERR, "FBVTclose: Couldn't set mode of VT %d", f->ttyno);
	}

	/* Switch back to original VT */

	if (ioctl(f->tty, VT_ACTIVATE, originaltty) == -1)
	{
		FBerror(FATAL | SYSERR, "FBVTclose: couldn't switch to VT %d", originaltty);
	}

	/* Close VT */

	close(f->tty+1);
	close(f->tty+2);
	if (close(f->tty) == -1)
	{
		FBerror(FATAL | SYSERR, "FBVTclose: failed to close VT");
	}
	f->tty = 0;
	/* Remove from ttylist */

	ttylist = FBListRemoveKey(ttylist, f->ttyno);
}

//
// looking for FB and current mode
//
int LookForFB(int mode)
{
	int m=0;
        /* Open the file for reading and writing */
        fbfd = open("/dev/fb0", O_RDWR);
        if (fbfd<=0) {
			if (verbose) {
                printf("Framebuffer device not available\n");
                printf("trying link with svgalib library ..\n\n");
			}
            return 0;
        }

        /* Get fixed screen information */
        if (ioctl(fbfd, FBIOGET_FSCREENINFO, &finfo)) {
            printf("Error reading fixed information\n");
			return 0;
        }

        /* Get variable screen information */
        if (ioctl(fbfd, FBIOGET_VSCREENINFO, &vinfo)) {
            printf("Error: reading variable information\n");
			return 0;
        }

		if (vinfo.bits_per_pixel != bpp*8U)
		{
			vinfo.bits_per_pixel = bpp*8;
	        if (ioctl(fbfd, FBIOPUT_VSCREENINFO, &vinfo)) {
                printf("Error: setting right BPP\n");
	            return 0; // not supported BPP
    	    }
		}
		GetModeSize(mode, &m);
		if (m==0)
		{
			printf("Error: not supported resolution\n");
			return 0;
		}
		vinfo.xres = X_width;
		vinfo.yres = Y_width;
        if (ioctl(fbfd, FBIOPUT_VSCREENINFO, &vinfo))
		{
			ioctl(fbfd, FBIOGET_VSCREENINFO, &vinfo);
			if (verbose) printf("Warning: resolution is %dx%dx%d\n",vinfo.xres,vinfo.yres,bpp*8);
   	    }

        /* and Re-Get variable screen information */
        if (ioctl(fbfd, FBIOGET_VSCREENINFO, &vinfo)) {
                printf("Error reading variable information\n");
        }

		switch(vinfo.xres)
		{
			case 320:
				m = 1;
				break;
			case 640:
				m = 2;
				break;
			case 800:
				m = 3;
				break;
			case 1024:
				m = 4;
				break;
			case 1280:
				m = 5;
				break;
			case 1600:
				m = 6;
				break;
			default:
                printf("Error not supported resolution\n");
				return 0;
		}
	    screensize = vinfo.xres * vinfo.yres * vinfo.bits_per_pixel / 8;
	    if (getenv("IOPERM") == NULL)
			if (ioperm(0x3b4, 0x3df - 0x3b4 + 1, 1))
			{
		    	printf("Cannot get permissions for I/O operations\n");
			    exit(1);
			}
		close(fbfd);
		if ((f=FBopen(NULL, 0, vinfo.xres, vinfo.yres))==NULL) {
			    printf("Error FBopen failed\n");
				return 0;
		}
		videobase = (FGPixel *)f->sbuf;
		return m;
}

FGPixel *vga_getgraphmem(void)
{
	return videobase = (FGPixel *) f->sbuf;
}

unsigned vga_getmemorysize(void)
{
	return finfo.smem_len/1024;
}

#endif

// --------------------------------------------------------------

static void signal_handler(int v)
{
	unsigned i;

	cleanup();
	printf("FastgGL: Signal %d (%s) received %s.\n", v, strsignal(v),
		   (v == SIGINT) ? " (ctrl-c pressed)" : "");

	for (i = 0; i < sizeof(sig2catch); i++)
		if (sig2catch[i] == v)
		{
			sigaction(v, old_signal_handler + i, NULL);
			raise(v);
			break;
		}
	if (i >= sizeof(sig2catch))
	{
		printf("svgalib: Aieeee! Illegal call to signal_handler, raising segfault.\n");
		raise(SIGSEGV);
	}
}

static void set_signals(void)
{
	unsigned i;
	struct sigaction siga;

	/* do our own interrupt handling */
	for (i = 0; i < sizeof(sig2catch); i++)
	{
		siga.sa_handler = signal_handler;
		siga.sa_flags = 0;
		zero_sa_mask(&(siga.sa_mask));
		sigaction((int) sig2catch[i], &siga, old_signal_handler + i);
	}
}

static void reset_signals(void)
{
	unsigned i;

	/* do our own interrupt handling */
	for (i = 0; i < sizeof(sig2catch); i++)
	{
		signal((int) sig2catch[i], SIG_DFL);
	}
}

int  (*ptr_vga_init)(void);
int  (*ptr_vga_setmode)(int);
int  (*ptr_vga_hasmode)(int);
void (*ptr_vga_setpage)(int);
vga_modeinfo * (*ptr_vga_getmodeinfo)(int);
unsigned char * (*ptr_vga_getgraphmem)(void);
int  (*ptr_vga_getmousetype)(void);
int  (*ptr_vga_setlinearaddressing)(void);
int  (*ptr_mouse_init)(char *dev, int type, int samplerate);
void (*ptr_mouse_close)(void);
int  (*ptr_mouse_update)(void);
void (*ptr_mouse_setposition)(int x, int y);
void (*ptr_mouse_setxrange)(int x1, int x2);
void (*ptr_mouse_setyrange)(int y1, int y2);
void (*ptr_mouse_setscale)(int s);
int  (*ptr_mouse_getx)(void);
int  (*ptr_mouse_gety)(void);
int  (*ptr_mouse_getbutton)(void);
int  *ptr_tty_fd;

static int linked=0;

static Dll *lib;

static int iskbd, vgamouse;

static int kbd_fd=-1;

typedef void (*vv)(void);
typedef int (*iv)(void);
typedef void (*vi)(int);
typedef void (*vii)(int,int);
typedef int (*ii)(int);

int LinkSvgalib(void)
{
	Widget ptr_disable;
	
	lib = new Dll("libvga.so.1", RTLD_LAZY | RTLD_GLOBAL);
	if (lib->GetResult()) return 0;
	ptr_vga_init = iv(lib->GetAddr("vga_init"));
	ptr_vga_setmode = ii(lib->GetAddr("vga_setmode"));
	ptr_vga_hasmode = ii(lib->GetAddr("vga_hasmode"));
	ptr_vga_setpage = vi(lib->GetAddr("vga_setpage"));
	ptr_vga_getmodeinfo = (vga_modeinfo *(*)(int)) lib->GetAddr("vga_getmodeinfo");
	ptr_vga_getgraphmem = (unsigned char * (*)(void))lib->GetAddr("vga_getgraphmem");
	ptr_vga_getmousetype = iv(lib->GetAddr("vga_getmousetype"));
	ptr_vga_setlinearaddressing = iv(lib->GetAddr("vga_setlinearaddressing"));

	ptr_mouse_init = (int (*)(char *,int,int))lib->GetAddr("mouse_init");
	ptr_mouse_close = vv(lib->GetAddr("mouse_close"));
	ptr_mouse_update = iv(lib->GetAddr("mouse_update"));
	ptr_mouse_setposition = vii(lib->GetAddr("mouse_setposition"));
	ptr_mouse_setxrange = vii(lib->GetAddr("mouse_setxrange"));
	ptr_mouse_setyrange = vii(lib->GetAddr("mouse_setyrange"));
	ptr_mouse_setscale = vi(lib->GetAddr("mouse_setscale"));
	ptr_mouse_getx = iv(lib->GetAddr("mouse_getx"));
	ptr_mouse_gety = iv(lib->GetAddr("mouse_gety"));
	ptr_mouse_getbutton = iv(lib->GetAddr("mouse_getbutton"));
	ptr_disable = (Widget)lib->GetAddr("vga_disabledriverreport");
	if (verbose==0) ptr_disable();
	ptr_tty_fd = (int *)lib->GetAddr("__svgalib_tty_fd");
	return 1;
}

void UnlinkSvgalib(void)
{
	if (linked) delete lib;
	linked = 0;
}

// set svgalib 
//
//
int LinuxStartup(int mode)
{
	int tmp=0;
#ifdef FRAMEBUFFER
	if (!nofb) tmp = LookForFB(mode);
#endif
	svgalib = 0;
	if (tmp==0) // no FB or FB mode available
	{
		if (!LinkSvgalib()) 
		{
			svgalib = -1;
			return 0; // no graphics layer present
		}
		else
		{
//			linux_mouse = ptr_mouse_init( "/dev/mouse", ptr_vga_getmousetype(), MOUSE_DEFAULTSAMPLERATE)==-1?0:1;
			ptr_vga_init();
			svgalib = 1;
		}
	}
	LinuxInit();
	return tmp?tmp:mode;
}

int LinuxInputOpen(void)
{
	// tty fd
	iskbd= 0;
	if (svgalib==1)
	{
		if ((kbd_fd = *ptr_tty_fd)==-1)
		{
			printf("Could not get tty\n");
			exit(-1);
		}
	}
#ifdef FRAMEBUFFER
	else kbd_fd = f->tty;
#endif
	
	if ((kbd_fd=GII_keyboard_init(kbd_fd)) < 0)
	{
		printf("Could not initialize keyboard\n");
		ioctl(kbd_fd, KDSKBMODE, K_XLATE);
		exit(-1);
	}
	iskbd = 1;
	
	if (svgalib==1)
	{
		RunGPM(0); // kill gpm
		vgamouse = ptr_mouse_init( "/dev/mouse", ptr_vga_getmousetype(), MOUSE_DEFAULTSAMPLERATE)==-1?0:1;
		if (vgamouse==0)
		{
			printf("FastGL: could not initialize svgalib mouse\n");
		}
		else
		{
			ptr_mouse_setxrange(0,X_max);
			ptr_mouse_setyrange(0,Y_max);
			ptr_mouse_setposition(X_width/2, Y_width/2);
			ptr_mouse_setscale(1);
		}
	}
#ifdef FRAMEBUFFER
	else if (svgalib==0)
	{
		RunGPM(1); // run gpm in SystemxMouse mode
		/* Open mouse */
		delay(800);
		FBmouseopen(X_width, Y_width);
		if (msefd!=-1)
		{
			gpmmouse = 1;
		}
		else printf("FastGL: could not initialize gpm mouse\n");
	}
	return (vgamouse || gpmmouse);
#endif
	return vgamouse;
}

void LinuxInputClose(void)
{
	if (iskbd) GII_close(); // close keyboard
	iskbd=0;
	ioctl(kbd_fd, KDSKBMODE, K_XLATE);
	if (vgamouse)
	{
		ptr_mouse_close();
		vgamouse = 0;
	}
#ifdef FRAMEBUFFER
	else if (gpmmouse)
	{
		FBmouseclose();
		gpmmouse = 0;
	}
#endif
	RestoreGPM();
}

void Snd(int a, int b)
{
	if (kbd_fd>=0) ioctl(kbd_fd, KDMKTONE, (b<<16)+(a&0xffff));
}

int LinuxGetEvent(int& type, int& key, int& x, int& y, int& buttons, int ismouse)
{
	int bb;
	if ((key = GII_get_key()) != 0)
	{
		type = KEYEVENT;
		x = y = buttons = 0;
		return 1;
	}
	if (ismouse==0) return 0; // no mouse driver
	if (vgamouse)
	{
		if (ptr_mouse_update())
		{
			x = ptr_mouse_getx();
			y = ptr_mouse_gety();
			bb = ptr_mouse_getbutton();
			buttons = 0;
			if (bb&MOUSE_RIGHTBUTTON) buttons |= 2;
			if (bb&MOUSE_LEFTBUTTON) buttons |=	1;
			key  = 0;
			type = MOUSEEVENT;
			return 1;
		}
	}
#ifdef FRAMEBUFFER
	else if (gpmmouse)
	{
		if (FBcheckmouse(type, key, x, y, buttons))
			return 1;
	}
#endif
	return 0;
}

int LinuxMemSize(int m)
{
#ifdef FRAMEBUFFER
	if (svgalib==0)
		return vga_getmemorysize();
#endif
	return ptr_vga_getmodeinfo(m)->maxpixels/1024*bpp;
}

void LinuxGraphClose(void)
{
#ifdef FRAMEBUFFER
	if (svgalib==0) FBclose(f);
	else
		if (svgalib==1)
#endif
			ptr_vga_setmode(TEXT);
	reset_signals();
}

int LinuxSetMode(int m, int mode, int& linear)
{
	if (svgalib==1)
	{
		if (!ptr_vga_hasmode(m))
		{
			printf("Mode not available.\n");
			return 0;
		}
		if (ptr_vga_setmode(m) == -1) 
		{
			printf("Check http://www.svgalib.org for newest SVGALIB driver!\a\n");
			return 0;
		}
		if (mode>1 && linear) if (ptr_vga_setlinearaddressing()	== -1)
		{
			printf("Could not set linear addressing.\n");
			linear = 0;
		}
		else linear = 1;
		videobase = (FGPixel *)ptr_vga_getgraphmem();
	}
#ifdef FRAMEBUFFER
	else
	{
		linear = 1;
		videobase = (FGPixel *)vga_getgraphmem();
	}
#endif
	set_signals();
	if (mode==1)  linear =	1; // in 320x200 allways linear
	if (linear) driver = 0; // no banking
	return 1;
}


static int n_arguments=1;
static pid_t old_pid, new_pid;
char *strs[3] = {"gpm","-k","-R MouseSystems"};
static char *arguments[8]={strs[0]};

static int GetGPM(int& i, char **argv)
{
	static char arg[80], *p;
	int pid;
	
	i = 0;
	FILE * f = fopen("/var/run/gpm.pid", "r");
	if (f)
	{
		fscanf(f, "%d", &pid);
		fclose(f);
		sprintf(arg, "/proc/%d/cmdline", pid);
		f = fopen(arg, "r");
		if (f)
		{
			i = 0;
			memset(arg, 0, sizeof(arg));
			fread(arg, 1, sizeof(arg), f);
			p = arg;
			while(strlen(p)!=0 && i<8)
			{
				argv[i] = p;
				p = p + strlen(p)+1;
				i++;
			}
			fclose(f);
			return pid;
		}
		return 0;
	}
	return 0;
}

static void execute(char **argv, int argc)
{
	char s[80]="";
	int i=0;

	while(argc--)
	{
		strcat(s, argv[i++]);
		strcat(s, " ");
	}
	strcat(s, "&");
	if (verbose) printf("Running '%s'\n", s);
	system(s);
}

//
// 1) save current gpm configuration
// 2) set new state of gpm where:
//	which=0 : run standard [svgalib] 'gpm -k'
//	which=1 : run in "SystemxMouse" mode
//
static void RunGPM(int which)
{
	int i;
	old_pid = new_pid = 0;
	
	// save for restore
	old_pid = GetGPM(n_arguments, arguments);
	if (which == 0 && old_pid == 0) return; // no job
	
	for (i=0; i<n_arguments; i++)
	{
		// return if SystemxMouse and GPM
		if (!stricmp(arguments[i], "MouseSystems") && which)
		{
			old_pid = 0; // do not restart
			execute(strs, 2); // kill
			execute(arguments, n_arguments); // refresh FIFO
			return;
		}
	}
	if (which==0)
	{
		old_pid = 0;
		return;
	}
	// kill current gpm = RESTART
	if (old_pid) execute(strs, 2);
	else
	{
		arguments[0]=strs[0];
		n_arguments = 1;
	}
	// and now run new gpm
	arguments[n_arguments] = strs[2];
	execute(arguments, n_arguments+1);
	new_pid = 1;
}

static void RestoreGPM(void)
{
	// kill current gpm
	if (new_pid) execute(strs, 2);
	if (old_pid) execute(arguments, n_arguments);
	new_pid = old_pid = 0;
}


