/*
    OpenGUI - Drawing & Windowing library

    Copyright (C) 1996,2000  Marian Krivos

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    nezmar@internet.alcatel.sk

    fastgl.h   - declarations of many objects for FastGL 
*/

#ifndef	__WINDOW_H
#define	__WINDOW_H

#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <time.h>

#ifdef __QNX__
#include <signal.h>
#include <process.h>
#include <sys/mouse.h>
#include <sys/dev.h>
#include <sys/proxy.h>
#include <sys/kernel.h>
#include <unistd.h>
#undef Yield
#endif

#include "base.h"

#pragma interface

#define	TRUE   1
#define	FALSE  0

#ifndef	M_PI
#define	M_PI	3.1415926
#endif

enum typ_item {	_DOUBLE='d', _INT='i', _STRING='s',	_COMMENT='c' };

#define	MAX_CONF_ITEMS	500

enum events	{ NOEVENT, KEYEVENT, MOUSEEVENT, QUITEVENT, ACCELEVENT,
	  TERMINATEEVENT, MOVEEVENT, CLICKRIGHTEVENT,
	  CLICKLEFTEVENT, DRAGLEFTEVENT, DRAGRIGHTEVENT,
	  GETFOCUSEVENT, LOSTFOCUSEVENT, INITEVENT,
	  BUTTONHOLDEVENT,  WINDOWRESIZEEVENT, WINDOWMOVEEVENT,
	  STARTDRAGLEFTEVENT, STARTDRAGRIGHTEVENT, REPAINTEVENT,
	  NOTIFYEVENT, ICONIZEEVENT,
	  // NOTE> this must be at end
 	  LASTEVENT};

enum bs	{ BDOWN, BUP, WHIDE, WVISIBLE, WDEACTIVE, WACTIVE};
enum BmpType { BMP_NONE, BMP_FILE, BMP_IMAGE, BMP_MEM };

// absolete
#define	RESIZEEVENT WINDOWRESIZEEVENT

#define	IM		 	CScheme->window_fore // CBLACK standard menu fore & background colors
#define	PM			CScheme->window_back

#define	BNORMAL		7
#define	BPUSHED		3
#define	BDISABLED	5
#define	BUNVISIBLE	6

#define	WFRAMED		0x0001
#define	WTITLED		0x0002
#define	WMODAL		0x0004
#define	WNOTIFY		0x0008
#define	WHIDEN		0x0010
#define	WUNMOVED	0x0020
#define	WMENU		0x0040
#define	WNOPICTO	0x0080
#define	WBITMAP		0x0100
#define	WCLICKABLE	0x0200
#define	WDIRTY		0x0400
#define	WTEMPHIDEN	0x0800
#define	WLOCKED		0x1000
#define	WTRIGGER	0x2000
#define	WLASTFOCUS	0x4000
#define	WSIZEABLE	0x8000
#define	WUSELAST	0x00010000
#define	WMINIMIZE	0x00020000
#define	WSTATUSBAR	0x00040000
#define	WEXIST		0x00080000
#define	WHORIZONTAL	0x00100000
#define	WVERTICAL	0x00200000
#define	WLINEMENU	0x00400000
#define	WCENTRED	0x00800000
#define	WESCAPE		0x01000000
#define	WALIGNED	0x02000000
#define	WFASTMOVE	0x04000000

#define	WMENUWINDOWTYPE	   0x80000000
#define	WICONWINDOWTYPE	   0x40000000

#define	WSTANDARD	(WCLICKABLE|WTITLED|WFRAMED)

#define	APP_WINDOWDATABASE		1
#define	APP_CFG					2
#define	APP_ENABLEALTX			4
#define	APP_MAGNIFIER			8
#define	APP_ROOTWINDOW			16
#define APP_ALL					0x1F

// some global fnc

extern "C" void	IError(char	*s,	int	flag);
extern "C" int GetMouseX(void);
extern "C" int GetMouseY(void);
extern "C" void	RemoveMousePointer(void);
extern "C" void ShowMousePointer(void);
void SetRepeatDelay(int	c1,	int	c2);
void SaveScreen(void);
void FlushInput(void);
void SetWindowMoveStyle(int);
extern long IntCounter; // counter of userevent pass

class BaseGui;
class Control;
class Window;
class Bitmap;
class EditBox;
class ButtonGroup;
class ListBox;
class PushButton;
class PointButton;
class CheckButton;
class BaseMenu;
class PushButton;
class Slider;

struct OverlayChain
{
	Window *wnd;
	OverlayChain *next;
};

//
// this object contains no. of the overlayed window relative to this wnd
//
class FGOverlay
{
	int		no;	// 0=none, -1=hiden, other is number of the overlayed windows
	OverlayChain *chain, *top, *curr;
  public:
	void deallocate(void)
	{
		int c=0;
		OverlayChain *p = chain, *pp;
		while(p)
		{
			pp = p->next;
			free(p);
			p = pp;
			c++;
		}
		assert(c == no);	// coherency test
		chain = 0;
		no = 0;
	}
	FGOverlay()
	{
		no = 0;
		chain = top = 0;
	}
	~FGOverlay()
	{
		deallocate();
	}
	void insert(Window *w)
	{
		OverlayChain *p = (OverlayChain *)malloc(sizeof(OverlayChain));
		assert(p);
		if (no==0) chain = top = p;
		else top = top->next = p;
		p->next = 0;
		no++;
		p->wnd = w;
		curr = chain;
	}
	Window * GetOverlapWindow(void)
	{
		curr = chain;
		return chain->wnd;
	}
	Window * GetOverlapNextWindow(void)
	{
		curr = curr->next;
		return curr->wnd;
	}
	int GetOverlapMap(void)
	{
		return no;
	}
};

typedef struct
{
	int		window_back, window_fore;
	int		active_title;
	int		inactive_title;
	int		wnd_bord1;
	int		wnd_bord2;
	int		wnd_bord3;
	int		statusbar;
	int		menu_back;
	int		menu_fore;
	int		menu_back_active;
	int		menu_fore_active;
	int		button_fore;
	int		button_back;
	int		button_fore_pushed;
	int		button_back_pushed;
	int		button_bord1;
	int		button_bord2;
	int		button_bord3;
	int		edit_back;
	int		edit_fore;
	int		edit_bord1;
	int		edit_bord2;
	int		slider;
	int		menuwindow_back;
	int		menuwindow_fore;
	int		pdmenu_back_active;
	int		pdmenu_fore_active;
	int		pdmenu_gray;
	int		edit_disable;
} ColorScheme;

extern ColorScheme * CScheme;

class Node {
	friend class Config;
	typ_item type;
	char name[36];
	int	i;
	char  s[132];
	double d;
	void ModifyNode(char * comment)	// Modify comment
	{
		type = _COMMENT;
		*name =	0;
		strncpy(s, comment, 128);
	}
	void ModifyNode(char * meno, char *	val) // Modify string
	{
		type = _STRING;
		strncpy(name, meno, 32);
		strncpy(s, val, 128);
	}
	void ModifyNode(char * meno, int val) // Modify integer
	{
		type = _INT;
		strncpy(name, meno, 32);
		i	 = val;
	}
	void ModifyNode(char * meno, double	val) // Modify double
	{
		type = _DOUBLE;
		strncpy(name, meno, 32);
		d	 = val;
	}
};

#ifndef __MSDOSa__
typedef unsigned int (*TimerCall)(unsigned int);

#define TIMER_RESOLUTION	10	/* Experimentally determined */

class FGTimer
{
		friend void 				HandleAlarm(int sig);
		int						state;
		static void				StartTicks(void);
	public:
		FGTimer(unsigned int val, TimerCall cb)
		{
			state = SetTimer(val, cb);
		}
		FGTimer(TimerCall cb)
		{
			state = SetTimer(0, cb);
		}
		~FGTimer()
		{
			TimerQuit();
		}
		static int			SetTimer(unsigned int val, TimerCall cb);
		static unsigned int GetTicks(void);
		static void			Delay(unsigned int);
		static int			TimerInit(void);
		static void			TimerQuit(void);
};
#endif

class Config {
	protected:
		int	find(char *	s);
		void CreateEmpty(void);
		char name[64];
		FILE *f;
		int	entries;
		int	change;
		Node pole[MAX_CONF_ITEMS];
		void ReadConfig(void);
		void WriteComment(char * s);
	public:
		
		void	Sync(void);
		int		ReadInt(char	* s);
		char	const * ReadString(char * s);
		double	ReadDouble(char * s);
		int		ReadInt(char	* s, int &p);
		int		ReadString(char * s,	char *p, int max);
		int		ReadString(char * s,	char **p);
		int		ReadDouble(char * s,	double &p);
		void	WriteInt(char * s, int);
		void	WriteString(char	* s, char *);
		void	WriteDouble(char	* s, double);

		Config(char	*);
		~Config();
};

//
//	data structure to event-driven ...
//

class GuiEvent
{
	protected:
		static	char *eventName[];
		int		type;
		int		key;
		int		buttons, x,	y, w, h;
		friend class BaseGui;
		friend class App;
		void AutoRepeatStart(Control *);
		void AutoRepeatDo(void);
	public:
		static void AutoRepeatEnd(void);
		Window	*wnd;	// address of destination
		int		guiType;
		Control *accel;
		GuiEvent(int a=NOEVENT,	int	b=0, int c=0, int d=0, int e=0, int f=0)
		{ 
			type= a;
			key	= b;
			x	= c;
			y	= d;
			w 	= e;
			h	= f;
			accel = 0;
		}
		int	Key(void)  const { return key;	}
		char 	*Name(void)  const { if (type<0 || type>=LASTEVENT) return eventName[LASTEVENT]; return eventName[type]; }
		int	GetX(void)  const { return	x; }
		int	GetY(void)  const { return	y; }
		int	GetW(void)  const { return	w; }
		int	GetH(void)  const { return	h; }
		int	Type(void)  const { return	type; }
		int	Buttons(void)  const {	return buttons;	}
		GuiEvent * GetUserEvent(int);
		void   TranslateUserEvent(void);
		static void	InitUserEvent(void);
		static void	CloseUserEvent(void);
		static void	GetDragVector(int &a, int &b, int &c, int &d);
		static void	SetDragShape(void(*a)(int,int,int,int)=0);
};

typedef	void (*Widget)(void);
typedef	void (*ControlCall)(Control *);
typedef	void (*GuiHwnd)(GuiEvent *);
typedef	int	(*MainHwnd)(GuiEvent *);
typedef	void (*PCHARFNC)(char *);
typedef	Control * CallBack;

//
//	base objects for all graphics objects
//
class BaseGui {
	protected:
		int		 id;
		int 	 parameter1;
		int		 type;
		char 	*name;
		FGPixel	 ink, paper;
		int		 x,y,w,h;
		unsigned long	status;
		BaseGui	*last, * next;
		BaseGui(const BaseGui &)
		{
			printf("No COPY Constructor defined!\n");
			exit(-1);
		}
		static	  	int	counter;
		void		PositionCheck(int &x, int &y, int &w, int &h);
		BaseGui	* 	Over(BaseGui *od, BaseGui *po);
		BaseGui	* 	OverSprite(BaseGui *od,	int	x, int y, int w, int h);
		void	  	Resize(int dx, int dy);
		void	  	Relocate(int xx, int yy);
		static	  	BaseGui	* FindId(BaseGui *,	int	idw);
		BaseGui	* 	FindName(BaseGui *,	char *);
		static	  	BaseGui	* Find(BaseGui *p,GuiEvent *e);
		BaseGui	* 	FindEvent(BaseGui *p,GuiEvent *e);
		void Rechain(BaseGui *l, BaseGui *n);  // place to object chain
	public:
		int		  	IsVisible(void)  const	{ return !(status&(WHIDEN|WTEMPHIDEN));	}
		virtual	~BaseGui();
		enum Type {	WINDOW,	CHECKBUTTON, PUSHBUTTON, POINTBUTTON, EDITBOX, PROGRESSBAR,
			MENUBUTTON, MENUBUTTON2, MENUWINDOW, SLIDEBAR, PSEUDOWINDOW, LISTBOX, LABEL};
		void 	  	SetStatus(long s) {	status = s;	}
		BaseGui	* 	GetNext(void)  const {	return next; }
		BaseGui	* 	GetLast(void)  const {	return last; }
		long	  	GetStatus(void)  const	{ return status; }
		virtual		void MoveIn(void){}
		virtual		void MoveOut(void){}
		void		SetLast(BaseGui *l) { last	= l; }
		void		SetNext(BaseGui *l) { next	= l; }
		void        SetColors(void) { set_colors(ink, paper); }
		void		SetParam(int a) { parameter1 = a; }
		int			GetParam(void)  const { return parameter1; }
		int		  	GetX(void)  const { return	x; }
		int		  	GetY(void)  const { return	y; }
		int		  	GetW(void)  const { return	w; }
		int		  	GetH(void)  const { return	h; }
		int		  	GetId(void)  const	{ return id; }
		int		  	GetType(void)  const {	return type; }
		char	* 	GetName(void)  const {	return name; }
		void	  	SetName(char *s) { delete name;	name = new char[strlen(s)+1]; strcpy(name,s); }
		int		  	GetCnt(void)  const { return counter; }
		FGPixel	  	GetInk(void)  const { return (FGPixel)ink; }
		FGPixel	  	GetPaper(void)  const { return	(FGPixel)paper; }
		BaseGui(int	xs,	int	ys,	int	ws,	int	hs,	char *nm, int typ, int i, int p, long flag);
};

class DrawBuffer
{
	protected:
		int			w,h;
		FGPixel		*bitmap;
		int			type;
		void		copy(FGPixel *dst, FGPixel *src);
	public:
		int			transparent_color;
		void		BitmapDraw(int	a=0, int b=0, int c=-1,	int	d=-1);
		int			GetW(void) const { return w; }
		int			GetType(void) { return type; }
		int			GetH(void) const { return h; }
		FGPixel		*GetArray(void)  const { return bitmap; }
		void		clear(int c) { FGmemset(bitmap, c, w*h); }
		DrawBuffer(int ww=0, int hh=0, BmpType=BMP_MEM, int color=CBLACK, FGPixel *buf=0);
		DrawBuffer(int ww, int hh, FGPixel *buf);
//		DrawBuffer(Window *);
		virtual		~DrawBuffer();
};

#ifndef __GNUC__
#pragma	pack(1);
#endif

class Bitmap : public DrawBuffer {
	void init(void);
	struct RGB3
	{
		char bBlue;				 /* Blue component of the color definition */
		char bGreen;			 /* Green component of the color definition*/
		char bRed;				 /* Red component of the color definition  */
	};
	struct RGB4
	{
		char bBlue;				 /* Blue component of the color definition */
		char bGreen;			 /* Green component of the color definition*/
		char bRed;				 /* Red component of the color definition  */
		char fcOptions;			 /* Reserved, must be zero                 */
	};
	struct bitmap_file_header
	{
		unsigned short	usType;
		unsigned short	cbSizeLow, cbSizeHigh;
		short			reserve1, reserve2;
	};
	struct bitmap_info
	{
		unsigned short	imageOffset;
		short			reserve;
	};
	struct bitmap_info_header1
	{
	  unsigned long	 size;
	  unsigned short cx;			   /* Bit-map width in pels                  */
	  unsigned short cy;			   /* Bit-map height in pels                 */
	  unsigned short cPlanes;		   /* Number of bit planes                   */
	  unsigned short cBitCount;		   /* Number of bits per pel within a plane  */
	  unsigned long	 ulCompression;	   /* Compression scheme used to store the bitmap */
	  unsigned long	 cbImage;		   /* Length of bit-map storage data in chars*/
	  unsigned long	 cxResolution;	   /* x resolution of target device          */
	  unsigned long	 cyResolution;	   /* y resolution of target device          */
	  unsigned long	 cclrUsed;		   /* Number of color indices used           */
	  unsigned long	 cclrImportant;	   /* Number of important color indices      */
	};
	struct bitmap_info_header2
	{
	  unsigned long	 size;
	  unsigned long	 cx;			   /* Bit-map width in pels                  */
	  unsigned long	 cy;			   /* Bit-map height in pels                 */
	  unsigned short cPlanes;		   /* Number of bit planes                   */
	  unsigned short cBitCount;		   /* Number of bits per pel within a plane  */
	  unsigned long	 ulCompression;	   /* Compression scheme used to store the bitmap */
	  unsigned long	 cbImage;		   /* Length of bit-map storage data in chars*/
	  unsigned long	 cxResolution;	   /* x resolution of target device          */
	  unsigned long	 cyResolution;	   /* y resolution of target device          */
	  unsigned long	 cclrUsed;		   /* Number of color indices used           */
	  unsigned long	 cclrImportant;	   /* Number of important color indices      */
	};

		bitmap_file_header	bmfh;
		bitmap_info			bmi;
		union {
			bitmap_info_header1	bmih1;
			bitmap_info_header2	bmih2;
		} bmih;
		union {
			RGB3 rgb3[256];
			RGB4 rgb4[256];
		} rgb;

		char		*name;
		long		*cUsedTable;
		char		*original_name;
		void		turn_bitmap(int cnt, FGPixel *from, FGPixel *to);
		void		DoPalette(int);
	public:
		char *GetName(void) { return name; }
		char *GetOriginalName(void) { return original_name; }
		void SetOriginalName(char *s) { if (original_name) free(original_name); original_name = strdup(s); }
		Bitmap(char	*name);
		Bitmap(void	*memimage);
		Bitmap(Window *);
		Bitmap(int,int,FGPixel=CBLACK);
		~Bitmap();
		int			BitmapSave(char *);
};
#ifndef __GNUC__
#pragma	pack();
#endif

//
//	Base object fo controls (some buttons, menu & input boxes ..
//

class ButtonGroup
{
		int		count;
		int		size;
		int		type; // 1-check,point; 2-push; 0 undefined
		Control * 	curr;
		Control **	array;
	public:
		ButtonGroup(int s=32)
		{
			size = s;
			count = type = 0;
			curr = 0;
			array = new Control *[size];
		}
		~ButtonGroup() { delete [] array; }
		void AddToGroup(Control *p, int activ = 0);
		void RefreshGroup(Control *c);
		void Clear(void) { type = count = 0; curr = 0; }
};

extern Control * call_handler;

class Control :	public BaseGui {
	protected:
		ControlCall	fnc;			// handler
		Window		*owner;			// owner of this
		ButtonGroup 	*grp;		// if is group members
		int		key;			// hotkey
		int		mask;
		int		xr,	yr;			 // x,y real coordinates
		int		local_id;
		Bitmap		*icon;
		Control(int	xs,	int	ys,	int	ws,	int	hs,	char *nm, int typ, int key,	int	flag, Window *,	ControlCall f=0,	int	p=CGRAY1, int i=CDARK);
		Control(int	xs,	int	ys,	int	typ, int key, int flag,	Window *, Bitmap *,	ControlCall f=0);
		~Control();
		void		Underscore(int xo, int yo, int c);
		static		Control *	cControl; // current activated control
		void 		frame(void);
		void		SetActive(void) { call_handler = cControl = this; }
	public:
		void		RunHandler(void);
		ControlCall	GetHandler(void)  const { return fnc; }
		void		RegisterToGroup(ButtonGroup *g) { grp = g;}
		void		RemoveFromGroup(void) { grp = 0; }
		virtual		void SetName(char *s) {	BaseGui::SetName(s); draw(); }
		void		SetKey(int k) {	if (k>='a' && k<='z') k	= toupper(k); key=k; draw(); }
		int			GetKey(void) const { return key; }
		void		SetHandler(ControlCall w) {	fnc=w; }
		virtual		void draw(void) {};
		void		Enable(void);
		void		Disable(void);
		void		Relocate(int xx, int yy) { xr += xx; yr	+= yy; }
		virtual		void ClickUp(int a);  // if TRUE, so call handler
		virtual		void ClickDown(void) {}
		virtual		void ClickMove(int,int) {}
		void		SetTrigger(int a) {	if (a) status |= WTRIGGER;	else	status &= (~WTRIGGER);	draw();}
		int	   		GetTrigger(void) const { return status&WTRIGGER?1:0;	}
		static 		Control * ButtonFind(GuiEvent *e,int	n);
		int			Key(void) const {	return key;	}
		int	   		GetXr(void) const { return xr; }
		int	   		GetYr(void) const { return yr; }
		Window		*GetOwner(void) const { return owner;	}
		void		RefreshGroup(void);
		int			GetLocalId() const { return local_id; }
};

class Label : public Control {
	protected:
		virtual	void draw(void);
		int		font_size;
		int		is_transparent;
		int		high_light;
	public:
		Label(int xs, int ys, char	*nm, int key, Window *w, ControlCall f, int i, int p)
			: Control(xs, ys, 0, 0, nm, LABEL, key, BNORMAL, w,	f, p, i)
		{
			font_size = fontn;
			is_transparent = high_light = 0;
		 	draw();
		}
		void SetTransparent(void) { is_transparent = 1; draw(); }
		void MoveIn(void)
		{
			high_light = 1;
			draw();
		}
		void MoveOut(void)
		{
			high_light = 0;
			draw();
		}
};

class PushButton : public Control {
	protected:
			virtual	void draw(void);
	public:
		PushButton(int xs, int ys, int ws, int hs, char	*nm, int key, int flag,	Window *w, ControlCall f=0)
			: Control(xs, ys, ws, hs, nm, PUSHBUTTON, key, flag, w,	f)
			{
			 	draw();
			}
		PushButton(int xs, int ys, int key,	int	flag, Window *w, Bitmap	*bm, ControlCall f=0)
			: Control(xs, ys,PUSHBUTTON, key, flag,	w, bm, f)
			{
			 	draw();
			}
		void ClickDown(void);
		void ClickUp(int);
		void MoveIn(void);
		void MoveOut(void);
		void Push(void);
		void Release(void);
};

class CheckButton :	public Control {
			virtual	void draw(void);
			int		* variable;
	public:
		CheckButton(int m, int	xs,	int	ys,	char *nm, int key, int flag, Window	*w,	int	fg,	int	bg,	ControlCall f=0, int *var=0)
			: Control(xs, ys, strlen(nm)*8+24, 16, nm, CHECKBUTTON,	key, flag, w, f,bg,fg)
			{
				variable = var;
				mask = m;
				if (variable) SetTrigger(*variable&mask); else draw();
			}
		void ChangeItem(int *v) { variable = v; SetTrigger(*v&mask); draw(); }
};

class PointButton :	public Control {
			virtual	void draw(void);
			int		* variable;
	public:
		PointButton(int m, int	xs,	int	ys,	char *nm, int key, int flag, Window	*w,	int	fg,	int	bg,	ControlCall f=0, int *var=0)
			: Control(xs, ys, strlen(nm)*8+24, 16, nm, POINTBUTTON,	key, flag, w, f,bg,fg)
			{
				variable = var;
				mask = m;
				if (variable) SetTrigger(*variable&mask); else draw();
			}
		void ChangeItem(int *v) { variable = v; SetTrigger(*v&mask); draw(); }
};

extern EditBox *ininput;

class EditBox :	public Control {
	protected:
		virtual	void draw(void);
		void	draw(int);
		int		w1,w2,min,max,data_type;
		unsigned first:1,passwd:1,check_range:1,hex:1;
		double	mind,maxd;
		void  *	ptr;
		static char	buf[128], clip[128];
		static int is_clip;
		int	 pos, maxpos, iline, icol, offset,size;
		void TestRange(void)
		{
			if (check_range==0) return;
			if (data_type==0)
			{
				int	dato = *(int *)ptr;
				if (dato<min) *(int	*)ptr =	min;
				else if	(dato>max) *(int *)ptr = max;
			}
			else if	(data_type==2)
			{
				double dato	= *(double *)ptr;
				if (dato<mind) *(double	*)ptr =	mind;
				else if	(dato>maxd)	*(double *)ptr = maxd;
			}
		}
	public:
		~EditBox()
		{
			if (ininput	== this)
				ininput	= 0;
		}
		void MoveIn(void);
		void MoveOut(void);
		void ClickUp(int);
		int	 inputproc(int);
		void input(void);
		void PasswdMode(int m) { passwd = m; draw(); }
		void HexMode(int m) { hex = m; draw(); }
		void ChangeItem(int	*p)
		{
			assert(data_type==0);
			ptr	= p;
			draw();
		}
		void ChangeItem(char *p)
		{
			assert(data_type==1);
			ptr= p;
			draw();
		}
		void ChangeItem(double *p)
		{
			assert(data_type==2);
			ptr	= p;
			draw();
		}
		void SetSize(int ns)
		{
			pos = offset = 0;
			assert(ns>0 && ns<128);
			size = ns;
			draw();
		}
		EditBox(int xs,	int ys,	int ws1, int ws2, char *nm, int	key, Window *w,	int *pt, int ink, int paper, ControlCall f, int mn, int mx, int check);
		EditBox(int sz, int	xs,	int	ys,	int	ws1, int ws2, char *nm,	int	key, Window	*w,	char *pt, int ink, int paper, ControlCall f);
		EditBox(int xs,	int ys,	int ws1, int ws2, char *nm, int	key, Window *w,	double *pt, int	ink, int paper,	ControlCall f, double mn, double mx, int check);
};

class ProgressBar : public Control
{
		int		steps, value;
		int		sirka;
	public:
		void	draw(void);
		void	setProgress(int a) { if (a>=0 && a<=steps) value = a; draw(); }
		int		progress(void) const { return value; }
		ProgressBar(Window *w, int xx, int yy, int ww, int hh, int s);
};

class BaseMenu : public	Control	{
			virtual	void draw(void);
	public:
		void MoveIn(void);
		void MoveOut(void);
		virtual	void ClickUp(int a);  // if TRUE, so call handler
		BaseMenu(int x,	int	y, int ww, int hh, char	*nm, int key, Window *w, int bg, int fg, ControlCall f, int type)
			: Control(x,y,ww, hh, nm, type, key, BNORMAL, w, f,bg,fg)
			{
				draw();
			}
};

class Slider  : public Control
{
	protected:
		virtual void draw(void) {}
		int		minv,maxv,*val,steps,smer; // 0 je vodorovny, 1 zvysly
		void 	frame(int,int,int,int,int f=0);
		void	modify(int);
		void MoveIn(void);
		void MoveOut(void);
	public:
		Slider(Window *win, int xs, int ys, int ws, int hs, int minimal, int maximal, int s, int *v, ControlCall f) :
			Control(xs, ys, ws, hs, 0, SLIDEBAR, 0, BNORMAL, win, f)
		{
			if (minimal>maximal)
			{
				minv = maximal;
				maxv = minimal;
			}
			else
			{
				minv = minimal;
				maxv = maximal;
			}
			if (*v<minv) *v=minv;
			if (*v>maxv) *v=maxv;
			val  = v;
			steps= s;
			smer = h>w;
		}
};

class SlideBarH : public Slider 
{
// 3 - 16 - val - 3 - 38
	public:
		SlideBarH(int x, int y, int min, int max, int step, int *val, Window *win, ControlCall f=0):
			Slider(win,x,y,(max-min)/step+16+12+38,16,min,max,step,val,f)
		{
			draw();
		}
		void    redraw(void)
		{
			ControlCall tmp = fnc;
			fnc = 0;
			draw();
			fnc = tmp;
		}
		virtual void draw(void);
		virtual void ClickMove(int dx, int dy);
		virtual void ClickDown(void);
};

class SlideBarV : public Slider
{
// 3 - 16 - val - 3 - 38
	public:
		SlideBarV(int x, int y, int min, int max, int step, int *val, Window *win, ControlCall f=0):
			Slider(win,x,y,16,(max-min)/step+16+12+38,min,max,step,val,f)
		{
			draw();
		}
		void    redraw(void)
		{
			ControlCall tmp = fnc;
			fnc = 0;
			draw();
			fnc = tmp;
		}
		virtual void draw(void);
		virtual void ClickMove(int dx, int dy);
		virtual void ClickDown(void);
};

class ListBox : public Slider
{
		void	(*kresli)(int,int,int,int,void *);
		void	*dataptr;
		int		*p_position, *p_size; // for items
		int		slider;			  // for position of slider
		void	corecture(void)
		{
			slider = 0;
			if (*p_size && *p_position) slider = int (maxv * (((float)*p_position+1.) / *p_size));
			draw();
		}
		void	_init(int xs, int ys, int wpol, int hpol, int w, int h, int *pos, int *size, void (*drawone)(int,int,int,int,void *), void *data);
		void	draw(void);	// for slider
		static 	void lbCall(Control *);
	public:
		int		DoListBox(int key);
		virtual void ClickMove(int dx, int dy);
		virtual void ClickDown(void);
		int 	Draw(int i=0);	// for listbox
		int 	x,y,polx,poly,_curr,_count,fpol,onew,oneh,cx,cy;
			// flag bit 0 set if x+1 is down (other x+1 is right)
		int 	GetCurrent(void) const { return *p_position; }
		int 	GetFirstVisible(void) const { return fpol*polx; }
		int 	GetSize(void) const { return *p_size; }
		void 	SetSize(int);
		ListBox(int xs, int ys, int wpol, int hpol, int w, int h, int *cnt, int *pos, void (*drawone)(int,int,int,int,void *), Window *wind, void *data=0);
		ListBox(int xs, int ys, int wpol, int hpol, int w, int h, int cnt, void (*drawone)(int,int,int,int,void *), Window *wind, void *data=0);
		void 	Resize(int kolko);
		void 	SetToItem(int);
		void 	SetToItemRel(int);
		void 	RedrawItem(void)
		{
			kresli(x+cx*onew, y+cy*oneh, *p_position, 1, dataptr);
		}
		void 	Up(void);
		void 	Down(void);
		void 	Right(void);
		void 	Left(void);
		int  	Test(int, int);
};

class Window : public BaseGui {
		friend	class SlideBarH;
		friend	class SlideBarV;
		friend	class ListBox;
		friend	volatile void DestroyWindow(void);
		friend	void   GuiEvent::TranslateUserEvent(void);
		friend	int	isgmkey(int	k);
		friend	class Control;
	protected:
		int		tmp_ink, tmp_paper, tmp_font, tmp_ppop;
		int		c_menupos;
		int		first_draw;
		int		changed;
		FGPixel		*image;
		int		statuscolor;
		Window		**itself;
		void 	(*handler)(GuiEvent	*);
		BaseGui	*Buttony;
		BaseGui	*lastButton; // last in list
		int		nControls;	 // pocet buttonov priradenych oknu
		static	int	nWnd;
		int		wsur,hsur,xpos,ypos;
		int		xwrk,ywrk,wwrk,hwrk;
		Window	*iconized;
		DrawBuffer *icon;
		FGOverlay	ovr;
		int		icon_x, icon_y;
		static	int	allControls;
		static	int	hresult, rebuild;
		static	Window *current, *first;  // last in list of all windows
		int		OverWnd(void);
		static void	RepaintBlock(int x,	int	y, int w, int h)
		{
			RamToVideo(x,y,current->GetX()+x,current->GetY()+y,w,h);
//
// set the conditions to TRUE for the transparent mouse cursor
//
#if 0
			if (_over(x, y, x+w, y+h, GetMouseX(), GetMouseY(), 16, 16))
			{
				RemoveMousePointer();
				ShowMousePointer();
			}
#endif
		}
		void	init(void);
		long	WindowStatus(long status);
		void	WindowHide(void);
		void	WindowShow(void);
		void	WindowRecopy(void);
		int		OdkryteOkno(Window *pokial, int x, int y, int w, int h);
		int		Rozporcuj(int X1, int Y1, int X2, int Y2, int *in, int *out);
		int		GetXM(int);
		int		GetYM(void);
		void	(*CallWindowHook)(int x, int y, int& nx, int& ny, int);
		void	DrawTitleMem(int color); // old virtual
		void 	WindowUpdate(int dx, int dy, int all=0);
		void	intersect(int, int, int, int);
		static  int	_over(int, int, int, int, int xx, int yy, int ww, int hh);
		int		iconize(int xs, int ys);
		void	PiktoMem(int x, int y);
		void	BuildOverlays(Window *);
	public:
		void 	WindowUpdate(int x, int y, int w, int h);
		virtual	void draw(void);
		static	void nextwnd(void);
		void	SetChange(void) { changed = 1; }
		void	SetIconPosition(int x, int y) { icon_x = x; icon_y = y; }
		void	ResetChange(void) { changed = 0; }
		int		ShowNotify(void);
		void	InstallWindowHook(void(*hook)(int, int, int&, int&, int))
		{
			CallWindowHook = hook;
		}
		FGPixel *GetArray(void) const { return image; }
		int		IsIconized(void) const { return (int)iconized; }
		static	int	TestNullHandler(void);
		void	wputc(int);
		void	move(int x=0,	int	y=0) { wputc(-1); xpos=x; ypos=y; }
		int		GetWW(void) const {	return wsur; }
		int		GetHW(void) const {	return hsur; }
		static	Window * GetCurrent(void)	{ return current; }
		BaseGui	* GetCurrentControls(void) { return	Buttony; }
		void	WindowUpdateBlock(int x, int y, int w, int h);
		void	WindowRepaint(int	xr,	int	yr,	int	w, int h);
		void	WindowRepaintUser(int	xr,	int	yr,	int	w, int h);
		void	WindowClipUser(int& xr, int& yr, int& wr, int& hr);
		void	WindowClip(int& xr, int& yr, int& wr, int& hr);
		Window	* Context(void);
		static	void HideAll(void);
		static	void ShowAll(void);
		Window(Window **, int xs, int ys, int ws, int hs, char *nm,	GuiHwnd=0, int i=IM, int p=PM, int flag=WTITLED|WFRAMED|WCLICKABLE);
		Window(DrawBuffer *);
		Window(int x, int y, int w, int h);
		Window(Window **, FGPixel);	// root window
		~Window();
		PushButton	*AddPushButton(int xs, int ys, int ws, int hs, char *nm,	int	key=0, ControlCall f=0, int flag=BNORMAL);
		PushButton	*AddPushButton(int xs, int ys, int key, Bitmap *bm, ControlCall f=0, int flag=BNORMAL);
		SlideBarH	*AddSlideBarH(int xs, int ys, int min, int max, int step, int *val, ControlCall f=0);
		SlideBarV	*AddSlideBarV(int xs, int ys, int min, int max, int step, int *val, ControlCall f=0);
		CheckButton	*AddCheckButton(int	xs,	int	ys,	char *nm, int key=0, int * variable=0, ControlCall=0, int flag=BNORMAL);
		PointButton	*AddPointButton(int	xs,	int	ys,	char *nm, int key=0, int * variable=0, ControlCall=0, int flag=BNORMAL);
		CheckButton	*AddCheckButtonMask(int mask, int	xs,	int	ys,	char *nm, int key=0, int * variable=0, ControlCall=0, int flag=BNORMAL);
		PointButton	*AddPointButtonMask(int mask, int	xs,	int	ys,	char *nm, int key=0, int * variable=0, ControlCall=0, int flag=BNORMAL);
		EditBox		*AddEditBox(int	xs,	int	ys,	int	ws1, int ws2, char *nm,	int	key, int *p, ControlCall f, int min, int max);
		EditBox		*AddEditBox(int	xs,	int	ys,	int	ws1, int ws2, char *nm,	int	key, int *p, ControlCall f=0);
		EditBox		*AddEditBox(int	xs,	int	ys,	int	ws1, int ws2, char *nm,	int	key, char *p, ControlCall f=0);
		EditBox		*AddEditBox(int sz, int	xs,	int	ys,	int	ws1, int ws2, char *nm,	int	key, char *p, ControlCall f=0);
		EditBox		*AddEditBox(int	xs,	int	ys,	int	ws1, int ws2, char *nm,	int	key, double *p, ControlCall f, double min, double max);
		EditBox		*AddEditBox(int	xs,	int	ys,	int	ws1, int ws2, char *nm,	int	key, double *p, ControlCall f=0);
		BaseMenu	*AddBaseMenu(char *nm,	int	key=0, ControlCall f=0);
		ProgressBar	*AddProgressBar(int xs, int ys, int ws, int hs, int size);
		ListBox		*AddListBox(int xs, int ys, int wpol, int hpol, int w, int h, int *cnt, int *pos, void (*drawone)(int,int,int,int,void *), void *data=0);
		ListBox 	*AddListBox(int xs, int ys, int wpol, int hpol, int w, int h, int cnt, void (*drawone)(int,int,int,int,void *), void *data);
		Label		*AddLabel(int x, int y, char *s, int key=0, ControlCall f=0, int i=-1, int p=-1);
		void 		WindowMove(int,int);
		void 		WindowFocus(void);
		void 		WindowText(int	x, int y, char *s);
		void 		WindowText(int	x, int y, char *s, int color);
		void 		WindowText(int	x, int y, char *s, int color, int bk);
		void 		WindowBox(int x, int y, int a,	int	b);
		void 		WindowBox(int x, int y, int a,	int	b, int c);
		void 		WindowPixel(int x,	int	y, int c);
		void 		WindowPixel(int x,	int	y);
		unsigned 	WindowGetPixel(int	x, int y)
		{
			if (x<0	|| y<0 || x>wwrk ||	y>hwrk)	return 0x01000100;
			return image[(x+xwrk)+(y+ywrk)*GetW()];
		}
		void 		WindowRect(int	x, int y, int a, int b);
		void 		WindowRect(int	x, int y, int a, int b,	int	c);
		void 		WindowLine(int	x, int y, int a, int b);
		void 		WindowLine(int	x, int y, int a, int b,	int	c);
		void 		WindowDrawCircle(int x, int y,	int	r, int c);
		void 		WindowDrawCircle(int x, int y,	int	r);
		void 		WindowFillCircle(int x, int y,	int	r, int c);
		void 		WindowFillCircle(int x, int y,	int	r);
		void 		WindowDrawEllipse(int x, int y,	int	rx, int ry, int c);
		void 		WindowDrawEllipse(int x, int y,	int	rx, int ry);
		void 		WindowFillEllipse(int x, int y,	int	rx, int ry, int c);
		void 		WindowFillEllipse(int x, int y,	int	rx, int ry);
		void 		WindowDrawArc(int x, int y,	double ang1, double ang2, int r, int c);
		void 		WindowDrawArc(int x, int y,	double ang1, double ang2, int r);
		void 		WindowScrollDown(int p1, int p2, int p3, int p4 ,int p5);
		void 		WindowScrollUp(int	p1,	int	p2,	int	p3,	int	p4 ,int	p5);
		void 		WindowStatusBar(int, char *s, int c=IM);
		int	 		TitleFind(GuiEvent	*e);
		static 		Window *	WindowFind(GuiEvent	*e);
		int	 		printf(const char *, ...);
		int	 		printf(int,int,const char *, ...);
		void 		SendToWindow(GuiEvent *p);
		void 		WindowLock(void) {	status|=WLOCKED; }
		void 		WindowUnLock(void);
		void 		WindowResize(int, int);
		void 		WindowShape(int, int, int, int);
		void 		WindowVirtualRoot(int, int, int, int);
		void 		puts(char *s) { while(*s) wputc(*s++);	}
		void 		WindowCopyFrom(int x,	int	y, int xs, int ys,	int	w, int h, Window *p);
		void 		WindowPutBitmap(int x,	int	y, int xs, int ys,	int	w, int h, DrawBuffer *p);
		void 		WindowGetBitmap(int x,	int	y,	int	xs,	int	ys,	int	w, int h, DrawBuffer *p);
		void 		SetName(char *s)
      	{
			BaseGui::SetName(s);
			if (this==current) WindowStatus(WACTIVE);
			else WindowStatus(WDEACTIVE);
		}
		int 		WindowIconize(void);
		int 		WindowIconize(int, int);
		void 		WindowAttachIcon(DrawBuffer *ico) { icon = ico; }
		void 		WindowFillPolygon(int,	int[][2], int col=-1);
		void 		WindowDrawPolygon(int,	int[][2], int col=-1);
		void 		WindowSpline(int points[8]);
		void 		FlushInput(void);							// absolete !!!
		void 		WindowSetWorkRect(int,int,int,int);
		void 		WindowGetWorkRect(int&,int&,int&,int&);
		void 		RemoveControls(void);
};

#define	MAX_EVENT_QUEQUE	10	 // numbers of events to app per one user event ...

class App {
		static	GuiEvent *event, *queque;
		static	void (*DelayProc)(void);
		static	int	quequeIndex;
		static	void (*OnEverySecond)(int);
		static  Window *Root;
		static	MainHwnd appHandler;
	public:
		int		flags;
		int		video;
		int		Argc;
		int		background;
		char	**Argv;
		char	* name;
		static	char	*homedir;
		static  char	currdir[128];
		static	void SetDelayProc(void (*fnc)(void))
		{
			DelayProc =	fnc;
		}
		static	void CallDelayProc(void)
		{
			if (DelayProc) DelayProc();
			else delay(10);
		}
		App(int, int &,	char **	&, int b, int f=APP_ENABLEALTX);
		~App();
		static	void SendToApp(GuiEvent	*x)
		{
			if (quequeIndex<MAX_EVENT_QUEQUE)
			{
				memcpy(queque+quequeIndex++,x,sizeof(GuiEvent));
			}
		}
		void	Run(MainHwnd hwnd=0);
		void	Yield(void);
		static	void AppDone(void)
		{
			GuiEvent e(QUITEVENT);
			SendToApp(&e);
		}
		static	void Timer(void);
		static void	SetTimerProc(void ((*p)(int)))
		{
			OnEverySecond =	p;
		}
		static Window * GetRootWindow(void)
		{
			assert(Root);
			return Root;
		}
		void BroadcastMessage(GuiEvent *event);
};

extern App *cApp; // current app
extern Config *cCfg;
extern int FGx, FGy;

class MenuWindow : public Window {
		static GuiHwnd Proc;
		static MenuWindow *	CurrentMenu;
		static void	MenuWindowHandler(GuiEvent *p);
		int		offset;
		int		cid;
		void _initmw(GuiHwnd proc)
		{
			offset = 0;
			cid = -1;
			Proc = proc;
			CurrentMenu	= this;
		}
	public:
		MenuWindow(int x, int y, int w,	int	h, char	*name="MenuWindow", GuiHwnd proc=0, int bg=CScheme->menuwindow_back):
			Window(0,x,y,w,h,name,MenuWindowHandler,CScheme->menuwindow_fore,bg,WFRAMED|WCLICKABLE|WNOPICTO|WMENUWINDOWTYPE)
		{
			_initmw(proc);
		}
		MenuWindow(int w, int h, char	*name="MenuWindow", GuiHwnd proc=0, int bg=CScheme->menuwindow_back):
			Window(0,FGx,FGy,w,h,name,MenuWindowHandler,CScheme->menuwindow_fore,bg,WFRAMED|WCLICKABLE|WNOPICTO|WMENUWINDOWTYPE)
		{
			_initmw(proc);
		}
		~MenuWindow()
		{
			CurrentMenu	= 0;
		}
		int	GetXM(void)	const 
		{
			return 4;
		}
		int	GetYM(int a)
		{
			return (offset+=a,offset-a);
		}
		BaseMenu *AddMenu(char *nm,	int	key=0, ControlCall f=0);
		CheckButton	*AddCheckButton(char *nm, int key=0, int * variable=0, ControlCall f=0, int flag=BNORMAL);
		PointButton	*AddPointButton(char *nm, int key=0, int * variable=0, ControlCall f=0, int flag=BNORMAL);
		CheckButton	*AddCheckButtonMask(int, char *nm, int key=0, int * variable=0, ControlCall f=0, int flag=BNORMAL);
		PointButton	*AddPointButtonMask(int, char *nm, int key=0, int * variable=0, ControlCall f=0, int flag=BNORMAL);
		EditBox	*AddEditBox(int	ws1, int ws2, char *nm,	int	key, int *p,ControlCall f, int min, int max);
		EditBox	*AddEditBox(int	ws1, int ws2, char *nm,	int	key, int *p, ControlCall f=0);
		EditBox	*AddEditBox(int	ws1, int ws2, char *nm,	int	key, char *p, ControlCall f=0);
		EditBox	*AddEditBox(int sz, int	ws1, int ws2, char *nm,	int	key, char *p, ControlCall f=0);
		EditBox	*AddEditBox(int	ws1, int ws2, char *nm,	int	key, double	*p,	ControlCall f, double min, double max);
		EditBox	*AddEditBox(int	ws1, int ws2, char *nm,	int	key, double	*p,	ControlCall f=0);
		void Separator(void);
};

void SendEvent(Window *w, int event, int key=0,	int	x=0, int y=0);
void SendEvent(Window *w, int event, int x, int y, int, int);
int RegisterJob(Widget w);
unsigned int CalculateCRC(unsigned StartCRC, void *Addr, unsigned Size);

#endif
