/*
	linux/kernel/blk_drv/mcd2.c - Mitsumi CDROM driver

	Copyright (C) 1992  Martin Harriss

	martin@bdsi.com

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2, or (at your option)
	any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	HISTORY

	0.1	First attempt - internal use only
	0.2	Cleaned up delays and use of timer - alpha release
	0.3	Audio support added
	0.3.1 Changes for mitsumi CRMC LU005S march version
		   (stud11@cc4.kuleuven.ac.be)
        0.3.2 bug fixes to the ioctls and merged with ALPHA0.99-pl12
		   (Jon Tombs <jon@robots.ox.ac.uk>)
	0.3.3 Added more #defines and mcd2_setup()
   		   (Jon Tombs <jon@gtex02.us.es>)

	October 1993 Bernd Huebner and Ruediger Helsch, Unifix Software GmbH,
	Braunschweig, Germany: Total rework to speed up data read operation.
	Also enabled definition of irq and address from bootstrap, using the
	environment. linux/init/main.c must be patched to export the env.
	November 93 added code for FX001 S,D (single & double speed).
	February 94 added code for broken M 5/6 series of 16-bit single speed.
*/


#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/cdrom.h>
#include <linux/ioport.h>
#include <linux/string.h>
#include <linux/delay.h>

/* #define REALLY_SLOW_IO  */
#include <asm/system.h>
#include <asm/io.h>
#include <asm/segment.h>

#define MAJOR_NR MITSUMI_CDROM2_MAJOR
#include "blk.h"
#include <linux/mcd2.h>

#if 0
static int mcd2_sizes[] = { 0 };
#endif

static int mcd2Present = 0;

#if 0 
#define TEST1 /* <int-..> */
#define TEST2 /* do_mcd2_req */
#define TEST3 */ /* MCD2_S_state */
#define TEST4 /* QUICK_LOOP-counter */
#define TEST5 */ /* port(1) state */
#endif

#if 1
#define QUICK_LOOP_DELAY udelay(45)  /* use udelay */
#define QUICK_LOOP_COUNT 20
#else
#define QUICK_LOOP_DELAY
#define QUICK_LOOP_COUNT 140 /* better wait constant time */
#endif
/* #define DOUBLE_QUICK_ONLY */

#define CURRENT_VALID \
  (CURRENT && MAJOR(CURRENT -> dev) == MAJOR_NR && CURRENT -> cmd == READ \
   && CURRENT -> sector != -1)
#define MFL_STATUSorDATA (MFL_STATUS | MFL_DATA)
#define MCD2_BUF_SIZ 16
static volatile int mcd2_transfer_is_active;
static char mcd2_buf[2048*MCD2_BUF_SIZ];	/* buffer for block size conversion */
static volatile int mcd2_buf_bn[MCD2_BUF_SIZ], mcd2_next_bn;
static volatile int mcd2_buf_in, mcd2_buf_out = -1;
static volatile int mcd2_error;
static int mcd2_open_count;
enum mcd2_state_e {
  MCD2_S_IDLE,   /* 0 */
  MCD2_S_START,  /* 1 */
  MCD2_S_MODE, /* 2 */
  MCD2_S_READ,   /* 3 */
  MCD2_S_DATA,   /* 4 */
  MCD2_S_STOP,   /* 5 */
  MCD2_S_STOPPING /* 6 */
};
static volatile enum mcd2_state_e mcd2_state = MCD2_S_IDLE;
static int mcd2_mode = -1;
static int MCMD_DATA_READ2= MCMD_PLAY_READ;
#define READ_TIMEOUT 3000
#define WORK_AROUND_MITSUMI_BUG_92
#define WORK_AROUND_MITSUMI_BUG_93
#ifdef WORK_AROUND_MITSUMI_BUG_93
int mitsumi_bug_93_wait2 = 0;
#endif /* WORK_AROUND_MITSUMI_BUG_93 */

static short mcd2_port = MCD2_BASE_ADDR;
static int   mcd2_irq  = MCD2_INTR_NR;

static int McdTimeout2, McdTries2;
static struct wait_queue *mcd2_waitq = NULL;

static struct mcd2_DiskInfo DiskInfo2;
static struct mcd2_Toc Toc2[MAX_TRACKS];
static struct mcd2_Play_msf mcd2_Play;

static int audioStatus2;
static char mcd2DiskChanged;
static char tocUpToDate2;
static char mcd2Version;

static void mcd2_transfer(void);
static void mcd2_poll(void);
static void mcd2_invalidate_buffers(void);
static void do_mcd2_request(void);
static void hsg2msf2(long hsg, struct msf *msf);
static void bin2bcd2(unsigned char *p);
static int bcd2bin2(unsigned char bcd);
static int mcd2Status(void);
static void sendMcdCmd2(int cmd, struct mcd2_Play_msf *params);
static int getMcdStatus2(int timeout);
static int GetQChannelInfo2(struct mcd2_Toc *qp);
static int updateToc2(void);
static int GetDiskInfo2(void);
static int GetToc2(void);
static int getValue2(unsigned char *result);


void mcd2_setup(char *str, int *ints)
{
   if (ints[0] > 0)
      mcd2_port = ints[1];
   if (ints[0] > 1)      
      mcd2_irq  = ints[2];
#ifdef WORK_AROUND_MITSUMI_BUG_93
   if (ints[0] > 2)
      mitsumi_bug_93_wait2 = ints[3];
#endif /* WORK_AROUND_MITSUMI_BUG_93 */
}

 
static int
check_mcd2_change(dev_t full_dev)
{
   int retval, target;


#if 1	 /* the below is not reliable */
   return 0;
#endif  
   target = MINOR(full_dev);

   if (target > 0) {
      printk("mcd2: Mitsumi CD-ROM2 request error: invalid device.\n");
      return 0;
   }

   retval = mcd2DiskChanged;
   mcd2DiskChanged = 0;

   return retval;
}


/*
 * Do a 'get status' command and get the result.  Only use from the top half
 * because it calls 'getMcdStatus2' which sleeps.
 */

static int
statusCmd2(void)
{
	int st, retry;

	for (retry = 0; retry < MCD2_RETRY_ATTEMPTS; retry++)
	{

		outb(MCMD_GET_STATUS, MCD2PORT(0));	/* send get-status cmd */
		st = getMcdStatus2(MCD2_STATUS_DELAY);
		if (st != -1)
			break;
	}

	return st;
}


/*
 * Send a 'Play' command and get the status.  Use only from the top half.
 */

static int
mcd2Play(struct mcd2_Play_msf *arg)
{
	int retry, st;

	for (retry = 0; retry < MCD2_RETRY_ATTEMPTS; retry++)
	{
		sendMcdCmd2(MCMD_PLAY_READ, arg);
		st = getMcdStatus2(2 * MCD2_STATUS_DELAY);
		if (st != -1)
			break;
	}

	return st;
}


long
msf2hsg2(struct msf *mp)
{
	return bcd2bin2(mp -> frame)
		+ bcd2bin2(mp -> sec) * 75
		+ bcd2bin2(mp -> min) * 4500
		- 150;
}


static int
mcd2_ioctl(struct inode *ip, struct file *fp, unsigned int cmd,
						unsigned long arg)
{
	int i, st;
	struct mcd2_Toc qInfo;
	struct cdrom_ti ti;
	struct cdrom_tochdr tocHdr;
	struct cdrom_msf msf;
	struct cdrom_tocentry entry;
	struct mcd2_Toc *tocPtr;
	struct cdrom_subchnl subchnl;
#if 0
	struct cdrom_volctrl volctrl;
#endif

	if (!ip)
		return -EINVAL;

	st = statusCmd2();
	if (st < 0)
		return -EIO;

	if (!tocUpToDate2)
	{
		i = updateToc2();
		if (i < 0)
			return i;	/* error reading TOC */
	}

	switch (cmd)
	{
	case CDROMSTART:     /* Spin up the drive */
		/* Don't think we can do this.  Even if we could,
 		 * I think the drive times out and stops after a while
		 * anyway.  For now, ignore it.
		 */

		return 0;

	case CDROMSTOP:      /* Spin down the drive */
		outb(MCMD_STOP, MCD2PORT(0));
		i = getMcdStatus2(MCD2_STATUS_DELAY);

		/* should we do anything if it fails? */

		audioStatus2 = CDROM_AUDIO_NO_STATUS;
		return 0;

	case CDROMPAUSE:     /* Pause the drive */
		if (audioStatus2 != CDROM_AUDIO_PLAY)
			return -EINVAL;

		outb(MCMD_STOP, MCD2PORT(0));
		i = getMcdStatus2(MCD2_STATUS_DELAY);

		if (GetQChannelInfo2(&qInfo) < 0)
		{
			/* didn't get q channel info */

			audioStatus2 = CDROM_AUDIO_NO_STATUS;
			return 0;
		}

		mcd2_Play.start = qInfo.diskTime;	/* remember restart point */

		audioStatus2 = CDROM_AUDIO_PAUSED;
		return 0;

	case CDROMRESUME:    /* Play it again, Sam */
		if (audioStatus2 != CDROM_AUDIO_PAUSED)
			return -EINVAL;

		/* restart the drive at the saved position. */

		i = mcd2Play(&mcd2_Play);
		if (i < 0)
		{
			audioStatus2 = CDROM_AUDIO_ERROR;
			return -EIO;
		}

		audioStatus2 = CDROM_AUDIO_PLAY;
		return 0;

	case CDROMPLAYTRKIND:     /* Play a track.  This currently ignores index. */

		st = verify_area(VERIFY_READ, (void *) arg, sizeof ti);
		if (st)
			return st;

		memcpy_fromfs(&ti, (void *) arg, sizeof ti);

		if (ti.cdti_trk0 < DiskInfo2.first
			|| ti.cdti_trk0 > DiskInfo2.last
			|| ti.cdti_trk1 < ti.cdti_trk0)
		{
			return -EINVAL;
		}

		if (ti.cdti_trk1 > DiskInfo2.last)
			ti. cdti_trk1 = DiskInfo2.last;

		mcd2_Play.start = Toc2[ti.cdti_trk0].diskTime;
		mcd2_Play.end = Toc2[ti.cdti_trk1 + 1].diskTime;

#ifdef MCD2_DEBUG
printk("play: %02x:%02x.%02x to %02x:%02x.%02x\n",
	mcd2_Play.start.min, mcd2_Play.start.sec, mcd2_Play.start.frame,
	mcd2_Play.end.min, mcd2_Play.end.sec, mcd2_Play.end.frame);
#endif

		i = mcd2Play(&mcd2_Play);
		if (i < 0)
		{
			audioStatus2 = CDROM_AUDIO_ERROR;
			return -EIO;
		}

		audioStatus2 = CDROM_AUDIO_PLAY;
		return 0;

	case CDROMPLAYMSF:   /* Play starting at the given MSF address. */

		if (audioStatus2 == CDROM_AUDIO_PLAY) {
		  outb(MCMD_STOP, MCD2PORT(0));
		  i = getMcdStatus2(MCD2_STATUS_DELAY);
		  audioStatus2 = CDROM_AUDIO_NO_STATUS;
		}

		st = verify_area(VERIFY_READ, (void *) arg, sizeof msf);
		if (st)
			return st;

		memcpy_fromfs(&msf, (void *) arg, sizeof msf);

		/* convert to bcd */

		bin2bcd2(&msf.cdmsf_min0);
		bin2bcd2(&msf.cdmsf_sec0);
		bin2bcd2(&msf.cdmsf_frame0);
		bin2bcd2(&msf.cdmsf_min1);
		bin2bcd2(&msf.cdmsf_sec1);
		bin2bcd2(&msf.cdmsf_frame1);

		mcd2_Play.start.min = msf.cdmsf_min0;
		mcd2_Play.start.sec = msf.cdmsf_sec0;
		mcd2_Play.start.frame = msf.cdmsf_frame0;
		mcd2_Play.end.min = msf.cdmsf_min1;
		mcd2_Play.end.sec = msf.cdmsf_sec1;
		mcd2_Play.end.frame = msf.cdmsf_frame1;

#ifdef MCD2_DEBUG
printk("play: %02x:%02x.%02x to %02x:%02x.%02x\n",
mcd2_Play.start.min, mcd2_Play.start.sec, mcd2_Play.start.frame,
mcd2_Play.end.min, mcd2_Play.end.sec, mcd2_Play.end.frame);
#endif

		i = mcd2Play(&mcd2_Play);
		if (i < 0)
		{
			audioStatus2 = CDROM_AUDIO_ERROR;
			return -EIO;
		}

		audioStatus2 = CDROM_AUDIO_PLAY;
		return 0;

	case CDROMREADTOCHDR:        /* Read the table of contents header */
		st = verify_area(VERIFY_WRITE, (void *) arg, sizeof tocHdr);
		if (st)
			return st;

		tocHdr.cdth_trk0 = DiskInfo2.first;
		tocHdr.cdth_trk1 = DiskInfo2.last;
		memcpy_tofs((void *) arg, &tocHdr, sizeof tocHdr);
		return 0;

	case CDROMREADTOCENTRY:      /* Read an entry in the table of contents */

		st = verify_area(VERIFY_WRITE, (void *) arg, sizeof entry);
		if (st)
			return st;

		memcpy_fromfs(&entry, (void *) arg, sizeof entry);
		if (entry.cdte_track == CDROM_LEADOUT)
			/* XXX */
			tocPtr = &Toc2[DiskInfo2.last + 1];

		else if (entry.cdte_track > DiskInfo2.last
				|| entry.cdte_track < DiskInfo2.first)
			return -EINVAL;

		else
			tocPtr = &Toc2[entry.cdte_track];

		entry.cdte_adr = tocPtr -> ctrl_addr;
		entry.cdte_ctrl = tocPtr -> ctrl_addr >> 4;

		if (entry.cdte_format == CDROM_LBA)
			entry.cdte_addr.lba = msf2hsg2(&tocPtr -> diskTime);

		else if (entry.cdte_format == CDROM_MSF)
		{
			entry.cdte_addr.msf.minute = bcd2bin2(tocPtr -> diskTime.min);
			entry.cdte_addr.msf.second = bcd2bin2(tocPtr -> diskTime.sec);
			entry.cdte_addr.msf.frame = bcd2bin2(tocPtr -> diskTime.frame);
		}

		else
			return -EINVAL;

		memcpy_tofs((void *) arg, &entry, sizeof entry);
		return 0;

	case CDROMSUBCHNL:   /* Get subchannel info */

		st = verify_area(VERIFY_WRITE, (void *) arg, sizeof subchnl);
		if (st)
			return st;

		memcpy_fromfs(&subchnl, (void *) arg, sizeof subchnl);

		if (GetQChannelInfo2(&qInfo) < 0)
			return -EIO;

		subchnl.cdsc_audiostatus = audioStatus2;
		subchnl.cdsc_adr = qInfo.ctrl_addr;
		subchnl.cdsc_ctrl = qInfo.ctrl_addr >> 4;
		subchnl.cdsc_trk = bcd2bin2(qInfo.track);
		subchnl.cdsc_ind = bcd2bin2(qInfo.pointIndex);

		if (subchnl.cdsc_format == CDROM_LBA)
		{
			subchnl.cdsc_absaddr.lba = msf2hsg2(&qInfo.diskTime);
			subchnl.cdsc_reladdr.lba = msf2hsg2(&qInfo.trackTime);
		}

		else if (subchnl.cdsc_format == CDROM_MSF)
		{
			subchnl.cdsc_absaddr.msf.minute = bcd2bin2(qInfo.diskTime.min);
			subchnl.cdsc_absaddr.msf.second = bcd2bin2(qInfo.diskTime.sec);
			subchnl.cdsc_absaddr.msf.frame = bcd2bin2(qInfo.diskTime.frame);

			subchnl.cdsc_reladdr.msf.minute = bcd2bin2(qInfo.trackTime.min);
			subchnl.cdsc_reladdr.msf.second = bcd2bin2(qInfo.trackTime.sec);
			subchnl.cdsc_reladdr.msf.frame = bcd2bin2(qInfo.trackTime.frame);
		}

		else
			return -EINVAL;

		memcpy_tofs((void *) arg, &subchnl, sizeof subchnl);
		return 0;

	case CDROMVOLCTRL:   /* Volume control */
	/*
	 * This is not working yet.  Setting the volume by itself does
	 * nothing.  Following the 'set' by a 'play' results in zero
	 * volume.  Something to work on for the next release.
	 */
#if 0
		st = verify_area(VERIFY_READ, (void *) arg, sizeof(volctrl));
		if (st)
			return st;

		memcpy_fromfs(&volctrl, (char *) arg, sizeof(volctrl));
printk("VOL %d %d\n", volctrl.channel0 & 0xFF, volctrl.channel1 & 0xFF);
		outb(MCMD_SET_VOLUME, MCD2PORT(0));
		outb(volctrl.channel0, MCD2PORT(0));
		outb(0, MCD2PORT(0));
		outb(volctrl.channel1, MCD2PORT(0));
		outb(1, MCD2PORT(0));

		i = getMcdStatus2(MCD2_STATUS_DELAY);
		if (i < 0)
			return -EIO;

		{
			int a, b, c, d;

			getValue2(&a);
			getValue2(&b);
			getValue2(&c);
			getValue2(&d);
			printk("%02X %02X %02X %02X\n", a, b, c, d);
		}

		outb(0xF8, MCD2PORT(0));
		i = getMcdStatus2(MCD2_STATUS_DELAY);
		printk("F8 -> %02X\n", i & 0xFF);
#endif
		return 0;

	case CDROMEJECT:
 	       /* all drives can at least stop! */
		if (audioStatus2 == CDROM_AUDIO_PLAY) {
		  outb(MCMD_STOP, MCD2PORT(0));
		  i = getMcdStatus2(MCD2_STATUS_DELAY);
 		}
 
		audioStatus2 = CDROM_AUDIO_NO_STATUS;
 
		outb(MCMD_EJECT, MCD2PORT(0));
		/*
		 * the status (i) shows failure on all but the FX drives.
		 * But nothing we can do about that in software!
		 * So just read the status and forget it. - Jon.
		 */
		i = getMcdStatus2(MCD2_STATUS_DELAY);
		return 0;
	default:
		return -EINVAL;
	}
}


/*
 * Take care of the different block sizes between cdrom and Linux.
 * When Linux gets variable block sizes this will probably go away.
 */

static void
mcd2_transfer(void)
{
  if (CURRENT_VALID) {
    while (CURRENT -> nr_sectors) {
      int bn = CURRENT -> sector / 4;
      int i;
      for (i = 0; i < MCD2_BUF_SIZ && mcd2_buf_bn[i] != bn; ++i)
	;
      if (i < MCD2_BUF_SIZ) {
	int offs = (i * 4 + (CURRENT -> sector & 3)) * 512;
	int nr_sectors = 4 - (CURRENT -> sector & 3);
	if (mcd2_buf_out != i) {
	  mcd2_buf_out = i;
	  if (mcd2_buf_bn[i] != bn) {
	    mcd2_buf_out = -1;
	    continue;
	  }
	}
	if (nr_sectors > CURRENT -> nr_sectors)
	  nr_sectors = CURRENT -> nr_sectors;
	memcpy(CURRENT -> buffer, mcd2_buf + offs, nr_sectors * 512);
	CURRENT -> nr_sectors -= nr_sectors;
	CURRENT -> sector += nr_sectors;
	CURRENT -> buffer += nr_sectors * 512;
      } else {
	mcd2_buf_out = -1;
	break;
      }
    }
  }
}


/*
 * We only seem to get interrupts after an error.
 * Just take the interrupt and clear out the status reg.
 */

static void
mcd2_interrupt(int unused)
{
	int st;

	st = inb(MCD2PORT(1)) & 0xFF;
#ifdef TEST1
		printk("<int1-%02X>", st);
#endif
	if (!(st & MFL_STATUS))
	{
		st = inb(MCD2PORT(0)) & 0xFF;
#ifdef TEST1
		printk("<int0-%02X>", st);
#endif
		if ((st & 0xFF) != 0xFF)
		  mcd2_error = st ? st & 0xFF : -1;
	}
}


static void
do_mcd2_request(void)
{
#ifdef TEST2
  printk(" do_mcd2_request(%ld+%ld)\n", CURRENT -> sector, CURRENT -> nr_sectors);
#endif
  mcd2_transfer_is_active = 1;
  while (CURRENT_VALID) {
    if (CURRENT->bh) {
      if (!CURRENT->bh->b_lock)
	panic(DEVICE_NAME ": block not locked");
    }
    mcd2_transfer();
    if (CURRENT -> nr_sectors == 0) {
      end_request(1);
    } else {
      mcd2_buf_out = -1;		/* Want to read a block not in buffer */
      if (mcd2_state == MCD2_S_IDLE) {
	if (!tocUpToDate2) {
	  if (updateToc2() < 0) {
	    while (CURRENT_VALID)
	      end_request(0);
	    break;
	  }
	}
	mcd2_state = MCD2_S_START;
	McdTries2 = 5;
	SET_TIMER(mcd2_poll, 1);
      }
      break;
    }
  }
  mcd2_transfer_is_active = 0;
#ifdef TEST2
  printk(" do_mcd2_request ends\n");
#endif
}



static void
mcd2_poll(void)
{
  int st;


  if (mcd2_error) {
    if (mcd2_error & 0xA5) {
      printk("mcd2: I/O error 0x%02x", mcd2_error);
      if (mcd2_error & 0x80)
	printk(" (Door open)");
      if (mcd2_error & 0x20)
	printk(" (Disk changed)");
      if (mcd2_error & 0x04)
	printk(" (Read error)");
      printk("\n");
      mcd2_invalidate_buffers();
#ifdef WARN_IF_READ_FAILURE
      if (McdTries2 == 5)
	printk("mcd2: read of block %d failed\n", mcd2_next_bn);
#endif
      if (!McdTries2--) {
	printk("mcd2: read of block %d failed, giving up\n", mcd2_next_bn);
	if (mcd2_transfer_is_active) {
	  McdTries2 = 0;
	  goto ret;
	}
	if (CURRENT_VALID)
	  end_request(0);
	McdTries2 = 5;
      }
    }
    mcd2_error = 0;
    mcd2_state = MCD2_S_STOP;
  }



 immediately:
  switch (mcd2_state) {



  case MCD2_S_IDLE:
#ifdef TEST3
    printk("MCD2_S_IDLE\n");
#endif
    return;



  case MCD2_S_START:
#ifdef TEST3
    printk("MCD2_S_START\n");
#endif

    outb(MCMD_GET_STATUS, MCD2PORT(0));
    mcd2_state = mcd2_mode == 1 ? MCD2_S_READ : MCD2_S_MODE;
    McdTimeout2 = 3000;
    break;



  case MCD2_S_MODE:
#ifdef TEST3
    printk("MCD2_S_MODE\n");
#endif

    if ((st = mcd2Status()) != -1) {

      if (st & MST_DSK_CHG) {
	mcd2DiskChanged = 1;
	tocUpToDate2 = 0;
	mcd2_invalidate_buffers();
      }

    set_mode_immediately:

      if ((st & MST_DOOR_OPEN) || !(st & MST_READY)) {
	mcd2DiskChanged = 1;
	tocUpToDate2 = 0;
	if (mcd2_transfer_is_active) {
	  mcd2_state = MCD2_S_START;
	  goto immediately;
	}
	printk((st & MST_DOOR_OPEN) ? "mcd2: door open\n" : "mcd2: disk removed\n");
	mcd2_state = MCD2_S_IDLE;
	while (CURRENT_VALID)
	  end_request(0);
	return;
      }

      outb(MCMD_SET_MODE, MCD2PORT(0));
      outb(1, MCD2PORT(0));
      mcd2_mode = 1;
      mcd2_state = MCD2_S_READ;
      McdTimeout2 = 3000;

    }
    break;



  case MCD2_S_READ:
#ifdef TEST3
    printk("MCD2_S_READ\n");
#endif

    if ((st = mcd2Status()) != -1) {

      if (st & MST_DSK_CHG) {
	mcd2DiskChanged = 1;
	tocUpToDate2 = 0;
	mcd2_invalidate_buffers();
      }

    read_immediately:

      if ((st & MST_DOOR_OPEN) || !(st & MST_READY)) {
	mcd2DiskChanged = 1;
	tocUpToDate2 = 0;
	if (mcd2_transfer_is_active) {
	  mcd2_state = MCD2_S_START;
	  goto immediately;
	}
	printk((st & MST_DOOR_OPEN) ? "mcd2: door open\n" : "mcd2: disk removed\n");
	mcd2_state = MCD2_S_IDLE;
	while (CURRENT_VALID)
	  end_request(0);
	return;
      }

      if (CURRENT_VALID) {
	struct mcd2_Play_msf msf;
	mcd2_next_bn = CURRENT -> sector / 4;
	hsg2msf2(mcd2_next_bn, &msf.start);
	msf.end.min = ~0;
	msf.end.sec = ~0;
	msf.end.frame = ~0;
	sendMcdCmd2(MCMD_DATA_READ2, &msf);
	mcd2_state = MCD2_S_DATA;
	McdTimeout2 = READ_TIMEOUT;
      } else {
	mcd2_state = MCD2_S_STOP;
	goto immediately;
      }

    }
    break;


  case MCD2_S_DATA:
#ifdef TEST3
    printk("MCD2_S_DATA\n");
#endif

    st = inb(MCD2PORT(1)) & (MFL_STATUSorDATA);
  data_immediately:
#ifdef TEST5
    printk("Status %02x\n",st);
#endif
    switch (st) {

    case MFL_DATA:
#ifdef WARN_IF_READ_FAILURE
      if (McdTries2 == 5)
	printk("mcd2: read of block %d failed\n", mcd2_next_bn);
#endif
      if (!McdTries2--) {
	printk("mcd2: read of block %d failed, giving up\n", mcd2_next_bn);
	if (mcd2_transfer_is_active) {
	  McdTries2 = 0;
	  break;
	}
	if (CURRENT_VALID)
	  end_request(0);
	McdTries2 = 5;
      }
      mcd2_state = MCD2_S_START;
      McdTimeout2 = READ_TIMEOUT;
      goto immediately;

    case MFL_STATUSorDATA:
      break;

    default:
      McdTries2 = 5;
      if (!CURRENT_VALID && mcd2_buf_in == mcd2_buf_out) {
	mcd2_state = MCD2_S_STOP;
	goto immediately;
      }
      mcd2_buf_bn[mcd2_buf_in] = -1;
      READ_DATA(MCD2PORT(0), mcd2_buf + 2048 * mcd2_buf_in, 2048);
      mcd2_buf_bn[mcd2_buf_in] = mcd2_next_bn++;
      if (mcd2_buf_out == -1)
	mcd2_buf_out = mcd2_buf_in;
      mcd2_buf_in = mcd2_buf_in + 1 == MCD2_BUF_SIZ ? 0 : mcd2_buf_in + 1;
      if (!mcd2_transfer_is_active) {
	while (CURRENT_VALID) {
	  mcd2_transfer();
	  if (CURRENT -> nr_sectors == 0)
	    end_request(1);
	  else
	    break;
	}
      }

      if (CURRENT_VALID
	  && (CURRENT -> sector / 4 < mcd2_next_bn ||
	      CURRENT -> sector / 4 > mcd2_next_bn + 16)) {
	mcd2_state = MCD2_S_STOP;
	goto immediately;
      }
      McdTimeout2 = READ_TIMEOUT;
#ifdef DOUBLE_QUICK_ONLY
      if (MCMD_DATA_READ2 != MCMD_PLAY_READ)
#endif
      {
	int count= QUICK_LOOP_COUNT;
	while (count--) {
          QUICK_LOOP_DELAY;
	  if ((st = (inb(MCD2PORT(1))) & (MFL_STATUSorDATA)) != (MFL_STATUSorDATA)) {
#   ifdef TEST4
/*	    printk("Quickloop success at %d\n",QUICK_LOOP_COUNT-count); */
	    printk(" %d ",QUICK_LOOP_COUNT-count);
#   endif
	    goto data_immediately;
	  }
	}
#   ifdef TEST4
/*      printk("Quickloop ended at %d\n",QUICK_LOOP_COUNT); */
	printk("ended ");
#   endif
      }
      break;
    }
    break;



  case MCD2_S_STOP:
#ifdef TEST3
    printk("MCD2_S_STOP\n");
#endif

#ifdef WORK_AROUND_MITSUMI_BUG_93
    if (!mitsumi_bug_93_wait2)
      goto do_not_work_around_mitsumi_bug_93_1;

    McdTimeout2 = mitsumi_bug_93_wait2;
    mcd2_state = 9+3+1;
    break;

  case 9+3+1:
    if (McdTimeout2)
      break;

  do_not_work_around_mitsumi_bug_93_1:
#endif /* WORK_AROUND_MITSUMI_BUG_93 */

    outb(MCMD_STOP, MCD2PORT(0));

#ifdef WORK_AROUND_MITSUMI_BUG_92
    if ((inb(MCD2PORT(1)) & MFL_STATUSorDATA) == MFL_STATUS) {
      int i = 4096;
      do {
	inb(MCD2PORT(0));
      } while ((inb(MCD2PORT(1)) & MFL_STATUSorDATA) == MFL_STATUS && --i);
      outb(MCMD_STOP, MCD2PORT(0));
      if ((inb(MCD2PORT(1)) & MFL_STATUSorDATA) == MFL_STATUS) {
	i = 4096;
	do {
	  inb(MCD2PORT(0));
	} while ((inb(MCD2PORT(1)) & MFL_STATUSorDATA) == MFL_STATUS && --i);
	outb(MCMD_STOP, MCD2PORT(0));
      }
    }
#endif /* WORK_AROUND_MITSUMI_BUG_92 */

    mcd2_state = MCD2_S_STOPPING;
    McdTimeout2 = 1000;
    break;

  case MCD2_S_STOPPING:
#ifdef TEST3
    printk("MCD2_S_STOPPING\n");
#endif

    if ((st = mcd2Status()) == -1 && McdTimeout2)
      break;

    if ((st != -1) && (st & MST_DSK_CHG)) {
      mcd2DiskChanged = 1;
      tocUpToDate2 = 0;
      mcd2_invalidate_buffers();
    }

#ifdef WORK_AROUND_MITSUMI_BUG_93
    if (!mitsumi_bug_93_wait2)
      goto do_not_work_around_mitsumi_bug_93_2;

    McdTimeout2 = mitsumi_bug_93_wait2;
    mcd2_state = 9+3+2;
    break;

  case 9+3+2:
    if (McdTimeout2)
      break;

    st = -1;

  do_not_work_around_mitsumi_bug_93_2:
#endif /* WORK_AROUND_MITSUMI_BUG_93 */

#ifdef TEST3
    printk("CURRENT_VALID %d mcd2_mode %d\n",
	   CURRENT_VALID, mcd2_mode);
#endif

    if (CURRENT_VALID) {
      if (st != -1) {
	if (mcd2_mode == 1)
	  goto read_immediately;
	else
	  goto set_mode_immediately;
      } else {
	mcd2_state = MCD2_S_START;
	McdTimeout2 = 1;
      }
    } else {
      mcd2_state = MCD2_S_IDLE;
      return;
    }
    break;

  default:
    printk("mcd2: invalid state %d\n", mcd2_state);
    return;
  }

 ret:
  if (!McdTimeout2--) {
    printk("mcd2: timeout in state %d\n", mcd2_state);
    mcd2_state = MCD2_S_STOP;
  }

  SET_TIMER(mcd2_poll, 1);
}



static void
mcd2_invalidate_buffers(void)
{
  int i;
  for (i = 0; i < MCD2_BUF_SIZ; ++i)
    mcd2_buf_bn[i] = -1;
  mcd2_buf_out = -1;
}


/*
 * Open the device special file.  Check that a disk is in.
 */

int
mcd2_open(struct inode *ip, struct file *fp)
{
	int st;

	if (mcd2Present == 0)
		return -ENXIO;			/* no hardware */
	
	if (fp->f_mode & 2)			/* write access? */
		return -EROFS;

	if (!mcd2_open_count && mcd2_state == MCD2_S_IDLE) {

	mcd2_invalidate_buffers();

	st = statusCmd2();			/* check drive status */
	if (st == -1)
		return -EIO;			/* drive doesn't respond */

	if ((st & MST_READY) == 0)		/* no disk in drive */
	{
		printk("mcd2: no disk in drive\n");
		return -EIO;
	}

	if (updateToc2() < 0)
		return -EIO;

	}
	++mcd2_open_count;

	return 0;
}


/*
 * On close, we flush all mcd2 blocks from the buffer cache.
 */

static void
mcd2_release(struct inode * inode, struct file * file)
{
  if (!--mcd2_open_count) {
	mcd2_invalidate_buffers();
	sync_dev(inode->i_rdev);
	invalidate_buffers(inode -> i_rdev);
  }
}


static struct file_operations mcd2_fops = {
	NULL,			/* lseek - default */
	block_read,		/* read - general block-dev read */
	block_write,		/* write - general block-dev write */
	NULL,			/* readdir - bad */
	NULL,			/* select */
	mcd2_ioctl,		/* ioctl */
	NULL,			/* mmap */
	mcd2_open,		/* open */
	mcd2_release,		/* release */
	NULL,			/* fsync */
	NULL,			/* fasync */
	check_mcd2_change,	/* media change */
	NULL			/* revalidate */
};


/*
 * Test for presence of drive and initialize it.  Called at boot time.
 */

unsigned long
mcd2_init(unsigned long mem_start, unsigned long mem_end)
{
	int count;
	unsigned char result[3];

	if (mcd2_port <= 0 || mcd2_irq <= 0) {
	  printk("skip mcd2_init\n");
	  return mem_start;
	}

	printk("mcd2=0x%x,%d: ", mcd2_port, mcd2_irq);

	if (register_blkdev(MAJOR_NR, "mcd2", &mcd2_fops) != 0)
	{
		printk("Unable to get major %d for Mitsumi CD-ROM2\n",
		       MAJOR_NR);
		return mem_start;
	}

	if (check_region(mcd2_port, 4)) {
	  printk("Init failed, I/O port (%X) already in use\n",
		 mcd2_port);
	  return mem_start;
	}
	  
	blk_dev[MAJOR_NR].request_fn = DEVICE_REQUEST;
	read_ahead[MAJOR_NR] = 4;

	/* check for card */

	outb(0, MCD2PORT(1));			/* send reset */
	for (count = 0; count < 2000000; count++)
		(void) inb(MCD2PORT(1));		/* delay a bit */

	outb(0x40, MCD2PORT(0));			/* send get-stat cmd */
	for (count = 0; count < 2000000; count++)
		if (!(inb(MCD2PORT(1)) & MFL_STATUS))
			break;

	if (count >= 2000000) {
		printk("Init failed. No mcd2 device at 0x%x irq %d\n",
		     mcd2_port, mcd2_irq);
		return mem_start;
	}
	count = inb(MCD2PORT(0));		/* pick up the status */
	
	outb(MCMD_GET_VERSION,MCD2PORT(0));
	for(count=0;count<3;count++)
		if(getValue2(result+count)) {
			printk("mitsumi get version failed at 0x%d\n",
			       mcd2_port);
			return mem_start;
		}	

	if (result[0] == result[1] && result[1] == result[2])
		return mem_start;

	printk("Mitsumi status, type and version : %02X %c %x\n",
	       result[0],result[1],result[2]);

	if (result[1] == 'D') MCMD_DATA_READ2= 0xC1;

	mcd2Version=result[2];

	if (mcd2Version >=4)
		outb(4,MCD2PORT(2)); 	/* magic happens */

	/* don't get the IRQ until we know for sure the drive is there */

	if (request_irq(mcd2_irq, mcd2_interrupt, SA_INTERRUPT, "Mitsumi CD2"))
	{
		printk("Unable to get IRQ%d for Mitsumi CD-ROM2\n", mcd2_irq);
		return mem_start;
	}
	snarf_region(mcd2_port, 4);

	outb(MCMD_CONFIG_DRIVE, MCD2PORT(0));
	outb(0x02,MCD2PORT(0));
	outb(0x00,MCD2PORT(0));
	getValue2(result);

	outb(MCMD_CONFIG_DRIVE, MCD2PORT(0));
	outb(0x10,MCD2PORT(0));
	outb(0x04,MCD2PORT(0));
	getValue2(result);

	mcd2_invalidate_buffers();
	mcd2Present = 1;
	printk("\n");
	return mem_start;
}


static void
hsg2msf2(long hsg, struct msf *msf)
{
	hsg += 150;
	msf -> min = hsg / 4500;
	hsg %= 4500;
	msf -> sec = hsg / 75;
	msf -> frame = hsg % 75;

	bin2bcd2(&msf -> min);		/* convert to BCD */
	bin2bcd2(&msf -> sec);
	bin2bcd2(&msf -> frame);
}


static void
bin2bcd2(unsigned char *p)
{
	int u, t;

	u = *p % 10;
	t = *p / 10;
	*p = u | (t << 4);
}

static int
bcd2bin2(unsigned char bcd)
{
	return (bcd >> 4) * 10 + (bcd & 0xF);
}


/*
 * See if a status is ready from the drive and return it
 * if it is ready.
 */

static int
mcd2Status(void)
{
	int i;
	int st;

	st = inb(MCD2PORT(1)) & MFL_STATUS;
	if (!st)
	{
		i = inb(MCD2PORT(0)) & 0xFF;
		return i;
	}
	else
		return -1;
}


/*
 * Send a play or read command to the drive
 */

static void
sendMcdCmd2(int cmd, struct mcd2_Play_msf *params)
{
	outb(cmd, MCD2PORT(0));
	outb(params -> start.min, MCD2PORT(0));
	outb(params -> start.sec, MCD2PORT(0));
	outb(params -> start.frame, MCD2PORT(0));
	outb(params -> end.min, MCD2PORT(0));
	outb(params -> end.sec, MCD2PORT(0));
	outb(params -> end.frame, MCD2PORT(0));
}


/*
 * Timer interrupt routine to test for status ready from the drive.
 * (see the next routine)
 */

static void
mcd2StatTimer(void)
{
	if (!(inb(MCD2PORT(1)) & MFL_STATUS))
	{
		wake_up(&mcd2_waitq);
		return;
	}

	McdTimeout2--;
	if (McdTimeout2 <= 0)
	{
		wake_up(&mcd2_waitq);
		return;
	}

	SET_TIMER(mcd2StatTimer, 1);
}


/*
 * Wait for a status to be returned from the drive.  The actual test
 * (see routine above) is done by the timer interrupt to avoid
 * excessive rescheduling.
 */

static int
getMcdStatus2(int timeout)
{
	int st;

	McdTimeout2 = timeout;
	SET_TIMER(mcd2StatTimer, 1);
	sleep_on(&mcd2_waitq);
	if (McdTimeout2 <= 0)
		return -1;

	st = inb(MCD2PORT(0)) & 0xFF;
	if (st == 0xFF)
		return -1;

	if ((st & MST_BUSY) == 0 && audioStatus2 == CDROM_AUDIO_PLAY)
		/* XXX might be an error? look at q-channel? */
		audioStatus2 = CDROM_AUDIO_COMPLETED;

	if (st & MST_DSK_CHG)
	{
		mcd2DiskChanged = 1;
		tocUpToDate2 = 0;
		audioStatus2 = CDROM_AUDIO_NO_STATUS;
	}

	return st;
}


/*
 * Read a value from the drive.  Should return quickly, so a busy wait
 * is used to avoid excessive rescheduling.
 */

static int
getValue2(unsigned char *result)
{
	int count;
	int s;

	for (count = 0; count < 2000; count++)
		if (!(inb(MCD2PORT(1)) & MFL_STATUS))
			break;

	if (count >= 2000)
	{
		printk("mcd2: getValue2 timeout\n");
		return -1;
	}

	s = inb(MCD2PORT(0)) & 0xFF;
	*result = (unsigned char) s;
	return 0;
}


/*
 * Read the current Q-channel info.  Also used for reading the
 * table of contents.
 */

int
GetQChannelInfo2(struct mcd2_Toc *qp)
{
	unsigned char notUsed;
	int retry;

	for (retry = 0; retry < MCD2_RETRY_ATTEMPTS; retry++)
	{
		outb(MCMD_GET_Q_CHANNEL, MCD2PORT(0));
		if (getMcdStatus2(MCD2_STATUS_DELAY) != -1)
			break;
	}

	if (retry >= MCD2_RETRY_ATTEMPTS)
		return -1;

	if (getValue2(&qp -> ctrl_addr) < 0) return -1;
	if (getValue2(&qp -> track) < 0) return -1;
	if (getValue2(&qp -> pointIndex) < 0) return -1;
	if (getValue2(&qp -> trackTime.min) < 0) return -1;
	if (getValue2(&qp -> trackTime.sec) < 0) return -1;
	if (getValue2(&qp -> trackTime.frame) < 0) return -1;
	if (getValue2(&notUsed) < 0) return -1;
	if (getValue2(&qp -> diskTime.min) < 0) return -1;
	if (getValue2(&qp -> diskTime.sec) < 0) return -1;
	if (getValue2(&qp -> diskTime.frame) < 0) return -1;

	return 0;
}


/*
 * Read the table of contents (TOC) and TOC header if necessary
 */

static int
updateToc2()
{
	if (tocUpToDate2)
		return 0;

	if (GetDiskInfo2() < 0)
		return -EIO;

	if (GetToc2() < 0)
		return -EIO;

	tocUpToDate2 = 1;
	return 0;
}


/*
 * Read the table of contents header
 */

static int
GetDiskInfo2()
{
	int retry;

	for (retry = 0; retry < MCD2_RETRY_ATTEMPTS; retry++)
	{
		outb(MCMD_GET_DISK_INFO, MCD2PORT(0));
		if (getMcdStatus2(MCD2_STATUS_DELAY) != -1)
			break;
	}

	if (retry >= MCD2_RETRY_ATTEMPTS)
		return -1;

	if (getValue2(&DiskInfo2.first) < 0) return -1;
	if (getValue2(&DiskInfo2.last) < 0) return -1;

	DiskInfo2.first = bcd2bin2(DiskInfo2.first);
	DiskInfo2.last = bcd2bin2(DiskInfo2.last);

	if (getValue2(&DiskInfo2.diskLength.min) < 0) return -1;
	if (getValue2(&DiskInfo2.diskLength.sec) < 0) return -1;
	if (getValue2(&DiskInfo2.diskLength.frame) < 0) return -1;
	if (getValue2(&DiskInfo2.firstTrack.min) < 0) return -1;
	if (getValue2(&DiskInfo2.firstTrack.sec) < 0) return -1;
	if (getValue2(&DiskInfo2.firstTrack.frame) < 0) return -1;

#ifdef MCD2_DEBUG
printk("Disk Info: first %d last %d length %02x:%02x.%02x first %02x:%02x.%02x\n",
	DiskInfo2.first,
	DiskInfo2.last,
	DiskInfo2.diskLength.min,
	DiskInfo2.diskLength.sec,
	DiskInfo2.diskLength.frame,
	DiskInfo2.firstTrack.min,
	DiskInfo2.firstTrack.sec,
	DiskInfo2.firstTrack.frame);
#endif

	return 0;
}


/*
 * Read the table of contents (TOC)
 */

static int
GetToc2()
{
	int i, px;
	int limit;
	int retry;
	struct mcd2_Toc qInfo;

	for (i = 0; i < MAX_TRACKS; i++)
		Toc2[i].pointIndex = 0;

	i = DiskInfo2.last + 3;

	for (retry = 0; retry < MCD2_RETRY_ATTEMPTS; retry++)
	{
		outb(MCMD_STOP, MCD2PORT(0));
		if (getMcdStatus2(MCD2_STATUS_DELAY) != -1)
			break;
	}

	if (retry >= MCD2_RETRY_ATTEMPTS)
		return -1;

	for (retry = 0; retry < MCD2_RETRY_ATTEMPTS; retry++)
	{
		outb(MCMD_SET_MODE, MCD2PORT(0));
		outb(0x05, MCD2PORT(0));			/* mode: toc */
		mcd2_mode = 0x05;
		if (getMcdStatus2(MCD2_STATUS_DELAY) != -1)
			break;
	}

	if (retry >= MCD2_RETRY_ATTEMPTS)
		return -1;

	for (limit = 300; limit > 0; limit--)
	{
		if (GetQChannelInfo2(&qInfo) < 0)
			break;

		px = bcd2bin2(qInfo.pointIndex);
		if (px > 0 && px < MAX_TRACKS && qInfo.track == 0)
			if (Toc2[px].pointIndex == 0)
			{
				Toc2[px] = qInfo;
				i--;
			}

		if (i <= 0)
			break;
	}

	Toc2[DiskInfo2.last + 1].diskTime = DiskInfo2.diskLength;

	for (retry = 0; retry < MCD2_RETRY_ATTEMPTS; retry++)
	{
		outb(MCMD_SET_MODE, MCD2PORT(0));
		outb(0x01, MCD2PORT(0));
		mcd2_mode = 1;
		if (getMcdStatus2(MCD2_STATUS_DELAY) != -1)
                        break;
	}

#ifdef MCD2_DEBUG
for (i = 1; i <= DiskInfo2.last; i++)
printk("i = %2d ctl-adr = %02X track %2d px %02X %02X:%02X.%02X    %02X:%02X.%02X\n",
i, Toc2[i].ctrl_addr, Toc2[i].track, Toc2[i].pointIndex,
Toc2[i].trackTime.min, Toc2[i].trackTime.sec, Toc2[i].trackTime.frame,
Toc2[i].diskTime.min, Toc2[i].diskTime.sec, Toc2[i].diskTime.frame);
for (i = 100; i < 103; i++)
printk("i = %2d ctl-adr = %02X track %2d px %02X %02X:%02X.%02X    %02X:%02X.%02X\n",
i, Toc2[i].ctrl_addr, Toc2[i].track, Toc2[i].pointIndex,
Toc2[i].trackTime.min, Toc2[i].trackTime.sec, Toc2[i].trackTime.frame,
Toc2[i].diskTime.min, Toc2[i].diskTime.sec, Toc2[i].diskTime.frame);
#endif

	return limit > 0 ? 0 : -1;
}

