/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * MODULE : mx_help_target.C
 *
 * AUTHOR : David Miller
 *
 * DESCRIPTION: 
 * Module mx_hp_target.C 
 *
 *
 */

#include "mx_help.h"
#include "mx_device.h"
#include <mx_dialog_man.h>
#include <mx_hsearch_d.h>

extern mx_dialog_manager *global_dialog_manager;

mx_font *mx_help_target::font ;
mx_char_style *mx_help_target::cs1 ;
mx_char_style *mx_help_target::cs2 ;
mx_print_d   *mx_help_target::print_d;
mx_hsearch_d *mx_help_target::search_d;


/*-------------------------------------------------
 * FUNCTION: mx_help_target::mx_help_target
 *
 * DESCRIPTION: 
 * 
 *
 */

mx_help_target::mx_help_target(int &err)
{
    int iitem ;
    
    err = MX_ERROR_OK ;

    window            = NULL ;
    helpFile          = NULL ;
    allList           = NULL ;
    currentHeadOffset = 0 ;
    
    // Initialise file items to NULL 
    for(iitem=0;iitem<MX_HELP_MAX_HOLD_FILES;iitem++) helpFilePtrs[iitem] = NULL ;

    sprintf(directory,"%s/help",global_maxhome) ;

    if (font == NULL)
    {
		mx_colour colour;

		colour.name = "blue";

        font = new mx_font(err, get_default_roman_font(), 12, mx_normal);
        MX_ERROR_CHECK(err);
        
        cs1 = new mx_char_style(*font);
        cs2 = new mx_char_style(*font);
		cs2->colour = colour;
      
    }

    setLineAscender(cs1->get_ascender()) ;
        
    setLineHeight(cs1->get_ascender() + cs1->get_descender()) ;

    maxLineLength = 150 ;
    minLineLength = 70 ;
    idealLineCharLength = 60 ;

    leftMargin      = 4.0 ;
    lineSeparator   = 2.0 ;
    pixmapSize.x    = 20.0 ;
    pixmapSize.y    = 20.0 ;
    
    return ;
  abort:
    return ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_target::~mx_help_target
 *
 * DESCRIPTION: 
 * 
 *
 */

mx_help_target::~mx_help_target() 
{
    int iitem ;

    // delete the help files 
    for(iitem=0;iitem<MX_HELP_MAX_HOLD_FILES;iitem++)
    { 
        delete helpFilePtrs[iitem] ;
    }

    delete font ;
    delete cs1 ;
    delete cs2 ;
    delete allList ;
}


/*-------------------------------------------------
 * FUNCTION: mx_help_target::setFile
 *
 * DESCRIPTION: 
 * 
 *
 */

void mx_help_target::setFile(int      &err,
                             char     *fileName,
                             bool     newFile,
                             mx_point &topLeft,
                             bool     *backward,
                             bool     *forward) 
{
    bool found ;
    int  iitem ;
    char *fullFileName ;
    
    err = MX_ERROR_OK ;
    found = FALSE;
     
    // Look for the file in the existing set of files 

    for(iitem=0;iitem<MX_HELP_MAX_HOLD_FILES;iitem++) 
    {
        if(helpFilePtrs[iitem] != NULL) 
        {
            if( strcmp((helpFilePtrs[iitem])->getFileName(),fileName) == 0) 
            {
                helpFile = helpFilePtrs[iitem] ;
                found = TRUE ;
                break ;
            }
        }
    }
    
    if(!found) 
    {
        fullFileName = new char[strlen(directory) + strlen(fileName) + 2] ;
        
        sprintf(fullFileName,"%s/%s",directory,fileName) ;
        
        helpFile = new mx_help_file(err,fullFileName,fileName) ;
        MX_ERROR_CHECK(err) ;           
        delete [] fullFileName;
        
        helpFile->setHelpTarget(this) ;
        
        helpFile->read(err) ;
        MX_ERROR_CHECK(err) ;
        
        // Replace the existing file in the 
        
        delete helpFilePtrs[currentHeadOffset] ;
        helpFilePtrs[currentHeadOffset] = helpFile ;
        
        currentHeadOffset = currentHeadOffset + 1 ;
        if(currentHeadOffset == MX_HELP_MAX_HOLD_FILES) currentHeadOffset = 0 ;
        
    }

    if((allList == currentElement) && (allList != NULL)) 
    {
        allList->setPosition(topLeft) ;    
    }
    
    if(newFile) 
    {
        // add to all list of all files 
        
        mx_help_list *newList = new mx_help_list(fileName,allList) ;

        allList        = newList ;
        currentElement = newList ;
    }
   
    *backward  = (currentElement->getPrev() != NULL) ;
    *forward   = (currentElement->getNext() != NULL) ;

    return ;
abort:
    return ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_target::draw
 *
 * DESCRIPTION: 
 * 
 *
 */

void mx_help_target::draw(int &err,
                          mx_draw_event &event) 
{
    int iline,nlines ;
    mx_help_line *thisLine ;
    mx_doc_coord_t *linePos ;
    mx_device *device = event.getDevice();
    double tly,bry ;
    mx_doc_coord_t linkPos,stringStartPos ;
    int nlinks ;
    int linkOffset;
    char *lineString ;
    int stringStart ;
    mx_file_link *thisLink ;
    char savedValue ;
    int endLink ;
    char *outputString ;
    
    err = MX_ERROR_OK ;
    
    if(helpFile == NULL) return ;

    mx_doc_coord_t tl = event.getTL() ;
    mx_doc_coord_t br = event.getBR() ;
    
    if(br.sheet_number == 1) 
    {
        br.p.y += getTotalSize(err).y ;
        MX_ERROR_CHECK(err) ;
    }

    nlines = helpFile->getNlines() ;
    iline = 0 ;
    
    tly = tl.p.y - DMAX(lineHeight + lineSeparator,pixmapSize.y) ;
    bry = br.p.y + lineHeight + lineSeparator ;

    while(iline < nlines) 
    {
        thisLine = helpFile->getLine(err,iline) ;
        MX_ERROR_CHECK(err) ;
    
        linePos = thisLine->getLinePos() ;
                
        if(linePos->p.y > bry) break ;
        
        if(linePos->p.y > tly) 
        {
            linkOffset      = thisLine->getLinkOffset() ;
            endLink         = linkOffset + thisLine->getNlinks();
            lineString      = thisLine->getLine() ;
            stringStart     = 0 ;
            stringStartPos  = *linePos ;
            
            while(linkOffset < endLink) 
            {
                thisLink = helpFile->getLink(linkOffset) ;
                linkOffset++ ;

                if(thisLink->hasPixmap) 
                {
                    mx_doc_coord_t topLeft ;
                    mx_doc_coord_t bottomRight ;
                    
                    topLeft.p     = thisLink->extent.topLeft() ;
                    bottomRight.p = thisLink->extent.bottomRight() ;

                    device->fillRect(err,topLeft,bottomRight,defaultAngle) ;
                    MX_ERROR_CHECK(err) ;
                }
                else
                {
                    savedValue = lineString[thisLink->start] ;
                    lineString[thisLink->start] = 0 ;

                    /* output text up to link */
                    
                    outputString = lineString + stringStart ;
                    
                    if(*outputString != 0)
                    {
                        device->drawText(err,
                                         lineString + stringStart,
                                         stringStartPos,
                                         *cs1) ;
                        MX_ERROR_CHECK(err) ;                
                    }
                

                    lineString[thisLink->start] = savedValue ;
                
                    /* now output link text */

                    savedValue = lineString[thisLink->end + 1] ;
                    lineString[thisLink->end + 1] = 0 ;
                    
                    linkPos.p.x = thisLink->extent.xb ;
                    linkPos.p.y = thisLink->extent.yt ;
                    
                    outputString = lineString + thisLink->start ;
                    
                    if(*outputString != 0) 
                    {
                        device->drawText(err,
                                         lineString + thisLink->start,
                                         linkPos,
                                         *cs2) ;
                        MX_ERROR_CHECK(err) ;        
                    }

                    lineString[thisLink->end + 1] = savedValue ;

                    stringStartPos.p.x = thisLink->extent.xt ;
                    stringStartPos.p.y = thisLink->extent.yt ;
                    stringStart        = thisLink->end + 1 ;
                }
            }
            

            /* now draw what's left */

            device->drawText(err,
                             lineString + stringStart,
                             stringStartPos,
                             *cs1) ;
            MX_ERROR_CHECK(err) ;                
        }
        
        iline++ ;
    }
    
    return ;
abort:
    return ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_target::buttonPress
 *
 * DESCRIPTION: 
 * 
 *
 */

void mx_help_target::buttonPress(int &err,
                                 mx_button_event &event) 
{
    int ilink,nlinks ;
    mx_file_link *thisLink ;
    mx_doc_coord_t pt = event.getCoord() ;
    
    err = MX_ERROR_OK ;

    if(helpFile == NULL) return ;
    
    nlinks = helpFile->getNlinks() ;
    
    for(ilink = 0;ilink<nlinks;ilink++) 
    {
        thisLink = helpFile->getLink(ilink) ;

        // gone too far 
        if(thisLink->extent.yb > pt.p.y) break ;
        
        if( (thisLink->extent.yt > pt.p.y) && 
            (thisLink->extent.xb < pt.p.x) && 
            (thisLink->extent.xt > pt.p.x) ) 
        {
            window->setFile(err,thisLink->fileName,NULL) ;
            MX_ERROR_CHECK(err);
	    
	    break ;
        }
    }

    return ;
abort:
    return ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_target::backward
 *
 * DESCRIPTION: 
 * 
 *
 */
        

void mx_help_target::backward(int &err) 
{
    err = MX_ERROR_OK ;

    if( (currentElement == NULL) || (currentElement->getPrev() == NULL)) 
        return ;

    if(currentElement == allList) 
    {
        mx_point z = window->get_frame()->getTopLeft() ;
        currentElement->setPosition(z);
    }

    currentElement = currentElement->getPrev() ;

    mx_point currentPoint = currentElement->getPosition() ;
    window->setFile(err,
                    currentElement->getFileName(),
                    &currentPoint) ;
    MX_ERROR_CHECK(err);
    
    return ;
abort:
    return ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_target::forward
 *
 * DESCRIPTION: 
 * 
 *
 */
        

void mx_help_target::forward(int &err) 
{
    err = MX_ERROR_OK ;

    if( (currentElement == NULL) || (currentElement->getNext() == NULL)) return ;

    currentElement = currentElement->getNext() ;
    
    mx_point currentPoint = currentElement->getPosition() ;
    window->setFile(err,
                    currentElement->getFileName(),
                    &currentPoint) ;
    MX_ERROR_CHECK(err);
    
    return ;
abort:
    return ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_target::find
 *
 * DESCRIPTION: 
 * 
 *
 */

void mx_help_target::find(int &err)
{
    err = MX_ERROR_OK;
    
    if (search_d == NULL)
    {
        search_d = new mx_hsearch_d(window->get_widget());
    }
    
    search_d->centre();
    search_d->activate();
    if (search_d->run_modal() == yes_e)
    {
        window->setFile(err, search_d->selected_topic,NULL) ;
        MX_ERROR_CHECK(err);
    }
    search_d->deactivate();
    return;
    
  abort:
    return ;
}

/*-------------------------------------------------
 * FUNCTION: mx_help_target::print
 *
 * DESCRIPTION: 
 * 
 *
 */

void mx_help_target::print()
{
    char *file;
    char file_name[MAX_PATH_LEN];
    char exec_name[MAX_PATH_LEN];
    char num[10];
    char pname[MAX_PATH_LEN + 2];

	const char *current_printer;
	int  current_x_res;
	int  current_y_res;
	int  err = MX_ERROR_OK;
    
    if(print_d == NULL)
    {
        print_d = new mx_print_d(window->get_widget());
    }
    
    print_d->centre();
    

	current_printer = global_user_config->get_default_string(err, "printer", "lp");
	MX_ERROR_CLEAR(err);

	current_x_res = global_user_config->get_default_int(err, "printer_x_res", 300);
	MX_ERROR_CLEAR(err);

	current_y_res = global_user_config->get_default_int(err, "printer_y_res", 300);
	MX_ERROR_CLEAR(err);

    print_d->activate(1, (char *)current_printer, current_x_res, current_y_res, FALSE);
  
    if (print_d->run_modal() == yes_e)
    {
        file = helpFile->getFileName();
        if (file != NULL)
        {
            file[strlen(file) - 4] = 0;
            sprintf(file_name,"%s/help/%s.ps", global_maxhome, file) ;
            sprintf(exec_name,"%s/bin/lpr", global_maxhome) ;
            sprintf(num, "-#%d", print_d->num_copies) ;
            sprintf(pname, "-P%s", print_d->selected_printer) ;
            if (vfork() == 0)
            {
                execl(exec_name, "lpr", num, pname, file_name, NULL);
                
              // only get here if a nasty error happens
                exit(1);
            }
        }
    }
    print_d->deactivate();
    
    return ;
}











