/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * Module : mx_help.h
 *
 * AUTHOR : David Miller
 *
 * DESCRIPTION: 
 * Module mx_help.h 
 *
 *
 */

#ifndef MXHELP_H
#define MXHELP_H

#define DMAX(a,b) ((a)>(b) ? (a) : (b)) 
#define DMIN(a,b) ((b)>(a) ? (a) : (b)) 


#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>

#include <mx.h>


#include "rtf.h"
#include "mx_stsbar.h"
#include "mx_menubar.h"
#include "mx_xframe.h"
#include "mx_print_d.h"
#include "mx_hsearch_d.h"
#include "mx_font.h"
#include "mx_char_style.h"

class mx_editor;

#define MX_HELP_MAX_HOLD_FILES 10

typedef enum
{
        mx_line_left_justify_e,
        mx_line_right_justify_e,
        mx_line_centre_justify_e,
        mx_line_full_justify_e
}
mx_line_justify_t;


/*-------------------------------------------------
 * CLASS: mx_file_link
 *
 * DESCRIPTION: The link between help files
 * 
 *
 */

class mx_file_link 
{
public:
   int             start ;
   int             end ;
   char            fileName[13] ;
   mx_rect         extent ;
   bool            hasPixmap ;
protected:
private:
} ;

/*-------------------------------------------------
 * CLASS: mx_help_line
 *
 * DESCRIPTION: 
 * 
 *
 */

class mx_help_line
{
public:
    mx_help_line() ;
    ~mx_help_line() ;
    
    void setLine(int               &err,
                 int               lineSize, 
                 char              *lineStart,
                 double            indentSize,
                 double            fileOffset, 
                 mx_line_justify_t justification,
                 int               nlinks,
                 int               linkOffset) ;
    
    inline mx_doc_coord_t  *getLinePos(){return &linePos;};
    inline int     getJustify(){return justification;};
    inline char    *getLine(){return line;};
    inline int     getLineSize(){return lineSize;} ;

    inline int getLinkOffset(){return linkOffset;};
    inline int getNlinks(){return nlinks;};

protected:
private:
   char              *line ;
   int               lineSize ;
   mx_line_justify_t justification;
   mx_doc_coord_t    linePos ;
   int               nlinks ;
   int               linkOffset ;
} ;


class mx_help_target ;

/*-------------------------------------------------
 * CLASS: mx_help_file
 *
 * DESCRIPTION: 
 * 
 *
 */

class mx_help_file : public mx_rtf
{
public:

    mx_help_file(int &err,char *filename,char *filename2) ;   
    ~mx_help_file() ;
    inline char *getFileName(){return fileName;} ;
    
    inline void setHelpTarget(mx_help_target *t){helpTarget=t;};
    
    inline mx_point &getFileSize() {return fileSize;};
   
    inline int getNlines(){return nlines;};

    mx_help_line *getLine(int &err,
                          int iline) ;

    inline int          getNlinks(){return nlinks;};
    inline mx_file_link *getLink(int ilink) { return links + ilink ;};

    inline mx_point getFrameTopLeft(){return frameTopLeft;};
    inline void     setFrameTopLeft(mx_point &ip){frameTopLeft=ip;};
protected:
private:

   char *fileName ;
   char *fullFileName ;

   mx_point fileSize ;
   mx_point frameTopLeft ;

   bool processingHidden ;
   bool gettingLinkText ;

   int  paraBufferSize ;
   int  allocParaBufferSize ;
   char *paraBuffer ;

   int  hiddenBufferSize ;
   int  allocHiddenBufferSize ;
   char *hiddenBuffer ;

   // the number of lines in the document
   int           currentLine ;
   int           nlines ;
   int           nallocLines ;
   mx_help_line  **lines ;

   // the links in the file
   int           currentLink ;
   int           nlinks ;
   int           nallocLinks ;
   mx_file_link  *links ;

   mx_help_target *helpTarget ;

   void processMainOutput(int       &err,
                          RTFSgroup *group) ;

   void processNewParagraph(int                 &err,
                            RTFSgroup           *group,
                            rtfDestinationType  destination) ;
   void processNewTab(int                 &err,
                      RTFSgroup           *group,
                      rtfDestinationType  destination) ;
   void processSpecialCharacter(int                 &err,
                                RTFSgroup           *group,
                                char                 special,
                                rtfDestinationType  destination) ;

   void addToParagraph(int  &err,
                       char *chars,
                       int  charLen) ;

   void addToHidden(int  &err,
                    char *chars,        
                    int  charLen) ;

   void processHiddenBuffer(int &err) ;

   mx_point getIndentSize() ;

   void getLineLinks(int  lineStartOffset,
                   int   &startLink,
                   int   &nlineLinks,
                   int   &lineSize) ;

   void processLineLinks(int  lineStartOffset,
                         int  startLink,
                         int  nlineLinks,
                         char *line,
                         double xlineStart,
                         double ylineStart) ;

   void splitParagraphLines(int &err,
                            mx_line_justify_t justification) ;

   mx_file_link *nextFileLink() ;
   
   mx_help_line *nextHelpLine() ;
   
} ;

/*-------------------------------------------------
 * CLASS: mx_help_toolbar
 *
 * DESCRIPTION: 
 * 
 *
 */

class mx_help_toolbar : public mx_bar 
{
  public:
    mx_help_toolbar(mx_window *aWindow);
 
    static void activateCB(Widget, XtPointer, XtPointer);
    static void      armCB(Widget, XtPointer, XtPointer);
    static void   disarmCB(Widget, XtPointer, XtPointer);

    void set_forward_sensitive(bool isSensitive);
    void set_backward_sensitive(bool isSensitive);
    
  private:
    Widget backButton ;
    Widget forwardButton ;
    Widget findButton, printButton, closeButton;
};

/*-------------------------------------------------
 * CLASS: mx_help_menubar
 *
 * DESCRIPTION: 
 * 
 *
 */


class mx_help_menubar : public mx_menubar {

  public:
    mx_help_menubar(mx_window * parent);
    
    void set_forward_sensitive(bool isSensitive);
    void set_backward_sensitive(bool isSensitive);
    
  private:
    
    static void fileCB(Widget myWidget, XtPointer thisPtr, XtPointer callData);
    
    mx_menu_item  *forward_button;
    mx_menu_item  *backward_button;
    
};

/*-------------------------------------------------
 * CLASS: mx_help_window
 *
 * DESCRIPTION: 
 * 
 *
 */


class mx_help_window : public mx_window
{
  public:
    
    mx_help_window(int &err,char *rootFile, mx_editor *owner = NULL);
    ~mx_help_window();
    
    inline mx_help_target  *get_target() {return target;};
    inline mx_help_toolbar *get_toolbar()  {return tools;};
    inline mx_help_menubar *get_menubar()  {return menu;};
    inline mx_statusbar    *get_statusbar(){return status;};
    inline mx_scrollable_frame *get_frame() {return frame;};

    void setFile(int &err,char *fileName,mx_point *setPos) ;


    void setSensitive(bool setBackward,
                      bool setForward) ;
                      
    virtual void forced_close();
  private:
    
    mx_help_toolbar        *tools ;
    mx_statusbar           *status ;
    mx_help_menubar        *menu ;
    mx_scrollable_frame    *frame ;
    mx_help_target         *target ;

    void resetSize(int &err,
                   mx_point &newSize) ;

    mx_editor *owner;
};

typedef mx_help_file* mx_help_file_ptr ;
typedef mx_help_line* mx_help_line_ptr ;


/*-------------------------------------------------
 * CLASS: mx_help_list
 *
 * DESCRIPTION: 
 * 
 *
 */

class mx_help_list
{
public:
   mx_help_list(char *ifileName,mx_help_list *prev);
   ~mx_help_list() ;
   inline char     *getFileName(){return fileName;};
   inline void     setPosition(mx_point &pos){position=pos;};
   inline mx_point getPosition(){return position;};
   inline mx_help_list *getNext(){return next ;};
   inline mx_help_list *getPrev(){return prev ;};
private:

   mx_help_list    *next ;
   mx_help_list    *prev ;
   char            fileName[13] ;
   mx_point        position ;
};

/*-------------------------------------------------
 * CLASS: mx_help_target
 *
 * DESCRIPTION: 
 * 
 *
 */

class mx_help_target:public mx_frame_target
{
  public:
    
    mx_help_target(int &err);
    virtual ~mx_help_target();
    
    void setFile(int      &err,
		 char     *fileName,
		 bool     newFile,
		 mx_point &oldPos,
		 bool     *backward,
		 bool     *forward);
    inline char *getFileName(){return helpFile->getFileName();};
        
    inline mx_point getPixmapSize() {return pixmapSize ;};

    inline double getMaxMMLineLength(){return maxLineLength;};
    inline double getMinMMLineLength(){return minLineLength;}; 
    int    getLineIdealCharacterLength(){return idealLineCharLength;};
    inline double lineMMLength(char *string,int stringLength) {return cs1->nwidth(string,stringLength);};
    inline double getLineSeparator(){return lineSeparator;};
    inline double getLineHeight() {return lineHeight;};
    inline double getLineAscender() {return lineAscender;};
    inline double getLeftMargin() {return leftMargin;};
    inline void   setLineHeight(double ilh) {lineHeight=ilh;};
    inline void   setLineSeparator(double ilh) {lineSeparator=ilh;};
    inline void   setLineAscender(double ilh) {lineAscender=ilh;};
    inline void   setLeftMargin(double ilh) {leftMargin=ilh;};
    inline mx_help_file *getHelpFile() {return helpFile;};

    void draw(int &err,mx_draw_event &drawEvent) ;
    void buttonPress(int &err,mx_button_event &buttonEvent) ;
    
    void setWindow(mx_help_window *w) {window = w;};
    mx_help_window *getWindow(){return window;};


    void forward(int &err) ;
    void backward(int &err) ;
    void find(int &err);
    void print();

  protected:
  private:
    static mx_font         *font ;
    static mx_char_style   *cs1 ;
    static mx_char_style   *cs2 ;
    double                 maxLineLength ;
    double                 minLineLength ;
    int                    idealLineCharLength ;
    int                    currentHeadOffset ;
    char                   directory[100] ;
    mx_help_file_ptr       helpFilePtrs[MX_HELP_MAX_HOLD_FILES] ;
    mx_help_file           *helpFile ;
    mx_point               pixmapSize ;
    double                 lineHeight;
    double                 lineSeparator ;
    double                 lineAscender ;
    double                 leftMargin ;
    mx_help_window         *window ;
    mx_help_list           *allList ;
    mx_help_list           *currentElement ;
    static mx_print_d   *print_d;
    static mx_hsearch_d *search_d;
    

} ;

#endif








