/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifndef MX_FONT_H
#define MX_FONT_H
/*
 * MODULE/CLASS : mx_font & mx_font_mod
 *
 * AUTHOR : Andrew Haisley
 *
 * 
 *
 * DESCRIPTION:
 *
 * This class is a font class and includes methods for querying a font about
 * its metrics and such like..
 *
 *
 *
 */

#include <mx.h>
#include <mx_db.h>

#include <mx_hash.h>
#include <mx_sl_obj.h>


class mx_font;
class mx_font_mod;
class mx_font_metrics;
class mx_font_metrics_store;
class mx_font_family;

typedef enum 
{
    mx_normal = 0,
    mx_bold,
    mx_italic,
    mx_bold_italic
} 
mx_font_style_t;

#define MX_FONT_MAX_FAMILY_NAME_LEN 100


extern char *get_default_roman_font();
extern char *get_default_symbol_font();

class mx_font : mx_serialisable_object 
{
    MX_RTTI(mx_font_class_e)

    friend class mx_font_mod;

public:

    mx_font(int             &err, 
            char            *family_name,
            float           new_size = 12.0, 
            mx_font_style_t new_style = mx_normal);

    // use default copy constructor
    // use default assignment operator

    // default constructor
    mx_font();
    virtual ~mx_font();


    // methods for getting "stuff"

    // get_family returns a pointer to the actual string so don't delete it!!
    inline const char     *get_family() const;
    inline float           get_size() const;
    inline mx_font_style_t get_style() const;
    inline bool            get_bold() const;
    inline bool            get_italic() const;
           char           *get_x_fontname(int &err, float scale = 1.0);
    
    // used by the printer device driver 
    char *get_filename() const;
    char *get_ps_fontname() const;

    // height in MM above the baseline of the tallest character in this font
    // a -ve value -> below the baseline, +ve above
    float get_ascender() const;

    // height in MM below the baseline of the lowest character in this font
    // a -ve value -> below the baseline, +ve above
    float get_descender() const;

    // line height for font (ascender + descender)
    inline float get_line_height() const;

    // methods for setting things

    // These set various font attributes - they return whether the change
    // worked or not.
    bool set_family(int &err, const char *new_family_name);
    bool set_size(float new_size);
    bool set_style(int &err, mx_font_style_t new_style);
    void set_bold(int &err, bool flag);
    void set_italic(int &err, bool flag);
    bool set_nearest_family(int &err, const char *name);

    // the following are used to get information about font metrics, such
    // as the width of a character or string etc..

    inline float  width(const char *string) const;
    inline float  width(char character) const;
           uint32 em_width(char character) const;
    inline float  em_to_maxwell(uint32 em_width) const;
    inline uint32 em_to_maxwell_int(uint32 em_width) const;

    virtual void   serialise(int &err, uint8 **buffer);
    virtual void   unserialise(int &err, uint8 **buffer);
    virtual uint32 get_serialised_size(int &err);

           friend bool operator==(const mx_font &f1, const mx_font &f2);
    inline friend bool operator!=(const mx_font &f1, const mx_font &f2);

    // apply a font modification
    mx_font &operator+=(const mx_font_mod &m);

    // get a pointer to the font metrics storage object
    inline mx_font_metrics_store *get_font_metrics_store();

    // this is OBSOLETE, so so do not use it in new code!!!
    // instead query the font metrics storage mechanism for a font_family and
    // query that instead
    void family_has_B_or_I(int &err, const char *family_name, 
                           bool &b, bool &i); 

    // clears the cache of x font names
    void clear_x_font_name_cache();

    // has this font got a standard latin encoding?
    bool has_latin_encoding(int &err);

private:

    // private methods

    // reload the cached_x_fontname variable with relevant values
    void set_x_fontname();

    // For use inside a constructor to init all fields to initial values and
    // allocate space.
    void init_font();

    // private variables
    
    // This should be NULL when it has an invalid value. Do not delete this
    // value since it points to part of another data structure
    // (mx_font_metrics->family_name) 
    char            *typeface_family;

    mx_font_style_t typeface_style;
    float           typeface_size;            /* in points */

    // For optimisation, we save a pointer to the font family so that we don't
    // have to keep using the hash table. It should be set to NULL when it is
    // no longer valid.
    mx_font_family *font_family;

    // Also for optimisation, we store a pointer to the X font name, which are
    // cached in a static hash table. This should always be set to NULL when
    // invalid. We also store a scale, which is used for zooming documents in
    // and out of the display
    char            *cached_x_fontname;
    float           scale;

    // This contains a hash table of cached_x_fontnames and typeface_family
    // names, which are shared between fonts and never deleted.
    static mx_hash x_font_names;

    // This contains an object which manages all the font metrics objects in
    // the system
    static mx_font_metrics_store *font_metrics;

    // number of instances of mx_font
    static uint32 num_instances;
};


class mx_font_mod : public mx_serialisable_object 
{
    friend class mx_font;

    MX_RTTI(mx_font_mod_class_e)

public:
    mx_font_mod();
    virtual ~mx_font_mod();
    mx_font_mod(const mx_font &s1, const mx_font &s2);
    mx_font_mod(const mx_font_mod &s);

    // set all mods to TRUE
    mx_font_mod(const mx_font &s);

    void operator=(const mx_font_mod &other);
    mx_font_mod &operator+=(const mx_font_mod &other);

    void   serialise(int &err, uint8 **buffer);
    void   unserialise(int &err, uint8 **buffer);
    uint32 get_serialised_size(int &err);

    bool is_null() const;

    // sets the revert to default flag for any part of the mod which is
    // changing to the given default paragraph style
    void set_revert_to_default_flags(const mx_font &default_para_font);

    // changes any mods with a revert flag set so that result of the
    // modification is the new default character style
    void set_new_default_cs(const mx_font &default_para_font);

    void clear_revert_to_default_flags();

    // clears any mod flags in this mod which are set in the other_mod
    void clear_mods_in(const mx_font_mod &other_mod);

    // get/set/clear individual mods
    bool get_family_mod(const char *&new_family) const;
    bool get_size_mod(float &new_size) const;
    bool get_bold_mod(bool &bold_on) const;
    bool get_italic_mod(bool &italic_on) const;

    void set_family_mod(const char *new_family);
    void set_size_mod(float new_size);
    void set_bold_mod(bool bold_on);
    void set_italic_mod(bool italic_on);

    void clear_family_mod();
    void clear_size_mod();
    void clear_bold_mod();
    void clear_italic_mod();

private:

    void set_family(const char *s);

    // values to change
    bool             typeface_family_mod     : 1;
    bool             typeface_size_mod       : 1;
    bool             typeface_bold_mod       : 1;
    bool             typeface_italic_mod     : 1;

    // revert to default flags
    bool             typeface_family_mod_rev : 1;
    bool             typeface_size_mod_rev   : 1;
    bool             typeface_bold_mod_rev   : 1;
    bool             typeface_italic_mod_rev : 1;

    char            *typeface_family;
    float            typeface_size;
    bool             typeface_bold   : 1;
    bool             typeface_italic : 1;
};

inline
const char * mx_font::get_family() const
{
    return typeface_family;
}

inline
float mx_font::get_size() const
{
    return typeface_size;
}

inline
mx_font_style_t mx_font::get_style() const
{
    return typeface_style;
}

inline
float mx_font::em_to_maxwell(uint32 em_width) const
{
    // This width is the range 0 - 1000  - convert to 
    // Maxwell units (floating mm) for this point size
    return MX_POINTS_TO_MM( ((float)(em_width * get_size()))/1000.0 );
}

inline
float mx_font::width(const char *string) const
{
    uint32 res = 0;

    while(*string != '\0')
    {
        res += em_width(*string++);
    }

    return em_to_maxwell(res);
}

inline
float mx_font::width(char character) const
{
    return em_to_maxwell(em_width(character));
}

inline
uint32 mx_font::em_to_maxwell_int(uint32 em_width) const
{
    return (int)(MX_POINTS_TO_MM((float)(em_width*get_size())*10.0));
}

inline
bool mx_font::get_bold() const
{
    return typeface_style == mx_bold ||
        typeface_style == mx_bold_italic;
}

inline
bool mx_font::get_italic() const
{
    return typeface_style == mx_italic ||
        typeface_style == mx_bold_italic;
}

inline
float mx_font::get_line_height() const
{
    return get_ascender() - get_descender();
}

inline
mx_font_metrics_store *mx_font::get_font_metrics_store()
{
    return font_metrics;
}

inline
bool operator!=(const mx_font &f1, const mx_font &f2)
{
    return !(f1 == f2);
}

#endif // MX_FONT_H
