/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * MODULE/CLASS : mx_font
 *
 * AUTHOR : Tom Newton
 *
 * 
 *
 * DESCRIPTION:
 *
 * This class is a font class and includes methods for querying a font about
 * its metrics and such like..
 *
 *
 *
 */
#include <mx.h>
#include <mx_type1.h>
#include <mx_xfontname.h>
#include <mx_hash_iter.h>
#include "mx_font.h"
#include <mx_font_metrics.h>
#include <Intrinsic.h>

mx_font_metrics_store *mx_font::font_metrics = NULL;
mx_hash mx_font::x_font_names(20);
uint32 mx_font::num_instances = 0;

extern char *global_maxhome;

mx_font::mx_font()
{
    int err = MX_ERROR_OK;

    init_font(); 
    num_instances++;

    set_family(err, get_default_roman_font());
    MX_ERROR_CHECK(err);

    return;
abort:
    global_error_trace->print();
    MX_ERROR_CLEAR(err);
    mx_printf_warning("FATAL ERROR: Cannot find default font. Exiting.....");
    exit(EXIT_FAILURE);
    return;
}

char *get_default_roman_font()
{
#ifdef sun
	return "Adobe-Pstimes";
#else
	return "Adobe-Utopia";
#endif
}

char *get_default_symbol_font()
{
#ifdef sun
	return "Adobe-Symbol";
#else
	return "Daltonmaag-Symbol";
#endif
}

mx_font::mx_font(int  &err, 
                 char *family_name,
                 float new_size = 12.0, 
                 mx_font_style_t new_style = mx_normal)
{
    init_font(); 
    num_instances++;

    set_family(err, family_name); 
    MX_ERROR_CHECK(err);

    set_style(err, new_style); 
    MX_ERROR_CHECK(err);

    set_size(new_size); 

    return;
abort:
    return;
}

mx_font::~mx_font()
{
    num_instances--;
}

void mx_font::init_font()
{
    if(font_metrics == NULL) 
    {
        font_metrics = new mx_font_metrics_store;
    }

    typeface_size = 12.0;
    typeface_style = mx_normal;
    typeface_family = NULL;
    cached_x_fontname = NULL;
    scale = 1.0;
}

bool mx_font::set_family(int &err, const char *new_family_name)
{
    mx_font_family *ff;
    mx_font_metrics *fm;

    if (typeface_family == NULL || 
        strcmp(new_family_name, typeface_family) != 0)
    {
        ff = font_metrics->get_font_family(err, new_family_name);
        MX_ERROR_CHECK(err);

        fm = ff->get_font_metrics(get_style());
        if(fm == NULL) MX_ERROR_THROW(err, MX_FONT_STYLE_NOT_AVAILABLE);
        
        typeface_family = fm->get_name();
        
        font_family = ff;
        cached_x_fontname = NULL;
    }

    return TRUE;
abort:
    return FALSE;
}

bool mx_font::set_nearest_family(int &err, const char *new_family_name)
{
    mx_font_family *ff = font_metrics->get_nearest_font_family(err, new_family_name);
    MX_ERROR_CHECK(err);

    set_family(err, ff->get_name());
    if (err == MX_FONT_STYLE_NOT_AVAILABLE)
    {
        MX_ERROR_CLEAR(err);
        set_style(err, mx_normal);
    }
    MX_ERROR_CHECK(err);

    return TRUE;
abort:
    return FALSE;
}

bool mx_font::set_size(float new_size)
{
    if (!MX_FLOAT_EQ(typeface_size, new_size))
    {
        cached_x_fontname = NULL;
        typeface_size = new_size;
        return TRUE;
    }
    return FALSE;
}

bool mx_font::set_style(int &err, mx_font_style_t new_style)
{
    if(new_style != typeface_style)
    {
        if(font_family->get_font_metrics(new_style) == NULL)
        {
            MX_ERROR_THROW(err, MX_FONT_STYLE_NOT_AVAILABLE);
        }
        cached_x_fontname = NULL;
        typeface_style = new_style;
    }
    return TRUE;
abort:
    return FALSE;
}

uint32 mx_font::em_width(char character) const
{
    return font_family->get_font_metrics(typeface_style)->
        get_em_width((uint8)character);
}

char *mx_font::get_ps_fontname() const
{
    return font_family->get_font_metrics(get_style())->get_ps_name();
}

char *mx_font::get_filename() const
{
    return font_family->get_font_metrics(get_style())->get_file_name();
}

extern Widget global_top_level;

static inline char *get_x_res()
{
    static char x_res_str[20] = "";

    if (x_res_str[0] == '\0')
    {
        Screen *s = XtScreen(global_top_level);
        int xres = 
            (int)(0.5 + ((double)s->width / ((double)s->mwidth / 25.4)));

        sprintf(x_res_str, "%d", xres);
    }
    return x_res_str;
}

static inline char *get_y_res()
{
    static char y_res_str[20] = "";

    if (y_res_str[0] == '\0')
    {
        Screen *s = XtScreen(global_top_level);
        int yres = 
            (int)(0.5 + ((double)s->height / ((double)s->mheight / 25.4)));

        sprintf(y_res_str, "%d", yres);
    }
    return y_res_str;
}

void mx_font::set_x_fontname()
{
    int err = MX_ERROR_OK;
    char new_x_fontname[400];
    char size_string[20];
    char *unscaled_x_name = NULL;

    // convert our size into a string
    sprintf(size_string, "%ld", (uint32)(10.0 * typeface_size * scale + 0.5));

    // The type1 X fonts which are scalable have 0s 
    // for their sizes, so substitute the real size for
    // the fist 0 and put stars (*) in the others
    
    unscaled_x_name = font_family->get_font_metrics(get_style())->get_x_name();
    strcpy(new_x_fontname, unscaled_x_name);

    mx_xfontname_set_arg(err, new_x_fontname, mx_xfontname_point_size_e,
                         size_string);
    MX_ERROR_CHECK(err);
    mx_xfontname_set_arg(err, new_x_fontname, mx_xfontname_pixel_size_e, "0");
    MX_ERROR_CHECK(err);
    mx_xfontname_set_arg(err, new_x_fontname, mx_xfontname_x_res_e, 
                         get_x_res());
    MX_ERROR_CHECK(err);
    mx_xfontname_set_arg(err, new_x_fontname, mx_xfontname_y_res_e, 
                         get_y_res());
    MX_ERROR_CHECK(err);

    cached_x_fontname = (char *)x_font_names.get(err, new_x_fontname);
    if(err == MX_HASH_NOT_FOUND)
    {
        MX_ERROR_CLEAR(err);
        cached_x_fontname = mx_string_copy(new_x_fontname);

        x_font_names.add(err, cached_x_fontname, cached_x_fontname);
    }
    MX_ERROR_CHECK(err);

    return;
abort:
    global_error_trace->print();
    MX_ERROR_CLEAR(err);
    return;
}

char *mx_font::get_x_fontname(int &err, float new_scale)
{
    if(cached_x_fontname == NULL || !MX_FLOAT_EQ(new_scale, scale))
    {
        scale = new_scale;
        set_x_fontname();
    }

    return cached_x_fontname;
}

uint32 mx_font::get_serialised_size(int &err)
{
    return SLS_STRING(typeface_family) + SLS_FLOAT + SLS_ENUM;
}

void mx_font::serialise(int &err, uint8 **buffer)
{
    serialise_string(typeface_family, buffer);
    serialise_float(typeface_size, buffer);
    serialise_enum(typeface_style, buffer);
}

void mx_font::unserialise(int &err, uint8 **buffer)
{
    uint16 e;

    unserialise_string((char *)temp_buffer, buffer);
    set_family(err, (char *)temp_buffer);
    MX_ERROR_CHECK(err);
    unserialise_float(typeface_size, buffer);
    unserialise_enum(e, buffer);
    typeface_style = (mx_font_style_t)e;
abort:;
}

void mx_font::set_bold(int &err, bool flag)
{
    bool is_bold = get_bold();

    if ((flag && is_bold) || (!flag && !is_bold)) return;

    if(flag)
    {
        if(get_italic())
        {
            set_style(err, mx_bold_italic);
        }
        else
        {
            set_style(err, mx_bold);
        }
    }
    else
    {
        if(get_italic())
        {
            set_style(err, mx_italic);
        }
        else
        {
            set_style(err, mx_normal);
        }
    }
    MX_ERROR_CHECK(err);
    return;
abort:
    return;
}

void mx_font::set_italic(int &err, bool flag)
{
    bool is_italic = get_italic();

    if ((flag && is_italic) || (!flag && !is_italic)) return;

    if(flag)
    {
        if(get_bold())
        {
            set_style(err, mx_bold_italic);
        }
        else
        {
            set_style(err, mx_italic);
        }
    }
    else
    {
        if(get_bold())
        {
            set_style(err, mx_bold);
        }
        else
        {
            set_style(err, mx_normal);
        }
    }
    MX_ERROR_CHECK(err);
    return;
abort:
    return;
}

// height in MM above the baseline of the tallest character in this font
float mx_font::get_ascender() const
{
    return MX_POINTS_TO_MM(
        (font_family->get_font_metrics(get_style())->get_em_ascender() *
         typeface_size) / 1000.0);
}

// depth in MM below the baseline of the lowest character in this font
float mx_font::get_descender() const
{
    return MX_POINTS_TO_MM(
        (font_family->get_font_metrics(get_style())->get_em_descender() *
         typeface_size) / 1000.0);
}

void mx_font::family_has_B_or_I(int &err, const char *family_name, 
                                bool &b, bool &i)
{
    mx_font_family *ff = font_metrics->get_font_family(err, family_name);
    
    MX_ERROR_CHECK(err);
    ff->family_has_B_or_I(b, i);
    
abort:
    return;
}

void mx_font::clear_x_font_name_cache()
{
    mx_hash_iterator i(x_font_names);
    while (i.more()) delete [] (char *)i.data();
}

bool mx_font::has_latin_encoding(int &err)
{
    char buf[255];
    const char *x_name = this->get_x_fontname(err, scale);
    MX_ERROR_CHECK(err);

    mx_xfontname_get_arg(err, x_name, mx_xfontname_registry_e, buf);
    MX_ERROR_CHECK(err);

    if (strcasecmp(buf, "iso8859") != 0) return FALSE;

    mx_xfontname_get_arg(err, x_name, mx_xfontname_encoding_e, buf);
    MX_ERROR_CHECK(err);
    
    if (strcasecmp(buf, "1") != 0) return FALSE;

    return TRUE;
abort:
    return FALSE;
}

bool operator==(const mx_font &f1, const mx_font &f2)
{
   return ((f1.typeface_size == f2.typeface_size) &&
            (f1.typeface_style == f2.typeface_style) &&
            (strcmp(f1.typeface_family, f2.typeface_family) == 0));
}

mx_font &mx_font::operator+=(const mx_font_mod &m)
{
    int err = MX_ERROR_OK;
    if (m.typeface_family_mod)
    {
        set_family(err, m.typeface_family);
        if (err != MX_ERROR_OK)
        {
            const char *str = m.typeface_family;
            MX_ERROR_CLEAR(err);

            // skip the foundry
            while (*str != '-' && *str != '\0') str++;
            str++;

            set_nearest_family(err, str);
            MX_ERROR_CHECK(err);
        }
    }

    if (m.typeface_size_mod)
    {
        set_size(m.typeface_size);
    }

    if (m.typeface_bold_mod)
    {
        set_bold(err, m.typeface_bold);
        MX_ERROR_CHECK(err);
    }

    if (m.typeface_italic_mod)
    {
        set_italic(err, m.typeface_italic);
        MX_ERROR_CHECK(err);
    }

    return *this;
abort:
    global_error_trace->print();
    MX_ERROR_CLEAR(err);
    return *this;
}

mx_font_mod::mx_font_mod()
{
    bzero(this, sizeof(*this));
}

mx_font_mod::~mx_font_mod()
{
    if (typeface_family != NULL)
    {
        delete [] typeface_family;
    }
}

mx_font_mod::mx_font_mod(const mx_font &s1, const mx_font &s2)
{
    bzero(this, sizeof(*this));

    // if either typeface family is NULL, the font has not been
    // initialised properly, so we can't do very much
    if (s1.typeface_family == NULL || s2.typeface_family == NULL)
    {
        mx_printf_warning("FATAL ERROR: typeface family unset in mx_font_mod::mx_font_mod.....");
        exit(-1);
    }

    if (strcmp(s1.typeface_family, s2.typeface_family) != 0)
    {
        typeface_family_mod = TRUE;
        set_family(s2.typeface_family);
    }

    if (!MX_FLOAT_EQ(s1.typeface_size, s2.typeface_size))
    {
        typeface_size_mod = TRUE;
        typeface_size = s2.typeface_size;
    }

    if (s1.typeface_style != s2.typeface_style)
    {
        if (s1.get_bold() && !s2.get_bold())
        {
            typeface_bold_mod = TRUE;
            typeface_bold = FALSE;
        }
        else if (!s1.get_bold() && s2.get_bold())
        {
            typeface_bold_mod = TRUE;
            typeface_bold = TRUE;
        }

        if (s1.get_italic() && !s2.get_italic())
        {
            typeface_italic_mod = TRUE;
            typeface_italic = FALSE;
        }
        else if (!s1.get_italic() && s2.get_italic())
        {
            typeface_italic_mod = TRUE;
            typeface_italic = TRUE;
        }
    }
}

mx_font_mod::mx_font_mod(const mx_font_mod &o)
{
    memcpy(this, &o, sizeof(o));
    typeface_family = NULL;
    set_family(o.typeface_family);
}

mx_font_mod::mx_font_mod(const mx_font &font)
{
    mx_font f = font;
    typeface_family = NULL;
    set_family_mod(f.get_family());
    set_size_mod(f.get_size());
    set_bold_mod(f.get_bold());
    set_italic_mod(f.get_italic());
}

void mx_font_mod::operator=(const mx_font_mod &o)
{
    set_family(NULL);
    memcpy(this, &o, sizeof(o));
    typeface_family = NULL;
    set_family(o.typeface_family);
}

mx_font_mod &mx_font_mod::operator+=(const mx_font_mod &o)
{
    if (o.typeface_family_mod)
    {
        typeface_family_mod = TRUE;
        typeface_family_mod_rev = o.typeface_family_mod_rev;
        set_family(o.typeface_family);
    }

    if (o.typeface_size_mod)
    {
        typeface_size_mod = TRUE;
        typeface_size_mod_rev = o.typeface_size_mod_rev;
        typeface_size = o.typeface_size;
    }

    if (o.typeface_bold_mod)
    {
        typeface_bold_mod = TRUE;
        typeface_bold_mod_rev = o.typeface_bold_mod_rev;
        typeface_bold = o.typeface_bold;
    }

    if (o.typeface_italic_mod)
    {
        typeface_italic_mod = TRUE;
        typeface_italic_mod_rev = o.typeface_italic_mod_rev;
        typeface_italic = o.typeface_italic;
    }
    return *this;
}

void mx_font_mod::set_family(const char *s)
{
    if (typeface_family != NULL)
    {
        delete [] typeface_family;
        typeface_family = NULL;
    }
    if (s != NULL)
    {
        typeface_family = mx_string_copy(s);
    }
}

void mx_font_mod::serialise(int &err, uint8 **buffer)
{
    serialise_bool(typeface_family_mod, buffer);
    if (typeface_family_mod)
    {
        serialise_string(typeface_family, buffer);
    }

    serialise_bool(typeface_size_mod, buffer);
    if (typeface_size_mod)
    {
        serialise_float(typeface_size, buffer);
    }

    serialise_bool(typeface_bold_mod, buffer);
    if (typeface_bold_mod)
    {
        serialise_bool(typeface_bold, buffer);
    }

    serialise_bool(typeface_italic_mod, buffer);
    if (typeface_italic_mod)
    {
        serialise_bool(typeface_italic, buffer);
    }
}

void mx_font_mod::unserialise(int &err, uint8 **buffer)
{
    bool b;

    unserialise_bool(b, buffer);
    typeface_family_mod = b;
    if (typeface_family_mod)
    {
        unserialise_string((char *)temp_buffer, buffer);
        set_family((char *)temp_buffer);
    }

    unserialise_bool(b, buffer);
    typeface_size_mod = b;
    if (typeface_size_mod)
    {
        unserialise_float(typeface_size, buffer);
    }

    unserialise_bool(b, buffer);
    typeface_bold_mod = b;
    if (typeface_bold_mod)
    {
        unserialise_bool(b, buffer);
        typeface_bold = b;
    }

    unserialise_bool(b, buffer);
    typeface_italic_mod = b;
    if (typeface_italic_mod)
    {
        unserialise_bool(b, buffer);
        typeface_italic = b;
    }
}

uint32 mx_font_mod::get_serialised_size(int &err)
{
    int res;

    res = 4 * SLS_BOOL;

    if (typeface_family_mod)
    {
        res += SLS_STRING(typeface_family);
    }

    if (typeface_size_mod)
    {
        res += SLS_FLOAT;
    }

    if (typeface_bold_mod)
    {
        res += SLS_BOOL;
    }

    if (typeface_italic_mod)
    {
        res += SLS_BOOL;
    }

    return res;
}

bool mx_font_mod::is_null() const
{
    return
        !typeface_family_mod &&
        !typeface_size_mod &&
        !typeface_bold_mod &&
        !typeface_italic_mod;
}

void mx_font_mod::set_revert_to_default_flags(const mx_font &f)
{
    typeface_family_mod_rev = (
        typeface_family_mod && typeface_family != NULL &&
        strcmp(typeface_family, f.typeface_family) == 0);
    
    typeface_size_mod_rev = (typeface_size_mod && 
                             MX_FLOAT_EQ(typeface_size, f.typeface_size));

    typeface_bold_mod_rev = (typeface_bold_mod && 
                              (typeface_bold && f.get_bold() ||
                               !typeface_bold && !f.get_bold()));

    typeface_italic_mod_rev = (typeface_italic_mod && 
                              (typeface_italic && f.get_italic() ||
                               !typeface_italic && !f.get_italic()));
}

void mx_font_mod::set_new_default_cs(const mx_font &f)
{
    if (typeface_family_mod && typeface_family_mod_rev)
    {
        set_family((char *)f.get_family());
    }

    if (typeface_size_mod && typeface_size_mod_rev)
    {
        typeface_size = f.typeface_size;
    }

    if (typeface_bold_mod && typeface_bold_mod_rev)
    {
        typeface_bold = f.get_bold();
    }

    if (typeface_italic_mod && typeface_italic_mod_rev)
    {
        typeface_italic = f.get_italic();
    }
}

void mx_font_mod::clear_revert_to_default_flags()
{
    typeface_family_mod_rev = 
        typeface_size_mod_rev = 
        typeface_bold_mod_rev = 
        typeface_italic_mod_rev = FALSE; 
}

void mx_font_mod::clear_mods_in(const mx_font_mod &o)
{
    if (o.typeface_family_mod) clear_family_mod();
    if (o.typeface_size_mod) clear_size_mod();
    if (o.typeface_bold_mod) clear_bold_mod();
    if (o.typeface_italic_mod) clear_italic_mod();
}

bool mx_font_mod::get_family_mod(const char *&new_family) const
{
    if (typeface_family_mod) new_family = typeface_family;
    return typeface_family_mod;
}

bool mx_font_mod::get_size_mod(float &new_size) const
{
    if (typeface_size_mod) new_size = typeface_size;
    return typeface_size_mod;
}

bool mx_font_mod::get_bold_mod(bool &bold_on) const
{
    if (typeface_bold_mod) bold_on = typeface_bold;
    return typeface_bold_mod;
}

bool mx_font_mod::get_italic_mod(bool &italic_on) const
{
    if (typeface_italic_mod) italic_on = typeface_italic;
    return typeface_italic_mod;
}

void mx_font_mod::set_family_mod(const char *new_family)
{
    typeface_family_mod = TRUE;
    typeface_family_mod_rev = FALSE;
    set_family(new_family);
}

void mx_font_mod::set_size_mod(float new_size)
{
    typeface_size_mod = TRUE;
    typeface_size_mod_rev = FALSE;
    typeface_size = new_size;
}

void mx_font_mod::set_bold_mod(bool bold_on)
{
    typeface_bold_mod = TRUE;
    typeface_bold_mod_rev = FALSE;
    typeface_bold = bold_on;
}

void mx_font_mod::set_italic_mod(bool italic_on)
{
    typeface_italic_mod = TRUE;
    typeface_italic_mod_rev = FALSE;
    typeface_italic = italic_on;
}

void mx_font_mod::clear_family_mod()
{
    set_family(NULL);
    typeface_family_mod = typeface_family_mod_rev = FALSE;
}
    
void mx_font_mod::clear_size_mod()
{
    typeface_size = 0.0f;
    typeface_size_mod = typeface_size_mod_rev = FALSE;
}

void mx_font_mod::clear_bold_mod()
{
    typeface_bold = typeface_bold_mod = typeface_bold_mod_rev = FALSE;
}

void mx_font_mod::clear_italic_mod()
{
    typeface_italic = typeface_italic_mod = typeface_italic_mod_rev = FALSE;
}
