/*
 *  This file is part of the Maxwell Word Processor application.
 *  Copyright (C) 1996, 1997, 1998 Andrew Haisley, David Miller, Tom Newton
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdlib.h>
#include <stdio.h>
#include <mx.h>
#include <mx_db.h>
#include <mx_collection.h>
#include <mx_document.h>
#include <mx_disk_heap.h>

#include <X11/Intrinsic.h>

static mx_wp_document *doc;

Widget            global_top_level;
XtAppContext      global_app_context;

char             *global_maxhome;

char *locked_by;
char *locked_host;
pid_t locked_pid;

void test1()
{
    int             i, n, err;
    mx_sheet     *sheet;
    mx_text_area *page;

    mx_spline_coord_t     coords[4];
    mx_quadspline_t pp;

    printf("create a document\n");
    doc = mx_db_client_open_wp_doc(err, "test.doc", TRUE,
            &locked_by, &locked_host, locked_pid);
    MX_ERROR_CHECK(err);

    printf( "set description\n" );
    doc->set_info(err, "a test document", "andrew", "0.1");
    MX_ERROR_CHECK( err );

    printf("Number of sheets:");    
    scanf("%d", &n);

    printf( "add sheets - each with a page area\n" );

    for (i = 0; i < n; i++)
    {
        sheet = doc->add_sheet(err);
        MX_ERROR_CHECK( err );

        page = sheet->add_text_area(err);
        MX_ERROR_CHECK(err);

        pp.num_coords=4;
        pp.coords=coords;
        coords[0].p.x=0.0;
        coords[0].p.y=0.0;
        coords[0].flag=TRUE;
        coords[1].p.x=210.0;
        coords[1].p.y=0.0;
        coords[1].flag=TRUE;
        coords[2].p.x=210.0;
        coords[2].p.y=297.0;
        coords[2].flag=TRUE;
        coords[3].p.x=0.0;
        coords[3].p.y=297.0;
        coords[3].flag=TRUE;

        page->set_outline(err, pp);
        MX_ERROR_CHECK(err);

        printf(".");
        fflush(stdout);
    }

    printf("commit\n");

    doc->commit(err);
    MX_ERROR_CHECK(err);

    printf( "close document\n" );
    mx_db_client_close_wp_doc(err, doc);

    return;
abort:
    global_error_trace->print();
}

void test11()
{
    int             i, n, err;
    mx_sheet     *sheet;
    mx_text_area *page, *last_page = NULL;

    mx_spline_coord_t     coords[4];
    mx_quadspline_t pp;

    printf("create a document\n");
    doc = mx_db_client_open_wp_doc(err, "test.doc", TRUE,
            &locked_by, &locked_host, locked_pid);
    MX_ERROR_CHECK(err);

    printf( "set description\n" );
    doc->set_info(err, "a test document", "andrew", "0.1");
    MX_ERROR_CHECK( err );

    printf("Number of sheets:");    
    scanf("%d", &n);

    printf( "add sheets - each with a page area\n" );

    for (i = 0; i < n; i++)
    {
        sheet = doc->add_sheet(err);
        MX_ERROR_CHECK( err );

        sheet->set_int(err, "NUM", i);
        MX_ERROR_CHECK(err);

        page = sheet->add_text_area(err);
        MX_ERROR_CHECK(err);

        page->set_int(err, "NUM", i);
        MX_ERROR_CHECK(err);

        pp.num_coords=4;
        pp.coords=coords;
        coords[0].p.x=0.0;
        coords[0].p.y=0.0;
        coords[0].flag=TRUE;
        coords[1].p.x=210.0;
        coords[1].p.y=0.0;
        coords[1].flag=TRUE;
        coords[2].p.x=210.0;
        coords[2].p.y=297.0;
        coords[2].flag=TRUE;
        coords[3].p.x=0.0;
        coords[3].p.y=297.0;
        coords[3].flag=TRUE;

        page->set_outline(err, pp);
        MX_ERROR_CHECK(err);

        if (last_page != NULL)
        {
            page->insert_into_chain(err, last_page, NULL);
            MX_ERROR_CHECK(err);
        }

        last_page = page;

        printf(".");
        fflush(stdout);
    }

    printf("commit\n");

    doc->commit(err);
    MX_ERROR_CHECK(err);

    printf( "close document\n" );
    mx_db_client_close_wp_doc(err, doc);

    return;
abort:
    global_error_trace->print();
}

void test13()
{
    int             i, n, err;
    mx_sheet     *sheet;
    mx_text_area *page, *first_page = NULL;
    mx_point p1, p2;

    p1.x = 100;
    p1.y = 200;
    p2.x = 300;
    p2.y = 400;

    printf("create a document\n");
    doc = mx_db_client_open_wp_doc(err, "test.doc", TRUE,
            &locked_by, &locked_host, locked_pid);
    MX_ERROR_CHECK(err);

    printf( "set description\n" );
    doc->set_info(err, "a test document", "andrew", "0.1");
    MX_ERROR_CHECK( err );

    printf("Number of sheets:");    
    scanf("%d", &n);

    printf( "add sheets - each with a page area\n" );

    for (i = 0; i < n; i++)
    {
        sheet = doc->add_sheet(err);
        MX_ERROR_CHECK( err );

        sheet->set_int(err, "NUM", i);
        MX_ERROR_CHECK(err);

        page = sheet->add_text_area(err);
        MX_ERROR_CHECK(err);

        page->set_int(err, "NUM", i);
        MX_ERROR_CHECK(err);

        if (first_page != NULL)
        {
            page->link_to_area(err, first_page, p1, p2);
            MX_ERROR_CHECK(err);
        }
        else
        {
            first_page = page;
        }

        printf(".");
        fflush(stdout);
    }

    printf("commit\n");

    doc->commit(err);
    MX_ERROR_CHECK(err);

    printf( "close document\n" );
    mx_db_client_close_wp_doc(err, doc);

    return;
abort:
    global_error_trace->print();
}

void test2()
{
    int             i, j,  n, err, np;
    mx_sheet     *sheet;
    mx_text_area *page;

    const mx_quadspline_t *ol;

    printf("open document\n");
    doc = mx_db_client_open_wp_doc(err, "test.doc", TRUE,
            &locked_by, &locked_host, locked_pid);
    MX_ERROR_CHECK(err);

    printf( "description = %s\n", doc->get_description());

    printf( "get sheets\n" );

    n = doc->get_num_sheets(err);
    MX_ERROR_CHECK( err );

    printf("num sheets = %d\n", n);

    for (i = 0; i < n; i++)
    {
        printf("sheet %d\n", i);

        sheet = doc->sheet(err, i);
        MX_ERROR_CHECK(err);

        np = sheet->get_num_text_areas(err);
        MX_ERROR_CHECK(err);

        printf("num pages = %d\n", np);

        for (j = 0; j < np; j++)
        {
            page = sheet->text_area(err, j);
            MX_ERROR_CHECK(err);

            ol = page->get_outline_readonly(err);
            MX_ERROR_CHECK(err);

            for(uint32 k = 0; k < ol->num_coords; k++ ) 
            {
                printf( "%ld) %f,%f\n", k, ol->coords[k].p.x,ol->coords[k].p.y);
            }
        }
    }

    printf( "close document\n" );
    mx_db_client_close_wp_doc(err, doc);

    return;
abort:
    global_error_trace->print();
}

void test15()
{
    int             i, n, err;

    printf("open document\n");
    doc = mx_db_client_open_wp_doc(err, "test.doc", TRUE,
            &locked_by, &locked_host, locked_pid);
    MX_ERROR_CHECK(err);

    printf( "description = %s\n", doc->get_description());

    printf( "how many sheets to delete>" );
    scanf("%d", &n);

    for (i = 0; i < n; i++)
    {
        doc->delete_sheet_index(err, 0);
        MX_ERROR_CHECK(err);

    }

    printf("commit...1, rollback...2\n");
    scanf("%d", &i);
    if (i == 1)
    {
        printf( "commit\n" );
        doc->commit(err);
    }
    else
    {
        printf( "rollback\n" );
        doc->rollback(err);
        MX_ERROR_CHECK(err);

        printf( "commit\n" );
        doc->commit(err);
    }

    MX_ERROR_CHECK(err);

    printf( "close document\n" );
    mx_db_client_close_wp_doc(err, doc);

    return;
abort:
    global_error_trace->print();
}

void test12()
{
    int             n, err;
    mx_sheet     *sheet;
    mx_text_area *page;
    bool         b;

    printf("open document\n");
    doc = mx_db_client_open_wp_doc(err, "test.doc", TRUE,
            &locked_by, &locked_host, locked_pid);
    MX_ERROR_CHECK(err);

    printf( "description = %s\n", doc->get_description());

    printf( "get a sheet\n" );

    sheet = doc->sheet(err, 0);
    MX_ERROR_CHECK(err);

    page = sheet->text_area(err, 0);
    MX_ERROR_CHECK(err);

    while (TRUE)
    {
        n = page->get_int(err, "NUM");
        MX_ERROR_CHECK(err);

        printf("page number = %d\n", n);

        n = sheet->get_int(err, "NUM");
        MX_ERROR_CHECK(err);

        printf("sheet number = %d\n", n);

        b = page->next_chained_area(err);
        MX_ERROR_CHECK(err);

        if (b)
        {
            page = (mx_text_area *)page->next_in_chain(err);
            MX_ERROR_CHECK(err);

            sheet = (mx_sheet *)page->get_owner_object(err);
            MX_ERROR_CHECK(err);
        }
        else
        {
            break;
        }
    }

    mx_db_client_close_wp_doc(err, doc);
    MX_ERROR_CHECK(err);

    return;
abort:
    global_error_trace->print();
}

void test14()
{
    int             i, pn, n, err;
    mx_sheet     *sheet;
    mx_text_area *page, *linked_page;

    mx_area_link_t *l;

    printf("open document\n");
    doc = mx_db_client_open_wp_doc(err, "test.doc", TRUE,
            &locked_by, &locked_host, locked_pid);
    MX_ERROR_CHECK(err);

    printf( "description = %s\n", doc->get_description());

    printf( "get a sheet\n" );

    sheet = doc->sheet(err, 0);
    MX_ERROR_CHECK(err);

    page = sheet->text_area(err, 0);
    MX_ERROR_CHECK(err);

    n = page->number_linked_areas(err);
    MX_ERROR_CHECK(err);

    printf("NUMBER OF LINKED AREAS = %d\n", n);

    for (i = 0; i < n; i++)
    {
        l = page->linked_area(err, i);
        MX_ERROR_CHECK(err);

        linked_page = (mx_text_area *)l->linked_area;

        pn = linked_page->get_int(err, "NUM");
        MX_ERROR_CHECK(err);

        printf("page number = %d  lp = (%f,%f - %f,%f)\n", 
                pn,
                l->thispoint.x,
                l->thispoint.y,
                l->otherpoint.x,
                l->otherpoint.y);
    }

    mx_db_client_close_wp_doc(err, doc);
    MX_ERROR_CHECK(err);

    return;
abort:
    global_error_trace->print();
}

void test3()
{
    int              i, j, err;
    mx_sheet      *sheet;
    mx_image_area *area;
    unsigned char b[1024];

    printf("open a document\n");
    doc = mx_db_client_open_wp_doc(err, "test.doc", TRUE,
            &locked_by, &locked_host, locked_pid);
    MX_ERROR_CHECK(err);

    printf( "add a sheet\n" );

    sheet = doc->add_sheet(err);
    MX_ERROR_CHECK( err );

    printf( "add an image area\n" );

    area = sheet->add_image_area(err);
    MX_ERROR_CHECK(err);

    printf( "add a blob\n" );
    area->create_blob(err, "BLOBBY");
    MX_ERROR_CHECK(err);

#define SIK 4192

    printf( "set size to %dK\n", SIK);
    area->set_blob_size(err, "BLOBBY", SIK*1024);
    MX_ERROR_CHECK(err);

    printf( "get size = %ld\n", area->get_blob_size(err, "BLOBBY"));
    MX_ERROR_CHECK(err);

    while (1)
    {

        printf( "set data\n");
        for (i = 0; i < SIK; i++)
        {
            for (j = 0; j < 1024; j++)
            {
                b[j] = ((i + j) % 256);
            }
            area->set_blob_data(err, "BLOBBY", i * 1024, 1024, b);
            MX_ERROR_CHECK(err);
        }

        printf( "get data\n");
        for (i = 0; i < SIK; i++)
        {
            area->get_blob_data(err, "BLOBBY", i * 1024, 1024, b);
            MX_ERROR_CHECK(err);

            for (j = 0; j < 1024; j++)
            {
                if (b[j] != ((i + j) % 256))
                {
                printf("ERROR IN BYTE %d\n", i * 1024 + j);
                }
            }
        }
    }

    printf( "commit\n" );
    doc->commit(err);
    MX_ERROR_CHECK(err);

    printf( "close document\n" );
    mx_db_client_close_wp_doc(err, doc);

    return;
abort:
    global_error_trace->print();
}

void test4()
{
    int              i, j, err;
    mx_sheet      *sheet;
    mx_image_area *area;
    unsigned char b[1024];

    printf("open a document\n");
    doc = mx_db_client_open_wp_doc(err, "test.doc", TRUE,
            &locked_by, &locked_host, locked_pid);
    MX_ERROR_CHECK(err);

    printf( "get a sheet\n" );

    sheet = doc->sheet(err, 0);
    MX_ERROR_CHECK( err );

    printf( "get an image area\n" );

    area = sheet->image_area(err, 0);
    MX_ERROR_CHECK(err);

    printf( "get blob size = %ld\n", area->get_blob_size(err, "BLOBBY"));
    MX_ERROR_CHECK(err);

    printf( "get data\n");
    for (i = 128; i < 2048; i++)
    {
        area->get_blob_data(err, "BLOBBY", i * 1024, 1024, b);
        MX_ERROR_CHECK(err);

        for (j = 0; j < 1024; j++)
        {
            if (b[j] != ((i + j) % 256))
            {
                printf("ERROR IN BYTE %d (got %d expected %d)\n", 
                    i * 1024 + j,
                    b[j],
                    (i + j) % 256);
            }
        }
    }

    printf( "close document\n" );
    mx_db_client_close_wp_doc(err, doc);

    return;
abort:
    global_error_trace->print();
}

void test5()
{
    int    err;
    const mx_attribute *a;
    mx_sheet *s;

    printf("open a document\n");
    doc = mx_db_client_open_wp_doc(err, "test.doc", TRUE,
            &locked_by, &locked_host, locked_pid);
    MX_ERROR_CHECK(err);

    s = doc->sheet(err, 0);
    MX_ERROR_CHECK(err);

    while (TRUE)
    {
        s->set_int(err, MX_DB_OBJECT_OWNER_ID, 0);
        MX_ERROR_CHECK(err);

        a = s->get_readonly(err, MX_DB_OBJECT_OWNER_ID);
        MX_ERROR_CHECK(err);
    }

    return;
abort:
    global_error_trace->print();
}

void test6()
{
    int    err;
    int32 a[100];

    printf("open a document\n");

    doc = mx_db_client_open_wp_doc(err, "test.doc", TRUE,
            &locked_by, &locked_host, locked_pid);
    MX_ERROR_CHECK(err);

    while (TRUE)
    {
        doc->set_int_array(err, "INTA", a, 100);
        MX_ERROR_CHECK(err);
    }

    return;
abort:
    global_error_trace->print();
}

void test7()
{
    int    err;
    uint32 a[100];
    float f[100];

    printf("open a document\n");
    doc = mx_db_client_open_wp_doc(err, "test.doc", TRUE,
            &locked_by, &locked_host, locked_pid);
    MX_ERROR_CHECK(err);

    doc->set_id_array(err, "IDA", a, 100);
    MX_ERROR_CHECK(err);

    doc->set_float_array(err, "FA", f, 100);
    MX_ERROR_CHECK(err);

    while (TRUE)
    {
        doc->set_id_array_item(err, "IDA", 10, 100);
        MX_ERROR_CHECK(err);

        doc->set_float_array_item(err, "FA", 10, 100.0);
        MX_ERROR_CHECK(err);
    }

    return;
abort:
    global_error_trace->print();
}

void test8()
{
    int    err;
    int i;

    printf("open a document\n");
    doc = mx_db_client_open_wp_doc(err, "test.doc", TRUE,
            &locked_by, &locked_host, locked_pid);
    MX_ERROR_CHECK(err);

    while (TRUE)
    {
        for (i = 0; i < 2000; i++)
        {
            doc->add_id_array_item(err, "IDA", i);
            MX_ERROR_CHECK(err);

        }

        doc->delete_attribute(err, "IDA");
        MX_ERROR_CHECK(err);
    }

    return;
abort:
    global_error_trace->print();
}

void test9()
{
    int             i, n, err;
    mx_sheet     *sheet;


    while (TRUE)
    {
        doc = mx_db_client_open_wp_doc(err, "test.doc", TRUE,
            &locked_by, &locked_host, locked_pid);
        MX_ERROR_CHECK(err);

        n = doc->get_num_sheets(err);
        MX_ERROR_CHECK( err );

        for (i = 0; i < 2; i++)
        {
            sheet = doc->sheet(err, i);
            MX_ERROR_CHECK(err);

/*
            np = sheet->get_num_text_areas(err);
            MX_ERROR_CHECK(err);

            for (j = 0; j < np; j++)
            {
                page = sheet->text_area(err, j);
                MX_ERROR_CHECK(err);

                ol = page->get_outline_readonly(err);
                MX_ERROR_CHECK(err);
            }
*/
        }
        mx_db_client_close_wp_doc(err, doc);
        MX_ERROR_CHECK(err);

    }

    return;
abort:
    global_error_trace->print();
}

void test10()
{
    int err;

    while (TRUE)
    {
        doc = mx_db_client_open_wp_doc(err, "test.doc", TRUE,
            &locked_by, &locked_host, locked_pid);
        MX_ERROR_CHECK(err);

        mx_db_client_close_wp_doc(err, doc);
        MX_ERROR_CHECK(err);
    }

    return;
abort:
    global_error_trace->print();
}

void add_sheets(int &err)
{
    int n, i;
    mx_sheet     *s;
    mx_text_area *p;
    mx_spline_coord_t     coords[4];
    mx_quadspline_t pp;

    pp.num_coords=4;
    pp.coords=coords;
    coords[0].p.x=0.0;
    coords[0].p.y=0.0;
    coords[0].flag=TRUE;
    coords[1].p.x=210.0;
    coords[1].p.y=0.0;
    coords[1].flag=TRUE;
    coords[2].p.x=210.0;
    coords[2].p.y=297.0;
    coords[2].flag=TRUE;
    coords[3].p.x=0.0;
    coords[3].p.y=297.0;
    coords[3].flag=TRUE;


    printf("How many>");
    scanf("%d", &n);

    for (i = 0; i < n; i++)
    {
        s = doc->add_sheet(err);
        MX_ERROR_CHECK(err);

        p = s->add_text_area(err);
        MX_ERROR_CHECK(err);

        p->set_outline(err, pp);
        MX_ERROR_CHECK(err);
    }

abort:;
}


void delete_sheets(int &err)
{
    int      n, i;

    printf("How many>");
    scanf("%d", &n);

    for (i = 0; i < n; i++)
    {
        doc->delete_sheet_index(err, 0);
        MX_ERROR_CHECK(err);
    }

abort:;
}

void dump(int &err)
{
    int             i, n, np, j;
    mx_sheet     *sheet;
    mx_text_area *page;

    const mx_quadspline_t *ol;

    printf( "description = %s\n", doc->get_description());

    n = doc->get_num_sheets(err);
    MX_ERROR_CHECK( err );

    printf("num sheets = %d\n", n);

    for (i = 0; i < n; i++)
    {
        printf("sheet %d\n", i);

        sheet = doc->sheet(err, i);
        MX_ERROR_CHECK(err);

        np = sheet->get_num_text_areas(err);
        MX_ERROR_CHECK(err);

        printf("num pages = %d\n", np);

        for (j = 0; j < np; j++)
        {
            page = sheet->text_area(err, j);
            MX_ERROR_CHECK(err);

            ol = page->get_outline_readonly(err);
            MX_ERROR_CHECK(err);

            for(uint32 k = 0; k < ol->num_coords; k++ ) 
            {
                printf( "    %ld) %f,%f\n", k, ol->coords[k].p.x,ol->coords[k].p.y);
            }
        }
    }

    return;

abort:;
}

void leak_test_1(int &err)
{
    mx_db_object *s;
    int i;

    while (TRUE)
    {

        s = doc->add_sheet(err);
        MX_ERROR_CHECK(err);

        doc->delete_sheet_index(err, 0);
        MX_ERROR_CHECK(err);

        i++;

        if ((i % 1000) == 0)
        {
            printf("%d\n", i);
        }
    }
abort:;
}

void leak_test_2(int &err)
{
    mx_sheet *s;
    int i = 0;

    while (TRUE)
    {
        s = doc->add_sheet(err);
        MX_ERROR_CHECK(err);

        doc->rollback(err);
        MX_ERROR_CHECK(err);

        i++;

        if ((i % 1000) == 0)
        {
            printf("%d\n", i);
        }
    }
abort:;
}

void leak_test_3(int &err)
{
    mx_log *l;

    while (TRUE)
    {
        l = new mx_log(10, 10, 10);
        l->delete_obj(err, 123);

        l->delete_attr(err, 123, "fred");

        delete l;
    }
}

void leak_test_4(int &err)
{
    mx_hash *h;

    while (TRUE)
    {
        h = new mx_hash(10);

        h->add(err, "hello", "world");
        delete h;
    }
}

void p1(int &err)
{
    err = MX_HASH_NOT_FOUND;
}

void p2(int &err)
{
    p1(err);
    MX_ERROR_CHECK(err);
abort:;
}

void p3(int &err)
{
    p2(err);
    MX_ERROR_CHECK(err);
abort:;
}

void p4(int &err)
{
    p3(err);
    MX_ERROR_CHECK(err);
abort:;
}

void error_leak_test(int &err)
{
    while (TRUE)
    {
        err = MX_ERROR_OK;
        p4(err);
        MX_ERROR_CLEAR(err);
    }
}

static void blob_rollback_test(int &err)
{
    mx_wp_document *doc;
    mx_sheet *sheet;
    int i;
    unsigned char b[20*1024];

    // create a document
    printf("create a document\n");
    doc = mx_db_client_open_wp_doc(err, "blob.doc", TRUE,
            &locked_by, &locked_host, locked_pid);
    MX_ERROR_CHECK(err);

    printf("add a sheet\n");
    // add a sheet
    sheet = doc->add_sheet(err);
    MX_ERROR_CHECK( err );

    printf("add a blob to the sheet\n");
    // add a blob to the sheet
    sheet->create_blob(err, "BLOBBY");
    MX_ERROR_CHECK(err);

    printf("set some data in the blob\n");
    // set some data in the blob
    sheet->set_blob_size(err, "BLOBBY", 1024*10);
    MX_ERROR_CHECK(err);

    for (i = 0; i < 1024*10; i++)
    {
        b[i] = i % 256;
    }

    sheet->set_blob_data(err, "BLOBBY", 0, 1024*10, b);
    MX_ERROR_CHECK(err);

    printf("commit\n");
    // commit
    doc->commit(err);
    MX_ERROR_CHECK(err);

    // close doc
    mx_db_client_close_wp_doc(err, doc);
    MX_ERROR_CHECK(err);

    printf("open document\n");
    // open document
    doc = mx_db_client_open_wp_doc(err, "blob.doc", TRUE,
            &locked_by, &locked_host, locked_pid);
    MX_ERROR_CHECK(err);

    sheet = doc->sheet(err, 0);
    MX_ERROR_CHECK(err);

    printf("check blob is still the same\n");
    // check blob is still the same
    if (sheet->get_blob_size(err, "BLOBBY") != 1024*10)
    {
        printf("blob size wrong after re-open\n");
    }
    MX_ERROR_CHECK(err);

    sheet->get_blob_data(err, "BLOBBY", 0, 1024*10, b);
    MX_ERROR_CHECK(err);

    for (i = 0; i < 1024*10; i++)
    {
        if (b[i] != (i % 256))
        {
            printf("blob data wrong after re-open\n");
        }
    }

    printf("change it\n");
    // change it
    sheet->set_blob_size(err, "BLOBBY", 1024*20);
    MX_ERROR_CHECK(err);

    for (i = 0; i < 1024*20; i++)
    {
        b[i] = (i + 10) % 256;
    }

    sheet->set_blob_data(err, "BLOBBY", 0, 1024*20, b);
    MX_ERROR_CHECK(err);

    printf("add a new blob\n");
    // add a new blob
    sheet->create_blob(err, "BLOBBY2");
    MX_ERROR_CHECK(err);

    // set some data in the blob
    sheet->set_blob_size(err, "BLOBBY2", 1024*30);
    MX_ERROR_CHECK(err);

    // check it's all ok
    if (sheet->get_blob_size(err, "BLOBBY2") != 1024*30)
    {
        printf("second blob size wrong\n");
    }
    MX_ERROR_CHECK(err);

    sheet->get_blob_data(err, "BLOBBY", 0, 1024*20, b);
    MX_ERROR_CHECK(err);

    for (i = 0; i < 1024*20; i++)
    {
        if (b[i] != (i + 10) % 256)
        {
            printf("blob data wrong after change\n");
        }
    }

    printf("rollback\n");
    // rollback
    doc->rollback(err);
    MX_ERROR_CHECK(err);

    // make sure the new blob has gone
    (void)sheet->get_blob_size(err, "BLOBBY2");
    if (err == MX_ERROR_OK)
    {
        MX_ERROR_CLEAR(err);
        printf("error, new blob still there\n");
    }

    // make sure the old blob size & data is restored
    if (sheet->get_blob_size(err, "BLOBBY") != 10*1024)
    {
        printf("error, first blob size not restored\n");
    }

    sheet->get_blob_data(err, "BLOBBY", 0, 1024*10, b);
    MX_ERROR_CHECK(err);

    for (i = 0; i < 1024*10; i++)
    {
        if (b[i] != (i % 256))
        {
            printf("error, blob data not restored\n");
        }
    }

    // close document
    mx_db_client_close_wp_doc(err, doc);
    MX_ERROR_CHECK(err);

abort:;
}

void test16()
{
    int    err = MX_ERROR_OK;
    int    i;

    while( 1 ) 
    {
        printf( "open document......1\n");
        printf( "add N sheets.......2\n");
        printf( "delete N sheets....3\n");
        printf( "dump document......4\n");
        printf( "rollback...........5\n");
        printf( "commit.............6\n");
        printf( "close document.....7\n");
        printf( "leak test 1........8\n");
        printf( "leak test 2........9\n");
        printf( "log leak test......10\n");
        printf( "hash leak test.....11\n");
        printf( "error leak test ...12\n");
        printf( "blob rollback test.13\n");

        printf( "which>" );
        scanf( "%d",&i );

        switch( i ) 
        {
            case 1 : doc = mx_db_client_open_wp_doc(err, "test.doc", TRUE,
                                &locked_by, &locked_host, locked_pid);
                     MX_ERROR_CHECK(err);

                     doc->set_info(err, "a test document", "andrew", "0.1");
                     MX_ERROR_CHECK(err);
                     break;
            case 2 : add_sheets(err);
                     MX_ERROR_CHECK(err);
                     break;
            case 3 : delete_sheets(err);
                     MX_ERROR_CHECK(err);
                     break;
            case 4 : dump(err);
                     MX_ERROR_CHECK(err);
                     break;
            case 5 : doc->rollback(err);
                     MX_ERROR_CHECK(err);
                     break;
            case 6 : doc->commit(err);
                     MX_ERROR_CHECK(err);
                     break;
            case 7 : mx_db_client_close_wp_doc(err, doc);
                     MX_ERROR_CHECK(err);
                     break;
            case 8 : leak_test_1(err);
                     MX_ERROR_CHECK(err);
                     break;
            case 9 : leak_test_2(err);
                     MX_ERROR_CHECK(err);
                     break;
            case 10: leak_test_3(err);
                     MX_ERROR_CHECK(err);
                     break;
            case 11: leak_test_4(err);
                     MX_ERROR_CHECK(err);
                     break;
            case 12: error_leak_test(err);
                     MX_ERROR_CHECK(err);
                     break;
            case 13: blob_rollback_test(err);
                     MX_ERROR_CHECK(err);
                     break;
        }
    }
abort:
    global_error_trace->print();
}

void test17()
{
    mx_wp_document *doc;
    mx_sheet *sheet;
    mx_table_area *area;
    mx_text_area *ta;
    int nr, nc, x, y;
    int i, j, err = MX_ERROR_OK;

    while (TRUE)
    {
        printf( "open document......1\n");
        printf( "commit.............2\n");
        printf( "close..............3\n");
        printf( "create table.......4\n");
        printf( "print table........5\n");
        printf( "insert rows........6\n");
        printf( "insert columns.....7\n");
        printf( "delete rows........8\n");
        printf( "delete columns.....9\n");

        printf( "which>" );
        scanf( "%d",&i );

        switch( i ) 
        {
            case 1 :
                doc = mx_db_client_open_wp_doc(err, "table.doc", TRUE,
                            &locked_by, &locked_host, locked_pid);
                MX_ERROR_CHECK(err);
                break;
            case 2 :
                doc->commit(err);
                MX_ERROR_CHECK(err);
                break;
            case 3 :
                mx_db_client_close_wp_doc(err, doc);
                MX_ERROR_CHECK(err);
                break;
            case 4 :
                sheet = doc->add_sheet(err);
                MX_ERROR_CHECK( err );

                area = sheet->add_table_area(err);
                MX_ERROR_CHECK( err );

                area->set_num_rows(err, 10);
                MX_ERROR_CHECK( err );

                area->set_num_columns(err, 10);
                MX_ERROR_CHECK( err );

                for (i = 0; i < 10; i++)
                {
                    for (j = 0; j < 10; j++)
                    {
                        ta = area->get_cell(err, i, j);
                        MX_ERROR_CHECK( err );

                        ta->set_int(err, "row", i);
                        MX_ERROR_CHECK( err );

                        ta->set_int(err, "column", j);
                        MX_ERROR_CHECK( err );
                    }
                }

                break;
            case 5 :
                sheet = doc->sheet(err, 0);
                MX_ERROR_CHECK( err );

                area = sheet->table_area(err, 0);
                MX_ERROR_CHECK( err );

                nr = area->get_num_rows(err);
                MX_ERROR_CHECK( err );

                nc = area->get_num_columns(err);
                MX_ERROR_CHECK( err );

                for (i = 0; i < nr; i++)
                {
                    for (j = 0; j < nc; j++)
                    {
                        ta = area->get_cell(err, i, j);
                        MX_ERROR_CHECK( err );

                        x = ta->get_default_int(err, "row", -1);
                        MX_ERROR_CHECK( err );

                        y = ta->get_default_int(err, "column", -1);
                        MX_ERROR_CHECK( err );

                        printf("(%02d,%02d) ", x, y);
                    }
                    printf("\n");
                }

                break;
            case 6 :
                printf("how many rows>");
                scanf("%d",&i);
                printf("from row position>");
                scanf("%d",&j);

                sheet = doc->sheet(err, 0);
                MX_ERROR_CHECK( err );

                area = sheet->table_area(err, 0);
                MX_ERROR_CHECK( err );

                area->insert_rows(err, j, i);
                MX_ERROR_CHECK(err);
                
                break;
            case 7 :
                printf("how many columns>");
                scanf("%d",&i);
                printf("from column position>");
                scanf("%d",&j);

                sheet = doc->sheet(err, 0);
                MX_ERROR_CHECK( err );

                area = sheet->table_area(err, 0);
                MX_ERROR_CHECK( err );

                area->insert_columns(err, j, i);
                MX_ERROR_CHECK(err);
                
                break;
            case 8 :
                printf("delete how many rows>");
                scanf("%d",&i);
                printf("from row position>");
                scanf("%d",&j);

                sheet = doc->sheet(err, 0);
                MX_ERROR_CHECK( err );

                area = sheet->table_area(err, 0);
                MX_ERROR_CHECK( err );

                area->remove_rows(err, j, i);
                MX_ERROR_CHECK(err);
                
                break;
            case 9 :
                printf("delete how many columns>");
                scanf("%d",&i);
                printf("from column position>");
                scanf("%d",&j);

                sheet = doc->sheet(err, 0);
                MX_ERROR_CHECK( err );

                area = sheet->table_area(err, 0);
                MX_ERROR_CHECK( err );

                area->remove_columns(err, j, i);
                MX_ERROR_CHECK(err);
                break;
        }
    }

abort:
    global_error_trace->print();
}


static void thrash_attrs(int &err)
{
    mx_wp_document *doc;
    mx_sheet *sheets[10];

    int inta[10][100], i, sn, res, stra[10][100], start, end;
    char name[30], str[100];
    const char *cres;
    unsigned char *blob[10];
    int size, sizes[10];
    unsigned char buffer[102400];

    unlink("thrash.doc");
    doc = mx_db_client_open_wp_doc(err, "thrash.doc", TRUE,
            &locked_by, &locked_host, locked_pid);
    MX_ERROR_CHECK(err);

    for (i = 0; i < 10; i++)
    {
        sheets[i] = doc->add_sheet(err);
        MX_ERROR_CHECK( err );
        printf("sheet %d id = %d\n", i, sheets[i]->get_db_id(err));
        blob[i] = new unsigned char[100*1024];
        memset(blob[i], 0, 100*1024);

        sizes[i] = 100*1024;

        sheets[i]->create_blob(err, "BLOBBY");
        MX_ERROR_CHECK( err );

        sheets[i]->set_blob_size(err, "BLOBBY", 100*1024);
        MX_ERROR_CHECK( err );
    }

    srand(100);
    for (sn = 0; sn < 10; sn++)
    {
        for (i = 0; i < 100; i++)
        {
            inta[sn][i] = -1;
            stra[sn][i] = -1;
        }
    }

    while (TRUE)
    {
        //printf(".");
        //fflush(stdout);
        sn = abs(rand()) % 10;
        switch (abs(rand()) % 4)
        {
            case 0 :
                printf("ints\n");
                i = abs(rand()) % 100;
                inta[sn][i] = abs(rand());
                sprintf(name, "int%d", i);
                sheets[sn]->set_int(err, name, inta[sn][i]);
                MX_ERROR_CHECK(err);
                break;
            case 1 :
                printf("strings\n");
                i = abs(rand()) % 100;
                stra[sn][i] = abs(rand());
                sprintf(name, "str%d", i);
                sprintf(str, "this is string %d etc etc etc etc", stra[sn][i]);
                sheets[sn]->set_string(err, name, str);
                MX_ERROR_CHECK(err);

                cres = sheets[sn]->get_default_string(err, name, "none");
                MX_ERROR_CHECK(err);

                if (strcmp(cres, str) != 0)
                {
                    printf("!!string value %d,%d wrong. got '%s', expected '%s'\n",
                        i, sn, cres, str);
                    return;
                }
                break;
            case 2 :
                printf("blobs\n");
                // set some blob data
                size = abs(rand())%(100*1024);
                if (size == 0)
                {
                    break;
                }
                if (size > sizes[sn])
                {
                    memset(blob[sn] + sizes[sn], 0, 102400-sizes[sn]);
                }
                else
                {
                    memset(blob[sn] + size, 0, 102400-size);
                }
                sizes[sn] = size;

                printf("set sheet %d blob size to %d\n",sn,sizes[sn]);

                sheets[sn]->set_blob_size(err, "BLOBBY", sizes[sn]);
                MX_ERROR_CHECK(err);

                // try to re-create the blob
                sheets[sn]->create_blob(err, "BLOBBY");
                if (err == MX_DB_CLIENT_CACHE_DUPLICATE_BLOB) 
                {
                    MX_ERROR_CLEAR(err);
                }
                else 
                {
                    MX_ERROR_CHECK(err);
                    printf("blob %d has gone away\n", sn);
                }

                start = abs(rand())%sizes[sn];
                end = abs(rand())%sizes[sn];
                if (start==end)
                {
                    break;
                }
                else
                {
                    if (start>end)
                    {
                        i = start; start = end; end = i;
                    }
                    for (i = start; i < end; i++)
                    {
                        blob[sn][i] = abs(rand()) % 256;
                    }


                    sheets[sn]->set_blob_data(err, "BLOBBY", start, end-start, blob[sn]+start);
                    MX_ERROR_CHECK(err);

                    printf("set sheet %d blob section %d to %d\n",sn,start,end);
                }
                break;
            case 3 :
                // set a float array
                break;
        }

        // check the ints
        for (sn = 0; sn < 10; sn++)
        {
        for (i = 0; i < 100; i++)
        {
            sprintf(name, "int%d", i);
            res = sheets[sn]->get_default_int(err, name, -1);
            MX_ERROR_CHECK(err);
            if (inta[sn][i] != res)
            {
#if 0
                printf("integer value %d,%d wrong. got %d, expected %d\n",
                    i, sn, res, inta[i]);
#endif
                return;
            }
        }
        }

        // check the strings
        for (sn = 0; sn < 10; sn++)
        {
        for (i = 0; i < 100; i++)
        {
            sprintf(name, "str%d", i);
            cres = sheets[sn]->get_default_string(err, name, "none");
            MX_ERROR_CHECK(err);

            if (stra[sn][i] == -1)
            {
                if (strcmp(cres, "none") != 0)
                {
                    printf("string value %d,%d wrong. got %s, expected %s\n",
                        i, sn, cres, "none");
                    return;
                }
            }
            else
            {
                sprintf(str, "this is string %d etc etc etc etc", stra[sn][i]);
                if (strcmp(cres, str) != 0)
                {
                    printf("string value %d,%d wrong. got '%s', expected '%s'\n",
                        i, sn, cres, str);
                    return;
                }
            }
        }
        }

        if ((abs(rand()) % 80) == 20)
        {
            printf("commit.....");
            doc->commit(err);
            MX_ERROR_CHECK(err);

            mx_db_client_close_wp_doc(err, doc);
            MX_ERROR_CHECK(err);

            doc = mx_db_client_open_wp_doc(err, "thrash.doc", TRUE,
                        &locked_by, &locked_host, locked_pid);
            MX_ERROR_CHECK(err);

            for (sn = 0; sn < 10; sn++)
            {
                sheets[sn] = doc->sheet(err, sn + 2);
                MX_ERROR_CHECK( err );
            }

            printf("done\n");
        }

        // check blob data
        for (sn = 0; sn < 10; sn++)
        {
            sheets[sn]->get_blob_data(err, "BLOBBY", 0, sizes[sn], buffer);
            MX_ERROR_CHECK(err);

            size = sheets[sn]->get_blob_size(err, "BLOBBY");
            MX_ERROR_CHECK(err);

            if (size != sizes[sn])
            {            
                printf("sheet %d size wrong, found %d expected %d\n",
                    sn, sizes[sn], size);
            }

            for (i = 0; i < sizes[sn]; i++)
            {

                if (blob[sn][i] != buffer[i])
                {
                    printf("sheet %d, blob byte %d wrong, found %d expected %d, size = %d\n",
                        sn, i, buffer[i], blob[sn][i], sizes[sn]);
                    return;
                }
            }
        }

    }

abort:;
}

static void thrash_objects(int &err)
{
    err = MX_ERROR_OK;
}

static void thrash_temp_doc(int &err)
{
    while (TRUE)
    {
        doc = mx_db_client_open_temporary_wp_doc(err);
        MX_ERROR_CHECK(err);

        mx_db_client_close_wp_doc(err, doc);
        MX_ERROR_CHECK(err);
    }
abort:;
}

void test18()
{
    int i, err = MX_ERROR_OK;

    while (TRUE)
    {
        printf( "thrash attributes and blobs..1\n");
        printf( "thrash objects...............2\n");
        printf( "thrash temp doc..............3\n");

        printf( "which>" );
        scanf( "%d",&i );

        switch( i ) 
        {
            case 1 :
                thrash_attrs(err);
                MX_ERROR_CHECK(err);
                break;
            case 2 :
                thrash_objects(err);
                MX_ERROR_CHECK(err);
                break;
            case 3 :
                thrash_temp_doc(err);
                MX_ERROR_CHECK(err);
                break;
        }
    }

abort:
    global_error_trace->print();
}

void test19()
{
    int err = MX_ERROR_OK;

    doc = mx_db_client_open_temporary_wp_doc(err);
    MX_ERROR_CHECK(err);

    mx_db_client_close_wp_doc(err, doc);
    MX_ERROR_CHECK(err);

    return;
abort:
    global_error_trace->print();
}

void test20()
{
    mx_font *f;
    mx_char_style *c;
    mx_paragraph_style *p1;
    mx_paragraph_style *p2;

    int err = MX_ERROR_OK;

    while (TRUE)
    {
        // normal style - utopia 12
        f = new mx_font(err, "Utopia", 12, mx_normal);
        MX_ERROR_CHECK(err);

        c = new mx_char_style(*f);
        MX_ERROR_CHECK(err);

        p1 = new mx_paragraph_style();
        MX_ERROR_CHECK(err);

        p1->set_char_style(*c);
        p1->set_name("default");

        p2 = new mx_paragraph_style(*p1);
        MX_ERROR_CHECK(err);

        delete f;
        delete c;
        delete p1;
        delete p2;

    }
    return;
abort:;
    global_error_trace->print();
}

void test21()
{
    float p, q;
    int err = MX_ERROR_OK;
    mx_wp_document *doc1, *doc2;
    mx_image_area *ia;
    mx_paragraph_style *ps;
    mx_area *new_area;

    doc1 = mx_db_client_open_temporary_wp_doc(err);
    MX_ERROR_CHECK(err);

    doc2 = mx_db_client_open_temporary_wp_doc(err);
    MX_ERROR_CHECK(err);

    ps = doc1->get_default_style(err);
    MX_ERROR_CHECK(err);

    ia = new mx_image_area(doc1->get_docid(), ps);

    ia->set_outline(err, 123, 456);
    MX_ERROR_CHECK(err);

    new_area = ia->duplicate();

    p = new_area->get_width(err);
    MX_ERROR_CHECK(err);
    
    q = new_area->get_height(err);
    MX_ERROR_CHECK(err);
    
    printf("doc1 width = %f, height = %f\n", p, q);
    fflush(stdout);
    
    new_area->move_to_document(err, doc2);
    MX_ERROR_CHECK(err);

    p = new_area->get_width(err);
    MX_ERROR_CHECK(err);

    q = new_area->get_height(err);
    MX_ERROR_CHECK(err);
    
    printf("doc2 width = %f, height = %f\n", p, q);
    fflush(stdout);

    return;
abort:
    global_error_trace->print();
}

void test22()
{
	int err = MX_ERROR_OK;

	mx_disk_heap heap(err, 100 * 1024 * 1024);
	uint32 alloced[100];
	int i, j, x, n;
	uint8 buffer[100000];

	while (TRUE)
	{
		printf("."); fflush(stdout);

		for (i = 0; i < 100; i++)
		{
			alloced[i] = heap.alloc(err, 1024 + (i % 10) * 100);
			MX_ERROR_CHECK(err);
		}

		for (x = 0; i < 10; x++)
		{
			for (i = 0; i < 100; i++)
			{
				n = 1024 + (i % 10) * 100;
				memset(buffer, i, n);

				heap.copy_to(err, alloced[i], buffer, n);
				MX_ERROR_CHECK(err);
			}

			for (i = 0; i < 100; i++)
			{
				n = 1024 + (i % 10) * 100;
				memset(buffer, 0, n);

				heap.copy_from(err, alloced[i], buffer, n);
				MX_ERROR_CHECK(err);

				for (j = 0; j < n; j++)
				{
					if (buffer[j] != i)
					{
						printf("ERROR:block %d has wrong byte at index %d\n",
							i, j);
						return;
					}
				}
			}
		}

		for (i = 0; i < 100; i++)
		{
			heap.free(err, alloced[i]);
			MX_ERROR_CHECK(err);
		}
	}

abort:
    global_error_trace->print();
}

main(int argc, char *argv[])
{
    int    err;
    int    i;

    printf("sizeof(char_style)=%d\n", sizeof(mx_char_style));
    printf("sizeof(char_style_mod)=%d\n", sizeof(mx_char_style_mod));

    global_maxhome = getenv("MAXHOME");
    if (global_maxhome == NULL)
    {
        global_maxhome = new char[200];
        strcpy(global_maxhome, "/usr/local/maxwell");
    }

    for (i = 0; i < argc; i++)
    {
        printf("%s\n", argv[i]);
    }

    printf("block entry size = %d\n", sizeof(entry_t));
    printf("block header size = %d\n", sizeof(block_header_t));
    printf("number of entries per block = %d\n", NUM_BLOCK_ENTRIES);
    
    (void)mx_db_client_login(err);
    MX_ERROR_CHECK( err );

    while( 1 ) {
        printf( "add N pages to document.........1\n" );
        printf( "dump document contents..........2\n" );
        printf( "blob test.......................3\n" );
        printf( "check blob data.................4\n" );
        printf( "scalar attribute leak test......5\n" );
        printf( "vector attribute leak test 1....6\n" );
        printf( "vector attribute leak test 2....7\n" );
        printf( "vector attribute leak test 3....8\n" );
        printf( "read leak test..................9\n" );
        printf( "file leak test..................10\n" );
        printf( "add N chained pages to document.11\n" );
        printf( "dump chained pages..............12\n" );
        printf( "add N linked pages to document..13\n" );
        printf( "dump linked pages...............14\n" );
        printf( "delete N sheets from document...15\n" );
        printf( "transaction tests...............16\n" );
        printf( "table area tests................17\n" );
        printf( "thrash tests....................18\n" );
        printf( "open temp doc...................19\n" );
        printf( "styles..........................20\n" );
        printf( "move image area between docs....21\n" );
        printf( "disk heap test..................22\n" );

        printf( "which>" );
        scanf( "%d",&i );

        switch( i ) {
            case 1 : test1();
                     break;
            case 2 : test2();
                     break;
            case 3 : test3();
                     break;
            case 4 : test4();
                     break;
            case 5 : test5();
                     break;
            case 6 : test6();
                     break;
            case 7 : test7();
                     break;
            case 8 : test8();
                     break;
            case 9 : test9();
                     break;
            case 10 : test10();
                      break;
            case 11 : test11();
                      break;
            case 12 : test12();
                      break;
            case 13 : test13();
                      break;
            case 14 : test14();
                      break;
            case 15 : test15();
                      break;
            case 16 : test16();
                      break;
            case 17 : test17();
                      break;
            case 18 : test18();
                      break;
            case 19 : test19();
                      break;
            case 20 : test20();
                      break;
            case 21 : test21();
                      break;
            case 22 : test22();
                      break;
        }
    }
    return 0;
abort:
    global_error_trace->print();
    return -1;
}
