/* remap.c:  convert heightfield data to a different projection      
 *          read in array of height values,
 *          convert
 *          write out in specified format
 *           John P. Beale 7/18/95                           
 *          GIF library routines from gd library             */

#include <stdio.h>                      /* fopen() sprintf() etc. */
#include <stdlib.h>                     /* strtod() */
#include <math.h>                       /* math stuff */
#include <string.h>                     /* strcpy() */
#include "hcon.h"
#include "hf_io.h"

void hf_print();                        /* print out hf array to stdout */
void remap(PTYPE **hf, PTYPE **hf2);    /* remap array  */
void remapy(PTYPE **hf, PTYPE **hf2);    /* remap array  */

unsigned int xsize, ysize;     /* size of hf array (input xsize+2) */
unsigned int xsize2, ysize2;   /* size of rescaled array hf2 */
PTYPE *hf;                            /* height field array */
PTYPE *hf2;                    /* rescale array */
PTYPE fmin = 0, fmax = 1.0;                 /* scaling limits */
Boolean norenorm = FALSE;
Boolean negate = FALSE;

/* ---------------------------------- */

int main(int argc, char **argv)
{ char *usage = 
"remap: A Heightfield Converter v0.1a (c) <beale@jump.stanford.edu> 1995\n\
Usage: remap <infile> <outfile> \n\
  Determines filetype based on extension, eg:  remap file1.tga file1.pot\n\
  POT,TGA are 16-bit formats, MAT is Matlab floating point format.\n";
  
  char fname_in[160];
  char fname_out[160];
  char *ext1, *ext2;

  int rval;
  int inlen, outlen, otype, itype;

  if ((argc < 3) || (argc > 3)) {
	fprintf(stderr,"%s",usage);
	exit(1);
      }
 
  strcpy(fname_in,argv[1]);
  strcpy(fname_out,argv[2]);

  inlen = strlen(fname_in);
  outlen = strlen(fname_out);
  
  ext1 = (char *) malloc((size_t)sizeof(char)*5);
  ext2 = (char *) malloc((size_t)sizeof(char)*5);

  strncpy(ext1,(char *)(&fname_in[inlen-4]),4);
  strncpy(ext2,(char *)(&fname_out[outlen-4]),4);

  if ((ext1==NULL) || (ext2==NULL)) { 
    fprintf(stderr,"filenames must have type extension, eg. topo.gif\n");
    exit(1);
  }

  if       (strcmp(ext1,".gif")==0) itype = GIF;  
   else if (strcmp(ext1,".pot")==0) itype = POT;
   else if (strcmp(ext1,".tga")==0) itype = TGA;
   else if (strcmp(ext1,".pgm")==0) itype = PGM;
   else if (strcmp(ext1,".pg8")==0) itype = PG8;
   else if (strcmp(ext1,".mat")==0) itype = MAT;
   else if (strcmp(ext1,".oct")==0) itype = OCT;
   else if (strcmp(ext1,".png")==0) itype = PNG;
   else { 
     itype = PGM;       /* default type */
     fprintf(stderr,"Warning: %s has a missing or unknown file extension\n",argv[1]);
   }

  if       (strcmp(ext2,".gif")==0) otype = GIF;  
   else if (strcmp(ext2,".pot")==0) otype = POT;
   else if (strcmp(ext2,".tga")==0) otype = TGA;
   else if (strcmp(ext2,".pgm")==0) otype = PGM;
   else if (strcmp(ext2,".pg8")==0) otype = PG8;
   else if (strcmp(ext2,".mat")==0) otype = MAT;
   else if (strcmp(ext2,".oct")==0) otype = OCT;
   else if (strcmp(ext2,".png")==0) otype = PNG;
   else { 
     fprintf(stderr,"Error: %s has a missing or unknown file extension\n",argv[2]);
     exit(1);
   }

					/* read in heightfield */
   rval = read_hf(&hf, &xsize, &ysize, fname_in, itype);
   if (rval != 0)
     exit(rval);

   remapy((PTYPE **)&hf, (PTYPE **)&hf2);   /* rescale array to [0..1] */
   xsize = xsize2;
   ysize = ysize2;
   
   printf("%s: [%d x %d] \n",fname_out,xsize,ysize);
   
   if (xsize < 9) hf_print();  /* print small arrays */

   rval = write_hf(hf2, xsize, ysize, fname_out, otype);
   
   free(hf);            /* clear allocated memory */
   free(hf2);
   return(rval);    /* return a value */
} /* end main() */

void hf_print()                /* print out hf array on stdout */
{
  int ix,iy;

   printf("Heightfield array: %dx%d\n",xsize,ysize);
   for (iy = 0; iy<ysize; iy++) {
    printf("%d:",iy);
    for (ix = 0; ix<xsize; ix++) {
	printf(" %1.2f",El(hf,ix,iy));
    }
     printf("\n");
   }
}

/* -------------------------------------------------------------------- */
/* remap() --  remap in hf1 into hf2        */
/* -------------------------------------------------------------------- */

void remap(PTYPE **hf1, PTYPE **hf2)
{
 int xi;
 int x,y;
 float xn, yn, xnp, xni, ratio;
 size_t memspace;

 memspace = (size_t) xsize*ysize*sizeof(PTYPE);

 hf2[0] = (PTYPE *)malloc(memspace);
 if (hf2[0]==NULL) {
   fprintf(stderr,"could not allocate %lu bytes for output array.\n",
	 (unsigned long)memspace);
   exit(1);
 }

 xsize2 = xsize;        /* size of new image array */
 ysize2 = ysize;

#define spmap(xf)  sqrt(1-((xf)*(xf)))     /* mapping function */

   for (y = 0; y<ysize; y++) {
    for (x = 0; x<xsize; x++) {
      xn = (2.0 * x / xsize) - 1.0;    /* xn on [-1..1] */
      yn = (2.0 * y / ysize) - 1.0;
      xnp = spmap(yn) * xn;      /* xnp on [-1..1] */
      xni = (xnp+1.0)*(0.5*xsize);    /* xni on [0..xsize] */
      xi = floor(xni);
      ratio = xni - xi;
      if ((xi+1) >= xsize) xi = xsize-2;
      
      if (y==4) printf("%d:%d  ",x,xi);
      El(hf2[0],x,y) = ratio*El(hf1[0],xi+1,y) + (1-ratio)*El(hf1[0],xi,y);
    }
   }
   
   for (y = 0; y<ysize; y++) {         /* copy hf2 into hf1 */
    for (x = 0; x<xsize; x++) {
      El(hf1[0],x,y) = El(hf2[0],x,y);
    }
   }
   
   
   printf("\n");
}       /* end remap() */


void remapy(PTYPE **hf1, PTYPE **hf2)
{
 int yi;
 int x,y;
 float xn, yn, ynp, yni, ratio;
 size_t memspace;

 memspace = (size_t) xsize*ysize*sizeof(PTYPE);

 hf2[0] = (PTYPE *)malloc(memspace);
 if (hf2[0]==NULL) {
   fprintf(stderr,"could not allocate %lu bytes for output array.\n",
	 (unsigned long)memspace);
   exit(1);
 }

 xsize2 = xsize;        /* size of new image array */
 ysize2 = ysize;

#define spmap(xf)  sqrt(1-((xf)*(xf)))     /* mapping function */

   for (y = 0; y<ysize; y++) {
    for (x = 0; x<xsize; x++) {
      xn = (2.0 * x / xsize) - 1.0;    /* xn on [-1..1] */
      yn = (2.0 * y / ysize) - 1.0;
      ynp = spmap(xn) * yn;           /* ynp on [-1..1] */
      yni = (ynp+1.0)*(0.5*ysize);    /* yni on [0..xsize] */
      yi = floor(yni);
      ratio = yni - yi;
      if ((yi+1) >= ysize) yi = ysize-2;
      
      El(hf2[0],x,y) = ratio*El(hf1[0],x,yi+1) + (1-ratio)*El(hf1[0],x,yi);
    }
   }
   
   for (y = 0; y<ysize; y++) {         /* copy hf2 into hf1 */
    for (x = 0; x<xsize; x++) {
      El(hf1[0],x,y) = El(hf2[0],x,y);
    }
   }
   
   
   printf("\n");
}       /* end remapy() */

