/* hread.c: 
            read in a heightfield */
/*          GIF library routines from gd library             */

#include <stdio.h>                      /* fopen() sprintf() etc. */
#include <stdlib.h>                     /* strtod() */
#include <math.h>                       /* math stuff */
#include <string.h>                     /* strcpy() */
#include "hcon.h"
#include "hf_io.h"

int gen_sphere(double f_scale, int maxlevel, char *fname_out);

U xsize, ysize;     /* size of hf array (input xsize+2) */
PTYPE *hf;                            /* height field array */
PTYPE fmin = 0, fmax = 1.0;                 /* scaling limits */
Boolean norenorm = FALSE;
Boolean negate = FALSE;
Boolean rescale = FALSE;
Boolean raw_out = FALSE;

/* ---------------------------------- */

int main(int argc, char **argv)
{ char *usage = 
"orb: An HF -> Spherical HF converter v0.1a (c) John P. Beale 1995\n\
Usage: orb <infile> <outfile> <scale> <levels>\n\
  Available input formats: PGM|MAT|GIF|POT|TGA, output RAW or INC\n\
  Determines filetype based on extension, eg:  orb file1.tga file1.inc 1 4\n\
  POT,TGA are 16-bit formats, MAT is Matlab floating point format.\n\
  scale is radial extension of hf from unit sphere; eg 2.5 for huge bumps\n\
  levels is number of recursive tesselations of the sphere; watch out...\n\
  9th level will generate a 40 megabyte include file\n";
  
  char fname_in[160];
  char fname_out[160];
  char *ext1, *ext2;
  char *buf;
  int rval;
  int inlen, outlen, itype;
  int mlev = 4;
  double feature_scale = 0.1;

  if ((argc < 3) || (argc > 5)) {
	fprintf(stderr,"%s",usage);
	exit(1);
      }
 
  strcpy(fname_in,argv[1]);
  strcpy(fname_out,argv[2]);


  if (argc > 3) {
    feature_scale = (PTYPE) strtod(argv[3], &buf);
    }
  if (argc > 4) {
    mlev = (int) strtod(argv[4], &buf);
    }

  printf("Height scale = %5.3f  Recursion level = %d\n",feature_scale,mlev);
  inlen = strlen(fname_in);
  outlen = strlen(fname_out);
  
  ext1 = (char *) malloc((size_t)sizeof(char)*5);
  strncpy(ext1,(char *)(&fname_in[inlen-4]),4);
  ext2 = (char *) malloc((size_t)sizeof(char)*5);
  strncpy(ext2,(char *)(&fname_out[outlen-4]),4);

  if ((ext1==NULL)) { 
    fprintf(stderr,"filenames must have type extension, eg. topo.gif\n");
    exit(1);
  }

  if       (strcmp(ext1,".gif")==0) itype = GIF;  
   else if (strcmp(ext1,".pot")==0) itype = POT;
   else if (strcmp(ext1,".tga")==0) itype = TGA;
   else if (strcmp(ext1,".pgm")==0) itype = PGM;
   else if (strcmp(ext1,".pg8")==0) itype = PG8;
   else if (strcmp(ext1,".mat")==0) itype = MAT;
   else if (strcmp(ext1,".oct")==0) itype = OCT;
   else if (strcmp(ext1,".png")==0) itype = PNG;
   else { 
     itype = PGM;       /* default type */
     fprintf(stderr,"Warning: %s has a missing or unknown file extension\n",argv[1]);
   }

  if (strcmp(ext2,".raw")==0) raw_out = TRUE;

					/* read in heightfield */
   rval = read_hf(&hf, &xsize, &ysize, fname_in, itype);
   if (rval != 0)
     exit(rval);

   printf("%s: [%d x %d] \n",fname_in,xsize,ysize);
   if (raw_out)   printf("Generating raw triangle file...");
    else printf("Generating POV .inc triangle file...");
   fflush(stdout);

   gen_sphere(feature_scale, mlev ,fname_out);
   printf("\n"); 
  
   return(rval);    /* return a value */
} /* end main() */

void hf_print()                /* print out hf array on stdout */
{
  int ix,iy;

   printf("Heightfield array: %dx%d\n",xsize,ysize);
   for (iy = 0; iy<ysize; iy++) {
    printf("%d:",iy);
    for (ix = 0; ix<xsize; ix++) {
	printf(" %1.2f",El(hf,ix,iy));
    }
     printf("\n");
   }
}



