/* hf_io.c   do input/output on heightfield data  JPB july 22 1995 */

#include <stdio.h>                      /* fopen() sprintf() etc. */
#include <stdlib.h>                     /* strtod() */
#include <math.h>                       /* math stuff */
#include <string.h>                     /* strcpy() */
#include "hcon.h"
#include "hf_io.h"

extern U xsize, ysize;
extern PTYPE *hf;

int hf_write(char *fname)      /* print out hf array to PGM ascii */
{
  int ix,iy;
  long idx;
  FILE *fpout;
  int rval;
  int maxheight = 255;        /* maximum value of array */
                                                /* open input file */
  if ((fpout = fopen(fname,"w")) == NULL)
   { perror("Could not open output file ");
     exit(1);
   }

  fprintf(fpout,"P2\n");
  fprintf(fpout,"%d %d\n",xsize,ysize);
  fprintf(fpout,"%d\n",maxheight);

  idx = 0;
   for (iy = 0; iy<ysize; iy++) {
    for (ix = 0; ix<xsize; ix++) {
        fprintf(fpout,"%d ",(int) (maxheight*hf[idx++]));
    }
     fprintf(fpout,"\n");
   }
   rval = 0;
   return(rval);
}

/* oct_write()  --  print out hf array to Octave ascii fmt */

int oct_write(char *fname, char *mname)  
{
  int ix,iy;
  long idx;
  FILE *fpout;
  int rval;
                                                /* open input file */
  if ((fpout = fopen(fname,"w")) == NULL)
   { perror("Could not open output file ");
     exit(1);
   }

  fprintf(fpout,"# name: %s\n",mname);
  fprintf(fpout,"# type: matrix\n");
  fprintf(fpout,"# rows: %d\n",ysize);
  fprintf(fpout,"# columns: %d\n",xsize);
  
  idx = 0;
   for (iy = 0; iy<ysize; iy++) {
    for (ix = 0; ix<xsize; ix++) {
        fprintf(fpout,"%e ",hf[idx++]);
    }
     fprintf(fpout,"\n");
   }
   rval = 0;
   return(rval);
}


int hf_writeb(char *fname)                /* write hf array to PGM binary */
{
  int ix,iy;
  long idx;
  FILE *fpout;
  int rval;
  int maxheight = 255;        /* maximum value of array */
                                                /* open input file */
  if ((fpout = fopen(fname,"wb")) == NULL)
   { perror("Could not open output file ");
     exit(1);
   }

  fprintf(fpout,"P5\n");
  fprintf(fpout,"%d %d\n",xsize,ysize);
  fprintf(fpout,"%d\n",maxheight);

  idx = 0;
   for (iy = 0; iy<ysize; iy++) {
    for (ix = 0; ix<xsize; ix++) {
        fprintf(fpout,"%c",(BYTE)(255*hf[idx++]));
    }
   }
   rval = 0;
   return(rval);
} /* end hf_writeb() */

/* -----------------------------------------------------------   */
/* hf_read -- read in pgm data from file <fname> into hf array,  */
/* allocating memory to fit a (xsize x ysize) array.             */
/* readin PGM binary or ascii, or if neither, call read_mfile()  */
/* -----------------------------------------------------------   */

int hf_read(
PTYPE **hh, 
unsigned int *xs,
unsigned int *ys,
char *fname         
)

{
  unsigned long ix, iy;                  /* indexes to hf array */
  size_t hsize;                         /* # bytes memory needed */
  unsigned long npoints;            /* # points that should be in file */
  unsigned long points_read;        /* # points read from hf datafile */
  char sbuf[512],                       /* buffer for file input */
        dummy[2];                       /* used for \n and \0   */
  char *junk;                           /* unconverted part of a number */
  double a, maxval;
  FILE *fpin;                           /* input file pointer */
  BYTE fbin;                            /* PGM binary format flag */
  BYTE oneb;                            /* one byte from file */
  int rval;
                                                /* open input file */
  if ((fpin = fopen(fname,"rb")) == NULL)
   { perror("Could not open input file ");
     exit(1);
   }

  if (fread(&sbuf,sizeof(char),2,fpin) != 2)
    {   perror("Could not read file"); exit(1); }
  else {
        sbuf[2] = 0x00;
  }
  fbin = 0;                             /* assume PGM ascii */
  if (strcmp(sbuf,"P5") == 0) fbin = 1; /* PGM binary format */
  if (strcmp(sbuf,"GI") == 0) {
    fclose(fpin);
    rval = gread1(hh,xs,ys,fname);        /* read as GIF format file */
  } else if ((strcmp(sbuf,"P5") != 0) && (strcmp(sbuf,"P2") != 0)) {
                                              /* not PGM Ascii */
          fclose(fpin);   /* close file to reopen as binary */
          rval = read_mfile(hh,xs,ys,fname);
  } else {

  while ((fscanf(fpin,"%s",sbuf) != EOF) && sbuf[0] == '#') {
        printf("comment:");
        fscanf(fpin,"%[^\n]",sbuf);  /* read comment beyond '#' */
        fscanf(fpin,"%[\n]",dummy);  /* read newline */
        printf("%s\n",sbuf);            /* display comment */
        }

  xs[0] = (int) strtoul(sbuf, &junk, 10);
  fscanf(fpin,"%s",sbuf);               /* read ysize of array */
  ys[0] = (int) strtoul(sbuf, &junk, 10);
  fscanf(fpin,"%s\n",sbuf);               /* read maxval of array */
  maxval = strtod(sbuf, &junk);

  hsize = (size_t) ( xs[0] * ys[0] * sizeof(PTYPE));
  sprintf(sbuf,"Could not allocate memory for heightfield (%lu bytes)",
     (unsigned long)hsize);
  if ((hh[0] = (PTYPE *) malloc( hsize ) ) == NULL)   {
   fprintf(stderr,"%s\n",sbuf);
   exit(1);
  }
  memset(hh[0],(char) 0, hsize);

  points_read = 0;              /* haven't read any data points yet */
  npoints = (unsigned long int) (xs[0] * ys[0]);
  ix = 0;
  iy = 0;                       /* fill array 1..xsize, 1..size */

  if (fbin) {
     while ((fread(&oneb,sizeof(BYTE),1,fpin) == 1) && (points_read < npoints) ) {
        hh[0][points_read++] = (PTYPE) oneb / maxval;
     }
  } /* end if */
  else {
      while ((fscanf(fpin,"%s",sbuf) != EOF) && (points_read < npoints) ) {
        a = strtod( sbuf, &junk);
        hh[0][points_read++] = (PTYPE) a / maxval;
      }
  } /* end else */


  printf("%lu elements read.\n",points_read);
  if (points_read != npoints) {
        printf("Warning: %lu points missing.\n",npoints-points_read);
  }
  rval = fclose( fpin );

  } /* end else */
  return(rval);

} /* end hf_read */


/* -------------------------------------------------------------- */
/* read_mfile --  read in data matrix in Matlab Level 1.0 format  */
/*               and put data in header "x" and data array "hf"   */
/* -------------------------------------------------------------- */

int read_mfile(
PTYPE **hh, 
unsigned int *xs,
unsigned int *ys,
char *fname )
{
  char pname[80]="dat1\x00";           /* data matrix name, null-terminated*/
  int mn;
  Fmatrix hd;                           /* header */

  int a,b,c,i,j;
  int etype;
  int mtype,prec,rem,numtype;
  double tdbl;
  float tflt;
  FILE *fpin;

   /* --------- read in Matlab-format (Level 1.0) file ------- */

   etype = (NUM_FMT*1000) + (PREC*10);    /* expected data type flag */

   if ((fpin = fopen(fname,"rb")) == NULL) {
    perror("could not open file.\n");
    exit(1);
   }

   printf("Opening matlab file %s...\n",fname);

   a = fread(&hd, sizeof(Fmatrix),(size_t)1, fpin);          /* read header */
    if (a != 1) {
      perror("Problem reading matlab file header.\n");
      exit(1);
    }
   numtype = (hd.type % 10000)/1000;
   prec = (hd.type % 100)/10;
   mtype = (hd.type % 10);
   rem = (hd.type - (1000*numtype + 10*prec + mtype));
/*   printf("Data type flag: M%d P%d T%d R%d\n",
       numtype,prec,mtype,rem); */
   if ( (numtype!=NUM_FMT) || ((prec!=0)&&(prec!=1)) || (mtype!=0) )  {
     perror("Unsupported file format type\n");
     exit(1);
   }
   b = fread(pname, sizeof(char), (size_t)hd.namelen, fpin); /* read mx name */
   xs[0] = hd.ncols;
   ys[0] = hd.mrows;
   xsize = xs[0];
   mn = (hd.mrows) * (hd.ncols); /* size of array */
   if ((hh[0] = (PTYPE *) malloc(mn*sizeof(PTYPE)) ) == NULL) {
     perror("Could not allocate memory for input array.\n");
     exit(1);
   }
   if (prec==1) {               /* read in floats 1 by 1 */
     for (i=0;i<xs[0];i++) {
      for (j=0;j<ys[0];j++) {   /* format is columns saved first */
       c = fread(&tflt, sizeof(float), 1, fpin);
       if (c != 1) {
         perror("Trouble reading data from file.\n");
         exit(1);
       }
       El(hh[0],i,j) = (PTYPE) tflt;
      } /* end for j */
     } /* end for i */
   } else if (prec==0) {                     /* read doubles 1 by 1 and convert */

     for (i=0;i<xs[0];i++) {
      for (j=0;j<ys[0];j++) {   /* format is columns saved first */
       c = fread(&tdbl, sizeof(double), 1, fpin);
       if (c != 1) {
         perror("Trouble reading data from file.\n");
         exit(1);
       }
       El(hh[0],i,j) = (PTYPE) tdbl;
      } /* end for j */
     } /* end for i */

   } else {
        perror("Unsupported floating-point type\n");
        exit(1);
   } /* end else */

   fclose( fpin );

  return(0);

} /* end read_mfile */



/* --------------------------------------------------------------  */
/* write_mfile -- write out data matrix in Matlab Level 1.0 format */
/*               creates data header fhdr from xsize, ysize and    */
/*               writes data array 'hf'                            */
/* --------------------------------------------------------------  */

int write_mfile(char *fname, char *pname)
{
FILE *fpout;
Fmatrix fhdr;
int rv,mn,i,j,c;

   fpout = fopen(fname,"wb");
   if (fpout == NULL)
    { perror("Error opening file.\n");
      exit(1);
    }
   fhdr.type = NUM_FMT*1000 + PREC*10;
   fhdr.ncols = xsize;
   fhdr.mrows = ysize;
   fhdr.imagf = 0;
   fhdr.namelen = strlen(pname) + 1;

   rv = fwrite(&fhdr, sizeof(Fmatrix),(size_t)1, fpout);          /* write header */
   if (rv != 1) {
     perror("Error writing file.\n");
     exit(1);
   }
   rv = fwrite(pname, sizeof(char), (size_t)fhdr.namelen, fpout);   /* write mx name */
   if (rv != fhdr.namelen) {
     perror("Error writing file.\n");
     exit(1);
   }
   mn = fhdr.mrows * fhdr.ncols;

   for (i=0;i<xsize;i++) {
      for (j=0;j<ysize;j++) {   /* format is columns saved first */

       c = fwrite(&El(hf,i,j), sizeof(PTYPE), 1, fpout);
       if (c != 1) {
         perror("Trouble writing data to file.\n");
         exit(1);
       }
      } /* end for j */
   } /* end for i */

  if (fclose( fpout ) == EOF) {
        perror("Error closing file.\n");
        exit(1);
  }

  return(0);
} /* end write_mfile */

/* ------------------------------------------------------------------- 
  read_hf() --  read in a heightfield in GIF|POT|TGA|MAT|etc format
                 fname_in is filename string, itype is format type number
                 return dat in hfp, dimensions in (xsp, ysp)
  --------------------------------------------------------------------
*/

int read_hf(PTYPE **hfp, U* xsp, U* ysp, char *fname_in, int itype)
{
int rval; 

   if (itype == POT) {
    rval = pread(hfp, xsp, ysp, fname_in);
   } else if (itype == TGA) {
    rval = tga_read(hfp, xsp, ysp, fname_in);
   } else { 
       rval = hf_read(hfp,xsp,ysp,fname_in);  
     }
   return(rval);
}


/* ------------------------------------------------------------------- 
  write_hf() --  write out a heightfield in GIF|POT|TGA|MAT|etc format
                 fname_out is filename string, otype is format type number
                 hfield dat in hf, dimensions in (xsize, ysize)
  --------------------------------------------------------------------
*/

int write_hf(PTYPE *hf, U xsize, U ysize, char *fname_out, int otype)
{
 int rval;

   if (otype == GIF) {
     printf("Writing GIF file...\n");
     rval = gwrite(fname_out);
   } else if (otype == PGM) {
       printf("Writing PGM ascii...\n");
       rval = hf_write(fname_out);  /* save output file */
   } else if (otype == PG8) {
     printf("Writing PGM binary...\n");
     rval = hf_writeb(fname_out);  /* save output file */
   } else if (otype == POT) {
     printf("Writing POT file...\n");
     rval = pwrite(fname_out);
   } else if (otype == TGA) {
     printf("Writing TGA file...\n");
     rval = tga_write(hf, xsize, ysize, fname_out);
   } else if (otype == MAT) {
     printf("Writing Matlab file...\n");
     rval = write_mfile(fname_out,"topo");
   } else if (otype == OCT) {
     printf("Writing Octave file...\n");
     rval = oct_write(fname_out,"topo");
   } else rval = 1;

   return(rval);
} /* end write_hf() */

