#!/usr/bin/env python
## -*- mode: python; TAB: 4; -*-
###############################################################################
## Copyright (C) 1998 Klaus Alexander Seistrup @ Magnetic Ink, Ndola, ZAMBIA
##
## Author  : 1998 Klaus Alexander Seistrup <mutende@bigfoot.com>
## Created : Friday, 21st August 1998
## @(#) $Id: pydb.py,v 1.3 1998/08/30 08:13:38 root Exp root $
##
## This program is free software; you can redistribute it and/or modify it
## under the terms of the GNU General Public License as published by the Free
## Software Foundation; either version 2 of the License, or (at your option)
## any later version.
##
## This program is distributed in the hope that it will be useful, but WITH-
## OUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
## FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
## more details.
##
## You should have received a copy of the GNU General Public License along
## with this program; if not, write to the Free Software Foundation, Inc.,
## 675 Mass Ave, Cambridge, MA 02139, USA.
###############################################################################
"""
PyDBM -- a simple anydbm application

Usage: pydbm <key>          		# Retrieve value associated with <key>
  or:  pydbm <key> <val> ...		# Associate <val> with <key>

PyDBM will store or retrieve key/value pairs in a database whose name depends
on the invocation name.  The default database name is "~/.pydbm/pydbm.db" or,
if $PYDBM_HOME is defined, "$PYDBM_HOME/pydbm.db".  E.g.:

	% pydbm foo bar
	%

will associate the value "bar" with the key "foo".  A subsequent invocation
with foo as the only argument yields:

	% pydbm foo
	bar
	%

Multiple values assciayed with a keyword will be stored, and retrived, with
a TAB character separating the values.  E.g.:

	% pydbm foo bar fly
	% pydbm foo
	bar	fly
	%

If you link (or rename) pydbm to, say, pyaddr the name of the database will
be "$PYDBM_HOME/pyaddr.db" instead (suitable for e.g. a quick address look-up
thing).

BUGS: Keys are stored in UPPERCASE
      Doesn't ask before overwriting a key/value pair

TODO: Wildcard retrieval
      Better documentation
      Better error checking
      User configurable output formatting
      Some sort of semaphore locking when altering a database
      ...

Feed-back appreciated, you'll find my address below.

  -Mutende BaShiBanda

===============================================================================
Klaus Alexander Seistrup * Magnetic Ink * P.O. Box 73 124 * Ndola * Z A M B I A
http://www.bigfoot.com/~mutende/ * Soli Deo Gloria * mailto:mutende@bigfoot.com
===============================================================================
"""
###############################################################################
MAGIC = "@(#) $Id: pydb.py,v 1.3 1998/08/30 08:13:38 root Exp root $\0"

PGM, VER, REV, PL, DATE = "pydbm", 0, 1, 3, "25.VIII.1998"

import os, sys, string, anydbm          # I'm tempted to use gdbm, though...

# Shorthand(s)
eprint = lambda str="": sys.stderr.write (str + "\n")

# getenv -- get a value from the environment
def getenv (key):
	"""VAL = getenv(KEY)"""
	if os.environ.has_key (key):
		return os.environ[key]
	# end if
	return ""
# end def getenv

# opendb -- open a pydbm database
def opendb (dbName):
	"""DB = opendb(NAME)"""
	try:
		db = anydbm.open (dbName, "c", 0600)
	except anydbm.error, errMsg:
		reason = string.lower (errMsg[1])
		eprint ("Couldn't open %s: %s" % (dbName, reason))
		return None
	# end try
	if db.has_key ("__init__"):
		if db["__init__"] != MAGIC:		# TODO: Warn about possible
			pass						# format incompatibility???
		# end if
	else:
		db["__init__"] = MAGIC			# Store our magic cookie
	# end if
	return db
# end def opendb

# getval -- get a value from pydbm database
def getval (dbName, key):
	"""VAL = getval(DBNAME,KEY)"""
	val = ""
	db = opendb (dbName)
	if db:
		key = string.upper (key)
		if db.has_key (key):
			val = db[key]
		else:
			eprint ("No such key %s" % key)
		# end if
		db.close ()
	# end if
	return val
# end def getval

# setval -- set key=val in pydbm database
def setval (dbName, key, val):
	"""ERR = setval(DBNAME,KEY,VAL)"""
	db = opendb (dbName)
	if db:
		db[string.upper (key)] = val
		db.close ()
		return 0
	# end if
	return 1
# end def setval

# canonify -- had to call it something... ;)
def canonify (path):
	"""NAME = canonify('/FULL/PATH/NAME.EXT')"""
	try:
		i = string.rindex (path, "/")
	except ValueError:
		i = -1
	# end try
	if i != -1:
		path = path[i+1:]
	# end if
	try:
		i = string.rindex (path, ".")
	except ValueError:
		i = -1
	# end try
	if i != -1:
		path = path[:i]
	# end if
	return path
# end def canonify

# gethome -- find out where we live
def gethome (id="pydbm"):
	"""HOME = gethome (ID)"""
	home = getenv ("PYDBM_HOME")
	if not home:
		home = os.path.expanduser ("~") + "/.pydbm"
	# end if
	try:
		sbuf = os.stat (home)
	except os.error:
		try:
			os.mkdir (home)
		except os.error:
			eprint ("Cannot access %s, using current dir..." % home)
			home = "."
		# end try
	# end try
	if home[-1] == "/":
		return home[:-1]
	# end if
	return home
# end def gethome

# showhelp pipe help text thorugh $PAGER
def showhelp (hlp="Sorry, no help available\n"):
	"""Pipes HELP through pager"""
	pager = getenv ("PAGER")
	if not pager:						# If they haven't specified a pager
		pager = "more"					#  -- use more instead
	# end if
	try:
		fo = os.popen (pager, "w")
	except IOError:						# If we can't open $PAGER
		fo = sys.stderr					#  -- use stderr instead
	# end try
	fo.write (hlp + "\n")
	fo.flush ()
	if fo != sys.stderr:
		fo.close ()
	# end if
# end def showhelp

# main entry
def main (argc=1, argv=["pydbm"]):
	myName = canonify (argv[0])
	myHome = gethome (myName)
	dbName = "%s/%s.db" % (myHome, myName)
	if argc <= 1:
		eprint ("%s-%s %d.%d.%d (%s)\n" % (PGM, myName, VER, REV, PL, DATE))
		eprint ("Usage: %s <key>          	# get val")
		eprint ("  or:  %s <key> <val> ...	# set key=val")
		return 1
	elif argc == 2:						# 2 args: get info
		if "help" in argv:
			showhelp (__doc__)
			return 0
		# end if
		val = getval (dbName, argv[1])
		if val:
			print val
			return 0
		# end if
		return 1
	elif argc == 3:						# 3 args: set info
		return setval (dbName, argv[1], argv[2])
	# end if
	return setval (dbName, argv[1], string.joinfields (argv[2:], "\t"))
# end def main

if __name__ == "__main__":
	rc = main (len (sys.argv), sys.argv)
	sys.exit (rc)
# end if

## Soli Deo Gloria ############################################################
