/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB
   
   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.
   
   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.  The License grants you the right to 
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

/* Function with list databases, tables or fields */

#include "mysql_priv.h"
#include <my_dir.h>

static int mysql_find_files(THD *thd,const char *path,const char *wild,
			    bool dir);

/****************************************************************************
** Send list of databases
** A database is a directory in the mysql_data_home directory
****************************************************************************/


int
mysqld_show_dbs(THD *thd,const char *wild)
{
  Item_string field("",0);
  List<Item> field_list;
  char *end;

  field.name=(char*) sql_alloc(20+ (wild ? strlen(wild)+4: 0));
  field.max_length=NAME_LEN;
  end=strmov(field.name,"Database");
  if (wild && wild[0])
    strxmov(end," (",wild,")",NullS);
  field_list.push_back(&field);

  send_fields(thd,field_list,1);
  return mysql_find_files(thd,mysql_data_home,wild,1);
}


/***************************************************************************
** List all tables in a database
** A table is a .frm file in the current databasedir
***************************************************************************/

int mysqld_show_tables(THD *thd,const char *db,const char *wild)
{
  Item_string field("",0);
  List<Item> field_list;
  char path[FN_LEN],*end;

  field.name=(char*) sql_alloc(20+strlen(db)+(wild ? strlen(wild)+4:0));
  end=strxmov(field.name,"Tables in ",db,NullS);
  if (wild && wild[0])
    strxmov(end," (",wild,")",NullS);
  field.max_length=NAME_LEN;
  (void)sprintf(path,"%s/%s",mysql_data_home,db);
  field_list.push_back(&field);
  send_fields(thd,field_list,1);
  return mysql_find_files(thd,path,wild,0);
}


static int
mysql_find_files(THD *thd,const char *path,const char *wild,bool dir)
{
  uint i;
  char *ext;
  byte *pos;
  MY_DIR *dirp;
  FILEINFO *file;
  DBUG_ENTER("mysql_find_files");

  if (!(dirp = my_dir(path,MYF(MY_WME | (dir ? MY_WANT_STAT : 0)))))
    DBUG_RETURN(-1);

  for (i=0 ; i < (uint) dirp->number_off_files	; i++)
  {
    file=dirp->dir_entry+i;
    if (dir)
    {						/* Return databases */
      if (file->name[0] == '.' || !MY_S_ISDIR(file->mystat.st_mode) ||
	  (wild && wild[0] && wild_compare(file->name,wild)))
	continue;
    }
    else
    {						/* Return files */
      if (strcmp(ext=fn_ext(file->name),reg_ext))
	continue;
      *ext=0;
      if (wild && wild[0] && wild_compare(file->name,wild))
	continue;
    }
    pos=net_store_length(thd->packet,strlen(file->name));
    pos=strmov((char*) pos,file->name);
    if (net_write(&thd->net,thd->packet,(uint) (pos - thd->packet)))
      break;
  }
  send_eof(&thd->net);
  my_dirend(dirp);
  DBUG_RETURN(0);
}


int
mysqld_show_fields(THD *thd, const char *db, const char *table_name,
		  const char *wild)
{
  TABLE *table;
  char *pos;
  DBUG_ENTER("mysqld_show_fields");
  DBUG_PRINT("enter",("db: %s  table: %s",db,table_name));

  if (!(table = open_table(thd,db,table_name,table_name)))
  {
    send_error(&thd->net,NullS);
    DBUG_RETURN(1);
  }
  ha_info(&table->form,2);			// sync record count

  List<Item> field_list;
  field_list.push_back(new Item_empty_string("Field",NAME_LEN));
  field_list.push_back(new Item_empty_string("Type",40));
  field_list.push_back(new Item_empty_string("Null",1));
  field_list.push_back(new Item_empty_string("Key",3));
  field_list.push_back(new Item_empty_string("Default",NAME_LEN));
  field_list.push_back(new Item_empty_string("Extra",20));

	// Send first number of fields and records
  pos=net_store_length(thd->packet,field_list.elements);
  pos=net_store_length(pos,table->form.keyfile_info.records);
  (void) net_write(&thd->net, thd->packet,(uint) (pos-thd->packet));

  send_fields(thd,field_list,0);
  restore_record(&table->form,2);	// Get empty record

  FIELD *field,*end;
  for (field=table->field,end=table->field_end ; field != end; field++)
  {
    if (!wild || !wild[0] || !wild_compare(field->field_name,wild))
    {
      byte *pos;
      char tmp[MAX_FIELD_WIDTH],*end,*type;
      uint length;
      bool len,dec,blob;
      uint flags=field->flags;
      blob=len=dec=0;
      pos=net_store_data(thd->packet,field->field_name);
      switch (field->type) {
      case FIELD_TYPE_CHAR:		type="tinyint";   len=1; break;
      case FIELD_TYPE_SHORT:		type="smallint";  len=1; break;
      case FIELD_TYPE_INT24:		type="mediumint"; len=1; break;
      case FIELD_TYPE_LONG:		type="int";   len=1; break;
      case FIELD_TYPE_LONGLONG:		type="bigint"; len=1; break;
      case FIELD_TYPE_STRING:		type="char"; len=1; break;
      case FIELD_TYPE_VAR_STRING:	type="varchar"; len=1; break;
      case FIELD_TYPE_FLOAT:		type="float"; len=dec=1; break;
      case FIELD_TYPE_DOUBLE:		type="double"; len=dec=1; break;
      case FIELD_TYPE_NULL:		type="null"; break;
      case FIELD_TYPE_DECIMAL:		type="decimal"; len=dec=1; break;
      case FIELD_TYPE_TIMESTAMP:
	type="timestamp";
	len=1;
	flags&= ~(UNSIGNED_FLAG | ZEROFILL_FLAG);
	break;
      case FIELD_TYPE_DATE:		type="date"; break;
      case FIELD_TYPE_TIME:		type="time"; break;
      case FIELD_TYPE_TINY_BLOB:	type="tinyblob"; blob=1; break;
      case FIELD_TYPE_BLOB:		type="blob"; blob=1; break;
      case FIELD_TYPE_MEDIUM_BLOB:	type="mediumblob"; blob=1; break;
      case FIELD_TYPE_LONG_BLOB:	type="longblob"; blob=1; break;
      default:				type="Unknown"; break;
      }
      end=strmov(tmp,type);
      if (len)
      {
	*end++='(';
	end=int2str((long) field->length,end,10);
	if (dec)
	{
	  *end++=',';
	  end=int2str((long) field->decimals,end,10);
	}
	*end++=')';
      }
      if (flags & UNSIGNED_FLAG)
	end=strmov(end," unsigned");
      if (flags & ZEROFILL_FLAG)
	end=strmov(end," zerofill");

      pos=net_store_data(pos,tmp,(uint) (end-tmp));
      type=(flags & NOT_NULL_FLAG) ? "" : "YES";
      pos=net_store_data(pos,type);
      type=((field->flags & PRI_KEY_FLAG) ? "PRI" :
	    (field->flags & UNIQUE_KEY_FLAG) ? "UNI" :
	    (field->flags & MULTIPLE_KEY_FLAG) ? "MUL":"");
      pos=net_store_data(pos,type);
      tmp[0]=0;					// Get default value
      length=0;
      if ((flags & NOT_NULL_FLAG) && !blob && field->type != FIELD_TYPE_TIMESTAMP)
	length=r_unpackf_stripp(field,tmp);
      pos=net_store_data(pos,tmp,length);
      end=tmp;
      if (MTYP_TYPENR(field->unireg_type) == MTYP_NEXT_NUMBER)
	end=strmov(tmp,"auto_increment");
      pos=net_store_data(pos,tmp,(uint) (end-tmp));
      if (net_write(&thd->net,thd->packet,(uint) (pos - thd->packet)))
	DBUG_RETURN(1);
    }
  }
  send_eof(&thd->net);
  DBUG_RETURN(0);
}


int
mysqld_show_keys(THD *thd, const char *db, const char *table_name)
{
  TABLE *table;
  DBUG_ENTER("mysqld_show_keys");
  DBUG_PRINT("enter",("db: %s  table: %s",db,table_name));

  if (!(table = open_table(thd,db,table_name,table_name)))
  {
    send_error(&thd->net,NullS);
    DBUG_RETURN(1);
  }

  List<Item> field_list;
  Item *item;
  field_list.push_back(new Item_empty_string("Table",NAME_LEN));
  field_list.push_back(new Item_int("Non_unique",0,1));
  field_list.push_back(new Item_empty_string("Key_name",NAME_LEN));
  field_list.push_back(new Item_int("Seq_in_index",0,2));
  field_list.push_back(new Item_empty_string("Column_name",NAME_LEN));
  field_list.push_back(item=new Item_empty_string("Collation",1));
  item->maybe_null=1;
  field_list.push_back(item=new Item_int("Cardinality",0,11));
  item->maybe_null=1;
  field_list.push_back(item=new Item_int("Sub_part",0,3));

  send_fields(thd,field_list,1);

  FORM *form= &table->form;
  KEY *key_info=form->key_info;
  ha_info(form,2);
  for (uint i=0 ; i < form->keys ; i++,key_info++)
  {
    KEY_PART_INFO *key_part= key_info->key_part;
    byte  *pos,buff[10],*end;
    for (uint j=0 ; j < key_info->key_parts ; j++,key_part++)
    {
      pos=thd->packet;
      pos=net_store_data(pos,table->table_name);
      pos=net_store_data(pos,key_info->dupp_key ? "1" :"0", 1);
      pos=net_store_data(pos,key_info->name);
      end=int2str((long) (j+1),buff,10);
      pos=net_store_data(pos,buff,(uint) (end-buff));
      pos=net_store_data(pos,key_part->field ? key_part->field->field_name :
			 "");
      if (ha_option_flag[form->db_type] & HA_READ_ORDER)
	pos=net_store_data(pos,key_part->key_part_flag == 0 ? "A" : "D", 1);
      else
	pos=net_store_length(pos,NULL_LENGTH);
      if (form->db_type == DB_TYPE_ISAM && form->keyfile_info.rec_per_key[i])
      {
	ulong records=(form->keyfile_info.records /
		       form->keyfile_info.rec_per_key[i]);
	end=int2str((long) records,buff,10);
	pos=net_store_data(pos,buff,(uint) (end-buff));
      }
      else
	pos=net_store_length(pos,NULL_LENGTH);
      if (!key_part->field ||
	  key_part->length != key_part->field->packlength)
      {
	end=int2str((long) key_part->length,buff,10);
	pos=net_store_data(pos,buff,(uint) (end-buff));
      }
      else
	pos=net_store_length(pos,NULL_LENGTH);
      if (net_write(&thd->net,thd->packet,(uint) (pos - thd->packet)))
	DBUG_RETURN(1);
    }
  }
  send_eof(&thd->net);
  DBUG_RETURN(0);
}



/****************************************************************************
** Return only fields for API mysql_list_fields
** Use "show table wildcard" in mysql instead of this
****************************************************************************/

void
mysqld_list_fields(THD *thd, const char *table_name, const char *wild)
{
  TABLE *table;
  DBUG_ENTER("mysqld_list_fields");
  DBUG_PRINT("enter",("table: %s",table_name));

  if (!(table = open_table(thd,thd->db,table_name,table_name)))
    send_error(&thd->net,NullS);
  else
  {
    List<Item> field_list;

    FIELD *field,*end;
    for (field=table->field,end=table->field_end ; field != end; field++)
    {
      if (!wild || !wild[0] || !wild_compare(field->field_name,wild))
	field_list.push_back(new Item_field(field));
    }
    restore_record(&table->form,2);		// Get empty record
    send_fields(thd,field_list,2);
    field_list.delete_elements();
  }
  DBUG_VOID_RETURN;
}


/****************************************************************************
** Return info about all processes
** returns for each thread: thread id, user, host, db, command, info
****************************************************************************/

class thread_info :public Sql_alloc, public ilink {
public:
  ulong thread_id;
  uint	 command;
  char	*user,*host,*db,*query;
  bool locked;
};

#ifdef __GNUC__
template class I_List<thread_info>;
template class I_List_iterator<THD>;
#endif


void mysqld_list_processes(THD *thd)
{
  Item *field;
  List<Item> field_list;
  I_List<thread_info> thread_infos;
  DBUG_ENTER("mysql_list_processes");

  field_list.push_back(new Item_int("Id",0,7));
  field_list.push_back(new Item_empty_string("User",16));
  field_list.push_back(new Item_empty_string("Host",64));
  field_list.push_back(field=new Item_empty_string("db",NAME_LEN));
  field->maybe_null=1;
  field_list.push_back(new Item_empty_string("Command",16));
  field_list.push_back(field=new Item_empty_string("Info",100));
  field->maybe_null=1;
  send_fields(thd,field_list,1);

  VOID(pthread_mutex_lock(&LOCK_thread_count)); // For unlink from list
  if (!abort_loop)
  {
    I_List_iterator<THD> it(threads);
    THD *tmp;
    while ((tmp=it++))
    {
      if (tmp->net.fd >= 0)
      {
	thread_info *thd_info=new thread_info;

	thd_info->thread_id=tmp->thread_id;
	thd_info->user=sql_strdup(tmp->user);
	thd_info->host=sql_strdup(tmp->host);
	thd_info->db=0;
	if (tmp->db)
	  thd_info->db=sql_strdup(tmp->db);
	thd_info->command=(int) tmp->command;
	thd_info->locked=tmp->locked;
	thd_info->query=0;
	if (tmp->query)
	{
	  uint length=strlen(tmp->query);
	  if (length > 100)
	    length=100;
	  thd_info->query=(char*) sql_memdup(tmp->query,length+1);
	  thd_info->query[length]=0;
	}
	thread_infos.append(thd_info);
      }
    }
  }
  VOID(pthread_mutex_unlock(&LOCK_thread_count));

  thread_info *thd_info;
  while ((thd_info=thread_infos.get()))
  {
    byte *pos,buff[20],*end;
    end=int2str((long) thd_info->thread_id,buff,10);
    pos=net_store_data(thd->packet,buff,(uint) (end-buff));
    pos=net_store_data(pos,thd_info->user);
    pos=net_store_data(pos,thd_info->host);
    if (thd_info->db)
      pos=net_store_data(pos,thd_info->db);
    else
      pos=net_store_length(pos,NULL_LENGTH);
    if (thd_info->locked)
      pos=net_store_data(pos,"Locked");
    else
      pos=net_store_data(pos,command_name[thd_info->command]);
    if (thd_info->query)
      pos=net_store_data(pos,thd_info->query);
    else
      pos=net_store_length(pos,NULL_LENGTH);
    if (net_write(&thd->net,thd->packet,(uint) (pos - thd->packet)))
      break;
  }
  send_eof(&thd->net);
  DBUG_VOID_RETURN;
}
