/* Copyright (C) 1979-1997 TcX AB & Monty Program KB & Detron HB
   
   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.
   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.  The License grants you the right to 
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

/* Copy data from a textfile to table */

#include "mysql_priv.h"
#include "handler.h"
#include <my_dir.h>

class READ_INFO {
  File	file;
  byte	*buffer,			/* Buffer for read text */
	*end_of_buff;			/* Data in bufferts ends here */
  uint	buff_length,			/* Length of buffert */
	max_length;			/* Max length of row */
  char	*field_term_ptr,*line_term_ptr;
  uint  field_term_length,line_term_length,enclosed_length;
  int	field_term_char,line_term_char,enclosed_char,escape_char;
  int	*stack,*stack_pos;
  bool  found_end_of_line,eof;
  IO_CACHE cache;
public:
  bool error,line_cuted,found_null,enclosed;
  byte	*row_start,			/* Found row starts here */
	*row_end;			/* Found row ends here */

  READ_INFO(File file,uint tot_length,
	    String &field_term,String &line_term,String &enclosed,int escape);
  ~READ_INFO();
  int read_field();
  int read_fixed_length(void);
  int next_line(void);
  char unescape(char chr);
  terminator(char *ptr,uint length);
};


typedef struct st_copy_info {
  ulong records;
  ulong deleted;
  ulong copied;
  enum enum_duplicates handle_duplicates;
  int escape_char;
} COPY_INFO;

static int read_fixed_length(THD *thd,COPY_INFO &info,TABLE *table,
			     List<Item> &fields, READ_INFO &read_info);
static int read_sep_field(THD *thd,COPY_INFO &info,TABLE *table,
			  List<Item> &fields, READ_INFO &read_info,
			  String &enclosed);


int mysql_load(THD *thd,sql_exchange *ex,tname_t *table_list,
	       List<Item> &fields, enum enum_duplicates handle_duplicates,
	       ulong rows_to_skipp)
{
  char name[FN_REFLEN];
  File file;
  TABLE *table;
  int error;
  String *field_term=ex->field_term,*escaped=ex->escaped,*enclosed=ex->enclosed;
  DBUG_ENTER("mysql_load");

  if (escaped->length() > 1 || enclosed->length() > 1)
  {
    my_message(ER(ER_WRONG_FIELD_TERMINATORS),MYF(0));
    DBUG_RETURN(-1);
  }

  if (!(table=table_list->table))
  {						/* Not already open */
    if (!(table = open_ltable(thd,table_list->real_name,table_list)))
      DBUG_RETURN(-1);
  }
  TABLE **table_ptr= &table;
  FORM *form= &table->form; 
  form->reginfo.update=1;			/* Write lock */
  if (!(thd->lock=mysql_lock_tables(thd,table_ptr,1)))
    DBUG_RETURN(-1);

  if (!fields.elements)
  {
    FIELD *field,*end;
    for (field=table->field,end=table->field_end; field != end; field++)
      fields.push_back(new Item_field(field));
  }
  else
  {						// Part field list
    if (setup_fields(thd,table_list,fields,1) < 0)
      DBUG_RETURN(-1);
    if (thd->dupp_field)
    {
      my_printf_error("Field '%s' specified twice",MYF(0),
		      thd->dupp_field->field_name);
      DBUG_RETURN(-1);
    }
  }

  uint tot_length=0;
  bool use_blobs=0,use_timestamp=0;
  List_iterator<Item> it(fields);

  while (Item_field *field=(Item_field*) it++)
  {
    tot_length+=field->field->length;
    if (field->field->flags & BLOB_FLAG)
      use_blobs=1;
    if (!field_term->length() && !(field->field->flags & NOT_NULL_FLAG))
      mark_as_not_null(field->field);
    if (field->field->str == form->record[0]+form->time_stamp-1)
      use_timestamp=1;
  }
  if (use_blobs && !ex->line_term->length() && !field_term->length())
  {
    my_message(ER(ER_BLOBS_AND_NO_TERMINATED),MYF(0));
    DBUG_RETURN(-1);
  }

  if (!dirname_length(ex->file_name))
    (void) sprintf(name,"%s/%s/%s",mysql_data_home,thd->db,ex->file_name);
  else
  {
    MY_STAT stat_info;
    strmov(name,ex->file_name);
    if (!my_stat(name,&stat_info,MYF(MY_FAE)))
      DBUG_RETURN(-1);
    if (!(stat_info.st_mode & S_IROTH))
    {
      my_printf_error(MYF(0),ER_TEXTFILE_NOT_READABLE,name);
      DBUG_RETURN(-1);
    }      
  }

  if ((file=my_open(name,O_RDONLY,MYF(MY_WME))) < 0)
    DBUG_RETURN(-1);

  COPY_INFO info;
  bzero(&info,sizeof(info));
  info.handle_duplicates=handle_duplicates;
  info.escape_char=escaped->length() ? (*escaped)[0] : INT_MAX;

  READ_INFO read_info(file,tot_length,*field_term,
		      *ex->line_term, *enclosed,info.escape_char);
  if (read_info.error)
  {
    my_close(file,MYF(0));
    DBUG_RETURN(-1);				// Can't allocate buffers
  }

  restore_record(form,2);

  thd->count_cuted_fields=1;			/* calc cuted fields */
  thd->cuted_fields=0L;	
  if (ex->line_term->length() && field_term->length())
  {
    while (rows_to_skipp--)
    {
      if (read_info.next_line())
	break;
    }
  }
  if (!(error=test(read_info.error)))
  {
    uint save_time_stamp=form->time_stamp;
    if (use_timestamp)
      form->time_stamp=0;
    VOID(ha_extra(form,HA_EXTRA_WRITE_CACHE));
    if (!field_term->length() && !enclosed->length())
      error=read_fixed_length(thd,info,table,fields,read_info);
    else
      error=read_sep_field(thd,info,table,fields,read_info,*enclosed);
    if (ha_extra(form,HA_EXTRA_NO_CACHE))
      error=1;
    form->time_stamp=save_time_stamp;
    mysql_unlock_tables(thd->lock); thd->lock=0;
  }

  my_close(file,MYF(0));
  free_blobs(form);				/* if pack_blob was used */
  thd->count_cuted_fields=0;			/* Don`t calc cuted fields */
  if (error)
    DBUG_RETURN(-1);				// Error on read
  sprintf(name,ER(ER_LOAD_INFO),info.records,info.deleted,
	  info.records-info.copied,thd->cuted_fields);
  send_ok(&thd->net,info.copied+info.deleted,0L,name);
  DBUG_RETURN(0);
}


	/* Check if there is more uniq keys after field */

static int last_uniq_key(FORM *form,uint keynr)
{
  while (++keynr < form->keys)
    if (!form->key_info[keynr].dupp_key)
      return 0;
  return 1;
}


static int write_record(TABLE *table,COPY_INFO &info)
{
  int error;
  char *key=0;
  FORM *form= &table->form;

  if (abort_loop)
  {
    my_error(ER_SERVER_SHUTDOWN,MYF(0));
    return 1;
  }
  info.records++;
  if (info.handle_duplicates == DUP_REPLACE)
  {
    int key_nr,end;
    char *save_pos;
    while ((error=ha_write(form,form->record[0])))
    {
      if (error != HA_WRITE_SKIPP)
	goto err;
      if ((key_nr = ha_keyerror(form,error)) < 0)
      {
	error=HA_WRITE_SKIPP;			/* Database can't find key */
	goto err;
      }
      if (ha_extra(form,HA_EXTRA_FLUSH_CACHE))	/* Not neaded with NISAM */
      {
	error=my_errno;
	goto err;
      }
      if (!key)
	key=(char*) my_alloca(ha_max_key_length);
      key_copy(key,form,(uint) key_nr,0);
      if ((error=ha_rkey(form,form->record[1],key_nr,key,0, HA_READ_KEY_EXACT)))
	goto err;
      if (last_uniq_key(form,key_nr))
      {
	if ((error=ha_update(form,form->record[1],form->record[0])))
	  goto err;
	info.copied++;
	info.deleted++;
	break;					/* Update logfile and count */
      }
      else if ((error=ha_delete(form,form->record[1])))
	goto err;
      info.deleted++;
    }
    if (key)
      my_afree(key);
  }
  else if ((error=ha_write(form,form->record[0])))
  {
    if (info.handle_duplicates != DUP_IGNORE ||
	error != HA_ERR_FOUND_DUPP_KEY)
    {
      ha_error(&table->form,error,MYF(0));
      return 1;
    }
  }
  else
    info.copied++;
  if (form->next_number_field)
  {						// Clear for next record
    FIELD *field=form->next_number_field;
    f_fieldinit(field->str,(uint) field->pack_flag,(uint) field->length);    
  }
  return 0;

err:
  if (key)
    my_afree(key);
  return error;
}


/****************************************************************************
** Read of rows of fixed size + optional garage + optonal newline
****************************************************************************/

static int
read_fixed_length(THD *thd,COPY_INFO &info,TABLE *table,List<Item> &fields,
		  READ_INFO &read_info)
{
  List_iterator<Item> it(fields);
  Item_field *sql_field;
  DBUG_ENTER("read_fixed_length");

  /* No fields can be null in this format. mark all fields as not null */
  while ((sql_field= (Item_field*) it++))
  {
    if (!(sql_field->field->flags & NOT_NULL_FLAG))
      mark_as_not_null(sql_field->field);
  }

  while (!read_info.read_fixed_length())
  {
    it.rewind();
    char *pos=read_info.row_start;
#ifdef HAVE_purify
    read_info.row_end[0]=0;
#endif
    while ((sql_field= (Item_field*) it++))
    {
      FIELD *field=sql_field->field;
      if (pos == read_info.row_end)
      {
	thd->cuted_fields++;			/* Not enough fields */
	f_fieldinit(field->str,(uint) field->pack_flag,(uint) field->length);
      }
      else
      {
	uint length;
	char save_chr;
	if ((length=(uint) (read_info.row_end-pos)) > field->length)
	  length=field->length;
	save_chr=pos[length]; pos[length]='\0';
	f_packfield(field->str,pos,field->pack_flag,(uint) field->length,
		    field->intervall);
	pos[length]=save_chr;
	if ((pos+=length) > read_info.row_end)
	  pos=read_info.row_end;	/* Fills rest with space */
      }
    }
    if (pos != read_info.row_end)
      thd->cuted_fields++;			/* To long row */
    if (write_record(table,info))
      DBUG_RETURN(1);
    if (read_info.next_line())			// Skipp to next line
      break;
    if (read_info.line_cuted)
      thd->cuted_fields++;			/* To long row */
  }
  DBUG_RETURN(test(read_info.error));
}



static int
read_sep_field(THD *thd,COPY_INFO &info,TABLE *table,
	       List<Item> &fields, READ_INFO &read_info,
	       String &enclosed)
{
  List_iterator<Item> it(fields);
  Item_field *sql_field;
  uint enclosed_length;
  DBUG_ENTER("read_sep_field");

  enclosed_length=enclosed.length();

  for (;;it.rewind())
  {
    while (sql_field=(Item_field*) it++)
    {
      uint length;
      char *pos;

      if (read_info.read_field())
	break;
      pos=read_info.row_start;
      length=(uint) (read_info.row_end-pos);
      FIELD *field=sql_field->field;

      if (!read_info.enclosed &&
	  (enclosed_length && length == 4 && !memcmp(pos,"NULL",4)) ||
	  (length == 1 && read_info.found_null))
      {
	if (f_is_blob(field->pack_flag))
	  pack_blob(field,"",0);
	else
	  f_fieldinit(field->str,(uint) field->pack_flag,(uint) field->length);
	if (!(sql_field->field->flags & NOT_NULL_FLAG))
	  mark_as_null(sql_field->field);
	else
	  thd->cuted_fields++;
	read_info.found_null=0;
	continue;
      }
      if (!(sql_field->field->flags & NOT_NULL_FLAG))
	mark_as_not_null(sql_field->field);
      if (f_is_blob(field->pack_flag))
	pack_blob(field,read_info.row_start,length);
      else
      {
	read_info.row_end[0]=0;
	packf(read_info.row_start,field);
      }
    }
    if (read_info.error)
      break;
    if (sql_field)
    {						// Last record
      if (sql_field == (Item_field*) fields.head())
	break;
      for ( ; sql_field ; sql_field=(Item_field*) it++)
      {
	if (!(sql_field->field->flags & NOT_NULL_FLAG))
	  mark_as_null(sql_field->field);
	FIELD *field=sql_field->field;
	f_fieldinit(field->str,(uint) field->pack_flag,(uint) field->length);
      }
    }
    if (write_record(table,info))
      DBUG_RETURN(1);
    if (read_info.next_line())			// Skipp to next line
      break;
    if (read_info.line_cuted)
      thd->cuted_fields++;			/* To long row */
  }
  DBUG_RETURN(test(read_info.error));
}


/* Unescape all escape characters, mark \N as null */

char
READ_INFO::unescape(char chr)
{
  switch(chr) {
  case 'n': return '\n';
  case 't': return '\t';
  case 'r': return '\r';
  case 'b': return '\b';
  case '0': return 0;				// Ascii null
  case 'N': found_null=1;
    /* fall through */
  default:  return chr;
  }
}


	/* Read a line using buffering */
	/* If last line is empty (in line mode) then it isn't outputed */


READ_INFO::READ_INFO(File file, uint tot_length, String &field_term, 
		     String &line_term, String &enclosed, int escape)
  :file(file),escape_char(escape)
{
  field_term_ptr=(char*) field_term.ptr();
  field_term_length= field_term.length();
  line_term_ptr=(char*) line_term.ptr();
  line_term_length= line_term.length();
  enclosed_char= (enclosed_length=enclosed.length()) ? enclosed[0] : INT_MAX;
  field_term_char= field_term_length ? field_term_ptr[0] : INT_MAX;
  line_term_char= line_term_length ? line_term_ptr[0] : INT_MAX;
  error=eof=found_end_of_line=found_null=line_cuted=0;
  buff_length=tot_length;


  /* Set of a stack for unget if long terminators */
  uint length=max(field_term_length,line_term_length)+1;
  stack=stack_pos=(int*) sql_alloc(sizeof(int)*length);

  if (!(buffer=my_malloc(buff_length+1,MYF(0))))
    error=1;
  else
  {
    end_of_buff=buffer+buff_length;
    if (init_io_cache(&cache,file,0,READ_CACHE,0L,1,MYF(MY_WME)))
    {
      my_free(buffer,MYF(0));
      error=1;
    }
  }
}


READ_INFO::~READ_INFO()
{
  if (!error)
  {
    end_io_cache(&cache);
    my_free(buffer,MYF(0));
    error=1;
  }
}


#define GET (stack_pos != stack ? *--stack_pos : my_b_get(&cache))
#define PUSH(A) *(stack_pos++)=(A)


inline int READ_INFO::terminator(char *ptr,uint length)
{
  int chr=0;					// Keep gcc happy
  uint i;
  for (i=1 ; i < length ; i++)
  {
    if ((chr=GET) != *++ptr)
      break;
  }
  if (i == length)
    return 1;
  PUSH(chr);
  while (i-- > 1)
    PUSH(*--ptr);
  return 0;
}


int READ_INFO::read_field()
{
  int chr,found_enclosed_char;
  char *to,*new_buffer;

  if (found_end_of_line)
    return 1;					// One have to call next_line

  if ((chr=GET) == my_b_EOF)
  {
    found_end_of_line=eof=1;
    return 1;
  }

  to=buffer;
  if (chr == enclosed_char)
  {
    found_enclosed_char=enclosed_char;
    *to++=(char) chr;				// If error
  }
  else
  {
    found_enclosed_char= INT_MAX;
    PUSH(chr);
  }

  for (;;)
  {
    while ( to < end_of_buff) 
    {
      if ((chr=GET) == my_b_EOF)
	goto found_eof;
      if (chr == escape_char)
      {
	if ((chr=GET) == my_b_EOF)
	{
	  *to++= (char) escape_char;
	  goto found_eof;
	}
	*to++ =unescape((char) chr);
	continue;
      }
      if (chr == line_term_char)
      {
	if (terminator(line_term_ptr,line_term_length))
	{					// Maybe unexpected linefeed
	  enclosed=0;
	  found_end_of_line=1;
	  row_start=buffer;
	  row_end=  to;
	  return 0;
	}
      }
      if (chr == found_enclosed_char)
      {
	if ((chr=GET) == found_enclosed_char)
	{					// Remove dupplicated
	  *to++ = (char) chr;
	  continue;
	}
	// End of enclosed field if followed by field_term or line_term
	if (chr == my_b_EOF ||
	    chr == line_term_char && terminator(line_term_ptr,line_term_length))
	{					// Maybe unexpected linefeed
	  enclosed=1;
	  found_end_of_line=1;
	  row_start=buffer+1;
	  row_end=  to;
	  return 0;
	}
	if (chr == field_term_char &&
	    terminator(field_term_ptr,field_term_length))
	{
	  enclosed=1;
	  row_start=buffer+1;
	  row_end=  to;
	  return 0;
	}
      }
      else if (chr == field_term_char && found_enclosed_char == INT_MAX)
      {
	if (terminator(field_term_ptr,field_term_length))
	{
	  enclosed=0;
	  row_start=buffer;
	  row_end=  to;
	  return 0;
	}
      }
      *to++ = chr;
    }
    /*
    ** We come here if buffer is too small. Enlarge it and continue
    */
    if (!(new_buffer=my_realloc(buffer,buff_length+1+IO_SIZE,
				MYF(MY_WME))))
      return (error=1);
    to=new_buffer + (to-buffer);
    buffer=new_buffer;
    buff_length+=IO_SIZE;
    end_of_buff=buffer+buff_length;
  }

found_eof:
  enclosed=0;
  found_end_of_line=eof=1;
  row_start=buffer;
  row_end=to;
  return 0;
}


int READ_INFO::read_fixed_length()
{
  int chr;
  char *to;
  if (found_end_of_line)
    return 1;					// One have to call next_line

  to=row_start=buffer;
  while (to < end_of_buff)
  {
    if ((chr=GET) == my_b_EOF)
      goto found_eof;
    if (chr == escape_char)
    {
      if ((chr=GET) == my_b_EOF)
      {
	*to++= (char) escape_char;
	goto found_eof;
      }
      *to++ =unescape((char) chr);
      continue;
    }
    if (chr == line_term_char)
    {
      if (terminator(line_term_ptr,line_term_length))
      {					// Maybe unexpected linefeed
	found_end_of_line=1;
	row_end=  to;
	return 0;
      }
    }
    *to++ = chr;
  }
  row_end=to;					// Found full line
  return 0;

found_eof:
  found_end_of_line=eof=1;
  row_start=buffer;
  row_end=to;
  return to == buffer ? 1 : 0;
}


int READ_INFO::next_line()
{
  line_cuted=0;
  if (found_end_of_line)
  {
    found_end_of_line=0;
    return eof;
  }
  found_end_of_line=0;
  if (eof)
    return 1;
  if (!line_term_length)
    return 0;					// No lines

  for (;;)
  {
    int chr;
    if ((chr=GET) == my_b_EOF)
    {
      eof=1;
      return 1;
    }
    if (chr == escape_char)
    {      
      line_cuted=1;	
      if (GET == my_b_EOF)
	return 1;
      continue;
    }
    if (chr == line_term_char && terminator(line_term_ptr,line_term_length))
      return 0;
    line_cuted=1;
  }
}
