/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB
   
   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.
   
   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.  The License grants you the right to 
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

/* Insert of records */

#include "mysql_priv.h"

static int check_null_fields(THD *thd,TABLE *entry);

/*
  Check if insert fields are correct
  Resets form->time_stamp if a timestamp value is set
*/

static int
check_insert_fields(THD *thd,TABLE *table,List<Item> &fields,List<Item> values)
{
  bool all_fields;
  tname_t table_list;
  table_list.name=table->table_name;
  table_list.table=table;
  table_list.next=0;

  if ((all_fields= (fields.elements == 0)))
  {
    if (values.elements != table->form.fields)
    {
      my_error(ER_WRONG_VALUE_COUNT,MYF(0));
      return -1;
    }
    table->form.time_stamp=0;			// This should be saved
  }
  else
  {						// Part field list
    if (fields.elements != values.elements)
    {
      my_error(ER_WRONG_VALUE_COUNT,MYF(0));
      return -1;
    }
    if (setup_fields(thd,&table_list,fields,1) < 0)
      return -1;
    if (thd->dupp_field)
    {
      my_printf_error("Field '%s' specified twice",MYF(0),
		      thd->dupp_field->field_name);
      return -1;
    }
    if (table->form.timestamp_field &&		// Don't set timestamp if used
	table->form.timestamp_field->query_id == thd->query_id)
      table->form.time_stamp=0;			// This should be saved
  }
  if (setup_fields(thd,&table_list,values,0) < 0)
    return -1;
  return 0;
}


int mysql_insert(THD *thd,tname_t *table_list,List<Item> &fields,
		 List<Item> &values)
{
  int error;
  ulong id;
  FORM	*form;
  TABLE *table,**table_ptr;
  DBUG_ENTER("mysql_insert");
  uint save_time_stamp;

  if (!(table=table_list->table))
  {						/* Not already open */
    if (!(table = open_ltable(thd,table_list->real_name,table_list)))
      DBUG_RETURN(-1);
  }
  table_ptr= &table;
  form= &table->form;
  form->reginfo.update=1;			/* Write lock */
  if (!(thd->lock=mysql_lock_tables(thd,table_ptr,1)))
    DBUG_RETURN(-1);

  save_time_stamp=form->time_stamp;
  if (check_insert_fields(thd,table,fields,values))
    goto err;

  /*
  ** Fill in the given fields and dump it to the table file
  */
  form->next_number_field=form->found_next_number_field;
  if (fields.elements)
  {
    restore_record(form,2);			// Get empty record
    if (fill_record(fields,values) || check_null_fields(thd,table))
    {
      form->next_number_field=0;
      goto err;
    }
  }
  else
  {
    form->record[0][0]=form->record[2][0];	// Fix delete marker
    if (fill_record(table,values))
    {
      form->next_number_field=0;
      goto err;
    }
  }
  error=ha_write(form,form->record[0]);
  id=0;
  if (form->next_number_field)
  {
    id=(ulong) valfield_l(form->next_number_field);
    form->next_number_field=0;
  }
  mysql_unlock_tables(thd->lock); thd->lock=0;
  form->time_stamp=save_time_stamp;		// Restore auto timestamp pointer

  if (error)
  {
    int key=ha_keyerror(form,error);
    if (key >= 0)
    {
      char *buff= (char*) my_alloca(ha_max_key_length);
      uint length=key_unpack(buff,form,(uint) key,0);
      uint max_length=sizeof(thd->net.last_error)-strlen(ER(ER_DUP_ENTRY));
      if (length > max_length)
	strmov(buff+max_length-4,"...");
      my_error(ER_DUP_ENTRY,MYF(0),buff,key+1);
      my_afree(buff);
    }
    else
      ha_error(form,error,MYF(0));
    DBUG_RETURN(-1);
  }
  send_ok(&thd->net,1L,id);
  DBUG_RETURN(0);

err:
  form->time_stamp=save_time_stamp;		// Restore auto timestamp pointer
  DBUG_RETURN(-1);
}

/******************************************************************************
	Check that all fields with arn't null_fields are used
	if DONT_USE_DEFAULT_FIELDS isn't defined use default value for not
	set fields.
******************************************************************************/

static int check_null_fields(THD *thd __attribute__((unused)),
			     TABLE *entry __attribute__((unused)))
{
#ifdef DONT_USE_DEFAULT_FIELDS
  for (FIELD *field=entry->field ; field ; field=field->next)
  {
    if (field->query_id != thd->query_id &&
	field->field->type != MTYP_TIMESTAMP &&
	!f_maybe_null(field->field->flag) &&
	field->field != entry->form.next_number_field)
    {
      my_printf_error(BAD_NULL_ERROR,MYF(0),field->field_name);
      return 1;
    }
  }
#endif
  return 0;
}


/***************************************************************************
** store records in INSERT ... SELECT *
***************************************************************************/

int 
select_insert::prepare(List<Item> &values)
{
  FORM *form=&table->form;
  DBUG_ENTER("select_insert::prepare");

  if (check_insert_fields(current_thd,table,*fields,values))
    DBUG_RETURN(-1);

  if (fields->elements)
  {
    restore_record(form,2);			// Get empty record
  }
  else
    form->record[0][0]=form->record[2][0];	// Fix delete marker
  form->next_number_field=form->found_next_number_field;
  current_thd->count_cuted_fields=1;			/* calc cuted fields */
  current_thd->cuted_fields=0;
  ha_extra(form,HA_EXTRA_WRITE_CACHE);
  DBUG_RETURN(0);
err:
  DBUG_RETURN(-1);
}

select_insert::~select_insert()
{
  table->form.time_stamp=save_time_stamp;
  table->form.next_number_field=0;
  current_thd->count_cuted_fields=0;
}


int select_insert::send_data(List<Item> &values)
{
  int error;
  records++;
  if (fields->elements)
    fill_record(*fields,values);
  else
    fill_record(table,values);

  if ((error=ha_write(&table->form,table->form.record[0])))
  {
    if (error == HA_ERR_FOUND_DUPP_KEY)
      duplicates++;
    else
    {
      ha_error(&table->form,error,MYF(0));
      return(-1);
    }
  }
  if (table->form.next_number_field)
  {						// Clear for next record
    FIELD *field=table->form.next_number_field;
    f_fieldinit(field->str,(uint) field->pack_flag,(uint) field->length);    
  }
  return 0;
}


void select_insert::send_error(char *err)
{
  ::send_error(&current_thd->net,err);
  VOID(ha_extra(&table->form,HA_EXTRA_NO_CACHE));
}


void select_insert::send_eof()
{
  int error;
  THD *thd=current_thd;
  if ((error=ha_extra(&table->form,HA_EXTRA_NO_CACHE)))
  {
    ha_error(&table->form,error,MYF(0));
    ::send_error(&thd->net,NullS);
  }
  else
  {
    char buff[160];
    sprintf(buff,ER(ER_INSERT_INFO),records,duplicates,thd->cuted_fields);
    ::send_ok(&thd->net,records-duplicates,0L,buff);
  }
}
