/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB
   
   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.
   
   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.  The License grants you the right to 
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

/* Basic functions neaded by many modules */

#include "mysql_priv.h"
#include "select_q.h"
#include <m_ctype.h>
#include <my_dir.h>
#include <hash.h>
#include <nisam.h>

#define MAX_DBKEY_LENGTH (NAME_LEN*2+2)
#ifndef EXTRA_DEBUG
#define EXTRA_DEBUG				// Let this be one a while
#endif

static int key_cache_used=0;
TABLE *unused_tables;				/* Used by mysql_test */
HASH open_cache;				/* Used by mysql_test */


static FIELD *find_field_in_table(THD *thd,TABLE *table,FIELD *fields,
				    const string where);
static int open_unireg_entry(TABLE *entry,const char *db,const char *name,
			     const char *alias);
static bool insert_fields(THD *thd,tname_t *tables, char *table_name,
			  List_iterator<Item> *it);
static void free_cache_entry(TABLE *entry);
static void mysql_rm_tmp_tables(void);
static bool test_if_null(FIELD *field);
static bool add_unireg_cond(COND *cond,SELECT_ROW **pos,bool and_level,
			    uint row_flag);

static byte *cache_key(const byte *record,uint *length)
{
  TABLE *entry=(TABLE*) record;
  *length=entry->key_length;
  return entry->key;
}

void table_cache_init(void)
{
  VOID(init_hash(&open_cache,table_cache_size,0,0,cache_key,
		 (void (*)(void*)) free_cache_entry));
  mysql_rm_tmp_tables();
}


void table_cache_free(void)
{
  close_cached_tables();
  if (!open_cache.records)			// Safety first
    free_hash(&open_cache);
}


uint cached_tables(void)
{
  return open_cache.records;
}

#ifdef EXTRA_DEBUG
static void check_unused(void)
{
  uint count=0,index=0;
  TABLE *link,*start_link;

  if ((start_link=link=unused_tables))
  {
    do
    {
      if (link != link->next->prev || link != link->prev->next)
      {
	DBUG_PRINT("error",("Unused_links arn't linked properly"));
	return;
      }
    } while (count++ < open_cache.records && (link=link->next) != start_link);
    if (link != start_link)
    {
      DBUG_PRINT("error",("Unused_links arn't connected"));
    }
  }
  for (index=0 ; index < open_cache.records ; index++)
  {
    TABLE *entry=(TABLE*) hash_element(&open_cache,index);
    if (!entry->in_use)
      count--;
  }
  if (count != 0)
  {
    DBUG_PRINT("error",("Unused_links dosen't match open_cache: diff: %d",
			count));
  }
}
#else
#define check_unused()
#endif

int mysql_init_db(char *DB)
{
  char	path[FN_REFLEN];
  DBUG_ENTER("mysql_init_db");

  (void)sprintf(path,"%s/%s",mysql_data_home,DB);
  if (access(path,F_OK))
  {
    my_error(ER_BAD_DB_ERROR,MYF(0),DB);
    DBUG_RETURN(-1);
  }
  DBUG_RETURN(0);
}


/******************************************************************************
** Send name and type of result to client.
** Sum fields has table name empty and field_name.
******************************************************************************/

void
send_fields(THD *thd,List<Item> &list,uint flag)
{
  FIELD *field,tmp_field;
  byte *pos;
  List_iterator<Item> it(list);
  Item *item;
  char buff[80];
  String tmp(buff,sizeof(buff)),*res;

  if (flag & 1)
  {
    pos=net_store_length(thd->packet,list.elements);
    (void) net_write(&thd->net, thd->packet,(uint) (pos-thd->packet));
  }
  LINT_INIT(field);
  while ((item=it++))
  {
    field=item->make_field(&tmp_field);

    pos=net_store_data(thd->packet,field->table_name);
    pos=net_store_data(pos,item->name);
    *pos++=3; int3store(pos,field->length); pos+=3;
    *pos++=1; *pos++= (char) field->type;
    *pos++=2; *pos++= (char) field->flags; *pos++= (char) field->decimals;
    if (flag & 2)
    {						// Send default value
      res=item->str(&tmp);
      if (!res)
	pos=net_store_length(pos,NULL_LENGTH);
      else
	pos=net_store_data(pos,res->ptr(),res->length());
    }
    if (net_write(&thd->net,thd->packet,(uint) (pos - thd->packet)))
      break;
  }
  send_eof(&thd->net);
}


/*****************************************************************************
 *	 Functions to free open table cache
 ****************************************************************************/


void intern_close_table(TABLE *entry)
{						// Free all structures
  free_io_cache(entry);
  if (entry->form.file)
    VOID(closefrm(&entry->form));		// close file
  my_free(entry->key,MYF(0));
}


static void free_cache_entry(TABLE *entry)
{
  DBUG_ENTER("free_cache_entry");

  intern_close_table(entry);
  if (!entry->in_use)
  {
    entry->next->prev=entry->prev;		/* remove from used chain */
    entry->prev->next=entry->next;
    if (entry == unused_tables)
    {
      unused_tables=unused_tables->next;
      if (entry == unused_tables)
	unused_tables=0;
    }
    check_unused();
  }
  my_free((gptr) entry,MYF(0));
  DBUG_VOID_RETURN;
}


void free_io_cache(TABLE *entry)
{
  if (entry->io_cache)
  {
    close_cacheed_file(entry->io_cache);
    my_free((gptr) entry->io_cache,MYF(0));
    entry->io_cache=0;
  }
}

	/* Close all tables with arn't in use by any thread */

void close_cached_tables(void)
{
  VOID(pthread_mutex_lock(&LOCK_open));
  while (unused_tables)
    VOID(hash_delete(&open_cache,(byte*) unused_tables));
  if (!open_cache.records)
  {
    end_key_cache();				/* No tables in memory */
    key_cache_used=0;
  }
  VOID(pthread_mutex_unlock(&LOCK_open));
}


/* Put all tables used by thread in free list */

void close_thread_tables(THD *thd)
{
  TABLE *table,*next;
  DBUG_ENTER("close_thread_tables");

  if (thd->lock)
  {
    mysql_unlock_tables(thd->lock);
    thd->lock=0;
  }
  /* VOID(pthread_sigmask(SIG_SETMASK,&thd->block_signals,NULL)); */
  VOID(pthread_mutex_lock(&LOCK_open));

  for (table=thd->open_tables ; table ; table=next)
  {
    next=table->next;
    if (thd->version != reload_version)		/* loaddb or rm has happend */
      VOID(hash_delete(&open_cache,(byte*) table));
    else
    {
      table->in_use=0;
      if (unused_tables)
      {
	table->next=unused_tables;		/* Link in last */
	table->prev=unused_tables->prev;
	unused_tables->prev=table;
	table->prev->next=table;
      }
      else
	unused_tables=table->next=table->prev=table;
    }
  }
  thd->open_tables=0;
  check_unused();
  VOID(pthread_mutex_unlock(&LOCK_open));
  /*  VOID(pthread_sigmask(SIG_SETMASK,&thd->signals,NULL)); */
  DBUG_VOID_RETURN;
}

	/* move table first in unused links */

static void relink_unused(TABLE *entry)
{
  if (entry != unused_tables)
  {
    entry->prev->next=entry->next;		/* Remove from unused list */
    entry->next->prev=entry->prev;
    entry->next=unused_tables;			/* Link in unused tables */
    entry->prev=unused_tables->prev;
    unused_tables->prev->next=entry;
    unused_tables->prev=entry;
    unused_tables=entry;
    check_unused();
  }
}



/******************************************************************************
** open a table
** Uses a cache of open tables to find a table not in use.
******************************************************************************/


TABLE *open_table(THD *thd,const char *db,const char *table,const char *alias)
{
  reg1	TABLE *entry;
  char	key[MAX_DBKEY_LENGTH];
  uint	key_length;
  DBUG_ENTER("open_table");

  /* find a unused entry in the open table cache */

  key_length=(uint) (strmov(strmov(key,db)+1,table)-key)+1;

  VOID(pthread_mutex_lock(&LOCK_open));

  for (entry=(TABLE*) hash_search(&open_cache,key,key_length) ;
       entry && entry->in_use ;
       entry = (TABLE*) hash_next(&open_cache,key,key_length)) ;
  if (entry)
  {
    if (entry == unused_tables)
    {						// First unused
      unused_tables=unused_tables->next;	// Remove from link
      if (entry == unused_tables)
	unused_tables=0;
    }
    entry->prev->next=entry->next;		/* Remove from unused list */
    entry->next->prev=entry->prev;
    if (strcmp(entry->table_name,alias))
    {
      uint i;
      my_free((gptr) entry->form.reginfo.alias,MYF(0));
      entry->table_name=entry->form.reginfo.alias=my_strdup(alias,MYF(MY_WME));
      for (i=0 ; i < entry->form.fields ; i++)
	entry->field[i].table_name=entry->table_name;
    }
  }
  else
  {
    /* Free cache if too big */
    while (open_cache.records >= table_cache_size && unused_tables)
      VOID(hash_delete(&open_cache,(byte*) unused_tables));

    VOID(pthread_mutex_unlock(&LOCK_open));	/* Don't wait for slow open */

    /* make a new entry */
    entry=(TABLE*) my_malloc(sizeof(*entry),MYF(MY_FAE | MY_ZEROFILL));
    entry->key=my_memdup(key,key_length,MYF(MY_FAE));
    entry->key_length= key_length;
    if (open_unireg_entry(entry,db,table,alias))
    {
      entry->next=entry->prev=entry;
      free_cache_entry(entry);
      DBUG_RETURN(NULL);
    }
    VOID(pthread_mutex_lock(&LOCK_open));
    if (!key_cache_used)
    {
      key_cache_used=1;
      ha_key_cache();
    }
    VOID(hash_insert(&open_cache,(byte*) entry));
  }

  entry->in_use=thd;
  entry->version=thd->version;
  check_unused();
  VOID(pthread_mutex_unlock(&LOCK_open));
  entry->next=thd->open_tables;		/* Link into simple list */
  thd->open_tables=entry;
  entry->tmp_table = 0;
  entry->tablenr=thd->current_tablenr++;
  entry->used_fields=0;
  entry->form.reginfo.ref_fields=0;		/* Some precation */
  entry->form.reginfo.update=0;			/* Assume read */
  DBUG_RETURN(entry);
}

/****************************************************************************
** Reopen an already open table because the definition has changed
** Returns 0 if ok.
** If table can't be reopened, the entry is unchanged.
****************************************************************************/

int reopen_table(THD *thd,TABLE *table)
{
  TABLE tmp;
  char *db=table->key;
  char *table_name=strend(table->key)+1;
  DBUG_ENTER("reopen_table");

  if (open_unireg_entry(&tmp,db,table_name,table->table_name))
    DBUG_RETURN(1);
  VOID(pthread_mutex_lock(&LOCK_open));

  free_io_cache(table);
  if (table->form.file)
    VOID(closefrm(&table->form));		// close file
  memcpy((char*) &table->form,(char*) &tmp.form,sizeof(tmp.form));
  table->table_name=	tmp.table_name;
  table->null_flags=	tmp.null_flags;
  table->field=		tmp.field;
  table->field_end=	tmp.field_end;
  table->version=	thd->version;
  for (FIELD *field=tmp.field ; field != tmp.field_end ; field++)
    field->table=table;
  VOID(pthread_mutex_unlock(&LOCK_open));
  DBUG_RETURN(0);
}


/****************************************************************************
	open_unireg_entry
**	Purpose : Load a table definition from file and open unireg table
**	Args	: entry with DB and table given
**	Returns : 0 if ok
*/

static int open_unireg_entry(TABLE *entry,const char *db,const char *name,
			     const char *alias)
{
  uint index,null_count,null_length,flag,key;
  FIELD *fields,*field,**key_fields;
  FORM	*form;
  char path[FN_REFLEN],*table_name;
  uchar *null_flags;
  DBUG_ENTER("open_unireg_entry");

  (void)sprintf(path,"%s/%s/%s",mysql_data_home,db,name);
  form= &entry->form;
  if (openfrm(path,"",
	      (uint) (HA_OPEN_KEYFILE | HA_OPEN_RNDFILE | HA_GET_INDEX |
		      HA_TRY_READ_ONLY),
	      READ_KEYINFO | COMPUTE_TYPES | EXTRA_RECORD,
	      form))
    DBUG_RETURN(1);
  VOID(ha_extra(form,HA_EXTRA_NO_READCHECK));		// Not needed in SQL
  entry->field=fields=form->field;
  table_name=entry->table_name=form->reginfo.alias=my_strdup(alias,MYF(MY_WME));
  null_count=0;
  entry->null_flags=null_flags=(uchar*) form->record[0]+form->reclength-
    (form->null_fields+7)/8;

  /* Add sql types to fields */

  for (field=fields,index=0 ; index < form->fields ; index++,field++)
  {
    field->table_name= table_name;
    field->field_name= form->fieldnames.type_names[index];
    field->field_name[0]=tolower(field->field_name[0]);
    field->packlength= packlength(field);
    flag=field->pack_flag;
    if (f_is_alpha(flag))
    {
      if (f_is_blob(flag))
      {
	field->length=field->packlength-sizeof(char*);
	switch (field->length) {
	case 1: field->type=FIELD_TYPE_TINY_BLOB;	break;
	case 2: field->type=FIELD_TYPE_BLOB;		break;
	case 3: field->type=FIELD_TYPE_MEDIUM_BLOB;	break;
	default: field->type=FIELD_TYPE_LONG_BLOB;	break;
	}
      }
      else if (f_is_packed(flag) && !field->intervall)
	field->type= (enum_field_types) f_packtype(flag);
      else if (f_is_packed(flag) || field->length < 3 ||
	       !(form->db_create_options & HA_OPTION_PACK_RECORD))
	field->type= FIELD_TYPE_STRING;
      else
	field->type= FIELD_TYPE_VAR_STRING;
    }
    else
      field->type= (enum enum_field_types) f_packtype(flag);

    if (!f_maybe_null(flag))
    {
      field->flags|= NOT_NULL_FLAG;
    }
    else
    {
      field->null_pos=null_flags+null_count/8;;
      field->null_bit=1 << (null_count & 7);
      null_count++;
    }
    if (f_is_blob(flag))
      field->flags|= BLOB_FLAG;
    if (f_is_zerofill(flag))
      field->flags|= ZEROFILL_FLAG;
    if (f_is_num(flag) && ! f_is_dec(flag))
      field->flags|= UNSIGNED_FLAG;
    field->decimals=f_decimals(flag);
    field->table= entry;
  }
  entry->field_end=field;

  /* Update form->usable_parts and field->key_parts */

  uint primary_key= MAX_KEY;
  for (key=0 ; key < form->keys ; key++)
  {
    if (!strcmp(form->key_info[key].name,"PRIMARY"))
    {
      primary_key=key;
      break;
    }
    else if (!form->key_info[key].dupp_key && primary_key == MAX_KEY)
      primary_key=key;				// Use first unique as primary
  }
  for (key=0 ; key < form->keys ; key++)
  {
    KEY *keyinfo= &form->key_info[key];
    KEY_PART_INFO *keypart=keyinfo->key_part;
    uint usable_parts=0;

    if (keypart->field)				// Something wrong with key
    {
      if (key != primary_key)
	keypart->field->flags |=
	  (!keyinfo->dupp_key &&
	   keypart->field->packlength ==
	   keyinfo->key_length ? UNIQUE_KEY_FLAG : MULTIPLE_KEY_FLAG);
      keypart->field->key_parts|= (1L << key);

      for (uint part=0; part < keyinfo->key_parts; part++,keypart++)
      {
	if (!keypart->field)
	  break;				// Rest are unusable
	if (keypart->key_part_flag)
	  break;				// Can't use HA_REVERSE_SORT
	usable_parts++;
	keypart->field->flags|= PART_KEY_FLAG;
	if (key == primary_key)
	  keypart->field->flags|= PRI_KEY_FLAG;
      }
    }
    entry->form.key_info[key].usable_key_parts=usable_parts;
  }

  /* Mark null fields as null */
  null_length=(null_count+7)/8;
  bfill(null_flags,null_length,255);	/* Default null field */
  bfill(null_flags+(uint) (form->record[1]-form->record[0]),null_length,255);
  bfill(null_flags+(uint) (form->record[2]-form->record[0]),null_length,255);
  DBUG_RETURN(0);
}


/*****************************************************************************
** open all tables in list
*****************************************************************************/

int open_tables(THD *thd,tname_t *tables)
{
  DBUG_ENTER("open_tables");
  for ( ; tables ; tables=tables->next)
    if (!(tables->table=open_table(thd,thd->db,tables->real_name,
				   tables->name)))
      DBUG_RETURN(-1);
  DBUG_RETURN(0);
}

TABLE *open_ltable(THD *thd,const char *table,tname_t *table_list)
{
  TABLE *entry;
  if ((entry=open_table(thd,thd->db,table,table)))
  {
    table_list->table=entry;
    table_list->name=(char*) table;
    table_list->next=0;
  }
  return entry;
}

/*
** Open all tables in list and locks them for read.
** The lock will automaticly be freed by the close_thread_tables
*/

int open_and_lock_tables(THD *thd,tname_t *tables)
{
  if (open_tables(thd,tables) || lock_tables(thd,tables))
    return -1;
  return 0;
}

int lock_tables(THD *thd,tname_t *tables)
{
  if (tables)
  {
    uint count=0;
    tname_t *table;
    for (table = tables ; table ; table=table->next)
      count++;
    TABLE **start,**ptr;
    ptr=start=(TABLE**) sql_alloc(sizeof(TABLE*)*count);
    for (table = tables ; table ; table=table->next)
      *(ptr++)= table->table;
    if (!(thd->lock=mysql_lock_tables(thd,start,count)))
      return -1;
  }
  return 0;
}


/*****************************************************************************
** find field in list or tables. if field is unqualifed and unique,
** return unique field
******************************************************************************/

static FIELD *
find_field_in_table(THD *thd,TABLE *table,Item_field *item,
		    const string where)
{
  FIELD *field,*end,*found=0;
  char *table_name=item->table_name;
  char *name=item->field_name;

  for (field=table->field,end=table->field_end ;
       field != end ;
       field++)
  {
    if (!my_strcasecmp(field->field_name, name))
    {
      if (!table_name)
      {
	if (found)
	{
	  my_error(ER_NON_UNIQ_ERROR,MYF(0),item->full_name(),where);
	  return (FIELD*) 0;
	}
	found=field;
      }
      else if (!strcmp(field->table_name,table_name))
      {
	found=field;
	break;
      }
    }
  }
  if (found)
  {
    if (thd->set_query_id)
    {
      if (found->query_id != thd->query_id)
      {
	found->query_id=thd->query_id;
	found->table->used_fields++;
      }
      else
	thd->dupp_field=found;
      return found;
    }
  }
  my_error(ER_BAD_FIELD_ERROR,MYF(0),item->name,(char*) where);
  return (FIELD*) 0;
}


FIELD *
find_field_in_tables(THD *thd,Item_field *item,tname_t *tables,
		     const string where)
{
  FIELD *found=0;
  char *table_name=item->table_name;
  char *name=item->field_name;

  if (table_name)
  {						/* Qualified field */
    for (; tables && strcmp(tables->name,table_name) ; tables=tables->next) ;
    if (!tables)
    {
      my_printf_error("Unknown table '%s' in %s",MYF(0),table_name,where);
      return (FIELD*) 0;
    }
    return find_field_in_table(thd,tables->table,item,where);
  }

  for (; tables ; tables=tables->next)
  {
    FIELD *field,*end;
    for (field= tables->table->field,end=tables->table->field_end;
	 field != end;
	 field++)
    {
      if (!my_strcasecmp(field->field_name, name))
      {
	if (found)
	{
	  if (!where)
	    break;
	  my_error(ER_NON_UNIQ_ERROR,MYF(0),name,where);
	  return (FIELD*) 0;
	}
	found=field;
      }
    }
  }
  if (found)
  {
    if (thd->set_query_id)
    {
      if (found->query_id != thd->query_id)
      {
	found->query_id=thd->query_id;
	found->table->used_fields++;
      }
      else
	thd->dupp_field=found;
    }
    return found;
  }
  my_error(ER_BAD_FIELD_ERROR,MYF(0),item->full_name(),where);
  return (FIELD*) 0;
}

Item **
find_item_in_list(Item *find,List<Item> &items,string where)
{
  List_iterator<Item> li(items);
  Item **found=0,*item;
  char *field_name=0,*table_name=0;
  if (find->type() == Item::FIELD_ITEM  || find->type() == Item::REF_ITEM)
  {
    field_name= ((Item_ident*) find)->field_name;
    table_name= ((Item_ident*) find)->table_name;
  }

  while ((item=li++))
  {
    if (field_name && item->type() == Item::FIELD_ITEM)
    {
      if (!my_strcasecmp(((Item_field*) item)->name,field_name))
      {
	if (!table_name)
	{
	  if (found)
	  {
	    if ((*found)->eq(item))
	      continue;				// Same field twice (Access?) 
	    if (where)
	      my_error(ER_NON_UNIQ_ERROR,MYF(0),find->full_name(),where);
	    return (Item**) 0;
	  }
	  found=li.ref();
	}
	else if (!strcmp(((Item_field*) item)->table_name,table_name))
	{
	  found=li.ref();
	  break;
	}
      }
    }
    else if (!my_strcasecmp(item->name,find->name))
    {
      found=li.ref();
      break;
    }
  }
  if (!found && where)
    my_error(ER_BAD_FIELD_ERROR,MYF(0),find->full_name(),where);
  return found;
}


/****************************************************************************
**	Check that all given fields exists and fill struct with current data
****************************************************************************/

int setup_fields(THD *thd,tname_t *tables,List<Item> &fields,bool set_query_id)
{
  reg2 Item *item;
  List_iterator<Item> it(fields);
  DBUG_ENTER("setup_fields");

  thd->set_query_id=set_query_id;
  while ((item=it++))
  {
    if (item->type() == Item::FIELD_ITEM &&
	((Item_field*) item)->field_name[0] == '*')
    {
      if (insert_fields(thd,tables,((Item_field*) item)->table_name,&it))
	DBUG_RETURN(-1);
    }
    else if (item->fix_fields(thd,tables,"field_list"))
      DBUG_RETURN(-1);
  }
  DBUG_RETURN(0);
}


/****************************************************************************
**	This just drops in all fields instead of current '*' field
**	Returns pointer to last inserted field if ok
****************************************************************************/


static bool
insert_fields(THD *thd,tname_t *tables, char *table_name,
	      List_iterator<Item> *it)
{
  tname_t *table;
  uint found;
  DBUG_ENTER("insert_fields");

  found=0;
  for (table=tables ; table ; table=table->next)
  {
    if (!table_name || !strcmp(table_name,table->name))
    {
      FIELD *field,*end;
      for (field=table->table->field,end=table->table->field_end;
	   field != end;
	   field++)
      {
	Item_field *item= new Item_field(field);
	if (!found++)
	  (void) it->replace(item);
	else
	  it->after(item);
	if (field->query_id == thd->query_id)
	  thd->dupp_field=field;
	field->query_id=thd->query_id;
      }
      /* All fields are used */
      table->table->used_fields=table->table->form.fields;
    }
  }
  if (!found)
    my_printf_error("Unknown table '%s' used",MYF(0),table_name);
  DBUG_RETURN(!found);
}

int setup_conds(THD *thd,tname_t *tables,COND *conds)
{
  DBUG_ENTER("setup_conds");
  thd->set_query_id=1;
  thd->cond_count=0;
  if (conds)
  {
    DBUG_RETURN(conds->fix_fields(thd,tables,"where_clause"));
  }
  DBUG_RETURN(0);
}

uint count_select_levels(COND *cond,uint *deep)
{
  cond->marker=0;
  if (cond->type() != Item_func::COND_ITEM)
    return 1;
  (*deep)++;
  uint count=2;					// Two rows for '(' ')'
  List_iterator<Item> li(((Item_cond*) cond)->list);
  Item *item;
  while ((item=li++))
    count+=count_select_levels(item,deep);
  return count;
}

  

/*****************************************************************************
  set and reset a bit in last field (for null-fields)
*****************************************************************************/

void mark_as_null(FIELD *field)
{
  (*field->null_pos)|=(uchar) field->null_bit;
}

void mark_as_not_null(FIELD *field)
{
  (*field->null_pos)&= (uchar) ~(uint) field->null_bit;
}

static bool test_if_null(FIELD *field)
{
  return ((*field->null_pos) & (uchar) field->null_bit) != 0;
}


bool test_if_null_offset(FIELD *field,int offset)
{
  return ((field->null_pos[offset]) & (uchar) field->null_bit) != 0;
}


bool test_safe_null(FIELD *field)
{
  return f_maybe_null(field->pack_flag) && test_if_null(field);
}


void copy_field_from_tmp_record(FIELD *field,int offset)
{
  memcpy(field->str,field->str+offset,field->packlength);
  if (field->null_pos)
  {
    field->null_pos[0]= ((field->null_pos[0] &
			  (uchar) ~(uint) field->null_bit) |
			 field->null_pos[offset] &
			 (uchar) field->null_bit);
  }
}



/******************************************************************************
** Fill a record with data (for INSERT or UPDATE)
** Returns : 1 if some field has wrong type
******************************************************************************/

int
fill_record(List<Item> &fields,List<Item> &values)
{
  List_iterator<Item> f(fields),v(values);
  Item *value;
  Item_field *field;
  DBUG_ENTER("fill_record");

  while ((field=(Item_field*) f++))
  {
    value=v++;
    if (value->save_in_field(field->field))
      DBUG_RETURN(1);
  }
  DBUG_RETURN(0);
}


int
fill_record(TABLE *table,List<Item> &values)
{
  List_iterator<Item> v(values);
  Item *value;
  DBUG_ENTER("fill_record");

  FIELD *field,*end;
  for (field=table->field,end=table->field_end; field != end ; field++)
  {
    value=v++;
    if (value->save_in_field(field))
      DBUG_RETURN(1);
  }
  DBUG_RETURN(0);
}


static void mysql_rm_tmp_tables(void)
{
  uint index;
  char	*tmpdir,filePath[FN_REFLEN];
  MY_DIR *dirp;
  FILEINFO *file;
  DBUG_ENTER("mysql_rm_tmp_tables");

  if (!(tmpdir=getenv("TMPDIR")))	/* Use this if possibly */
    tmpdir=P_tmpdir;			/* Use system default */

  /* See if the directory exists */
  if (!(dirp = my_dir(tmpdir,MYF(MY_WME | MY_DONT_SORT))))
    DBUG_VOID_RETURN;

  /*
  ** Remove all SQLxxx tables from directory
  */

  for (index=2 ; index < (uint) dirp->number_off_files	; index++)
  {
    file=dirp->dir_entry+index;
    if (!bcmp(file,"SQL",3))
    {
      sprintf(filePath,"%s/%s",tmpdir,file->name);
      VOID(my_delete(filePath,MYF(MY_WME)));
    }
  }
  my_dirend(dirp);
  DBUG_VOID_RETURN;
}


/*
** mysql can't yet create index on the fly. all index must be created with
** CREATE TABLE.
** CREATE INDEX is implemented as an CHECK for if an index exists.
** If the CREATE INDEX statement fails then the table should be recreated
** if the index is needed
*/

int mysql_create_index(THD *thd,const char *table_name,
		       List<key_part_spec> &columns,
		       bool unique)
{
  TABLE *table;
  DBUG_ENTER("mysql_create_index");
  if (!(table = open_table(thd,thd->db,table_name,table_name)))
  {
    send_error(&thd->net,NullS);
    DBUG_RETURN(1);
  }
  List_iterator<key_part_spec> cols(columns);
  for (uint i=0 ; i < table->form.keys ; i++)
  {
    if (unique)
    {
      if (table->form.key_info[i].dupp_key ||
	  table->form.key_info[i].usable_key_parts != columns.elements)
	continue;
    }
    else if (ha_option_flag[(uint) table->form.db_type] & HA_READ_ORDER)
    {
      if ((uint) table->form.key_info[i].key_parts < columns.elements)
	continue;
    }
    else if ((uint) table->form.key_info[i].key_parts != columns.elements)
      continue;

    cols.rewind();
    key_part_spec *name;
    KEY_PART_INFO *key_part=table->form.key_info[i].key_part;
    for (; name=cols++ ;key_part++)
    {
      if (my_strcasecmp(name->field_name,key_part->field->field_name))
	break;
      if (name->length && key_part->length != name->length)
	break;
    }
    if (name)
      continue;
    send_ok(&thd->net);				// Found compatible key
    DBUG_RETURN(0);
  }
  my_error(ER_NO_SUCH_INDEX,MYF(0),table_name);
  DBUG_RETURN(-1);
}

/*****************************************************************************
	unireg support functions
*****************************************************************************/

	/* make a select from mysql info
	   Error is set as following:
	   0 = ok
	   1 = Got some error (out of memory?)
	   */

SELECT *make_unireg_select(TABLE **table,uint table_count,uint head,
			   uint const_tables, COND *conds, int *error)
{
  SELECT *select;
  uint count,brace_deep;
  SELECT_ROW *s_pos;
  SELECT_STACK *select_stack;
  DBUG_ENTER("make_unireg_select");

  *error=0;
  if (!conds)
    DBUG_RETURN(0);
  *error= 1;
  brace_deep=0;
  count=count_select_levels(conds,&brace_deep);
  if (!my_multi_malloc(MYF(MY_WME | MY_ZEROFILL),
		       &select,sizeof(*select),
		       &s_pos,sizeof(SELECT_ROW)*(count+1),
		       &select_stack,sizeof(SELECT_STACK)*3,
		       NullS))
    DBUG_RETURN(0);
  select->reg_read=select->reg_used=const_tables | (1 << head);
  select->const_reg=const_tables;
  select->reg_skipped= ~select->reg_used;
  select->maxreg=table_count;
  select->forms=table;
  select->stack=select_stack;
  select->max_brace_level=brace_deep;
  select->head=table[head];
  select->row=s_pos;

  if (add_unireg_cond(conds,&s_pos,0,SELECT_FLAG_AND))
    goto err;
  count=(uint) (s_pos - select->row);
  s_pos->flag=0;				/* End marker */
  if (!count)
  {
    my_free((gptr) select,MYF(0));
    *error=0;
    DBUG_RETURN(0);
  }

  if (table[head]->io_cache)
  {
    memcpy((gptr) &select->file,(gptr) table[head]->io_cache,sizeof(IO_CACHE));
    select->records=(select->file.end_of_file/
		     select->head->form.keyfile_info.ref_length);
  }
  *error=0;
  DBUG_RETURN(select);

err:
  bzero((gptr) &select->file,sizeof(select->file));
  end_select(select);
  DBUG_RETURN(0);
}


static bool
add_unireg_cond(COND *cond,SELECT_ROW **pos,bool and_level,uint row_flag)
{
  if (cond->type() == Item::COND_ITEM)
  {
    if (and_level)
    {
      (*pos)++ ->flag = SELECT_FLAG_LEFT_BRACE | SELECT_FLAG_NO_FIELDS |
	row_flag;
      row_flag= SELECT_FLAG_AND;
    }
    bool new_and_level= (((Item_cond*) cond)->functype() ==
			 Item_func::COND_AND_FUNC);
    List_iterator<Item> li(*((Item_cond*) cond)->argument_list());
    Item *item;
    while ((item=li++))
    {
      if (add_unireg_cond(item,pos,new_and_level,row_flag))
	return 1;
      row_flag= (new_and_level) ? SELECT_FLAG_AND : SELECT_FLAG_OR;
    }
    if (and_level)
      (*pos)++ ->flag = SELECT_FLAG_RIGHT_BRACE | SELECT_FLAG_NO_FIELDS |
	SELECT_FLAG_AND;
    return 0;
  }

  SELECT_ROW *s_pos= (*pos)++;
  bool like_flag;
  s_pos->flag=row_flag;
  enum Item_func::Functype functype=  ((Item_func*) cond)->functype();
  Item_field *cond_field=(Item_field*) (((Item_func*) cond)->arguments()[0]);
  FIELD *reg_field;

  /* Assume first is field */
  s_pos->field_type=SEL_FIELD_FIELD;
  s_pos->field.field.regnr= (int) cond_field->field->table->tablenr;
  reg_field=s_pos->field.field.regfield=cond_field->field;
  s_pos->reg_used=(1 <<  s_pos->field.field.regnr);

  if (functype == Item_func::ISNULL_FUNC ||
      functype == Item_func::ISNOTNULL_FUNC)
  {
    /* UNIREG Doesn't handle null, change to compare of empty string or 0 */
    s_pos->flag|= (functype == Item_func::ISNULL_FUNC) ? 1 : 2;
    if (f_is_alpha(reg_field->pack_flag))
    {
      char buff[MAX_FIELD_WIDTH],result[MAX_FIELD_WIDTH];
      f_fieldinit(buff,reg_field->pack_flag,reg_field->length);
      uint length=f_unpackfield_stripp(result,buff,reg_field->pack_flag,
				       reg_field->length,reg_field->intervall);
      s_pos->diff_type=SEL_FIELD_CONST_STR;
      s_pos->diff.str=sql_memdup(result,length+1);
      s_pos->cmp_type=SEL_CMP_STR;
    }
    else
    {
      s_pos->diff_type=SEL_FIELD_CONST_NUM;      
      s_pos->diff.nr=0.0;
      s_pos->cmp_type=SEL_CMP_NUM;
    }
    return 0;
  }

  Item *value=(Item*) ((Item_func*) cond)->arguments()[1];
  if (value->type() == Item::NULL_ITEM &&
      (functype == Item_func::EQ_FUNC ||
       functype == Item_func::NE_FUNC))
  {
    s_pos->flag|= (functype == Item_func::EQ_FUNC) ? 1 : 2;
    if (f_is_alpha(reg_field->pack_flag))
    {
      char buff[MAX_FIELD_WIDTH],result[MAX_FIELD_WIDTH];
      f_fieldinit(buff,reg_field->pack_flag,reg_field->length);
      uint length=f_unpackfield_stripp(result,buff,reg_field->pack_flag,
				       reg_field->length,reg_field->intervall);
      s_pos->diff_type=SEL_FIELD_CONST_STR;
      s_pos->diff.str=sql_memdup(result,length+1);
      s_pos->cmp_type=SEL_CMP_STR;
    }
    else
    {
      s_pos->diff_type=SEL_FIELD_CONST_NUM;      
      s_pos->diff.nr=0.0;
      s_pos->cmp_type=SEL_CMP_NUM;
    }
    return 0;
  }

  if (value->type() == Item::FIELD_ITEM)
  {
      s_pos->diff_type= SEL_FIELD_FIELD;
      s_pos->diff.field.regnr=(int) ((Item_field*) value)->field->
	table->tablenr;
      s_pos->diff.field.regfield= ((Item_field*) value)->field;
      s_pos->reg_used|=(1 <<  s_pos->diff.field.regnr);
  }
  like_flag= functype == Item_func::LIKE_FUNC || functype == Item_func::NOTLIKE_FUNC;
  if (value->type() == Item::STRING_ITEM ||
      f_is_alpha(reg_field->pack_flag) &&
      value->type() == Item::FIELD_ITEM || like_flag)
  {
    if (f_is_binary(reg_field->pack_flag))
      s_pos->cmp_type=SEL_CMP_BINARY_STR;
    else
      s_pos->cmp_type=SEL_CMP_STR;
    if (value->type() == Item::STRING_ITEM)
    {
      uint field_length=reg_field->length;
      const char *str= ((Item_string*) value)->str_value.c_ptr();
      size_t str_length=((Item_string*) value)->str_value.length();
      string errpos;

      s_pos->diff_type=SEL_FIELD_CONST_STR;
      if (like_flag)
      {
	if (strchr(str,wild_many) || (wild_prefix && strchr(str,wild_prefix)))
	  s_pos->cmp_type=SEL_CMP_WILD_STR;
	else
	  like_flag=0;				// Format value
      }
      if (field_length < str_length && !like_flag)
      {
	s_pos->flag|=SELECT_FLAG_ALWAYS_FALSE;
	s_pos->diff.str=(char*) str;
      }
      else
      {
	s_pos->diff.str=sql_alloc(field_length+1);
	memcpy(s_pos->diff.str,str,str_length+1);
	if (f_is_num(reg_field->pack_flag) && !like_flag)
	{
	  if (format_number(reg_field->pack_flag,field_length,
			    s_pos->diff.str,str_length,&errpos))
	    s_pos->flag|=SELECT_FLAG_ALWAYS_FALSE;
	}
      }
      if (s_pos->cmp_type != SEL_CMP_BINARY_STR)
	caseup_str(s_pos->diff.str);
    }
    else if (value->type() != Item::FIELD_ITEM)
    {
      my_printf_error("Value for \'%s\' does not match field or op type",
		      MYF(0),
		      cond_field->full_name());
      return 1;
    }
  }
  else
  {
    s_pos->cmp_type=SEL_CMP_NUM;
    if (value->type() != Item::FIELD_ITEM)
    {
      s_pos->diff_type=SEL_FIELD_CONST_NUM;
      s_pos->diff.nr=value->val();
    }
  }
  switch (functype) {
    case Item_func::EQ_FUNC: s_pos->flag|=1; break;
    case Item_func::NE_FUNC: s_pos->flag|=2; break;
    case Item_func::LT_FUNC: s_pos->flag|=3; break;
    case Item_func::LE_FUNC: s_pos->flag|=4; break;
    case Item_func::GT_FUNC: s_pos->flag|=5; break;
    case Item_func::GE_FUNC: s_pos->flag|=6; break;
    case Item_func::LIKE_FUNC:
      s_pos->flag|=1 | SELECT_FLAG_WILD_COMPARE; break;
    case Item_func::NOTLIKE_FUNC:
      s_pos->flag|=2 | SELECT_FLAG_WILD_COMPARE; break;
    default: break;				/* Impossible */
  }
  return 0;
}

/*****************************************************************************
*	Approximate how many records will be read from table
*****************************************************************************/

ulong get_quick_record_count(SELECT *select,uint table,uint keys)
{
  DBUG_ENTER("get_quick_record_count");
  select->head=select->forms[table];
  if (select->quick)
  {
    my_free((gptr) select->quick,MYF(0));
    select->quick=0;
  }
  if (test_quick_select(select,keys) == 1)
    DBUG_RETURN(select->quick->records);
  DBUG_PRINT("warning",("Couldn't use record count on const keypart"));
  DBUG_RETURN(~0L);				/* This shouldn't happend */
}

/****************************************************************************
** Store a String in a field
** This is neaded because a String don't have to be null_terminated
** This is never called for blob fields.
****************************************************************************/

void store_string(String &str,FIELD *field)
{
  uint str_length;
  if (!(str_length=str.length()))
  {
    f_fieldinit(field->str,field->pack_flag,field->length);
    return;
  }

  // Optimize the normal case :A non packed string field
  if (f_is_alpha(field->pack_flag) && !f_is_packed(field->pack_flag))
  {
    uint length=min(str_length,field->length);
    memcpy(field->str,(char*) str.ptr(),length);
    bfill(field->str+length,field->length-length,' ');
    return;
  }

  char *ptr=(char*) str.ptr();
  char buff[MAX_FIELD_WIDTH];

  if (str[str_length])
  {
    if (str.alloced_length() < str_length)
    {
      uint length=min(MAX_FIELD_WIDTH-1,str_length);
      memcpy(buff,ptr,length); buff[length]=0;
      ptr=buff;
    }
    else ptr[str_length]=0;
  }
  f_packfield(field->str,ptr,field->pack_flag,(uint) field->length,
	      field->intervall);
}



/*
** Invalidate any cache entries that are for some DB
** We can't use hash_delete when looping hash_elements, so mark them and
** delete what's unused.
*/

void remove_db_from_cache(const string db)
{
  for (uint index=0 ; index < open_cache.records ; index++)
  {
    TABLE *entry=(TABLE*) hash_element(&open_cache,index);
    if (!strcmp(entry->key,db))
    {
      entry->version=0L;			/* Free when thread is ready */
      if (!entry->in_use)
	relink_unused(entry);
    }
  }
  while (unused_tables && !unused_tables->version)
    VOID(hash_delete(&open_cache,(byte*) unused_tables));
}


void remove_table_from_cache(const char *db,const char *table)
{
  char key[MAX_DBKEY_LENGTH];
  uint key_length;
  TABLE *entry;

  key_length=(uint) (strmov(strmov(key,db)+1,table)-key)+1;
  for (entry=(TABLE*) hash_search(&open_cache,key,key_length) ;
       entry;
       entry = (TABLE*) hash_next(&open_cache,key,key_length))
  {
      entry->version=0L;			/* Free when thread is ready */
      if (!entry->in_use)
	relink_unused(entry);
      else
	entry->in_use->some_tables_deleted=1;
  }
  while (unused_tables && !unused_tables->version)
    VOID(hash_delete(&open_cache,(byte*) unused_tables));
}
