/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB
   
   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.
   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.  The License grants you the right to 
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

/* Functions to read, write and lock records */

#include "mysql_priv.h"

static int rr_quick(READ_RECORD *info);
static int rr_rnext(READ_RECORD *info);
static int rr_sequental(READ_RECORD *info);
static int rr_from_tempfile(READ_RECORD *info);
static int rr_from_cache(READ_RECORD *info);
static int init_rr_cache(READ_RECORD *info);
static int rr_cmp(uchar *a,uchar *b);

	/* init struct for read with info->read_record */

void init_read_record(READ_RECORD *info,FORM *reg_form,int key,
		      IO_CACHE *tempfile,SELECT *select)
{
  DBUG_ENTER("init_read_record");

  bzero((byte*) info,sizeof(*info));
  info->form=reg_form;
  info->forms= &info->form;		/* Only one form */
  info->record=reg_form->record[0];
  info->ref_length=reg_form->keyfile_info.ref_length;
  info->key=key-1;
  info->select=select;
  reg_form->found=1;			/* Record is allways read */
  reg_form->status=0;			/* And it's allways found */

  if (select && my_b_inited(&select->file))
    tempfile= &select->file;
  if (select && select->quick && (! tempfile || !tempfile->buffer))
  {
    DBUG_PRINT("info",("using rr_quick"));
    info->read_record=rr_quick;
  }
  else if (info->key >= 0)
  {
    DBUG_PRINT("info",("using rr_next"));
    info->read_record=rr_rnext;
    ha_reset(reg_form);
  }
  else if (tempfile && tempfile->buffer) /* Test if ref-records was used */
  {
    DBUG_PRINT("info",("using rr_from_tempfile"));
    info->read_record=rr_from_tempfile;
    info->io_cache=tempfile;
    info->io_cache->end_of_file=my_seek(tempfile->file,0L,MY_SEEK_END,MYF(0));
    reinit_io_cache(info->io_cache,READ_CACHE,0L,0);
    info->ref_pos=reg_form->keyfile_info.ref.refpos;

#ifndef _MSC_VER
    if (! (specialflag & SPECIAL_NO_NEW_FUNC) &&
	my_default_record_cache_size &&
	(reg_form->db_stat & HA_READ_ONLY || ! reg_form->reginfo.update) &&
	reg_form->reclength*(reg_form->keyfile_info.records+
			     reg_form->keyfile_info.deleted) >
	(ulong) RECORD_CACHE_SIZE * 16 &&
	info->io_cache->end_of_file/info->ref_length*reg_form->reclength >
	(ulong) my_default_record_cache_size*2 &&
	!reg_form->has_blob)
      if (! init_rr_cache(info))
      {
	DBUG_PRINT("info",("using rr_from_cache"));
	info->read_record=rr_from_cache;
      }
#endif
  }
  else
  {
    DBUG_PRINT("info",("using rr_sequential"));
    info->read_record=rr_sequental;
    ha_reset(reg_form);
    if (reg_form->reginfo.update == 0)
      VOID(ha_extra(reg_form,HA_EXTRA_CACHE));		/* Quick reads */
  }
  DBUG_VOID_RETURN;
} /* init_read_record */


void end_read_record(READ_RECORD *info)
{					/* free cache if used */
  if (info->cache)
  {
    my_free_lock(info->cache,MYF(0));
    info->cache=0;
  }
}

	/* Read a record from head-database */

static int rr_quick(READ_RECORD *info)
{
  return get_quick_record(info->select);
}

static int rr_rnext(READ_RECORD *info)
{
  return (ha_rnext(info->form,info->record,info->key));
}

static int rr_sequental(READ_RECORD *info)
{
  return (ha_r_rnd(info->form,info->record,info->ref_pos));
}

static int rr_from_tempfile(READ_RECORD *info)
{
  if (my_b_read(info->io_cache,info->ref_pos,info->ref_length))
    return HA_ERR_END_OF_FILE;			/* End of file */
  if (TEST_IF_LASTREF(info->ref_pos,info->ref_length))
    return(HA_ERR_END_OF_FILE);			/* File ends with this */
  return (ha_r_rnd(info->form,info->record,info->ref_pos));
} /* rr_from_tempfile */


	/* cacheing of records from a database */

#if !defined(_MSC_VER) || MAX_REFLENGTH != 4

static int init_rr_cache(READ_RECORD *info)
{
  DBUG_ENTER("init_rr_cache");

  info->struct_length=3+info->ref_length;
  info->reclength=max(info->form->reclength,sizeof(short))+
    info->struct_length+1;
  info->cache_records=my_default_record_cache_size/info->reclength;

  if (!(info->cache=my_malloc_lock(info->cache_records*info->reclength,MYF(0))))
    DBUG_RETURN(1);
  DBUG_PRINT("info",("Allocated buffert for %d records",info->cache_records));
  info->read_positions=info->cache+(info->reclength - info->ref_length)*
    info->cache_records;
  info->start_positions=info->cache+(info->reclength- info->struct_length)*
    info->cache_records;

  info->cache_pos=info->cache_end=info->cache;
  DBUG_RETURN(0);
} /* init_rr_cache */


static int rr_from_cache(READ_RECORD *info)
{
  reg1 uint i;
  uint length;
  ulong rest_of_file;
  int16 error;
  int lock_error;
  byte *position,*ref_position,*record_pos;
  ulong record;

  for (;;)
  {
    if (info->cache_pos != info->cache_end)
    {
      if (info->cache_pos[0])
      {
	shortget(error,info->cache_pos+1);
      }
      else
      {
	error=0;
	memcpy(info->record,info->cache_pos+1,(size_t) info->form->reclength);
      }
      info->cache_pos+=info->reclength;
      return ((int) error);
    }
    length=info->cache_records*info->ref_length;
    rest_of_file=info->io_cache->end_of_file - my_b_tell(info->io_cache);
    if ((ulong) length > rest_of_file)
      length=(uint) rest_of_file;
    if (!length || my_b_read(info->io_cache,info->read_positions,length))
      return HA_ERR_END_OF_FILE;		/* End of file */

    length/=info->ref_length;
    position=info->start_positions;
    ref_position=info->read_positions;
    for (i=0 ; i < length ; i++)
    {
      if (memcmp(ref_position,last_ref,(size_s) info->ref_length) == 0)
      {					/* End of file */
	if (! (length=i))
	  return HA_ERR_END_OF_FILE;	/* Last record and no in buffert */
	break;
      }
      memcpy(position,ref_position,(size_s) info->ref_length);
      int3store(position+info->ref_length,(long) i);
      position+=info->struct_length;
      ref_position+=info->ref_length;
    }
    qsort(info->start_positions,length,info->struct_length,(qsort_cmp) rr_cmp);

    ref_position=info->start_positions;
    position=info->cache+info->reclength - info->struct_length;
    for (i=0 ; i < length ; i++)
    {
      memcpy(position,ref_position,(size_s) info->struct_length);
      position+=info->reclength;
      ref_position+=info->struct_length;
    }

    lock_error=1;
    if (info->form->db_stat & HA_BLOCK_LOCK)
      lock_error=ha_lock(info->form,F_RDLCK);
    position=info->cache+info->reclength - info->struct_length;
    for (i=0 ; i < length ; i++)
    {
      memcpy(info->ref_pos,position,(size_s) info->ref_length);
      record=uint3korr(position+info->ref_length);
      record_pos=info->cache+record*info->reclength;
      if ((error=(int16) ha_r_rnd(info->form,record_pos+1,info->ref_pos)))
      {
	*record_pos=1;
	shortstore(record_pos+1,error);
      }
      else
	*record_pos=0;
      position+=info->reclength;
    }
    info->cache_end=(info->cache_pos=info->cache)+length*info->reclength;
    if (info->form->db_stat & HA_BLOCK_LOCK && ! lock_error)
      VOID(ha_lock(info->form,F_UNLCK));
  }
} /* rr_from_cache */


static int rr_cmp(uchar *a,uchar *b)
{
#if defined(WORDS_BIGENDIAN) && MAX_REFLENGTH == 4
  if (a[0] != b[0])
    return (int) a[0] - (int) b[0];
  if (a[1] != b[1])
    return (int) a[1] - (int) b[1];
  if (a[2] != b[2])
    return (int) a[2] - (int) b[2];
  return (int) a[3] - (int) b[3];
#else
  long la,lb;

  longget (la,a);
  longget (lb,b);

  return la - lb;
#endif
}

#endif /* _MSC_VER */
