/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB
   
   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.
   
   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.  The License grants you the right to 
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

#include "mysql_priv.h"
#include "sql_lex.h"
#include <nisam.h>
#include <thr_alarm.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/resource.h>
#include <getopt.h>
#ifdef HAVE_SYS_UN_H
#  include <sys/un.h>
#endif
#include <netdb.h>
#ifdef HAVE_SELECT_H
#  include <select.h>
#endif
#include <sys/utsname.h>
#ifdef HAVE_LINUXTHREADS
#include <gnu/types.h>
#define THR_KILL_SIGNAL SIGINT
#else
#include <my_pthread.h>			// For thr_setconcurency()
#define THR_KILL_SIGNAL SIGUSR2		// Can't use this with LinuxThreads
#endif

static int unix_sock= -1,ip_sock= -1;
static uint max_connections;
static ulong opt_specialflag=SPECIAL_ENGLISH;
static ulong query_id=1L,thread_id=1L;
static string opt_logname=0;
static char mysql_home[FN_REFLEN];
static pthread_key_t THR_THD,THR_NET;
static pthread_cond_t COND_thread_count=PTHREAD_COND_INITIALIZER;
static pthread_attr_t connection_attrib;
static pthread_t select_thread;
static bool opt_log,opt_noacl;

uint mysql_port;
uint table_cache_size;
uint test_flags,thread_count=0,select_errors=0;
uint keybuff_size,sortbuff_size,current_pid;
bool opt_endinfo;
bool volatile abort_loop;
ulong reload_version=1L;		/* Increments on each reload */
ulong specialflag;
char mysql_data_home[FN_REFLEN],language[LIBLEN],
     reg_ext[FN_EXTLEN],blob_newline,f_fyllchar;
char **errmesg;				/* Error messages */
byte last_ref[MAX_REFLENGTH];		/* Index ref of keys */
string mysql_unix_port;
DATE_FORMAT dayord;
double log_10[32];			/* 10 potences */
I_List<THD> threads;
time_t start_time;
pthread_key_t THR_MALLOC;
pthread_mutex_t LOCK_mysql_createDB=PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t LOCK_Acl=PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t LOCK_open=PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t LOCK_thread_count=PTHREAD_MUTEX_INITIALIZER;
pthread_t signal_thread;


static void *signal_hand(void *arg);
static void get_options(int argc,char **argv);
static int set_changeable_var(string str);
static void set_all_changeable_vars(void);
static void handle_connections(void);
static void *handle_one_connection(void *arg);
static sig_handler end_thread(int sig);
static string get_hostname(struct sockaddr_in *remote,char **ip);
static void application_end(void);
#ifdef HAVE_GETRLIMIT
static void set_maximum_open_files(uint max_file_limit);
#endif

static void close_connections(void)
{
  NET net;
  DBUG_ENTER("close_connections");

  abort_loop=1;				// mark abort for threads

  /* kill first connection thread */ 
  VOID(pthread_mutex_lock(&LOCK_thread_count));
  while (select_thread)
  {
    struct timespec abstime;
    (void) pthread_kill(select_thread,THR_CLIENT_ALARM);
#ifdef HAVE_TIMESPEC_TS_SEC
    abstime.ts_sec=time(NULL)+1;		// Bsd 2.1
    abstime.ts_nsec=0;
#else
    abstime.tv_sec=time(NULL)+1;		// Linux or Solairs
    abstime.tv_nsec=0;
#endif
    VOID(pthread_cond_timedwait(&COND_thread_count,&LOCK_thread_count,
				&abstime));
  }
  VOID(pthread_mutex_unlock(&LOCK_thread_count));

  /* Abort listening to new connections */
  if (ip_sock >= 0)
  {
    VOID(shutdown(ip_sock,2));
    VOID(close(ip_sock));
  }
#ifdef HAVE_SYS_UN_H
  if (unix_sock >= 0)
  {
    VOID(shutdown(unix_sock,2));
    VOID(close(unix_sock));
    VOID(unlink(mysql_unix_port));
  }
#endif
  end_thr_alarm();			 // Don't allow alarms

  VOID(net_init(&net,0));
  for (;;)
  {
    THD *tmp;
    VOID(pthread_mutex_lock(&LOCK_thread_count)); // For unlink from list
    if (!(tmp=threads.get()))
      break;
    if ((net.fd=tmp->net.fd) >= 0)
    {
      fprintf(stderr,ER(ER_FORCING_CLOSE),my_progname,
	      tmp->thread_id,tmp->user);
      close_connection(&net,0,0);
    }
    VOID(pthread_mutex_unlock(&LOCK_thread_count));
  }
  net_end(&net);

  /* abort all threads */
  while (thread_count)
    VOID(pthread_cond_wait(&COND_thread_count,&LOCK_thread_count));
  VOID(pthread_mutex_unlock(&LOCK_thread_count));

  table_cache_free();
  lex_free();				/* Free some memory */
  mysql_log.close();
  DBUG_VOID_RETURN;
}

	/* Force server down. kill all connections and threads and exit */

static sig_handler kill_server(int sig)
{
  DBUG_ENTER("kill_server");
  signal(sig,SIG_IGN);
  if (sig == SIGQUIT || sig == 0)
    fprintf(stderr,ER(ER_NORMAL_SHUTDOWN),my_progname);
  else
    fprintf(stderr,ER(ER_GOT_SIGNAL),my_progname,sig);

  close_connections();
  fprintf(stderr,ER(ER_SHUTDOWN_COMPLETE),my_progname);
  if (sig != SIGQUIT)
  {
    unireg_abort(1);
  }
  else
  {
    application_end();			/* Free application */
    unireg_end(0);
  }
  pthread_exit(0);
  DBUG_VOID_RETURN;
}


/****************************************************************************
** Init IP and UNIX socket
****************************************************************************/

static void set_ports()
{
  char	*env;
  if (!mysql_port)
  {						// Get port if not from commandline
    struct  servent *serv_ptr;
    mysql_port = MYSQL_PORT;
    if ((serv_ptr = getservbyname("mysql", "tcp")))
      mysql_port = ntohs((u_short) serv_ptr->s_port);
    if ((env = getenv("MYSQL_TCP_PORT")))
      mysql_port = (uint) atoi(env);
  }
  if (!mysql_unix_port)
  {
    mysql_unix_port = MYSQL_UNIX_ADDR;
    if ((env = getenv("MYSQL_UNIX_PORT")))
      mysql_unix_port = env;
  }
}


static void server_init(void)
{
  struct sockaddr_in	IPaddr;
#ifdef HAVE_SYS_UN_H
  struct sockaddr_un	UNIXaddr;
#endif
  int	arg=1;
  DBUG_ENTER("server_init");

  set_ports();
  DBUG_PRINT("general",("IP Socket is %d",mysql_port));
  ip_sock = socket(AF_INET, SOCK_STREAM, 0);
  if (ip_sock < 0)
  {
    perror(ER(ER_IPSOCK_ERROR));
    unireg_abort(1);
  }
  bzero(&IPaddr, sizeof(IPaddr));
  IPaddr.sin_family = AF_INET;
  IPaddr.sin_addr.s_addr = htonl(INADDR_ANY);
  IPaddr.sin_port = (unsigned short) htons((unsigned short) mysql_port);
  (void) setsockopt(ip_sock,SOL_SOCKET,SO_REUSEADDR,(char*)&arg,sizeof(arg));
  for(;;)
  {
    if (bind(ip_sock, (struct sockaddr *)&IPaddr, sizeof(IPaddr)) >= 0)
      break;
    perror("Can't start server : IP Bind ");	/* Had a loop here */
    unireg_abort(1);
  }
  VOID(listen(ip_sock,5));

#if defined(HAVE_SYS_UN_H) && !defined(HAVE_mit_thread)
  /*
  ** Create the UNIX socket
  */
  DBUG_PRINT("general",("UNIX Socket is %s",mysql_unix_port));

  if ((unix_sock = socket(AF_UNIX, SOCK_STREAM, 0)) < 0)
  {
    perror("Can't start server : UNIX Socket ");
    unireg_abort(1);
  }
  bzero(&UNIXaddr, sizeof(UNIXaddr));
  UNIXaddr.sun_family = AF_UNIX;
  strmov(UNIXaddr.sun_path, mysql_unix_port);
  VOID(unlink(mysql_unix_port));
  (void) setsockopt(unix_sock,SOL_SOCKET,SO_REUSEADDR,(char*)&arg,sizeof(arg));
  if (bind(unix_sock, (struct sockaddr *)&UNIXaddr, sizeof(UNIXaddr)) < 0)
  {
    perror("Can't start server : UNIX Bind ");
    unireg_abort(1);
  }
  VOID(listen(unix_sock,5));
#endif
  DBUG_PRINT("info",("server started"));
  DBUG_VOID_RETURN;
}


void yyerror(char *s)
{
  NET *net;
  char *yytext;
  net=(NET*) pthread_getspecific(THR_NET);
  yytext=(char*) current_lex->tok_start;
  net_printf(net,ER(ER_PARSE_ERROR), s, yytext ? (char*) yytext : "",
	     current_lex->yylineno);
}


void close_connection(NET *net,char *error,bool lock)
{
  int fd;
  if ((fd=net->fd) != -1)
  {
    if (lock)
      VOID(pthread_mutex_lock(&LOCK_thread_count));
    if (error)
      send_error(net,error);
    net->fd= -1;
    VOID(shutdown(fd,2));
    VOID(close(fd));
    net_end(net);
    if (lock)
      VOID(pthread_mutex_unlock(&LOCK_thread_count));
  }
}

	/* Called when a thread is aborted */
	/* ARGSUSED */

static sig_handler end_thread(int sig __attribute__((unused)))
{
  THD *thd=current_thd;
  DBUG_ENTER("end_thread");
  if (thd)
  {
    VOID(pthread_mutex_lock(&LOCK_thread_count));
    sql_free(&thd->alloc);
    thread_count--;
    delete thd;
    VOID(pthread_cond_signal(&COND_thread_count)); /* Tell main we are ready */
    VOID(pthread_mutex_unlock(&LOCK_thread_count));
    pthread_exit(NULL);
  }
  DBUG_VOID_RETURN;
}


/******************************************************************************
** Setup a signal thread with handles all signals
** Because linux doesn't support scemas use a mutex to check that
** the signal thread is ready before continuing
******************************************************************************/

static pthread_mutex_t LOCK_signal_init=PTHREAD_MUTEX_INITIALIZER;

static void init_signals(void)
{
  sigset_t set;
  pthread_attr_t thr_attr;

  VOID(sigset(THR_KILL_SIGNAL,end_thread));
  VOID(sigfillset(&set));
  VOID(sigdelset(&set,THR_CLIENT_ALARM));
  VOID(sigdelset(&set,THR_KILL_SIGNAL));		/* For interrupts */
  VOID(pthread_sigmask(SIG_SETMASK,&set,NULL));

  VOID(pthread_attr_init(&thr_attr));
  VOID(pthread_attr_setscope(&thr_attr,PTHREAD_SCOPE_SYSTEM));
  VOID(pthread_attr_setdetachstate(&thr_attr,PTHREAD_CREATE_DETACHED));
  pthread_attr_setstacksize(&thr_attr,65536L);

  VOID(pthread_mutex_lock(&LOCK_signal_init));
  VOID(pthread_create(&signal_thread,&thr_attr,signal_hand,0));
  VOID(pthread_attr_destroy(&thr_attr));
  VOID(pthread_mutex_lock(&LOCK_signal_init));	/* wait for signal handler */
  VOID(pthread_mutex_unlock(&LOCK_signal_init));/* don't use this anymore */
}


/* ARGSUSED */
static void *signal_hand(void *arg __attribute__((unused)))
{
  sigset_t set;
  int sig;

  VOID(signal(SIGPIPE,SIG_IGN));		/* We don't want this */
  init_thr_alarm();				/* Setup alarm handler */
#if SIGINT != THR_KILL_SIGNAL
  VOID(sigemptyset(&set));			/* Setup up SIGINT for debug */
  VOID(sigaddset(&set,SIGINT));			/* For debugging */
  VOID(pthread_sigmask(SIG_UNBLOCK,&set,NULL));
#endif
  VOID(sigfillset(&set));			/* Catch all signals */
#ifndef USE_ONE_SIGNAL_HAND
  VOID(sigdelset(&set,THR_SERVER_ALARM));	/* For alarms */
#endif
  VOID(sigdelset(&set,THR_KILL_SIGNAL));	/* For interrupts */
  VOID(sigdelset(&set,SIGINT));			/* For debugging */
  VOID(sigdelset(&set,SIGPIPE));		/* For warnings */

  VOID(pthread_mutex_unlock(&LOCK_signal_init));

  for (;;)
  {
    while (sigwait(&set,&sig) == EINTR) ;
    DBUG_PRINT("info",("Got signal %d",sig));
    switch (sig) {
    case SIGSEGV:
    case SIGBUS:
    case SIGQUIT:
    case SIGKILL:
    case SIGTERM:
      kill_server(sig);
      break;
    case SIGHUP:
      reload_acl_and_cache();			/* Read system files */
      break;
#ifdef USE_ONE_SIGNAL_HAND
    case THR_SERVER_ALARM:
      process_alarm(sig);
      break;
#endif
    default:
      printf("Warning: Got signal: %d\n",sig);
      break;
    }
  }
  return 0;
}


	/* ARGSUSED */
static int my_message_sql(const char *str, myf MyFlags __attribute__((unused)))
{
  NET *net;
  DBUG_ENTER("my_message_sql");
  DBUG_PRINT("error",("Message: '%s'",str));
  if ((net=(NET*) pthread_getspecific(THR_NET)))
  {
    if (!net->last_error[0])			// Return only first message
      VOID(strnmov(net->last_error,str,sizeof(net->last_error)));
  }
  else
    fprintf(stderr,"%s: %s\n",my_progname,str);
  DBUG_RETURN(0);
}

int main(int argc, char **argv)
{
  char	path[FN_LEN];
  FILE	*pidFile;
  struct sched_param sched_param;
  DBUG_ENTER("main");
  DBUG_PROCESS(argv[0]);
  DEBUGGER_OFF;

  start_time=time((time_t*) 0);
  bzero(&sched_param,sizeof(sched_param));
  sched_param.sched_priority=CONNECT_PRIOR;
  VOID(pthread_setschedparam(pthread_self(),SCHED_OTHER,&sched_param));

  /* Parameter for threads created for connections */
  VOID(pthread_attr_init(&connection_attrib));
  VOID(pthread_attr_setdetachstate(&connection_attrib,
				   PTHREAD_CREATE_DETACHED));
  pthread_attr_setstacksize(&connection_attrib,65536L);
  sched_param.sched_priority=WAIT_PRIOR;
  VOID(pthread_attr_setschedparam(&connection_attrib,&sched_param));

  MY_INIT(argv[0]);		/* init my_lib */
  get_options(argc,argv);
#if defined(HAVE_GETRLIMIT) && !defined(__linux__)
  /* connections and databases neads files */
  set_maximum_open_files(10+(int) (max_connections*4+table_cache_size*2));
#endif
  unireg_init(opt_specialflag); /* Set up extern variabels */
  lex_init();
  init_errmessage();		/* Read error messages from file */
  mysys_uses_curses=0;
#ifdef USE_REGEX
  regex_init();
#endif
  select_thread=pthread_self();

  /*
  ** We have enough space for fiddling with the argv, continue
  */
  VOID(umask(0));
  if (my_setwd(mysql_data_home,MYF(MY_WME)))
  {
    unireg_abort(1);
  }
  strmov(mysql_data_home,".");		// all paths are relative from here
  server_init();
  table_cache_init();
  {
    struct utsname uts_name;
    if (uname(&uts_name) < 0)
      strmov(uts_name.nodename,"mysql");
    (void)sprintf(path,"%s/%s.pid",mysql_data_home,uts_name.nodename);
    if (opt_log)
      mysql_log.open(opt_logname ? opt_logname : uts_name.nodename);
  }

  /*
    init signals & alarm
    After this we can't quit by a simple unireg_abort
    */
  init_signals();
  if (!opt_noacl && acl_init())
  {
    select_thread=0;
    (void) pthread_kill(signal_thread,SIGQUIT);
    pthread_exit(0);
    exit(0);
  }
  if ((pidFile = my_fopen(path,O_WRONLY,MYF(MY_WME))))
  {
    fprintf(pidFile,"%d",current_pid);
    VOID(my_fclose(pidFile,MYF(0)));
    VOID(chmod(path,0644));
  }
  VOID(umask(0));

  if (pthread_key_create(&THR_THD,NULL) || pthread_key_create(&THR_NET,NULL) ||
      pthread_key_create(&THR_MALLOC,NULL))
  {
    my_message("Can't create thread-keys",MYF(0));
    (void) my_delete(path,MYF(0));		// Not neaded anymore
    select_thread=0;
    (void) pthread_kill(signal_thread,SIGQUIT);
    pthread_exit(0);
  }

  error_handler_hook = my_message_sql;
#ifdef HAVE_THR_SETCONCURRENCY
  VOID(thr_setconcurrency(20));			// We are iobound
#endif
  printf(ER(ER_READY),my_progname);
  handle_connections();

  VOID(pthread_attr_destroy(&connection_attrib));
  (void) my_delete(path,MYF(0));		// Not neaded anymore

  VOID(pthread_mutex_lock(&LOCK_thread_count));
  select_thread=0;				// For close_connections
  VOID(pthread_cond_signal(&COND_thread_count));
  VOID(pthread_mutex_unlock(&LOCK_thread_count));

  pthread_exit(0);
  DBUG_RETURN(0);
}

	/* Handle new connections and spawn new process to handle them */

static void handle_connections(void)
{
  int sock,new_sock;
  uint opt,pkt_len;
  fd_set readFDs,clientFDs;
  pthread_t tid;
  THD *thd;
  struct sockaddr_in cAddr;
  uint max_used_connection= (uint) (max(ip_sock,unix_sock)+1);
  struct rand_struct rand;
  char scramble[9],*passwd;
  DBUG_ENTER("handle_connections");

  randominit(&rand,(ulong) start_time);
  FD_ZERO(&clientFDs);
  FD_SET(ip_sock,&clientFDs);
#ifdef HAVE_SYS_UN_H
  FD_SET(unix_sock,&clientFDs);
#endif
  pkt_len=0;

  DBUG_PRINT("general",("Waiting for connections."));
  while (!abort_loop)
  {
    memcpy(&readFDs,&clientFDs,sizeof(readFDs));
#ifdef HPUX
    if (select(max_used_connection,(int*) &readFDs,0,0,0) < 0)
      continue;
#else
    if (select((int) max_used_connection,&readFDs,0,0,0) < 0)
    {
      if (errno != EINTR)
      {
	if (!select_errors++ && !abort_loop)
	  fprintf(stderr,"mysql: Got error %d from select\n",errno);
      }
      continue;
    }
#endif
    /*
    ** Is this a new connection request
    */

#ifdef HAVE_SYS_UN_H
    if (FD_ISSET(unix_sock,&readFDs))
      sock = unix_sock;
    else
#endif
      sock = ip_sock;

    for (;;)
    {
      size_socket length=sizeof(struct sockaddr_in);
      new_sock = accept(sock, (struct sockaddr *)&cAddr, &length);
      if (new_sock != -1 || errno != EINTR)
	break;
    }
    if (new_sock < 0)
    {
      perror("Error in accept ");
      continue;
    }
    {
      size_socket dummyLen;
      struct sockaddr dummy;
      dummyLen = sizeof(struct sockaddr);
      if (getsockname(new_sock,&dummy, &dummyLen) < 0)
      {
	perror("Error on new connection socket");
	VOID(shutdown(new_sock,2));
	VOID(close(new_sock));
	continue;
      }
    }

    /*
    ** Don't allow too many connections
    */

    if (!(thd= new THD))
    {
      VOID(shutdown(new_sock,2)); VOID(close(new_sock));
      continue;
    }
    if (net_init(&thd->net,new_sock))
    {
      close_connection(&thd->net,ER(ER_OUT_OF_RESOURCES));
      delete thd;
      continue;
    }
    NET *net=&thd->net;				// For easy ref
    if (thread_count > max_connections || abort_loop)
    {
      close_connection(net,ER(ER_CON_COUNT_ERROR));
      delete thd;
      continue;
    }

    /*
    ** store the connection details
    */

    DBUG_PRINT("general",("New connection received on %d", new_sock));
    if (sock == ip_sock)
    {
      size_socket addrLen;

      addrLen = sizeof(struct sockaddr);
      if (getpeername(new_sock, (struct sockaddr *) &thd->remote, &addrLen))
      {
	close_connection(net,ER(ER_BAD_HOST_ERROR));
	delete thd;
	continue;
      }
#ifdef TO_BE_REMOVED
      addrLen = sizeof(struct sockaddr);
      if (getsockname(new_sock, (struct sockaddr *) &thd->local, &addrLen))
      {
	close_connection(net,ER(ER_BAD_HOST_ERROR));
	delete thd;
	continue;
      }
#endif
      if (!(thd->host=get_hostname(&thd->remote,&thd->ip)))
      {
	close_connection(net, ER(ER_BAD_HOST_ERROR));
	delete thd;
	continue;
      }
    }
    else
    {
      DBUG_PRINT("general",("Host = UNIX domain"));
      thd->host = my_strdup(LOCAL_HOST,MYF(0));
      thd->ip=0;
      bzero(&thd->local, sizeof(struct sockaddr));
      bzero(&thd->remote,sizeof(struct sockaddr));
    }

    opt=1;
    VOID(setsockopt(new_sock,SOL_SOCKET,SO_KEEPALIVE,
		    (char *) &opt, sizeof(opt)));
    {
      char buff[32],*end;
      end=strmov(buff,SERVER_VERSION)+1;
      int4store((uchar*) end,thread_id);
      end+=4;
      for (uint i=0; i < 8 ; i++)
	scramble[i]= *end++ = (char) (rnd(&rand)*94+33);
      scramble[8]=0;
      if (net_write_command(net,PROTOCOL_VERSION,buff,(uint) (end-buff)) ||
	  (pkt_len=net_read(net)) == packet_error || pkt_len < 6)
      {
	close_connection(net,ER(ER_HANDSHAKE_ERROR));
	delete thd;
	continue;
      }
    }
    thd->client_capabilities=uint2korr(net->buff);
    thd->max_packet_length=uint3korr(net->buff+2);
    thd->user = my_strdup((char*) net->buff+5, MYF(MY_FAE));
    passwd= strend((char*) net->buff+5)+1;
    thd->thread_id=thread_id++;
    thd->master_access=acl_getroot(thd->host, thd->ip, thd->user, passwd,
				   scramble);
    thd->rand=rand;
    DBUG_PRINT("general",
	       ("Capabilities: %d  packet_length: %d  User = %s  Access: %d",
		thd->client_capabilities, thd->max_packet_length, thd->user,
		thd->master_access));
    if (thd->master_access & NO_ACCESS)
    {
      close_connection(net,ER(ER_ACCESS_DENIED_ERROR));
      delete thd;
      continue;
    }
    send_ok(net);


    /* Start a new thread to handle connection */
    VOID(pthread_mutex_lock(&LOCK_thread_count));
    threads.append(thd);

    if ((errno=pthread_create(&tid,&connection_attrib,handle_one_connection,
			      (void*) thd)))
    {
      DBUG_PRINT("error",
		 ("Can't create thread to handle request (error %d)",
		  errno));
      delete thd;
      VOID(pthread_mutex_unlock(&LOCK_thread_count));
      close_connection(net,ER(ER_OUT_OF_RESOURCES));
      continue;
    }
    thread_count++;
    DBUG_PRINT("info",(("Thread created")));
    VOID(pthread_mutex_unlock(&LOCK_thread_count));
  }
  DBUG_VOID_RETURN;
}


static void *handle_one_connection(void *arg)
{
  THD *thd=(THD*) arg;
  NET *net;
  sigset_t set;
  DBUG_ENTER("handle_one_connection");

  my_thread_init();
  VOID(pthread_setspecific(THR_THD,(void*) thd));
  VOID(pthread_setspecific(THR_MALLOC,(void*) &thd->alloc));
  VOID(pthread_setspecific(THR_NET,(void*) &thd->net));
  VOID(sigemptyset(&set));			/* Get mask in use */
  VOID(pthread_sigmask(SIG_UNBLOCK,&set,&thd->block_signals));
  VOID(sigaddset(&thd->block_signals,THR_KILL_SIGNAL));

  net= &thd->net;
  thd->db_access=NO_ACCESS;
  thd->db=0;

  mysql_log.write(CONNECT,"%s@%s",thd->user,thd->host);
  while (!net->error && net->fd >= 0 && !abort_loop)
  {
    VOID(pthread_mutex_lock(&LOCK_thread_count));
    thd->query_id=query_id++;
    VOID(pthread_mutex_unlock(&LOCK_thread_count));
    if (do_command(thd))
      break;
  }
  VOID(pthread_mutex_lock(&LOCK_thread_count));
  if (net->error && net->fd >= 0)
  {
    fprintf(stderr,"Aborted connection %d to db: '%s' user: '%s'\n",
	    net->fd,(thd->db ? thd->db : "unconnected"),
	    thd->user);
  }
  VOID(pthread_mutex_unlock(&LOCK_thread_count));
  close_connection(net);
  end_thread(0);
  DBUG_RETURN(0);
}


THD *_current_thd(void)
{
  return (THD*) pthread_getspecific(THR_THD);
}

/******************************************************************************
** handle start options
******************************************************************************/

enum options {OPT_ISAM_LOG=256,OPT_SKIPP_NEW,OPT_SKIPP_GRANT,OPT_SKIPP_LOCK,
	      OPT_SKIPP_UNSAFE,OPT_USE_LOCKING,OPT_SOCKET};

static struct option long_options[] =
{
  {"basedir",	required_argument, 0, 'b'},
  {"datadir",	required_argument, 0, 'h'},
  {"debug",	optional_argument, 0, '#'},
  {"debug-info",no_argument,	   0, 'T'},
  {"help",	no_argument,	   0, '?'},
  {"log",	optional_argument, 0, 'l'},
  {"language",	required_argument, 0, 'L'},
  {"log-isam",	optional_argument, 0, (int) OPT_ISAM_LOG},
  {"port",	required_argument, 0, 'P'},
  {"socket",	required_argument, 0, (int) OPT_SOCKET},
  {"set-variable",required_argument, 0, 'O'},
  {"skip-new-routines", no_argument,0, (int) OPT_SKIPP_NEW},
  {"skip-grant-tables", no_argument,0, (int) OPT_SKIPP_GRANT},
  {"skip-locking",	no_argument,0, (int) OPT_SKIPP_LOCK},
  {"skip-unsafe-select",no_argument,0, (int) OPT_SKIPP_UNSAFE},
  {"use-locking",	no_argument,0, (int) OPT_USE_LOCKING},
  {"version",	no_argument,	   0, 'V'},
  {0, 0, 0, 0}
};

static struct st_changeable_vars {
  string name;
  uint *varptr;
  uint def_value,min_value,max_value,sub_size,block_size;
} changeable_vars[] = {
  { "keybuffer",&keybuff_size,KEY_CACHE_SIZE,MALLOC_OVERHEAD,(uint) ~0,
      MALLOC_OVERHEAD, IO_SIZE },
  { "max_allowed_packet",(uint*) &max_allowed_packet,65536,16384,(uint) ~0,
    MALLOC_OVERHEAD,1024},
  { "net_buffer_length",(uint*) &net_buffer_length,16384,4096,1024*1024,
    MALLOC_OVERHEAD,1024},
  { "max_connections", &max_connections,90,1,16384,0,1}, 
  { "table_cache", &table_cache_size,64,1,16384,0,1},
  { "recordbuffer",&my_default_record_cache_size,RECORD_CACHE_SIZE,
		 IO_SIZE*2+MALLOC_OVERHEAD,(uint) ~0,MALLOC_OVERHEAD,IO_SIZE },
  { "sortbuffer",&sortbuff_size,MAX_SORT_MEMORY,
	       MIN_SORT_MEMORY+MALLOC_OVERHEAD,(uint) ~0,MALLOC_OVERHEAD,1 },
  { NullS,(uint*) 0,0,0,0,0,0,} };


static void print_version(void)
{
  printf("%s  Ver %s for %s on %s\n",my_progname,
	 SERVER_VERSION,SYSTEM_TYPE,MACHINE_TYPE);
}

static void usage(void)
{
  print_version();
  puts("Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB.");
  puts("All rights reserved. Se the file PUBLIC for licence information.");
  puts("This software comes with ABSOLUTELY NO WARRANTY: see the file PUBLIC for details.\n");
  puts("Starts the mysql server\n");

  printf("Usage: %s [OPTIONS]\n", my_progname);
  puts("\n\
  -b, --basedir=path	path to installation directory\n\
  -h, --datadir=path	path to the database root\n\
  -#, --debug=...       output debug log. Often this is 'd:t:o,filename`\n\
  -T, --debug-info	print some debug info at exit\n\
  -?, --help		display this help and exit\n\
  -L, --language=...	client error messages in given language\n\
  -l, --log[=filename]	log connections and queries to file\n\
      --log-isam[=filename]\n\
			log all isam changes to file\n\
  -P, --port=...	Port number to use for connection\n\
  -O, --set-variable var=option
			give a variable an value. --help lists variables\n\
  --skip-new-routines	don't use new possible wrong routines.
  --skip-grant-tables	start without grant tables. This gives anyone FULL\n\
			ACCESS to all tables!\n\
  --skip-locking	don't use system locking. To use isamchk one has\n\
			to shut down the server.\n\
  --skip-unsafe-select  skipp unsafe select optimizations.\n\
  --socket=...		Socket file to use for connection\n\
  -V, --version		output version information and exit\n");

  printf("Current base_dir: %s\n",mysql_home);
  printf("Current data_dir: %s\n",mysql_data_home);
  printf("Current language: %s\n",language);
  if (opt_logname)
    printf("Current logfile:  %s\n",opt_logname);
  set_ports();
  printf("TCP port:         %d\n",mysql_port);
#if defined(HAVE_SYS_UN_H) && !defined(HAVE_mit_thread)
  printf("Unix socket:      %d\n",mysql_unix_port);
#endif
  if (my_disable_locking)
    puts("\nsystem locking is not in use");
  if (opt_noacl)
    puts("\nGrant tables are not used. All users have full access rights");
  printf("\nPossibly variables to option --set-variable (-O) are:\n");
  for (uint i=0 ; changeable_vars[i].name ; i++)
    printf("%-20s  current value: %u\n",
	   changeable_vars[i].name,
	   *changeable_vars[i].varptr);
};


	/* Initiates DEBUG - but no debugging here ! */

static void get_options(int argc,char **argv)
{
  int c,option_index=0;
  char *pos,l_buff[LIBLEN];
  set_all_changeable_vars();

  VOID(strmov(language,LANGUAGE));	/* Default-language */
  VOID(strmov(mysql_data_home,DATADIR));
  if (!(pos = getenv("MY_BASEDIR_VERSION")))
    pos=DEFAULT_MYSQL_HOME;
  VOID(strmov(mysql_home,pos));
#ifdef HAVE_mit_thread
  my_disable_locking=1;
#endif

  while ((c=getopt_long(argc,argv,"b:h:#::T?l::L:O:P:S::VI?",
			long_options, &option_index)) != EOF)
  {
    switch(c) {
    case '#':
      DBUG_PUSH(optarg ? optarg : "d:t:o");
      opt_endinfo=1;				/* unireg: memory allocation */
      break;
    case 'b':
      strmov(mysql_home,optarg);
      break;
    case 'l':
      opt_log=1;
      opt_logname=optarg;			// Use hostname.log if null
      break;
    case 'h':
      strmov(mysql_data_home,optarg);
      break;
    case 'L':
      strmov(language,optarg);
      break;
    case 'O':
      if (set_changeable_var(optarg))
      {
	usage();
	exit(1);
      }
      break;
    case 'P':
      mysql_port= (unsigned int) atoi(optarg);
      break;
    case OPT_SOCKET:
      mysql_unix_port= optarg;
      break;
    case 'V':
      print_version();
      exit(0);
    case 'I':
    case '?':
      usage();
      exit(0);
    case 'T':
      test_flags= optarg ? (uint) atoi(optarg) : (uint) ~0;
      break;
    case 'S':
      if (!optarg)
	opt_specialflag|= SPECIAL_NO_NEW_FUNC;
      else if (!strcmp(optarg,"l"))
	my_disable_locking=1;
      else if (!strcmp(optarg,"u"))
	opt_specialflag|=SPECIAL_NO_UNSAFE_OPT;
      else if (!strcmp(optarg,"g"))
	opt_noacl=1;
      else
      {
	usage();
	exit(1);
      }
      break;
    case (int) OPT_ISAM_LOG:
      if (optarg)
	nisam_log_filename=optarg;
      (void) ni_log(1);
      break;
    case (int) OPT_SKIPP_NEW:
      opt_specialflag|= SPECIAL_NO_NEW_FUNC;    
      break;
    case (int) OPT_SKIPP_GRANT:
      opt_noacl=1;
      break;
    case (int) OPT_SKIPP_LOCK:
      my_disable_locking=1;
      break;
    case (int) OPT_SKIPP_UNSAFE:
      opt_specialflag|=SPECIAL_NO_UNSAFE_OPT;
      break;
    case (int) OPT_USE_LOCKING:
      my_disable_locking=0;
      break;
    }
  }
  if (argc != optind)
  {
    fprintf(stderr,"%s: Too many parameters\n",my_progname);
    usage();
    exit(1);
  }
  (void) fn_format(mysql_home,mysql_home,"","",16); // Remove symlinks
  convert_dirname(mysql_home);
  convert_dirname(mysql_data_home);
  convert_dirname(language);
  (void) my_load_path(mysql_home,mysql_home,""); // Resolve current dir
  (void) my_load_path(mysql_data_home,mysql_data_home,mysql_home);
  char buff[FN_REFLEN];
  if (test_if_hard_path(SHAREDIR))
    strmov(buff,SHAREDIR);
  else
    strxmov(buff,mysql_home,SHAREDIR,NullS);
  convert_dirname(buff);
  strmov(l_buff,language);
  (void) my_load_path(language,l_buff,buff);
}

	/* Give a changeable variable value according to user-given-string */

static int set_changeable_var(string str)
{
  reg1 int i;
  long num;
  char endchar;
  string names[sizeof(changeable_vars)/sizeof(changeable_vars[0])],pos;
  TYPELIB typelib;

  if (str)
  {
    if (!(pos=strchr(str,'=')))
      fprintf(stderr,"Can't find '=' in expression '%s' to option -O\n",str);
    else
    {
      pos[0]=0;
      for (i=0 ; (names[i]=changeable_vars[i].name) ; i++) ;
      typelib.count= (uint) i;
      typelib.type_names=names;
      if ((i=find_type(str,&typelib,2)-1) >=0)
      {
	num=(long) atol(pos+1); endchar=strend(pos+1)[-1];
	if (endchar == 'k' || endchar == 'K')
	  num*=1024;
	else if (endchar == 'm' || endchar == 'M')
	  num*=1024L*1024L;
	if (num < (long) changeable_vars[i].min_value)
	  num=(long) changeable_vars[i].min_value;
	else if ((unsigned long) num >
		 (unsigned long) changeable_vars[i].max_value)
	  num=(long) changeable_vars[i].max_value;
	*changeable_vars[i].varptr=((uint) num-changeable_vars[i].sub_size)/
	  changeable_vars[i].block_size;
	(*changeable_vars[i].varptr)*=changeable_vars[i].block_size;
	return(0);
      }
      pos[0]='=';
      fprintf(stderr,"Can't set varable to option -O in expression: '%s'\n",
	      str);
    }
  }
  return(1);
}


	/* set all changeable variables */

static void set_all_changeable_vars(void)
{
  reg1 int i;
  for (i=0 ; changeable_vars[i].name ; i++)
    *changeable_vars[i].varptr=changeable_vars[i].def_value;
  return;
} /* set_all_changeable_vars */


	/* Called by unireg_end before exit */

static void application_end(void)
{
  acl_free();
}

	/* Called by databases to get interrupt safe code */

void dont_break()
{
}

void allow_break()
{
}


	/* ARGSUSED */
void my_remember_signal(int sig __attribute__((unused)),
			sig_handler (*func)(int)  __attribute__((unused)))
{}


#ifdef HAVE_GETRLIMIT
static void set_maximum_open_files(uint max_file_limit)
{
  struct rlimit rlimit;
  ulong old_cur;

  if (!getrlimit(RLIMIT_NOFILE,&rlimit))
  {
    old_cur=rlimit.rlim_cur;
    rlimit.rlim_cur=rlimit.rlim_max=max_file_limit;
    if (setrlimit(RLIMIT_NOFILE,&rlimit))
    {
      fprintf(stderr,"Warning: Error from setrlimit: Max open files is %ld\n",
	      old_cur);
    }
    else
    {
      VOID(getrlimit(RLIMIT_NOFILE,&rlimit));
      if ((uint) rlimit.rlim_cur != max_file_limit)
	fprintf(stderr,
		"Warning: setrlimit gave ok, but didn't change limits. Max open files is %ld\n",(ulong) rlimit.rlim_cur);
    }
  }
}
#endif


static string get_hostname(struct sockaddr_in *remote,char **ip)
{
  int tmp_errno;
  struct hostent tmp_hostent,*hp;
  char *name;

#ifdef HAVE_GETHOSTBYADDR_R
  char buff[2048];
#ifdef HAVE_purify
  bzero(buff,sizeof(buff));		// Bug in purify
#endif
  *ip=0;
  if (!(hp=gethostbyaddr_r((char*) &remote->sin_addr,sizeof(remote->sin_addr),
			   AF_INET,
			   &tmp_hostent,buff,sizeof(buff),&tmp_errno)))
  {
    DBUG_PRINT("error",("gethostbyaddr_r returned %d",tmp_errno));
    return 0;
  }
#else
  VOID(pthread_mutex_lock(&LOCK_thread_count));
  if (!(hp=gethostbyaddr((char*) &remote->sin_addr,sizeof(remote->sin_addr),AF_INET)))
  {
    DBUG_PRINT("error",("gethostbyaddr returned %d",errno));
    VOID(pthread_mutex_unlock(&LOCK_thread_count));
    return 0;
  }
#endif
  struct in_addr in;
  (void) memcpy(&in.s_addr, *hp->h_addr_list, sizeof (in.s_addr));
  *ip=my_strdup(inet_ntoa(in),MYF(0));
  name=my_strdup(hp->h_name,MYF(0));
  DBUG_PRINT("general",("Host: %s  ip: %s", name, *ip));
#ifndef HAVE_GETHOSTBYADDR_R
  VOID(pthread_mutex_unlock(&LOCK_thread_count));
#endif
  return name;
}


/*****************************************************************************
** Instansiate templates
*****************************************************************************/

#ifdef __GNUC__
/* Used templates */
template class I_List<THD>;
#endif
