/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB
   
   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.
   
   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.  The License grants you the right to 
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

/* locking functions for mysql */

#include "mysql_priv.h"
#include <hash.h>
#ifndef MASTER
#include "../srclib/merge/mrgdef.h"		/* Includes isam & thr_lock */
#else
#include "../merge/mrgdef.h"
#endif

extern HASH open_cache;

static MYSQL_LOCK *get_lock_data(TABLE **table,uint count);
static int lock_databases(TABLE **table,uint count);
static int unlock_databases(TABLE **table,uint count);


MYSQL_LOCK *mysql_lock_tables(THD *thd,TABLE **tables,uint count)
{
  MYSQL_LOCK *sql_lock;
  thd->locked=1;
  for (;;)
  {
    if ((sql_lock = get_lock_data(tables,count)))
    {
      thr_multi_lock(sql_lock->locks,sql_lock->lock_count);
      if (lock_databases(tables,count))
      {
	thr_multi_unlock(sql_lock->locks,sql_lock->lock_count);
	my_free((gptr) sql_lock,MYF(0));
	sql_lock=0;
	break;
      }
    }
    if (!thd->some_tables_deleted)
      break;
    /* some table was altered or deleted. reopen tables marked deleted */

    thd->some_tables_deleted=0;
    mysql_unlock_tables(sql_lock);
    sql_lock=0;
    for (uint i=0 ; i < count ; i++)
    {
      if (!tables[i]->version)
      {						// Table must be reopened
	if (reopen_table(thd,tables[i]))
	  return 0;
      }
    }
  }
  thd->locked=0;
  if (abort_loop)
  {
    my_error(ER_SERVER_SHUTDOWN,MYF(0));
    mysql_unlock_tables(sql_lock);
    sql_lock=0;
  }
  return sql_lock;
}


static int lock_databases(TABLE **tables,uint count)
{
  reg1 uint i;
  int lock_type,error;
  DBUG_ENTER("lock_databases");

  for (i=1 ; i <= count ; i++, tables++)
  {
    lock_type=F_WRLCK;				/* Lock exclusive */
    if ((*tables)->form.db_stat & HA_READ_ONLY ||
	! ((*tables)->form.reginfo.update & ~REG_PROG))
      lock_type=F_RDLCK;
    if ((error=ha_lock(&(*tables)->form,lock_type)))
    {
      for ( ; i-- ; tables--)
	VOID(ha_lock(&(*tables)->form,F_UNLCK));
      my_error(ER_CANT_LOCK,MYF(ME_BELL+ME_OLDWIN+ME_WAITTANG),error);
      DBUG_RETURN(error);
    }
    else
      (*tables)->form.db_stat &= ~ HA_BLOCK_LOCK;
  }
  DBUG_RETURN(0);
}


void mysql_unlock_tables(MYSQL_LOCK *sql_lock,bool no_free)
{
  VOID(unlock_databases(sql_lock->table,sql_lock->table_count));
  thr_multi_unlock(sql_lock->locks,sql_lock->lock_count);
  if (!no_free)
    my_free((gptr) sql_lock,MYF(0));
}

/*
  Unlock some of the tables locked my mysql_lock_tables
  This will work even if get_lock_data fails (next unlock will free all)
  */

void mysql_unlock_some_tables(TABLE **table,uint count)
{
  MYSQL_LOCK *sql_lock;
  if ((sql_lock = get_lock_data(table,count)))
    mysql_unlock_tables(sql_lock);
  VOID(unlock_databases(table,count));
}

	/* unlock a set of databases */

static int unlock_databases(TABLE **table,uint count)
{
  int error,error_code;
  DBUG_ENTER("unlock_databases");

  error_code=0;
  for (; count-- ; table++)
  {
    if ((error=ha_lock(&(*table)->form,F_UNLCK)))
      error_code=error;
  }
  if (error_code)
    my_error(ER_CANT_LOCK,MYF(ME_BELL+ME_OLDWIN+ME_WAITTANG),error_code);
  DBUG_RETURN(error_code);
}


/*
** Get lock structures from table structs and initialize locks
*/


static MYSQL_LOCK *get_lock_data(TABLE **table,uint count)
{
  uint i,tables;
  int lock_type;
  MYSQL_LOCK *sql_lock;
  THR_LOCK_DATA **locks;

  for (i=tables=0 ; i < count ; i++)
  {
    if (table[i]->form.db_type == DB_TYPE_MRG_ISAM)
    {
      MRG_INFO *db= (MRG_INFO*) table[i]->form.file;
      tables+=db->tables;
    }
    else if (table[i]->form.db_type == DB_TYPE_ISAM)
      tables++;
  }

  if (!(sql_lock= (MYSQL_LOCK*)
	my_malloc(sizeof(*sql_lock)+
		  (sizeof(THR_LOCK_DATA*)+sizeof(table))*tables,MYF(0))))
    return 0;
  locks=sql_lock->locks=(THR_LOCK_DATA**) (sql_lock+1);
  sql_lock->table=(TABLE**) (locks+tables);
  sql_lock->table_count=count;
  sql_lock->lock_count=tables;
  memcpy(sql_lock->table,table,sizeof(table[0])*count);

  for (i=0 ; i < count ; i++)
  {
    lock_type= (table[i]->form.reginfo.update) ? F_WRLCK : F_RDLCK;

    if (table[i]->form.db_type == DB_TYPE_MRG_ISAM)
    {
      MRG_INFO *db= (MRG_INFO*) table[i]->form.file;
      MRG_TABLE *file;

      for (file=db->open_tables ; file != db->end_table ; file++)
      {
	*(locks++)= &file->table->lock;
	file->table->lock.type=lock_type;
      }
    }
    else if (table[i]->form.db_type == DB_TYPE_ISAM)
    {
      N_INFO *db=(N_INFO*) table[i]->form.file;
      *(locks++)= &db->lock;
      db->lock.type=lock_type;
    }
  }
  return sql_lock;
}
