/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB
   
   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.
   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.  The License grants you the right to 
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

/*  Hantering av f{lt f|r registerhanteing & formhantering */

#include "mysql_priv.h"
#include <m_ctype.h>
#ifdef HAVE_FCONVERT
#include <floatingpoint.h>
#endif

struct decstr {
  uint nr_length,nr_dec,sign,extra;
};

	/* Functions defined in this file */

static void NEAR_F number_dec(struct decstr *sdec,const char *str);
static uint NEAR_F unpack_number(char *to,uchar *from,uint inputflag,
				 uint length);
#define INC_ERROR_COUNTER current_thd->cuted_fields++

	/* Calc length of a packed field */

uint f_packlength(uint length, uint flag)
{
  if (f_is_blob(flag))
    return f_packlength(length,f_is_packed(flag))+sizeof(char*);
  switch ((enum enum_field_types) f_packtype(flag)) {
  case FIELD_TYPE_DECIMAL: return (length);
  case FIELD_TYPE_CHAR	: return 1;
  case FIELD_TYPE_SHORT : return 2;
  case FIELD_TYPE_INT24 : return 3;
  case FIELD_TYPE_TIME:	  return 3;
  case FIELD_TYPE_TIMESTAMP:
  case FIELD_TYPE_DATE:
  case FIELD_TYPE_LONG	: return sizeof(int32);
  case FIELD_TYPE_FLOAT : return sizeof(float);
  case FIELD_TYPE_DOUBLE: return sizeof(double);
  case FIELD_TYPE_LONGLONG: return 8;	/* Don't crash if no longlong */
  case FIELD_TYPE_NULL	: return 0;
  default: return 0;				// Impossible
  }
} /* f_packlength */


	/* Store default for field in record */

void f_fieldinit(char *pos, uint inputflag, uint length)
{
  char *init;

  init= "0";
  if (f_is_alpha(inputflag))
  {
    if (!f_is_packed(inputflag))
    {
      bfill(pos,length,f_fyllchar);
      return;
    }
    else if (f_is_intervall(inputflag))		/* Intervall */
    {
      init="1";					/* 1 as default */
    }
    else if (f_is_blob(inputflag))
    {
      bzero(pos,f_packlength(length,inputflag));
      return;
    }
    inputflag|=FIELDFLAG_NUMBER;		/* Pack as num */
  }
  f_packfield(pos,init,inputflag,length,(TYPELIB*) 0);
} /* f_fieldinit */


	/* Give field a value from a string */
	/* Blobs are ignored */

void f_packfield(byte *to, const byte *from, uint inputflag, uint length,
		 TYPELIB *intervall)
{
  if (f_is_alpha(inputflag))
  {
    if (!f_is_packed(inputflag))
      copystrn(to,from,length);			/* Copy and fill with space */
    else
    {						/* Intervall or bit-field */
      ulonglong value;
      if (intervall)				/* If not NULL-packed */
      {
	if (f_is_intervall(inputflag))
	{
	  int tmp=find_type((char*) from,intervall,1+2+4);
	  if (tmp <= 0)
	  {
	    value=0L;
	    INC_ERROR_COUNTER;
	  }
	  else
	    value= (ulonglong) (uint) tmp;
	}
	else
	  value= find_bit_type((char*) from,intervall);
	switch ((enum enum_field_types) f_packtype(inputflag)) {
	case FIELD_TYPE_CHAR:
	  to[0]=(char) (uchar) value;
	  break;
	case FIELD_TYPE_SHORT:
	{
	  short tmp= (short) value;
	  shortstore(to,tmp);
	  break;
	}
	case FIELD_TYPE_INT24:
	  int3store(to,(long) value);
	  break;
	case FIELD_TYPE_LONG:
	  longstore(to,(long) value);
	  break;
	case FIELD_TYPE_LONGLONG:
	  longlongstore(to,value);
	  /* fall through */
	default:				/* No warnings from GCC */
	  break;
	}
      }
      long date;
      switch ((enum enum_field_types) f_packtype(inputflag)) {
      case FIELD_TYPE_DATE:
	date=(long) str_to_date(from,strlen(from));
	longstore(to,date);
	break;
      case FIELD_TYPE_TIME:
	date=(long) str_to_time(from,strlen(from));
	int3store(to,date);
	break;
      default:					// No warnings from GCC */
	break;
      }
    }
  }
  else
  {
    while (isspace(*from))
      from++;
    if (current_thd->count_cuted_fields)
    {
      if (!isdigit(*from) && *from != '-' && *from != '+')
	INC_ERROR_COUNTER;
    }
    switch ((enum enum_field_types) f_packtype(inputflag)) {
    case FIELD_TYPE_DECIMAL:
      {
	reg3 int i;
	uint dec;
	char fyllchar;
	struct decstr index;

	if ((dec= f_decimals(inputflag)))
	  dec++;				/* Calculate pos of '.' */
	if (f_is_zerofill(inputflag))
	{
	  fyllchar = '0';
	  while (*from == '0')			/* Skippa f|rtecken */
	    from++;
	}
	else fyllchar=' ';
	number_dec(&index,from);

	for (i=(int) (index.nr_length+index.extra+index.sign - (length-dec)) ;
	     i > 0 ;
	     i--)
	{
	  if (*from++ == '0')			/* Skippa m|jliga f|rnollor */
	  {
	    index.nr_length--;
	    continue;
	  }
	  bfill(to,length,'?');			/* Wrong info */
	  INC_ERROR_COUNTER;
	  return;
	}
	for (i=(int) (length-dec-index.nr_length-index.extra-index.sign) ;
	     i-- > 0 ;)
	  *to++ = fyllchar;
	if (index.sign)
	  *to++ = *from++;
	if (index.extra)
	  *to++ = '0';
	for (i=(int) index.nr_length ; i-- > 0 ; )
	  if (*from != ',')			/* skipp thousand marker */
	    *to++ = *from++;
	  else
	    from++;
	if (dec--)
	{
	  *to++ ='.';
	  if (index.nr_dec) from++;		/* Skippa '.' */
	  for (i=(int) min(index.nr_dec,dec) ; i-- > 0 ; ) *to++ = *from++;
	  for (i=(int) (dec-min(index.nr_dec,dec)) ; i-- > 0 ; ) *to++ = '0';
	}
	if (current_thd->count_cuted_fields)
	{
	  for ( ; *from ; from++)
	    if (*from != ' ')
	    {
	      INC_ERROR_COUNTER;
	      break;
	    }
	}
      }
      break;
    case FIELD_TYPE_CHAR:
    {
      long tmp= strtol(from,NULL,10),min_val,max_val;
      if (f_is_dec(inputflag))
      {
	min_val=-128L; max_val=127L;
      }
      else
      {
	min_val=0L; max_val=255L;	
      }
      if (tmp < min_val)
      {
	tmp=min_val;
	INC_ERROR_COUNTER;
      }
      else if (tmp > max_val)
      {
	tmp=max_val;
	INC_ERROR_COUNTER;
      }
      *to= (char) (unsigned char) tmp;
      break;
    }
    case FIELD_TYPE_SHORT:
    {
      long tmp= strtol(from,NULL,10);
      if (f_is_dec(inputflag))
      {
	if (tmp < INT_MIN16)
	{
	  tmp=INT_MIN16;
	  INC_ERROR_COUNTER;
	}
	else if (tmp > INT_MAX16)
	{
	  tmp=INT_MAX16;
	  INC_ERROR_COUNTER;
	}
      }
      else
      {
	if (tmp < INT_MIN16)
	{				/* allow negative as complements */
	  tmp=0;
	  INC_ERROR_COUNTER;
	}
	else if (tmp > (uint16) ~0)
	{
	  tmp= (long) (uint16) ~0;
	  INC_ERROR_COUNTER;
	}
      }
      shortstore(to,(uint16) tmp);
      break;
    }
    case FIELD_TYPE_LONG:
    case FIELD_TYPE_DATE:
    {
      long tmp;
      errno=0;
      if (f_is_dec(inputflag))
	tmp= strtol(from,NULL,10);
      else
	tmp= (long) strtoul(from,NULL,10);
      if (errno)
	INC_ERROR_COUNTER;
      longstore(to,tmp);
      break;
    }
#ifdef HAVE_LONG_LONG
    case FIELD_TYPE_LONGLONG:
    {
      longlong tmp;
      errno=0;
      if (f_is_dec(inputflag))
	tmp= strtoll(from,NULL,10);
      else
	tmp= (longlong) strtoull(from,NULL,10);
      if (errno)
	INC_ERROR_COUNTER;
      longlongstore(to,tmp);
      break;
    }
#endif
    case FIELD_TYPE_INT24:
    case FIELD_TYPE_TIME:
    {
      long tmp;
      errno=0;
      if (f_is_dec(inputflag))
      {
	tmp= strtol(from,NULL,10);
	if (tmp < INT_MIN24)
	{
	  tmp=INT_MIN24;
	  INC_ERROR_COUNTER;
	}
	else if (tmp > INT_MAX24)
	{
	  tmp=INT_MAX24;
	  INC_ERROR_COUNTER;
	}
      }
      else
      {
	tmp= (long) strtoul(from,NULL,10);
	if (tmp < 0.0)
	{
	  tmp=0;
	  INC_ERROR_COUNTER;
	}
	else if (tmp >= (double) (ulong) (1L << 24))
	{
	  tmp= ~0L;
	  INC_ERROR_COUNTER;
	}
      }
      int3store(to,tmp);
      break;
    }
    case FIELD_TYPE_FLOAT:
      {
	float j; j = (float) atof(from); memcpy(to,(byte*) &j,sizeof(j));
      }
      break;
    case FIELD_TYPE_DOUBLE:
      {
	double j;
	j= atof(from);
	doublestore(to,j);
      }
      break;
    case FIELD_TYPE_TIMESTAMP:
      {
	long skr= (long) str_to_timestamp(from,strlen(from));
	longstore(to,skr);
	break;
      }
    case FIELD_TYPE_NULL:
      break;
    case FIELD_TYPE_STRING:
    case FIELD_TYPE_VAR_STRING:
    case FIELD_TYPE_BLOB:
    case FIELD_TYPE_LONG_BLOB:
    case FIELD_TYPE_MEDIUM_BLOB:
    case FIELD_TYPE_TINY_BLOB:
      break;					// Impossible
    }
  }
} /* f_packfield */


	/* unpack field to string */

void f_unpackfield(char *to, char *from, uint inputflag, uint length,
		   TYPELIB *intervall)
{
  reg4 uint i;
  uint len;
  char buff[MAX_FIELD_WIDTH],*end;

  if (!f_is_packed(inputflag))
  {
    memcpy(to,from,(size_t) length);
    *(to+length)='\0';
    return;
  }
  if (f_is_alpha(inputflag) && ! f_is_blob(inputflag))
  {
    ulonglong value;
    switch ((enum enum_field_types) f_packtype(inputflag)) {
    case FIELD_TYPE_CHAR:
      value=(ulonglong) (uchar) from[0];
      break;
    case FIELD_TYPE_SHORT:
      {
	short j ; shortget(j,from); value= (ulonglong) (unsigned short) j;
      }
      break;
    case FIELD_TYPE_INT24:
      value= (ulonglong) uint3korr(from);
      break;
    case FIELD_TYPE_LONG:
      {
	long j ; longget(j,from); value= (ulonglong) (ulong) j ;
      }
      break;
    case FIELD_TYPE_LONGLONG:
      {
	longlong j ; longlongget(j,from); value= (ulonglong) j ;
      }
      break;
    case FIELD_TYPE_DATE:
    {
      long tmp;
      longget(tmp,from);
      sprintf(to,"%04d-%02d-%02d",tmp/10000,(tmp/100 % 100),tmp % 100);
      return;
    }
    case FIELD_TYPE_TIME:
    {
      long tmp= uint3korr(from);
      sprintf(to,"%02d:%02d:%02d",tmp/10000,(tmp/100 % 100),tmp % 100);
      return;
    }
    case FIELD_TYPE_NULL:
    default:					/* Don't get any warnings */
      value=0L;
      break;
    }
    if (value)
    {
      if (f_is_intervall(inputflag))
      {
	if (value != 0L && (uint) value <= (uint) intervall->count)
	{
	  end=strnmov(to,intervall->type_names[(uint) value-1],length);
	  VOID(strfill(end,length-(uint) (end-to),f_fyllchar));
	}
	else
	  VOID(strfill(to,length,'?'));
      }
      else
	make_bit_type(to,value,intervall,length);
    }
    else
      VOID(strfill(to,length,f_fyllchar));
    return;
  }
  if ((len=unpack_number(buff,(uchar*) from,inputflag,length)) > length)
    VOID(strfill(to,length,'?'));
  else
  {
    register char fyllchar;
    fyllchar = f_is_zerofill(inputflag) ? (char) '0' : (char) ' ';
    for (i=length-len ; i-- > 0 ; ) *to++ = fyllchar;
    from=buff;
    for (i=len ; i-- > 0 ; ) *to++ = *from++;
    *to=0;
  }
} /* f_unpackfield */


	/* Unpack a packed number to a string */
	/* Returns length of string or INT_MAX */

static uint NEAR_F unpack_number(char *to, uchar *from, uint inputflag,
				 uint length)
{
  switch ((enum enum_field_types) f_packtype(inputflag)) {
  case FIELD_TYPE_CHAR:
  {
    long value;
    if (f_is_dec(inputflag))
      return (uint) (int2str((long) *((signed char*) from),to,-10)-to);
    return (uint) (int2str((long) *((uchar*) from),to,10)-to);
  }
  case FIELD_TYPE_SHORT:
    if (f_is_dec(inputflag))
    {
      short j; shortget(j,from);
      return (uint) (int2str((long) j,to,-10)-to);
    }
    else
    {
      short j; shortget(j,from);
      return (uint) (int2str((long) (uint16) j,to,10)-to);
    }
  case FIELD_TYPE_LONG:
  {
    long j; longget(j,from);
    return (uint) (int2str(j,to,f_is_dec(inputflag) ? -10 : 10)-to);
  }
#ifdef HAVE_LONG_LONG
  case FIELD_TYPE_LONGLONG:
  {
    longlong j; longlongget(j,from);
    return (uint) (longlong2str(j,to,f_is_dec(inputflag) ? -10 : 10)-to);
  }
#endif

  case FIELD_TYPE_INT24:
  {
    long j=f_is_dec(inputflag) ? sint3korr(from) : (long) uint3korr(from);
    return (uint) (int2str(j,to,-10)-to);
  }
  case FIELD_TYPE_FLOAT:
  {
    uint decimals=f_decimals(inputflag);
    float j;
    memcpy((byte*) &j,(char*) from,sizeof(j));
    if (fabs((double) j) >= 1E31)
      return(INT_MAX);
#ifdef HAVE_FCONVERT
    {
      char buff[70],*pos,*start;
      uint head=(uint) (length-decimals-test(decimals>0));
      int decpt,sign;
      VOID(sfconvert(&j,(int) decimals,&decpt,&sign,buff));
      start=to;
      if (sign)
      {
	*to++='-';
	head--;
      }
      if (decpt > (int) head || head == 0 && decimals == 0)
	return INT_MAX;			/* Too long field */
      pos=buff;
      if (decpt < 0)
      {					/* value is < 0 */
	*to++='0';
	if (!decimals)
	  return (uint) (to-start);
	*to++='.';
	if ((uint) -decpt > decimals)
	  decpt= - (int) decimals;
	decimals=(uint) ((int) decimals+decpt);
	while (decpt++ < 0)
	  *to++='0';
      }
      else if (decpt == 0)
      {
	if (head)
	  *to++= '0';
	if (!decimals)
	  return (uint) (to-start);
	*to++='.';
      }
      else
      {
	while (decpt-- > 0)
	  *to++= *pos++;
	if (!decimals)
	  return (uint) (to-start);
	*to++='.';
      }
      while (decimals--)
	*to++= *pos++;
      return (uint) (to-start);
    }
#else
    sprintf(to,"%.*lf",decimals,j);
    return strlen(to);
#endif
  }
  case FIELD_TYPE_DOUBLE:
  {
    double j;
    uint decimals=f_decimals(inputflag);
    doubleget (j,from);
    if (fabs(j) >= 1E31)
      return(INT_MAX);
#ifdef HAVE_FCONVERT
    {
      char buff[70],*pos,*start;
      uint head=(uint) (length-decimals-test(decimals>0));
      int decpt,sign;
      VOID(fconvert(j,(int) decimals,&decpt,&sign,buff));
      start=to;
      if (sign)
      {
	*to++='-';
	head--;
      }
      if (decpt > (int) head || head == 0 && decimals == 0)
	return INT_MAX;			/* Too long field */
      pos=buff;
      if (decpt < 0)
      {					/* value is < 0 */
	*to++='0';
	if (!decimals)
	  return (uint) (to-start);
	*to++='.';
	if ((uint) -decpt > decimals)
	  decpt= - (int) decimals;
	decimals=(uint) ((int) decimals+decpt);
	while (decpt++ < 0)
	  *to++='0';
      }
      else if (decpt == 0)
      {
	if (head)
	  *to++= '0';
	if (!decimals)
	  return (uint) (to-start);
	*to++='.';
      }
      else
      {
	while (decpt-- > 0)
	  *to++= *pos++;
	if (!decimals)
	  return (uint) (to-start);
	*to++='.';
      }
      while (decimals--)
	*to++= *pos++;
      return (uint) (to-start);
    }
#else
    sprintf(to,"%.*lf",decimals,j);
    return strlen(to);
#endif
  }
  case FIELD_TYPE_TIMESTAMP:
  {
    uint len,pos;
    long temp,part_time;
    time_t time_arg;
    struct tm *l_time;
#ifdef _REENTRANT
    struct tm tm_tmp;
#endif

    if (!length)
      length=12;			/* Get max length */
    longget(temp,from);
    if (temp == 0L)
    {					/* Zero time is "000000" */
      VOID(strfill(to,length,'0'));
      return length;
    }
    time_arg=(time_t) temp;
#ifdef _REENTRANT
    l_time=localtime_r(&time_arg,&tm_tmp);
#else
    l_time=localtime(&time_arg);
#endif
    for (len=pos=0; len+1 < length ; len+=2,pos++)
    {
      bool year_flag=0;
      switch (dayord.pos[pos]) {
      case 0: part_time=l_time->tm_year % 100; year_flag=1; break;
      case 1: part_time=l_time->tm_mon+1; break;
      case 2: part_time=l_time->tm_mday; break;
      case 3: part_time=l_time->tm_hour; break;
      case 4: part_time=l_time->tm_min; break;
      case 5: part_time=l_time->tm_sec; break;
      default: part_time=0; break;
      }
      if (year_flag && (length == 8 || length == 14))
      {
	if (part_time < 70)
	{
	  *to++='2'; *to++='0';
	}
	else
	{
	  *to++='1'; *to++='9';
	}
	len+=2;
      }
      *to++=(char) ('0'+((uint) part_time/10));
      *to++=(char) ('0'+((uint) part_time % 10));
    }
    *to=0;
    return len;
  }
  case FIELD_TYPE_NULL:
    sprintf(to,"%.*lf",(int) f_decimals(inputflag),0.0);
    return strlen(to);
  case FIELD_TYPE_DECIMAL:			/* This never happends */
  case FIELD_TYPE_STRING:
  case FIELD_TYPE_VAR_STRING:
  case FIELD_TYPE_BLOB:
  case FIELD_TYPE_LONG_BLOB:
  case FIELD_TYPE_MEDIUM_BLOB:
  case FIELD_TYPE_TINY_BLOB:
  case FIELD_TYPE_DATE:
  case FIELD_TYPE_TIME:
    break;

  }
  return(INT_MAX);
}


	/* unpack a field with space compress */
	/* Skipps endspace for asciifields and startspace for numbers */
	/* Returns length of field */

uint f_unpackfield_stripp(char *to, char *from, uint inputflag, uint max_length,
			  TYPELIB *intervall)
{
  reg4 uint i;
  uint len;
  char buff[MAX_FIELD_WIDTH],*end;

  if (!f_is_packed(inputflag))
  {						/* Not packed field */
    if (f_is_num(inputflag))
    {						/* If number */
      while (max_length && from[0] == ' ')	/* Skipp pre-space */
      {
	from++;
	max_length--;
      }
    }
    else
    {						/* String */
      while (max_length && from[max_length-1] == ' ')	/* Skipp end_space */
	max_length--;
    }
    memcpy(to,from,(size_t) max_length);
    *(to+max_length)='\0';
    return max_length;
  }
  if (f_is_alpha(inputflag) && ! f_is_blob(inputflag))
  {						/* Packed intervall */
    ulonglong value;
    switch ((enum enum_field_types) f_packtype(inputflag)) {
    case FIELD_TYPE_CHAR:
      value=(ulonglong) (uchar) from[0];
      break;
    case FIELD_TYPE_SHORT:
      {
	short j ; shortget(j,from);
	value= (ulonglong) (unsigned short) j;
      }
      break;
    case FIELD_TYPE_INT24:
      value= (ulonglong) uint3korr(from);
      break;
    case FIELD_TYPE_LONG:
      {
	long j ; longget(j,from); value= (ulonglong) (ulong) j ;
      }
      break;
    case FIELD_TYPE_LONGLONG:
      {
	longlong j ; longlongget(j,from); value= (ulonglong) j ;
      }
      break;
    case FIELD_TYPE_DATE:
    {
      long tmp;
      longget(tmp,from);
      sprintf(to,"%04d-%02d-%02d",tmp/10000,(tmp/100 % 100),tmp % 100);
      return 10;
    }
    case FIELD_TYPE_TIME:
    {
      long tmp= uint3korr(from);
      sprintf(to,"%02d:%02d:%02d",tmp/10000,(tmp/100 % 100),tmp % 100);
      return 8;
    }
    case FIELD_TYPE_NULL:
      /* fall through */
    default:			/* No warnings from GCC */
      value=0L;
      break;
    }
    if (value)
    {
      if (f_is_intervall(inputflag))
      {
	if (value != 0 && (uint) value <= (uint) intervall->count)
	  end=strmake(to,intervall->type_names[(uint) value-1],max_length);
	else
	  end=strfill(to,max_length,'?');
	return (uint) (end-to);
      }
      else
      {
	make_bit_type(to,value,intervall,0);
	return strlen(to);
      }
    }
    else
    {
      to[0]='\0';
      return 0;
    }
  }
  if ((len=unpack_number(buff,(uchar*) from,inputflag,max_length))
      > max_length)
    VOID(strfill(to,max_length,'?'));
  else
  {
    if (f_is_zerofill(inputflag))
      for (i=max_length-len ; i-- > 0 ; ) *to++ = '0';
    else
      max_length=len;
    memcpy(to,buff,(size_t) len);
    to[len]=0;
  }
  return max_length;
} /* f_unpackfield_stripp */


	/* Convert data between two fields. Blobs are suported by copying
	   pointers and converting lengths */

void f_konv(FIELD *to, FIELD *from)
{
  string to_str,from_str;
  uint to_flag,to_length,fr_flag,from_length;
  char buff[MAX_FIELD_WIDTH+1];

  to_flag=to->pack_flag; fr_flag=from->pack_flag;
  if (f_is_blob(to_flag) || f_is_blob(fr_flag))
  {
    conv_blob(to,from);
    return;
  }
  to_str=to->str; to_length=to->length;
  from_str=from->str; from_length=from->length;
  if (f_is_alpha(to_flag))
  {
    if (!f_is_packed(to_flag) && !f_is_packed(fr_flag))
    {						/* Both ascii fields */
      memcpy(to_str,from_str,
	     (size_t) (from_length=min(to_length,from_length)));
      bfill(to_str+from_length,to_length-from_length,' ');
      return;
    }
  }
  else if (f_is_equ(to_flag) == f_is_equ(fr_flag) &&
	   !((to_flag ^ fr_flag) & FIELDFLAG_ZEROFILL) &&
	   (to_length == from_length || f_is_packed(to_flag)))
  {						/* Both packed numbers */
    memcpy(to_str,from_str,to->packlength);
    return;
  }
  from_length=f_unpackfield_stripp(buff,from_str,fr_flag,from_length,
				   from->intervall);
  if (f_is_num(fr_flag) && buff[0] == '?')
  {						/* Wrong number */
    VOID(strmov(buff,"0"));			/* Change ???? -> 0 */
    from_length=1;
  }
  from_str=buff;
  if (f_is_num(fr_flag))
    while (from_length-- > to_length && *from_str == '0')
      from_str++;
  f_packfield(to_str,from_str,to_flag,to_length,to->intervall);
  return;
} /* f_konv */


	/*
	  functions to convert blobs
	  When a unpacked ascii-field is copied to a blob, the end space is
	  removed, everything else is copyed binary.
	  */

void conv_blob(FIELD *to, FIELD *from)
{
  uint to_flag=to->pack_flag,from_flag=from->pack_flag;

  if (f_is_blob(to_flag) && f_is_blob(from_flag))
  {
    uint to_blob_size= to->packlength-sizeof(char*);
    if (f_is_packed(to_flag) == f_is_packed(from_flag))
    {						/* Identical blobs */
      memcpy(to->str,from->str,(size_t) to_blob_size+sizeof(char*));
    }
    else
    {						/* Cut or extend length */
      ulong length=(ulong) valfield_l(from);
      length&= ((ulong) 1L << (to_blob_size*8))-1;
      packnrf((double) length,to);
      memcpy(to->str+to_blob_size,
	     from->str+from->packlength-sizeof(char*),
	     sizeof(char*));
    }
  }
  else if (f_is_blob(to_flag))
  {						/* Copy to a blob */
    char *pos,*end;
    uint length;
    if (f_is_alpha(from_flag) && ! f_is_packed(from_flag))
    {					/* Remove endspace from ascii */
      for (pos=from->str ,end=pos+from->length;
	   end > pos && end[-1] == ' ';
	   end --) ;
      length=(uint) (end-pos);
    }
    else
      length=from->length;
    packnrf((double) length,to);
    memcpy(to->str+to->packlength-sizeof(char*),&from->str,
	   sizeof(char*));
  }
  else
  {
    uint length=(uint) valfield_l(from);
    if (!length)
      bfill(to->str,to->packlength,(pchar) (f_is_packed(to_flag) ? '\0' : ' '));
    else
    {
      char *blob;
      uint copy_length=min(length,to->packlength);
      memcpy(&blob,from->str+from->packlength-sizeof(char*),sizeof(char*));
      memcpy(to->str,blob,copy_length);
      if (to->packlength > copy_length)
	bfill(to->str+copy_length,to->packlength-copy_length,
	      (pchar) (f_is_packed(to_flag) ? '\0' : ' '));
    }
  }
}


	/* Copy a string until '\0` and fill with space */

void copystrn(char *pos, const char *pos2, uint length)
{
  reg1 string end;

  end=strnmov(pos,pos2,length);
  bfill(end,length-(uint) (end-pos),' ');
  if (current_thd->count_cuted_fields)
  {
    end=(string) pos2+(int) (end-pos);
    while (*end && isspace(*end))
      end++;
    if (*end)
      INC_ERROR_COUNTER;
  }
  return;
} /* copystrn */


	/* Calculate length of number and it's parts */

static void NEAR_F number_dec(struct decstr *sdec, const char *str)
{
  reg2 uint length;

  sdec->sign=sdec->extra=length=0;
  if (*str == '-' || *str == '+')		/* sign */
  {
    sdec->sign=1;
    str++;
  }
  for (;;)
  {
    while (isdigit(*str))
    {
      length++;
      str++;
    }
    if (*str != ',')
      break;
    str++;					/* Skipp separator */
  }
  if (!(sdec->nr_length=length))
    sdec->extra=1;				/* We must put one 0 before */
  length=0;
  if (*str++ == '.')
  {
    while (isdigit(*str))
    {
      str++;
      length++;
    }
  }
  sdec->nr_dec=length;
} /* dumber_dec */



/* -------------------------------------------------------------------- */

	/* Funktioner som anv{nder sig av FIELD */

/* -------------------------------------------------------------------- */

	/* Ber{knar l{ngden p} ett packat f{lt */

uint packlength(const FIELD *field)
{
  if (f_is_blob(field->pack_flag))
    return f_packlength((uint) field->length,field->pack_flag);
  switch ((enum enum_field_types) f_packtype(field->pack_flag)) {
  case FIELD_TYPE_DECIMAL: return (field->length);
  case FIELD_TYPE_CHAR	: return 1;
  case FIELD_TYPE_SHORT : return 2;
  case FIELD_TYPE_INT24 : return 3;
  case FIELD_TYPE_TIME  : return 3;
  case FIELD_TYPE_TIMESTAMP	:
  case FIELD_TYPE_DATE:
  case FIELD_TYPE_LONG	: return sizeof(int32);
  case FIELD_TYPE_FLOAT : return sizeof(float);
  case FIELD_TYPE_DOUBLE: return sizeof(double);
  case FIELD_TYPE_LONGLONG: return 8;
  case FIELD_TYPE_NULL	: return 0;
  case FIELD_TYPE_STRING:
  case FIELD_TYPE_VAR_STRING:
  case FIELD_TYPE_BLOB:
  case FIELD_TYPE_LONG_BLOB:
  case FIELD_TYPE_MEDIUM_BLOB:
  case FIELD_TYPE_TINY_BLOB:
    break;					// Impossible
  }
  return 0;					/* We never comes here */
} /* packlength */


	/* Packar upp ett registerf{lt till str{ng-representation */
	/* OBS!  Samma l{ngd som i registret */

void r_unpackf(FIELD *field, char *pos)
{
  f_unpackfield(pos,field->str,(uint) field->pack_flag,(uint) field->length,
		field->intervall);
  return;
} /* r_unpackf */


	/* Unpack a field with space compress */

uint r_unpackf_stripp(FIELD *field, char *pos)
{
  return f_unpackfield_stripp(pos,field->str, (uint) field->pack_flag,
			      (uint) field->length, field->intervall);
} /* r_unpackf_stripp */


	/* Packar upp ett registerf{lt till ett tal */

double valfield(FIELD *field)
{
  reg1 uint flag;
  reg2 char *str;
  char buff[MAX_FIELD_WIDTH];
  DBUG_ENTER("valfield");

  flag=field->pack_flag;
  str=field->str;

  if (f_is_alpha(flag) && f_is_packed(flag) && ! f_is_blob(flag))
  {
    if ((enum enum_field_types) f_packtype(flag) != FIELD_TYPE_DATE &&
	(enum enum_field_types) f_packtype(flag) != FIELD_TYPE_TIME)
    {
      VOID(f_unpackfield_stripp(buff,str,flag,(uint) field->length,
				field->intervall));
      str=buff;					/* Value is now here */
      flag=0;					/* Not packed anymore */
    }
  }
  switch ((enum enum_field_types) f_packtype(flag)) {
  case FIELD_TYPE_DECIMAL:
    {
      reg4 char temp;
      reg5 double nr;
      temp= *(str+field->length); *(str+field->length) = '\0';
      nr=atod(str);
      *(str+field->length)=temp;
      DBUG_RETURN(nr);
    }
  case FIELD_TYPE_CHAR:
    if (f_is_dec(flag))
    {
      DBUG_RETURN ((double) (int) *str);
    }
    DBUG_RETURN ((double) *((uchar*) str));
  case FIELD_TYPE_SHORT:
    if (f_is_dec(flag))
    {
      short j; shortget(j,str);
      DBUG_RETURN ((double) j);
    }
    else
    {
      short j; shortget(j,str);
      DBUG_RETURN ((double) (uint16) j);
    }
  case FIELD_TYPE_LONG:
  case FIELD_TYPE_DATE:
    if (f_is_dec(flag))
    {
      long j; longget(j,str);
      DBUG_RETURN ((double) j);
    }
    else
    {
      ulong j; longget(j,str);
      DBUG_RETURN ((double) j);
    }
  case FIELD_TYPE_LONGLONG:
    if (f_is_dec(flag))
    {
      longlong j; longlongget(j,str);
      DBUG_RETURN ((double) j);
    }
    else
    {
      ulonglong j; longlongget(j,str);
      DBUG_RETURN ((double) j);
    }
  case FIELD_TYPE_INT24:
  case FIELD_TYPE_TIME:
  {
    long j=f_is_dec(flag) ? sint3korr(str) : (long) uint3korr(str);
    DBUG_RETURN ((double) j);
  }
  case FIELD_TYPE_FLOAT:
    {
      float j; memcpy((byte*) &j,str,sizeof(j));
      DBUG_RETURN ((double) j);
    }
  case FIELD_TYPE_DOUBLE:
    {
      double j;
      doubleget(j,str);
      DBUG_RETURN (j);
    }
  case FIELD_TYPE_NULL:
    DBUG_RETURN(0.0);
  case FIELD_TYPE_TIMESTAMP:
    {
      uint len,pos;
      long temp,part_time;
      time_t time_arg;
      struct tm *l_time;
      double res;
#ifdef _REENTRANT
      struct tm tm_tmp;
#endif

      longget(temp,str);
      if (temp == 0L)
	DBUG_RETURN(0.0);				/* No time */
      time_arg=(time_t) temp;
#ifdef _REENTRANT
      l_time=localtime_r(&time_arg,&tm_tmp);
#else
      l_time=localtime(&time_arg);
#endif
      res=0.0;
      for (pos=len=0; len+1 < (uint) field->length ; len+=2,pos++)
      {
	bool year_flag=0;
	switch (dayord.pos[pos]) {
	case 0: part_time=l_time->tm_year % 100; year_flag=1 ; break;
	case 1: part_time=l_time->tm_mon+1; break;
	case 2: part_time=l_time->tm_mday; break;
	case 3: part_time=l_time->tm_hour; break;
	case 4: part_time=l_time->tm_min; break;
	case 5: part_time=l_time->tm_sec; break;
	default: part_time=0; break;
	}
	if (year_flag && (field->length == 8 || field->length == 14))
	{
	  res=res*10000+part_time+((part_time < 70) ? 2000 : 1900);
	  len+=2;
	}
	else
	  res=res*100.0+part_time;
      }
      DBUG_RETURN(res);
    }
    case FIELD_TYPE_STRING:
    case FIELD_TYPE_VAR_STRING:
    case FIELD_TYPE_BLOB:
    case FIELD_TYPE_LONG_BLOB:
    case FIELD_TYPE_MEDIUM_BLOB:
    case FIELD_TYPE_TINY_BLOB:
      break;					// Impossible
  }
  DBUG_RETURN(0.0);			/* We never comes here */
} /* valfield */


	/* Packar upp ett registerf{lt till ett (long) integer */

long valfield_l(FIELD *field)
{
  reg1 uint flag;
  reg2 char *str;
  char buff[MAX_FIELD_WIDTH];
  DBUG_ENTER("valfield_l");

  flag=field->pack_flag;
  str=field->str;

  if (f_is_alpha(flag) && f_is_packed(flag) && ! f_is_blob(flag))
  {
    if ((enum enum_field_types) f_packtype(flag) != FIELD_TYPE_DATE &&
	(enum enum_field_types) f_packtype(flag) != FIELD_TYPE_TIME)
    {
      VOID(f_unpackfield_stripp(buff,str,flag,(uint) field->length,
				field->intervall));
      str=buff;					/* Value is now here */
      flag=0;					/* Not packed anymore */
    }
  }
  switch ((enum enum_field_types) f_packtype(flag)) {
  case FIELD_TYPE_DECIMAL:
    {
      reg4 char temp;
      reg5 long nr;
      temp= *(str+field->length); *(str+field->length) = '\0';
      nr=atoi(str);
      *(str+field->length)=temp;
      DBUG_RETURN(nr);
    }
  case FIELD_TYPE_CHAR:
    if (f_is_dec(flag))
    {
      DBUG_RETURN ((long) *str);
    }
    DBUG_RETURN ((long) *((uchar*) str));
  case FIELD_TYPE_SHORT:
    if (f_is_dec(flag))
    {
      short j; shortget(j,str);
      DBUG_RETURN ((long) j);
    }
    else
    {
      uint16 j; shortget(j,str);
      DBUG_RETURN ((long) j);
    }
  case FIELD_TYPE_LONG:
  case FIELD_TYPE_DATE:
    {
      long j; longget(j,str);
      DBUG_RETURN (j);
    }
  case FIELD_TYPE_LONGLONG:
    {
      longlong j; longlongget(j,str);
      DBUG_RETURN ((long) j);
    }
  case FIELD_TYPE_INT24:
  case FIELD_TYPE_TIME:
  {
    long j=f_is_dec(flag) ? sint3korr(str) : (long) uint3korr(str);
    DBUG_RETURN ((long) j);
  }
  case FIELD_TYPE_FLOAT:
    {
      float j; memcpy((byte*) &j,str,sizeof(j));
      DBUG_RETURN ((long) j);
    }
  case FIELD_TYPE_DOUBLE:
    {
      double j;
      doubleget(j,str);
      DBUG_RETURN ((long) j);
    }
  case FIELD_TYPE_NULL:
    DBUG_RETURN(0);
  case FIELD_TYPE_TIMESTAMP:
    DBUG_RETURN((long)valfield(field)); /* Not normaly used */
  case FIELD_TYPE_STRING:
  case FIELD_TYPE_VAR_STRING:
  case FIELD_TYPE_BLOB:
  case FIELD_TYPE_LONG_BLOB:
  case FIELD_TYPE_MEDIUM_BLOB:
  case FIELD_TYPE_TINY_BLOB:
    break;					// Impossible
  }
  DBUG_RETURN(0L);			/* We never comes here */
} /* valfield_l */



	/* Give field a value from a string */
	/* Blobs are ignored */

void packf(const char *str, FIELD *field)
{
  DBUG_ENTER("packf");
  f_packfield(field->str,str,field->pack_flag,(uint) field->length,
	      field->intervall);
  DBUG_VOID_RETURN;
} /* packf */


	/* Ger ett f{lt ett v{rde efter ett tal */

void packnrf(double nr, FIELD *field)
{
  reg2 uint flag;
  int  dec;
  reg3 char *pos;
  DBUG_ENTER("packnrf");
  DBUG_PRINT("enter",("nr: %lf",nr));

  flag=field->pack_flag;
  pos= field->str;
  if (f_is_alpha(flag) && ! f_is_blob(flag))
  {
    char buff[MAX_FIELD_WIDTH];
    sprintf(buff,"%.*lg",min(field->length,DBL_DIG+5),nr);
    if (!f_is_packed(flag))
      copystrn(pos,buff,(uint) field->length);
    else
      f_packfield(field->str,buff,flag,(uint) field->length,field->intervall);
  }
  else
  {
    switch ((enum enum_field_types) f_packtype(flag)) {
    case FIELD_TYPE_DECIMAL:
      {
	reg4 uint i,length;
	char fyllchar,*str;
	char buff[MAX_FIELD_WIDTH];

	fyllchar = f_is_zerofill(flag) ? (char) '0' : (char) ' ';
	dec=f_decimals(flag);
	sprintf(buff,"%.*lf",dec,nr);
	length=strlen(buff);

	if (length > field->length)
	{
	  bfill(pos,(uint) field->length,'?');
	  INC_ERROR_COUNTER;
	}
	else
	{
	  for (i=field->length-length ; i-- > 0 ;)
	    *pos++ = fyllchar;
	  str=buff;
	  for (i=length ; i-- > 0 ; ) *pos++ = *str++;
	}
      }
      break;
    case FIELD_TYPE_CHAR:
    {
      int tmp;
      double min_val,max_val;
      if (f_is_dec(flag))
      {
	min_val=-128.0; max_val=127.0;
      }
      else
      {
	min_val=0.0; max_val=255.0;	
      }
      nr=floor(nr+0.5);
      if (nr < min_val)
      {
	tmp=(int) min_val;
	INC_ERROR_COUNTER;
      }
      else if (nr > max_val)
      {
	tmp=(int) max_val;
	INC_ERROR_COUNTER;
      }
      else
	tmp= (int) nr;
      *pos= (char) (unsigned char) tmp;
      break;
    }
    case FIELD_TYPE_SHORT:
      {
	int tmp;
	nr=floor(nr+0.5);
	if (f_is_dec(flag))
	{
	  if (nr < INT_MIN16)
	  {
	    tmp=INT_MIN16;
	    INC_ERROR_COUNTER;
	  }
	  else if (nr > INT_MAX16)
	  {
	    tmp=INT_MAX16;
	    INC_ERROR_COUNTER;
	  }
	  else
	    tmp=(short) nr;
	}
	else
	{
	  if (nr < 0.0)
	  {
	    tmp=0;
	    INC_ERROR_COUNTER;
	  }
	  else if (nr > (uint16) ~0)
	  {
	    tmp= (int) (uint16) ~0;
	    INC_ERROR_COUNTER;
	  }
	  else
	    tmp=(int) (unsigned short) nr;
	}
	shortstore(pos,(uint16) tmp);
      }
      break;
    case FIELD_TYPE_LONG:
    case FIELD_TYPE_DATE:
    {
      long tmp;
      nr=floor(nr+0.5);
      if (f_is_dec(flag))
      {
	if (nr < INT_MIN32)
	{
	  tmp=INT_MIN32;
	  INC_ERROR_COUNTER;
	}
	else if (nr > INT_MAX32)
	{
	  tmp=INT_MAX32;
	  INC_ERROR_COUNTER;
	}
	else
	  tmp=(long) nr;
      }
      else
      {
	if (nr < 0.0)
	{
	  tmp=0;
	  INC_ERROR_COUNTER;
	}
	else if (nr > (double) (ulong) ~0L)
	{
	  tmp= ~0L;
	  INC_ERROR_COUNTER;
	}
	else
	  tmp=(long) (unsigned long) nr;
      }
      longstore(pos,tmp);
      break;
    }
#ifdef HAVE_LONG_LONG
    case FIELD_TYPE_LONGLONG:
    {
      longlong tmp;
      nr=floor(nr+0.5);
      if (f_is_dec(flag))
      {
	if (nr < (double) LONGLONG_MIN)
	{
	  tmp=LONGLONG_MIN;
	  INC_ERROR_COUNTER;
	}
	else if (nr > (double) LONGLONG_MAX)
	{
	  tmp=LONGLONG_MAX;
	  INC_ERROR_COUNTER;
	}
	else
	  tmp=(longlong) nr;
      }
      else
      {
	if (nr < 0.0)
	{
	  tmp=0;
	  INC_ERROR_COUNTER;
	}
	else if (nr > (double) ~ (ulonglong) 0)
	{
	  tmp= ~(ulonglong) 0;
	  INC_ERROR_COUNTER;
	}
	else
	  tmp=(longlong) (ulonglong) nr;
      }
      longlongstore(pos,tmp);
      break;
    }
#endif
    case FIELD_TYPE_INT24:
    case FIELD_TYPE_TIME:
    {
      long tmp;
      nr=floor(nr+0.5);
      if (f_is_dec(flag))
      {
	if (nr < INT_MIN24)
	{
	  tmp=INT_MIN24;
	  INC_ERROR_COUNTER;
	}
	else if (nr > INT_MAX24)
	{
	  tmp=INT_MAX24;
	  INC_ERROR_COUNTER;
	}
	else
	  tmp=(long) nr;
      }
      else
      {
	if (nr < 0.0)
	{
	  tmp=0;
	  INC_ERROR_COUNTER;
	}
	else if (nr >= (double) (ulong) (1L << 24))
	{
	  tmp= ~0L;
	  INC_ERROR_COUNTER;
	}
	else
	  tmp=(long) (unsigned long) nr;
      }
      int3store(pos,tmp);
      break;
    }
    case FIELD_TYPE_FLOAT:
      dec=f_decimals(flag);
      if (dec != 31)
	nr=floor(nr*log_10[dec]+0.5)/log_10[dec]; /* To fixed point */
      {
	float i; i = (float) nr; memcpy(pos,(byte*) &i,sizeof(i));
      }
      break;
    case FIELD_TYPE_DOUBLE:
      dec=f_decimals(flag);
      if (dec != 31)
	nr=floor(nr*log_10[dec]+0.5)/log_10[dec]; /* To fixed point */
      doublestore(pos,nr);
      break;
    case FIELD_TYPE_NULL:
      break;					/* Not implemented */
    case FIELD_TYPE_TIMESTAMP:
      {
	char buff[20];
	uint length= (uint) (int2str((long) nr,buff,10) -buff);
	if (length & 1)		// Should be even
	{			// Assume year 0-9   => 2000 - 2009
	  bmove_upp(buff+length+1,buff+length,length);
	  buff[0]='0';
	  length++;
	}
	long skr=(long) str_to_timestamp(buff,length);
	longstore(pos,skr);
	break;
      }
    case FIELD_TYPE_STRING:
    case FIELD_TYPE_VAR_STRING:
    case FIELD_TYPE_BLOB:
    case FIELD_TYPE_LONG_BLOB:
    case FIELD_TYPE_MEDIUM_BLOB:
    case FIELD_TYPE_TINY_BLOB:
      break;					// Impossible
    }
  }
  DBUG_VOID_RETURN;
} /* packnrf */


byte *get_blob_ptr(FIELD *field)
{
  byte *pos=field->str+(field->packlength-sizeof(char*));
  byte *blob_pos;
  memcpy(&blob_pos,pos,sizeof(char*));
  return blob_pos;
}


void store_ptr_in_blob(FIELD *field, const byte *blob, uint length)
{
  byte *pos=field->str+(field->packlength-sizeof(char*));
  if (!length)
    blob=0;
  memcpy(pos,&blob,sizeof(char*));
  packnrf((double) length,field);
}



void change_byte(byte *str, uint length, char from, char to)
{
  while (length-- != 0)
    if (*str == from)
      *str++=(byte) to;
    else
      str++;
}


int pack_blob(FIELD *field, byte *blob, uint length)
{
  byte *blob_pos;
  byte *pos=field->str+(field->packlength-sizeof(char*));
  memcpy(&blob_pos,pos,sizeof(char*));
  x_free(blob_pos);
  if (length)
    blob_pos=my_memdup(blob,length,MYF(0));
  else
    blob_pos=0;
  memcpy(pos,&blob_pos,sizeof(char*));
  if (blob_pos || !length)
  {
    packnrf((double) length,field);
    return 0;
  }
  packnrf(0.0,field);
  INC_ERROR_COUNTER;
  return 1;
}

	/* Free blobs allocated with pack_blob */

void free_blobs(FORM *form)
{
  FIELD *f,*end;
  gptr blob_pos;

  for (f=form->field,end=f+form->fields ; f != end ; f++)
  {
    if (f_is_blob(f->pack_flag))
    {
      gptr pos=f->str+(f->packlength-sizeof(char*));
      memcpy((gptr) &blob_pos,pos,sizeof(char*));
      x_free(blob_pos);
    }
  }
}
