/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB
   
   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.
   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.  The License grants you the right to 
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

/* Functions to convert between fields. Used by ALTER TABLE */

#include "mysql_priv.h"

static void conv_field_identical(FIELD *to,FIELD *from);
static void conv_field_alpha(FIELD *to,FIELD *from);
static void conv_field_1_pack(FIELD *to,FIELD *from);
static void conv_field_2_pack(FIELD *to,FIELD *from);
static void conv_field_4_pack(FIELD *to,FIELD *from);
static void conv_field_8_pack(FIELD *to,FIELD *from);
static void conv_field_default(FIELD *to,FIELD *from);
static void conv_num_to_char(FIELD *to,FIELD *from);
static void conv_num_to_short(FIELD *to,FIELD *from);
static void conv_num_to_long(FIELD *to,FIELD *from);
static void conv_num_to_float(FIELD *to,FIELD *from);
static void conv_int_to_float(FIELD *to,FIELD *from);
static void conv_int_to_double(FIELD *to,FIELD *from);

static void conv_field_identical(FIELD *to, FIELD *from)
{
  memcpy(to->str,from->str,(size_t) to->packlength);
}

static void conv_field_alpha(FIELD *to, FIELD *from)
{
  reg1 uint from_length;
  memcpy(to->str,from->str,(size_t) (from_length=from->length));
  bfill(to->str+from_length,to->length-from_length,' ');
}

static void conv_field_1_pack(FIELD *to, FIELD *from)
{
  to->str[0]=from->str[0];
}

static void conv_field_2_pack(FIELD *to, FIELD *from)
{
  to->str[0]=from->str[0];
  to->str[1]=from->str[1];
}

static void conv_field_3_pack(FIELD *to, FIELD *from)
{
  to->str[0]=from->str[0];
  to->str[1]=from->str[1];
  to->str[2]=from->str[2];
}

static void conv_field_4_pack(FIELD *to, FIELD *from)
{
  to->str[0]=from->str[0];
  to->str[1]=from->str[1];
  to->str[2]=from->str[2];
  to->str[3]=from->str[3];
}

static void conv_field_8_pack(FIELD *to, FIELD *from)
{
  to->str[0]=from->str[0];
  to->str[1]=from->str[1];
  to->str[2]=from->str[2];
  to->str[3]=from->str[3];
  to->str[4]=from->str[4];
  to->str[5]=from->str[5];
  to->str[6]=from->str[6];
  to->str[7]=from->str[7];
}

static void conv_num_to_char(FIELD *to, FIELD *from)
{
  to->str[0]=(char) valfield_l(from);
}

static void conv_num_to_short(FIELD *to, FIELD *from)
{
  short tmp=(short) valfield_l(from);
  shortstore(to->str,(unsigned short) tmp);
}

static void conv_num_to_int24(FIELD *to, FIELD *from)
{
  long tmp=valfield_l(from);
  int3store(to->str,tmp);
}

static void conv_num_to_long(FIELD *to, FIELD *from)
{
  long tmp=valfield_l(from);
  longstore(to->str,tmp);
}

static void conv_num_to_float(FIELD *to, FIELD *from)
{
  packnrf(valfield(from),to);		/* This handles floats and doubles */
}

static void conv_int_to_float(FIELD *to, FIELD *from)
{
  float tmp= (float) valfield_l(from);
  memcpy(to->str,(gptr) &tmp,sizeof(tmp));
}

static void conv_int_to_double(FIELD *to, FIELD *from)
{
  double tmp= valfield_l(from);
  doublestore(to->str,tmp);
}

	/* Dummy function when field isn't copyed */
	/* ARGSUSED */
void conv_field_skipp(FIELD *to, FIELD *from)
{
  return;
}

static void conv_field_default(FIELD *to, FIELD *from)
{
  reg1 uint from_length;
  char buff[MAX_FIELD_WIDTH+1],*from_str;

  from_length=f_unpackfield_stripp(buff,from->str,
				   from->pack_flag,(uint) from->length,
				   from->intervall);
  from_str=buff;
  if (f_is_num(from->pack_flag))
  {
    if (buff[0] == '?')
    {						/* Wrong number */
      VOID(strmov(buff,"0"));			/* Change ???? -> 0 */
      from_length=1;
    }
    while (from_length-- > to->length && *from_str == '0')
      from_str++;
  }
  f_packfield(to->str,from_str,to->pack_flag,(uint) to->length,to->intervall);
}


	/* Get a function to convert from one field-type to another */

void (*get_field_conv(FIELD *to,FIELD *from))(FIELD *,FIELD *)
{
  uint to_flag,to_length,fr_flag,from_length;

  to_flag=to->pack_flag; to_length=to->length;
  fr_flag=from->pack_flag; from_length=from->length;

  if (f_packtype(to_flag) == (int) FIELD_TYPE_NULL)
    return conv_field_skipp;
  if (f_is_blob(to_flag) || f_is_blob(fr_flag))
    return conv_blob;
  if (f_is_alpha(to_flag))
  {
    if (!f_is_packed(to_flag) && !f_is_packed(fr_flag))
    {						/* Both ascii fields */
      if (to_length <= from_length)
	return conv_field_identical;
      return conv_field_alpha;
    }
    if (f_is_packed(to_flag) != f_is_packed(fr_flag) ||
	f_is_intervall(to_flag) != f_is_intervall(fr_flag))
      return conv_field_default;
    {
      reg1 int i;
      TYPELIB *to_lib=to->intervall,*fr_lib=from->intervall;
      if (!to_lib && !fr_lib)			// The second test if for safety
	return conv_field_identical;
      if (to_lib->count < fr_lib->count)
	return conv_field_default;
      for (i=0 ; (uint) i < fr_lib->count ; i++)
	if (my_strcasecmp(to_lib->type_names[i],fr_lib->type_names[i]))
	  return conv_field_default;
    }
  }
  else if (f_is_equ(to_flag) != f_is_equ(fr_flag) ||
	   (!f_is_packed(to_flag) && (to_length != from_length ||
				      (to_flag & FIELDFLAG_ZEROFILL) !=
				      (fr_flag & FIELDFLAG_ZEROFILL))))
  {
    if (f_is_num(fr_flag) && f_is_packed(fr_flag) && f_is_packed(to_flag))
    {
      switch (f_packtype(to_flag)) {
      case FIELD_TYPE_CHAR:
	return conv_num_to_char;
      case FIELD_TYPE_SHORT:
	return conv_num_to_short;
      case FIELD_TYPE_INT24:
	return conv_num_to_int24;
      case FIELD_TYPE_LONG:
	return conv_num_to_long;
      case FIELD_TYPE_FLOAT:
	if (!f_is_dec(fr_flag) &&
	    f_packtype(fr_flag) != (int) FIELD_TYPE_DOUBLE &&
	    from_length <= 9)
	  return conv_int_to_float;		/* 31 bits long is enough */
	/* fall through */
      case FIELD_TYPE_DOUBLE:
	if (!f_is_dec(fr_flag) &&
	    f_packtype(fr_flag) != (int) FIELD_TYPE_FLOAT &&
	    f_packtype(fr_flag) != (int) FIELD_TYPE_DOUBLE &&
	    from_length <= 9)
	  return conv_int_to_double;		/* 31 bits long is enough */
	return conv_num_to_float;
      default:
	break;
      }}
    return conv_field_default;
  }

	/* Identical fields */
  switch (to->packlength) {
  default:					/* Unpacked num string */
  case 0: return conv_field_identical;
  case 1: return conv_field_1_pack;
  case 2: return conv_field_2_pack;
  case 3: return conv_field_3_pack;
  case 4: return conv_field_4_pack;
  case 8: return conv_field_8_pack;
  }
}
