#!@PERL@
#-----------------------------------------
# mysqlaccess
# ~~~~~~~~~~~
# Copyright (C) 1997 Yves.Carlier@rug.ac.be
#
# report the access-privileges for a USER from a  HOST to a DB
# by:   Yves.Carlier@rug.ac.be
#
# This script reports the access-rights a user has, if if connects
# to the given DB from the given host.
#
#---------------------------------------------------------------
#
# Release Note:
#
# 0.1-beta1: internal
# - first trial.
# 0.1-beta2: (1997-02-27)
# - complete rewrite of the granting-rules, based on the documentation
#   found in de FAQ.
# - IP-number and name for a host are equiv.
# 0.1-beta3: (1997-03-10)
# - more information on the rules which are used.
# - 'localhost' and the name/ip of the local machine are now equiv.
# 0.1-beta4: (1997-03-11)
# - inform the user if he has not enough priv. to read the mysql db
#
# 1.0-beta1: (1997-03-12)
# suggestions by Monty:
# - connect as superuser with superpassword.
# - mysqlaccess could also notice if all tables are empty. This means
#   that all user have full access!
# - It would be nice if one could optionally start mysqlaccess without
#   any options just the arguments 'user db' or 'host user db', where
#   host is 'localhost' if one uses only two arguments.
# 1.0-beta2: (1997-03-14)
# - bugfix: translation to reg.expr of \_ and \%.
# - bugfix: error in matching regular expression and string given
#           by user which resulted in
#           'test_123' being matched with 'test'
# 1.0-beta3: (1997-03-14)
# - bugfix: the user-field should not be treated as a sql-regexpr,
#           but as a plain string.
# - bugfix: the host-table should not be used if the host isn't empty in db
#                                         or  if the host isn't emty in user
#           (Monty)
# 1.0-beta4: (1997-03-14)
# - bugfix: in an expression "$i = $j or $k", the '=' binds tighter than the or
#           which results in problems...
#           (by Monty)
# - running mysqlaccess with "perl -w" gives less warnings...   ;-)
#---------------------------------------------------------------
#
# TODO:
# -wildcards and default values for parameters.
# -add some more functionality for DNS.
# -add a parameter 'report' for more reporting/information
# -maybe one should also be able to validate his password??
#
# suggestions by Monty:
#- Why not use the mysqlperl module this would make the following
#  things much easier:
#  (Actually I would like to use the DBD module, but until the new DBD
#   module is released it's better to stick with mysqlperl. At least this
#   will always work.)
#  In this case you don't have to do any parsing or be dependent on
#  the 'mysql' program.
#  The program would also be a lot faster (no big deal!)
#  --> version 2.0
#
#- The following indexes should maybe calculated from the select column
#  names. I will not rename any names, but some further version may add
#  more columns!
#  --> version 2.0
#
# ===============================================
#
# From the FAQ:
# ~~~~~~~~~~~~
# The host table is mainly to maintain a list of "secure" servers.
# At TCX hosts contain a list of all machines on local network. These are granted
# all privileges.
# Technically the user grant is calculated by:
#
#    1.First sort all entries by host by putting host without wildcards first,
#      after this host with wildcards and entries with host = ".
#      Under each host sort user by the same criterias.
#    2.Get grant for user from the "db" table.
#    3.If hostname is "empty" for the found entry, AND the privileges with
#      the privileges for the host in "host" table.
#      (Remove all which is not "Y" in both)
#    4.OR (add) the privileges for the user from the "user" table.
#     (add all privileges which is "Y" in "user")
#
#    When matching, use the first found match.
#
#

use Getopt::Long;
use Socket;
use Sys::Hostname;

$VERSION = "1.0-beta4 14 Mar. 1997";
$script  = "mysqlaccess";

# ****************************
# debugging flag
# can be set to 0,1,2
# a higher value gives more info
$|=1;
$DEBUG   = 0;

# ****************************
# temporary file
$tmpfile ="/tmp/mysqlaccess.$$";

# ****************************
# information on mysql
	$MYSQL     = '@bindir@/mysql';
	$ACCESS_DB = 'mysql';
	$ACCESS_H  = 'host';
	$ACCESS_U  = 'user';
	$ACCESS_D  = 'db';

# ***********************************
# translation-table for poss. answers
%Answer =  ('Y' =>  1 , 'N' =>  0
           , 1  => 'Y',  0  => 'N'
           ,'?' => '?'
           );

# ****************************
# information on used rules
%Rules = ( 'host' => 'no rules defined.'
         , 'user' => 'no rules defined.'
	 , 'db'   => 'no rules defined.'
	  );
$passwd_info="BEWARE:\t Everybody can access your DB as user '\$user',\n".
             "       \t WITHOUT supplying a password, by using the '-u' parameter";
$full_access='BEWARE:\t All grant-tables are empty, which gives full access to all users.';

# ****************************
# Record-layout of access-tables
%H = ( 'select_priv'   => 2
     , 'insert_priv'   => 3
     , 'update_priv'   => 4
     , 'delete_priv'   => 5
     , 'create_priv'   => 6
     , 'drop_priv'     => 7
     );
%D = ( 'select_priv'   => 3
     , 'insert_priv'   => 4
     , 'update_priv'   => 5
     , 'delete_priv'   => 6
     , 'create_priv'   => 7
     , 'drop_priv'     => 8
     );
%U = ( 'select_priv'   => 3
     , 'insert_priv'   => 4
     , 'update_priv'   => 5
     , 'delete_priv'   => 6
     , 'create_priv'   => 7
     , 'drop_priv'     => 8
     , 'reload_priv'   => 9
     , 'shutdown_priv' => 10
     , 'process_priv'  => 11
     , 'file_priv'     => 12
     );


# ****************************
# The access-rights:
# defaults to none
%Access = ( 'select_priv'  => 0
	  , 'insert_priv'  => 0
	  , 'update_priv'  => 0
 	  , 'delete_priv'  => 0
	  , 'create_priv'  => 0
	  , 'drop_priv'    => 0
	  , 'file_priv'    => 0
	  , 'reload_priv'  => 0
	  , 'shutdown_priv'=> 0
	  , 'process_priv' => 0
          );
%Access_db    = %Access;
%Access_user  = %Access;
%Access_host  = %Access;


# %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
#  mysqlaccess:
#  ~~~~~~~~~~~
#  Lets get to it,
#  and start the program by processing the parameters
# %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

# ---------------------------------
# give help with the -? option
if (@ARGV  and $ARGV[0] eq '-?') {
   help();
   exit 0;
}

# ----------------------------
# Get options from commandline
$Getopt::Long::ignorecase=0; #case sensitive options
GetOptions("help" => \$help
          ,"host|h=s" => \$host
          ,"user|u=s" => \$user
          ,"db|d=s"   => \$db
	  ,"superuser|U=s" => \$superuser
          ,"spassword|P=s" => \$spassword
	  );

# -----------------------------
# check for things which aren't
# declared as options:
# (user,db) -> ('localhost',user,db)
# or (host,user,db)
if ($#ARGV == 1) {
   print "$script called with 2 arguments:\n" if ($DEBUG);
   if (!defined($host)) { $host='localhost'; }
   if (!defined($user)) { $user=$ARGV[0]; }
   if (!defined($db))   { $db  =$ARGV[1]; }
}
if ($#ARGV == 2) {
   print "$script called with 3 arguments:\n" if ($DEBUG);
   if (!defined($host)) { $host=$ARGV[0]; }
   if (!defined($user)) { $user=$ARGV[1]; }
   if (!defined($db))   { $db  =$ARGV[2]; }
}

# ----------------------
# if no host is given
# assume we mean 'localhost'
if (!defined($host)) { $host='localhost'; }

#-----------------------
# get info/help if necc.
# and exit.
if ( (!defined($user) or !defined($host) or !defined($db)) or
     defined($help) ) {
  help();
  exit 0;
}

#---------------------------------
# inform user if he has not enough
# privileges to read the access-db
if (!HaveReadAccess()) {
    Print_Error_Access();
    exit 0;
}

# ----------------------------
# get hostname for IP-address
# get IP-address for hostname
$host_name = IP2Name($host);
$host_ip   = Name2IP($host);

	print "DEBUG $script:\n" if $DEBUG;
	print "  --host=$host\n" if $DEBUG;
        print "    hostname=$host_name\n" if $DEBUG;
        print "    host-ip =$host_ip\n" if $DEBUG;
	print "  --user=$user\n" if $DEBUG;
	print "  --db  =$db\n"  if $DEBUG;

# ----------------------------
# get hostname and local-ip
# for localhost
$localhost = LocalHost();
$local_ip  = Name2IP($localhost);
        print "  --localhost=$localhost\n" if ($DEBUG);
        print "  --local_ip =$local_ip\n" if ($DEBUG);


# ******************************************************************************
# first get sorted table of user-privileges
# -----------------------------------------
#    1.First sort all entries by host by putting host without wildcards first,
#      after this host with wildcards and entries with host = ".
#      Under each host sort user by the same criterias.
# ******************************************************************************
Sort_db_by_host_and_user();

# ******************************************************************************
# retrieve information on DB
#  check all records in mysql::db for matches with the triple (host,db,user)
#  first match is used.
# ******************************************************************************
#    2.Get grant for user from the "db" table.
$process_host_table=0;
Get_grant_from_db(); #set process_host


# ***********************************************************************
# retrieve information on HOST
#  check all records in mysql::host for matches with the tuple (host,db)
#
#  ' The host table is mainly to maintain a list of "secure" servers. '
# ***********************************************************************
#    3.If hostname is "empty" for the found entry, AND the privileges with
#      the privileges for the host in "host" table.
#      (Remove all which is not "Y" in both)
Get_grant_from_host();

# ***********************************************************************
# retrieve information on USER
#  check all records in mysql::user for matches with the tuple (host,user)
# ***********************************************************************
#    4.OR (add) the privileges for the user from the "user" table.
#     (add all privileges which is "Y" in "user")
Get_grant_from_user();
unlink($tmpfile);

# *******************************************************
#  Print access-rights on STDOUT
# *******************************************************
print Header();
Print_Access_rights();
exit 0;


#############################################################
#  FUNCTIONS  #
###############
# ==========================================================
# sub Sort_db_by_host_and_user:
# ==========================================================
sub Sort_db_by_host_and_user {
    my $debug=0;
    $debug=$DEBUG if ($DEBUG);
    my $CONNECT = ConnectAs();
    my @where;
    $where[1]="((FIELD not like '\\%') AND (FIELD <> ''))";
    $where[2]="((FIELD like '%\\%%') OR (FIELD like '%\\_%'))";
    $where[3]="(FIELD = '')";
    my $order   ="order by host,user";
    my $sql     ="$MYSQL $ACCESS_DB $CONNECT -s -e \"SELECT * FROM $ACCESS_U ";
    open(USER,"> $tmpfile") or die "Can not open temporary file $tmpfile for writing\n";
    #  first sorted by host
    foreach $i (1..3) {
	$SQL1 = $sql;
	$where = $where[$i];
	$where =~ s/FIELD/host/g;
	$SQL1 = "$SQL1 WHERE $where";
	# second sorted by user
	foreach $j (1..3) {
	    $where = $where[$j];
	    $where =~ s/FIELD/user/g;
	    $SQL2  = "$SQL1 AND $where";
	    $SQL2  = "$SQL2 $order\"";
	    print "SQL2=$SQL2\n" if ($debug>1);
	    open(MYSQL,"$SQL2 |");
	    $record=<MYSQL>;
	    while ($record=<MYSQL>) {
		print "read: $record" if ($debug>1);
		print USER $record;
	    }
	}
    }
    close(USER);
    return 1;
}

# ==========================================================
# sub Get_grant_from_db:
# ==========================================================
sub Get_grant_from_db {
    my $debug=0;
    $debug=$DEBUG if ($DEBUG);
    my $found_entry=0;		# found entry in DB-table which maches
				# if no entry's are found, no restrictions are done.
    my $CONNECT = ConnectAs();
    my $where = "";
    my $sql   = "$MYSQL $ACCESS_DB $CONNECT -s -e \"select * from $ACCESS_D $where;\"";
    open(DB,"$sql |");
    my $record=<DB>; #skip fieldnames
    while ($record=<DB> and !$found_entry) {
	$full_access='';
	print "db: $record" if ($debug);
	chop($record);
	@record=split(/\t/,$record);
	# check host and db
	# with possible wildcards in field
	# replace mysql-wildcards by reg-wildcards
	my $host_tpl = SQL2Reg($record[0]);
	my $db_tpl   = SQL2Reg($record[1]);
#	my $user_tpl = SQL2Reg($record[2]);
        my $user_tpl = $record[2];
	print "\thost_tpl : $record[0] => $host_tpl\n" if ($debug>1);
	print "\tdb_tpl   : $record[1] => $db_tpl\n" if ($debug>1);
	print "\tuser_tpl : $record[2] => $user_tpl\n" if ($debug>1);
	if ($debug>1) {
	    if ( Is_localhost($host_tpl)
                 or  MatchTemplate($host_tpl,$host_name)
                 or  MatchTemplate($host_tpl,$host_ip) )
	    { print "\t* host  [$host_name,$host_ip] matched with rule [$host_tpl]\n"; }
	    else			   { print "\t* host doesn't match\n"; }
	    if ( MatchTemplate($db_tpl,$db) )
	    { print "\t* db [$db] matched with rule [$db_tpl]\n"; }
	    else                       { print "\t* db doesn't match\n";}
	    if ( MatchTemplate($user_tpl,$user) )
	    { print "\t* user  [$user] matched with rule [$user_tpl]\n"; }
	    else                       { print "\t* user doesn't match\n"; }
	}

	if ( ( Is_localhost($host_tpl)
               or  MatchTemplate($host_tpl,$host_name)
               or  MatchTemplate($host_tpl,$host_ip) )
	     and ( MatchTemplate($db_tpl,$db) )
	     and ( MatchTemplate($user_tpl,$user) ) ) {
	    if ($host_tpl eq "") { $process_host_table=1; }
	    $found_entry=1;
	    $Rules{'db'} = $record;
	    foreach $field (keys(%D)) {
		printf "\t> %-15s: input=$record[$D{$field}]  =>  $Access{$field}($Answer{$Access_db{$field}}) -> ",$field if $debug;
		$Access_db{$field} = ($Access_db{$field} or $Answer{$record[$D{$field}]});
		print "$Access_db{$field}($Answer{$Access_db{$field}})\n" if $debug;
	    }
	    print "\n" if $debug;
	}
	else {
	    $Rules{'db'} = "no matching rule";
	}
    }
    # -------------------------------
    #  setting privileges to db-priv
    # -------------------------------
    if ($found_entry) {
	foreach $field (keys(%Access)) {
	    $Access{$field} = ($Access{$field} or $Access_db{$field});
	}
	print "Rights after parsing db-table..:\n" if ($DEBUG);
	Print_Access_rights() if ($DEBUG);
    }
    else {
	print "Rights after parsing db-table..:\n" if ($DEBUG);
	print "NO restriction found in db-table!!\n" if ($DEBUG);
	foreach $field (keys(%Access)) {
	    $Access{$field} = 1;
	}
	Print_Access_rights() if ($DEBUG);
    }
    return 1;
}

# ==========================================================
# sub Get_grant_from_host:
# ==========================================================
sub Get_grant_from_host {
    my $debug=0;
    $debug=$DEBUG if ($DEBUG);

    if ($process_host_table) {
	my $found_entry=0;
	my $CONNECT = ConnectAs();
	my $where = "";
	my $sql   = "$MYSQL $ACCESS_DB $CONNECT -s -e \"select * from $ACCESS_H $where;\"";
	open(HOST,"$sql |");
	my $record=<HOST>; #skip fieldnames
	while ($record=<HOST> and !$found_entry) {
	    $full_access='';
	    print "host: $record" if ($debug);
	    chop($record);
	    @record=split(/\t/,$record);
	    # check host and db
	    # with possible wildcards in field
	    # replace mysql-wildcards by reg-wildcards
	    my $host_tpl = SQL2Reg($record[0]);
	    my $db_tpl   = SQL2Reg($record[1]);
	    print "\thost_tpl : $record[0] -> $host_tpl\n" if ($debug>2);
	    print "\tdb_tpl: $record[1] -> $db_tpl\n" if ($debug>2);
	    if ($debug>1) {
		if ( Is_localhost($host_tpl)
		    or MatchTemplate($host_tpl,$host_name)
		    or MatchTemplate($host_tpl,$host_ip) )
		{ print "\t* host [$host_name,$host_ip] matched with rule [$host_tpl]\n"; }
		else
		{ print "\t* host doesn't match\n"; }
		if ( MatchTemplate($db_tpl,$db) )
		{ print "\t* db [$db] matched with rule [$db_tpl]\n"; }
		else
		{ print "\t* db doesn't match \n";}
	    }

	    if ( ( Is_localhost($host_tpl)
		  or MatchTemplate($host_tpl,$host_name)
		  or MatchTemplate($host_tpl,$host_ip) )
		and ( MatchTemplate($db_tpl,$db) ) ) {
		$found_entry=1;
		$Rules{'host'} = $record;
		foreach $field (keys(%H)) {
		    printf "\t> %-15s: input=$record[$H{$field}]  =>  $Access_host{$field}($Answer{$Access_host{$field}}) -> ",$field if ($debug);
		    $Access_host{$field} = ($Access_host{$field} or $Answer{$record[$H{$field}]});
		    print "$Access_host{$field}($Answer{$Access_host{$field}})\n" if ($debug);
		}
		print "\n" if ($debug);
	    }
	    else {
		$Rules{'host'} = "no matching rule";
	    }
	}
	if ($found_entry) {
	    foreach $field (keys(%Access)) {
		$Access{$field} = ($Access_host{$field} and $Access{$field});
	    }
	    print "Rights after parsing host-table..:\n" if ($debug);
	    Print_Access_rights() if ($debug);

	}
	else {
	    print "Rights after parsing host-table..:\n" if ($debug);
	    print "NO restrictions found in the host-table!!\n" if ($debug);
	    Print_Access_rights() if ($debug);

	}
    }
    else {
	$Rules{'host'} = 'not processed because host-field is not empty in user-table';
    }
    return 1;
}

# ==========================================================
# sub Get_grant_from_user:
# ==========================================================
sub Get_grant_from_user {
    my $debug=0;
    $debug=$DEBUG if ($DEBUG);
    my $found_entry=0;
    open(USER,"< $tmpfile");
    while ($record=<USER> and !$found_entry) {
	$full_access='';
	print "user: $record" if ($debug);
	chop($record);
	@record=split(/\t/,$record);
	# check host and db
	# with possible wildcards in field
	# replace mysql-wildcards by reg-wildcards
	my $host_tpl = SQL2Reg($record[0]);
#	my $user_tpl = SQL2Reg($record[1]);
        my $user_tpl = $record[1];
#	my $passwd   = SQL2Reg($record[2]);
	my $passwd   = $record[2];
	print "\thost_tpl : $record[0] -> $host_tpl\n" if ($debug>1);
	print "\tuser_tpl : $record[1] -> $user_tpl\n" if ($debug>1);
	print "\tpassword : $record[2] -> \n" if ($debug>1);
	if ($debug>1) {
	    if ( Is_localhost($host_tpl)
                 or MatchTemplate($host_tpl,$host_name)
                 or MatchTemplate($host_tpl,$host_ip) )
	    { print "\t* host  [$host_name,$host_ip] matched with rule [$host_tpl]\n"; }
	    else			   { print "\t* host doesn't match\n"; }
	    if ( MatchTemplate($user_tpl,$user) )
	    { print "\t* user  [$user] matched with rule [$user_tpl]\n"; }
	    else                       { print "\t* user doesn't match\n"; }
	    if ($passwd ne "" )        { print "\tPassword required\n"; }
	    else                       { print "\tNo password required\n"; }
	}

	if ( ( Is_localhost($host_tpl)
               or  MatchTemplate($host_tpl,$host_name)
               or  MatchTemplate($host_tpl,$host_ip) )
             and ( MatchTemplate($user_tpl,$user) ) ) {
	    $found_entry=1;
	    $Rules{'user'} = $record;
	    if ($passwd ne "") { $passwd_info="NOTE:\tA password is required for user '$user'"; }
            else               { $passwd_info =~ s/\$user/$user/; }
	    foreach $field (keys(%U)) {
		printf "\t> %-15s: input=$record[$U{$field}]  =>  $Access_user{$field}($Answer{$Access_user{$field}}) -> ",$field if $DEBUG;
		$Access_user{$field} = ($Access_user{$field} or $Answer{$record[$U{$field}]});
		print "$Access_user{$field}($Answer{$Access_user{$field}})\n" if $DEBUG;
	    }
	    print "\n" if $debug;
	}
	else {
	    $Rules{'user'}="no matching rule";
	}
    }
    # --------------------------------------------
    #  user-priv may be over-riden by db-priv,
    #  ie. user->N + db->Y ==> Y
    #      user->N + db->N ==> N
    #      user->Y + db->N ==> N
    #      user->Y + db->Y ==> Y
    # --------------------------------------------
    foreach $field (keys(%Access)) {
	$Access{$field} = ($Access_user{$field} or $Access{$field});
    }
    print "Rights after parsing user-table..:\n" if ($DEBUG);
    Print_Access_rights() if ($DEBUG);
    return 1;
}

# =====================================
# sub header:
#  print header info
# =====================================
sub Header {
    my $header;
    $header="$script Version $VERSION\n"
           ."By AIV-RUG, by Yves Carlier (Yves.Carlier\@rug.ac.be)\n"
           ."This software comes with ABSOLUTELY NO WARRENTY.\n";
    return $header;
}

# =====================================
# sub help:
#  print some information on STDOUT
# =====================================
sub help {
    print "\n";
    print Header();
    print "\n";
    print "Usage: $script OPTIONS [host] [user] [db]\n";
    print "\n";
    print "  -?, --help           display this help and exit\n";
    print "  -u, --user=#         user for login\n";
    print "  -h, --host=#         connection from host\n";
    print "  -d, --db=#           connection to database\n";
    print "  -U, --superuser=#    connect as superuser\n";
    print "  -P, --spassword=#    password for superuser\n";
    print "\n";
    print "  At least the user and the db must be given\n";
    print "  If no host is given, 'localhost' is assumed\n";
    print "\n";
    return 1;
}

# ======================================
# sub ConnectAs
#  if a superuser and a superpasswd is
#  given, use this to connect to the
#  access-rights DB.
# ======================================
sub ConnectAs {
    my $debug=0;
    my $connect="";
    if ($superuser) { $connect .= " -u $superuser"; }
    if ($spassword) { $connect .= " -p $spassword"; }
    print "Connect as: $connect\n" if ($debug);
    return $connect;
}

# ======================================
# sub HaveReadAccess
#  Does the user have read-access to
#  the mysql-db
# ======================================
sub HaveReadAccess {
    my $debug = 0;
    my $CONNECT = ConnectAs();
    my $SQL   = "$MYSQL $ACCESS_DB $CONNECT -e \"show tables\"";
    my $error = 'ERROR: Access denied';
    open(TEST,"$SQL 2>&1 |");
    $line = <TEST>;
    print "connect: $line" if ($debug);
    close(TEST);
    if ($line =~ /$error/) { return 0;}
    else                   { return 1;}
}

# ======================================
# sub Print_Access_rights:
#  print the access-rights on STDOUT
# ======================================
sub Print_Access_rights {
    my $column=2;

    # print table of access-rights
    print "\n";
    print "Access-rights\n";
    print "for USER '$user', at HOST '$host', to DB '$db'\n";
    print      "\t+-----------------+---+\t+-----------------+---+";
    foreach $field (keys(%Access)) {
	if ($column==2) { print "\n\t"; $column=1;}
	else            { print "\t"; $column=2;}
	printf "| %-15s | %s |",$field,$Answer{$Access{$field}},$Access{$field};
    }
    print      "\n";
    print      "\t+-----------------+---+\t+-----------------+---+\n";

    # inform if a password is necc.
    if ($passwd_info ne "") { print "$passwd_info !!\n"; }

    # inform if there are no rules ==> full access for everyone.
    if ($full_access ne "") { print "$full_access !!\n"; }

    # print the rules used
    print "\n";
    print "The following rules are used:\n";
    foreach $field (keys(%Rules)) {
	$Rules{$field} =~ s/\t/','/g;
	printf " %-5s : '%s'\n",$field,$Rules{$field};
    }
    return 1;
}

# ======================================
# sub Print_Error_Access:
# ======================================
sub Print_Error_Access {
    print Header();
    print "\n";
    print "Sorry,\n";
    print "You do not have enough access-rights to read the access-rights database\n";
    if ($superuser) {
       print "OR maybe the supplied superuser-id is invalid\n";
    }
    if ($spassword) {
       print "OR maybe the supplied password for the superuser is invalid\n";
    }
    print "\n";
    return 1;
}

# ======================================
# sub IP2Name
#  return the Name with the corr. IP-nmbr
#  (no aliases yet!!)
# ======================================
sub IP2Name {
    my ($ip) = @_;
    my ($name) = @_;
    my $debug = 0;
    print "DEBUG IP2Name():\n" if $debug;
    my ($a,$b,$c,$d) = split(/\./,$ip);
    print "* ip=$a.$b.$c.$d\n" if $debug;
    if ( int($a)==0 and int($b)==0 and int($c)==0 and int($d)==0 ) {
	# not an ip-number
	print "not an ip-number, returning $ip\n" if $debug;
	return $ip;
    }
    $ip = pack "C4",$a,$b,$c,$d;
    print "* ==>>$ip<<\n" if $debug;
    @addrs=(); #dummy
    ($name,$aliases,$addrtype,$length,@addrs) = gethostbyaddr($ip, AF_INET);
    print "* name=$name\n" if $debug;
    print "* aliases=$aliases\n" if $debug;
    print "* addrtype=$addrtype\n" if $debug;
    print "* length=$length\n" if $debug;
    if ($name ne "") { return "$name"; }
    else             { return undef; }
}

# ======================================
# sub Name2IP
#  return the IP-number of the host
# ======================================
sub Name2IP {
    my ($name) = @_;
    my $debug = 0;
    print "DEBUG Name2IP():\n" if $debug;
    print "* name=$name\n" if $debug;
    my ($name,$aliases,$addrtype,$length,@addrs) = gethostbyname($name);
    print "* aliases=$aliases\n" if $debug;
    print "* addrtype=$addrtype\n" if $debug;
    print "* length=$length\n" if $debug;
    my ($a,$b,$c,$d) = unpack('C4',$addrs[0]);
    my $ip = "$a.$b.$c.$d";
    print "* ip=$ip\n" if $debug;
    if ($ip ne "") { return "$ip"; }
    else           { return undef; }
}

# ========================================
# sub LocalHost
#  some special action has to be taken for
#  the localhost
# ========================================
sub LocalHost {
    my $host = hostname();
    my $debug = 0;
    print "localhost = $host\n" if $debug;
    return $host;
}

# ========================================
# check if the given hostname (or ip)
# corresponds with the localhost
# ========================================
sub Is_localhost {
    my ($host_tpl) = @_;
    my $debug=0;
    if (($host_name eq $localhost) or ($host_ip eq $local_ip)) {
	print "Checking for localhost\n" if $debug;
        print " 'localhost' =?= $host_tpl\n" if $debug;
	if ('localhost' =~ /$host_tpl/ ) { return 1;}
	else			         { return 0;}
    }
    else {
	print "Not checking for localhost\n" if $debug;
	return 0;
    }
}

# ============================================
# translage SQL-expressions to Reg-expressions
# ============================================
sub SQL2Reg {
    my ($expr) = @_;
    my $debug=0;
    print "SQL2Reg: $expr " if $debug;
    $expr  =~ s/\./\\./g;
    $expr  =~ s/\\%/!@#$YC#@!/g;
    $expr  =~ s/%/.*/g;
    $expr  =~ s/!@#$YC#@!/%/g;
    $expr  =~ s/\\_/!@#$YC#@!/g;
    $expr  =~ s/_/.+/g;
    $expr  =~ s/!@#$YC#@!/_/g;
    print " --> $expr\n" if $debug;
    return $expr;
}

# =============================================
# match a given string with a template
# =============================================
sub MatchTemplate {
    my ($tpl,$string) = @_;
    my $match=0;
    my $debug=0;
    print "MatchTemplate: $tpl <-> $string : " if ($debug);
    if ($string=~ /^$tpl$/ or $tpl eq '') { $match=1; }
    else                                  { $match=0;}
    print "result=$match\n" if ($debug);
    return $match;
}
