/* Copyright Abandoned 1996 TCX DataKonsult AB & Monty Program KB & Detron HB
   This file is public domain and comes with NO WARRANTY of any kind */

#include <global.h>
#include <errno.h>
#include <my_pthread.h>
#include <signal.h>
#ifdef HAVE_SYNCH_H
#include <synch.h>
#endif
#include <my_sys.h>
#include <m_string.h>
#include <queues.h>
#include "thr_alarm.h"

#ifndef ETIME 
#define ETIME ETIMEDOUT 
#endif

typedef struct st_alarm {
  ulong expire_time;
  int alarmed;					/* 1 when alarm is due */
  pthread_t thread;
} ALARM;


static pthread_mutex_t LOCK_alarm= PTHREAD_MUTEX_INITIALIZER;
static pthread_t signal_thread;
static sigset_t full_signal_set;
static QUEUE alarm_queue;
static alarm_aborted=1;

#ifdef USE_ALARM_THREAD
static pthread_cond_t COND_alarm=PTHREAD_COND_INITIALIZER;
static void *alarm_thread(void *arg);
#define reschedule_alarms() pthread_cond_signal(&COND_alarm)
#else
#define reschedule_alarms() pthread_kill(signal_thread,THR_SERVER_ALARM)
#endif

#if THR_CLIENT_ALARM != SIGALRM || defined(USE_ALARM_THREAD)
static sig_handler thread_alarm(int sig __attribute__((unused)));
#endif

static int compare_ulong(byte *a_ptr,byte* b_ptr)
{
  ulong a=*((ulong*) a_ptr),b= *((ulong*) b_ptr);
  return (a < b) ? -1  : (a == b) ? 0 : 1;
}

void init_thr_alarm(void)
{
  sigset_t s;
  DBUG_ENTER("init_thr_alarm");
  alarm_aborted=0;
  init_queue(&alarm_queue,100,offsetof(ALARM,expire_time),0,compare_ulong);
  sigfillset(&full_signal_set);			/* Neaded to block signals */
#if THR_CLIENT_ALARM != SIGALRM || defined(USE_ALARM_THREAD)
  sigset(THR_CLIENT_ALARM,thread_alarm);	/* int. thread system calls */
#endif
  sigemptyset(&s);
  sigaddset(&s, THR_SERVER_ALARM);
  signal_thread=pthread_self();
#if defined(USE_ALARM_THREAD)
  {
    pthread_attr_t thr_attr;
    pthread_attr_init(&thr_attr);
    pthread_attr_setscope(&thr_attr,PTHREAD_SCOPE_SYSTEM);
    pthread_attr_setdetachstate(&thr_attr,PTHREAD_CREATE_DETACHED);
    pthread_attr_setstacksize(&thr_attr,8196);    

    my_pthread_attr_setprio(&thr_attr,100);    	/* Very high priority */
    VOID(pthread_create(&signal_thread,&thr_attr,alarm_thread,NULL));
    VOID(pthread_attr_destroy(&thr_attr));
  }
#elif defined(USE_ONE_SIGNAL_HAND)
  pthread_sigmask(SIG_BLOCK, &s, NULL);		/* used with sigwait() */
  sigset(THR_SERVER_ALARM,process_alarm);
#else
  pthread_sigmask(SIG_UNBLOCK, &s, NULL);
  sigset(THR_SERVER_ALARM,process_alarm);
#endif
  DBUG_VOID_RETURN;
}

/*
** Request alarm after sec seconds.
** A pointer is returned with points to a non-zero int when the alarm has been
** given. This can't be called from the alarm-handling thread.
** Returns 0 if no more alarms are allowed (aborted by process)
*/

int *thr_alarm(uint sec)
{
  ALARM *alarm_data;
  ulong now;
  sigset_t old_mask;
  my_bool reshedule;
  DBUG_ENTER("thr_alarm");
  DBUG_PRINT("enter",("thread: %lx  sec: %d",pthread_self(),sec));

  now=(ulong) time((time_t*) 0);
  pthread_sigmask(SIG_BLOCK,&full_signal_set,&old_mask);
  pthread_mutex_lock(&LOCK_alarm);	/* Lock from threads & alarms */
  if (alarm_aborted)
  {					/* No signal thread */
    pthread_mutex_unlock(&LOCK_alarm);
    pthread_sigmask(SIG_SETMASK,&old_mask,NULL);
    DBUG_RETURN(0);
  }
  if (alarm_queue.elements == alarm_queue.max_elements)
  {
    fprintf(stderr,"Warning: thr_alarm queue is full\n");
    pthread_mutex_unlock(&LOCK_alarm);
    pthread_sigmask(SIG_SETMASK,&old_mask,NULL);
    DBUG_RETURN(0);
  }
  reshedule= (!alarm_queue.elements ||
	      (int) (((ALARM*) queue_top(&alarm_queue))->expire_time - now) >
	      (int) sec);

  alarm_data=(ALARM*) my_malloc(sizeof(ALARM),MYF(0));
  alarm_data->expire_time=now+sec;
  alarm_data->alarmed=0;
  alarm_data->thread=pthread_self();
  queue_insert(&alarm_queue,(byte*) alarm_data);

  /* Reschedule alarm if the current one has more than sec left */
  if (reshedule)
  {
    if (pthread_equal(pthread_self(),signal_thread))
      alarm(sec);
    else
      reschedule_alarms();			/* Reschedule alarms */
  }
  pthread_mutex_unlock(&LOCK_alarm);
  pthread_sigmask(SIG_SETMASK,&old_mask,NULL);
  DBUG_RETURN(&alarm_data->alarmed);
}


/*
** Remove alarm from list of alarms
*/

void thr_end_alarm(int *alarmed)
{
  ALARM *alarm_data;
  sigset_t old_mask;
  uint i;
  DBUG_ENTER("thr_end_alarm");

  pthread_sigmask(SIG_BLOCK,&full_signal_set,&old_mask);
  pthread_mutex_lock(&LOCK_alarm);

  alarm_data= (ALARM*) ((byte*) alarmed - offsetof(ALARM,alarmed));
  for (i=0 ; i < alarm_queue.elements ; i++)
  {
    if ((ALARM*) queue_element(&alarm_queue,i) == alarm_data)
    {
      my_free(queue_remove(&alarm_queue,i),MYF(0));
      break;
    }
  }
  if (alarm_aborted && !alarm_queue.elements)
    delete_queue(&alarm_queue);
  pthread_mutex_unlock(&LOCK_alarm);
  pthread_sigmask(SIG_SETMASK,&old_mask,NULL);
  DBUG_VOID_RETURN;
}

	/*
	  Come here when some alarm in queue is due.
	  Mark all alarms with are finnished in list.
	  Shedule alarms to be sent again after 1-10 sec (many alarms at once)
	  If alarm_aborted is set then all alarms are given and resent
	  every second.
	  */

sig_handler process_alarm(int sig __attribute__((unused)))
{
  sigset_t old_mask;
  ALARM *alarm_data;
  DBUG_ENTER("process_alarm");
  if (!pthread_equal(pthread_self(),signal_thread))
  {
#ifdef MAIN
    printf("thread_alarm\n"); fflush(stdout);
#endif
#ifdef DONT_REMEMBER_SIGNAL
    sigset(THR_CLIENT_ALARM,process_alarm);		/* int. thread system calls */
#endif
    DBUG_VOID_RETURN;
  }
#ifdef MAIN
  printf("process_alarm\n"); fflush(stdout);
#endif
#ifndef USE_ALARM_THREAD
  pthread_sigmask(SIG_SETMASK,&full_signal_set,&old_mask);
  pthread_mutex_lock(&LOCK_alarm);
#endif
  if (alarm_queue.elements)
  {
    if (alarm_aborted)
    {
      uint i;
      for (i=0 ; i < alarm_queue.elements ;)
      {
	alarm_data=(ALARM*) queue_element(&alarm_queue,i);
	alarm_data->alarmed=1;			/* Info to thread */
	if (pthread_equal(alarm_data->thread,signal_thread) ||
	    pthread_kill(alarm_data->thread, THR_CLIENT_ALARM))
	{
	  queue_remove(&alarm_queue,i);		/* No thread. Remove alarm */
	}
	else
	  i++;					/* Signal next thread */
      }
#ifndef USE_ALARM_THREAD
      if (alarm_queue.elements)
	alarm(1);				/* Signal soon again */
#endif
    }
    else
    {
      ulong now=(ulong) time((time_t*) 0);
      ulong next=now+10-(now%10);
      while ((alarm_data=(ALARM*) queue_top(&alarm_queue))->expire_time <= now)
      {
	alarm_data->alarmed=1;			/* Info to thread */
	if (pthread_equal(alarm_data->thread,signal_thread) ||
	    pthread_kill(alarm_data->thread, THR_CLIENT_ALARM))
	{
	  queue_remove(&alarm_queue,0);		/* No thread. Remove alarm */
	  if (!alarm_queue.elements)
	    break;
	}
	else
	{
	  alarm_data->expire_time=next;
	  queue_replaced(&alarm_queue);
	}
      }
#ifndef USE_ALARM_THREAD
      if (alarm_queue.elements)
	alarm((uint) (alarm_data->expire_time-now));
#endif
    }
  }
#ifndef USE_ALARM_THREAD
#if defined(DONT_REMEMBER_SIGNAL)
  sigset(THR_SERVER_ALARM,process_alarm);
#endif
  pthread_mutex_unlock(&LOCK_alarm);
  pthread_sigmask(SIG_SETMASK,&old_mask,NULL);
#endif
  DBUG_VOID_RETURN;
}


/*
** Shedule all alarms now.
** When all alarms are given, Free alarm memory and don't allow more alarms.
*/

void end_thr_alarm(void)
{
  DBUG_ENTER("end_thr_alarm");
  pthread_mutex_lock(&LOCK_alarm);
  if (!alarm_aborted)
  {
    alarm_aborted=1;				/* mark aborted */
    if (!alarm_queue.elements)
      delete_queue(&alarm_queue);
    if (pthread_equal(pthread_self(),signal_thread))
      alarm(1);					/* Shut down everything soon */
    else
      reschedule_alarms();
  }
  pthread_mutex_unlock(&LOCK_alarm);
  DBUG_VOID_RETURN;
}

/*
**  This is here for thread to get interruptet from read/write/fcntl
**  ARGSUSED
*/

#if THR_CLIENT_ALARM != SIGALRM || defined(USE_ALARM_THREAD)
static sig_handler thread_alarm(int sig)
{
#ifdef MAIN
  printf("thread_alarm\n"); fflush(stdout);
#endif
#ifdef DONT_REMEMBER_SIGNAL
  sigset(sig,thread_alarm);		/* int. thread system calls */
#endif
}
#endif


#ifdef HAVE_TIMESPEC_TS_SEC
#define tv_sec ts_sec
#define tv_nsec ts_nsec
#endif

/* set up a alarm thread with uses 'sleep' to sleep betwen alarms */

#ifdef USE_ALARM_THREAD
static void *alarm_thread(void *arg __attribute__((unused)))
{
  int error;
  struct timespec abstime;
#ifdef MAIN
  puts("Starting alarm thread");
#endif
  pthread_mutex_lock(&LOCK_alarm);
  for (;;)
  {
    if (alarm_queue.elements)
    {
      ulong sleep_time,now=time((time_t*) 0);
      if (alarm_aborted)
	sleep_time=now+1;
      else
	sleep_time= ((ALARM*) queue_top(&alarm_queue))->expire_time;
      if (sleep_time > now)
      {
	abstime.tv_sec=sleep_time;
	abstime.tv_nsec=0;
	if ((error=pthread_cond_timedwait(&COND_alarm,&LOCK_alarm,&abstime)) &&
	    error != ETIME && error != ETIMEDOUT)
	{
#ifdef MAIN
	  printf("Got error: %d from ptread_cond_timedwait\n",error);
#endif
	}
      }
    }
    else if (alarm_aborted)
      break;
    else if ((error=pthread_cond_wait(&COND_alarm,&LOCK_alarm)))
    {
#ifdef MAIN
      printf("Got error: %d from ptread_cond_wait\n",error);
#endif
    }
    process_alarm(0);
  }
  signal_thread=0;				/* For easy debugging */
  pthread_mutex_unlock(&LOCK_alarm);
  pthread_exit(0);
  return 0;					/* Impossible */
}
#endif


#ifdef MAIN

static pthread_cond_t COND_thread_count= PTHREAD_COND_INITIALIZER;
static pthread_mutex_t LOCK_thread_count=PTHREAD_MUTEX_INITIALIZER;
static uint thread_count;

static void *test_thread(void *arg)
{
  int i,param=*((int*) arg),*got_alarm,wait_time;
  time_t start_time;
  fd_set fd;
  FD_ZERO(&fd);

  printf("Tread %d (%lx) started\n",param,pthread_self()); fflush(stdout);
  for (i=1 ; i <= 10 ; i++)
  {
    wait_time=param ? 11-i : i;
    start_time=time((time_t*) 0);
    if (!(got_alarm=thr_alarm(wait_time)))
    {
      printf("Thread: %lx  Alarms aborted\n",pthread_self());
      break;
    }
    if (wait_time == 3)
    {
      printf("Thread: %lx  Simulation of no alarm needed\n",pthread_self());
      fflush(stdout);
    }
    else
    {
      while (!*got_alarm)
      {
	printf("Thread: %lx  Waiting %d sec\n",pthread_self(),wait_time);
	select(0,&fd,0,0,0);
      }
      if (wait_time == 7)
      {
	printf("Thread: %lx  Simulating alarm miss\n",pthread_self());
	fflush(stdout);
	select(0,&fd,0,0,0);			/* Simulate alarm-miss */
      }
    }
    printf("Thread: %lx  Sleeped for %d (%d) sec\n",pthread_self(),
	   time((time_t*) 0)-start_time, wait_time); fflush(stdout);
    thr_end_alarm(got_alarm);
    fflush(stdout);
  }
  pthread_mutex_lock(&LOCK_thread_count);
  thread_count--;
  VOID(pthread_cond_signal(&COND_thread_count)); /* Tell main we are ready */
  pthread_mutex_unlock(&LOCK_thread_count);
  free((gptr) arg);
  return 0;
}

pthread_mutex_t LOCK_signal_init=PTHREAD_MUTEX_INITIALIZER;

static void *signal_hand(void *arg __attribute__((unused)))
{
  sigset_t set;
  int sig,error,err_count=0;;

  init_thr_alarm();				/* Setup alarm handler */
  VOID(pthread_mutex_unlock(&LOCK_signal_init));
  sigfillset(&set);				/* Catch all signals */
#ifndef USE_ONE_SIGNAL_HAND
  VOID(sigdelset(&set,THR_SERVER_ALARM));	/* For alarms */
#endif
  for(;;)
  {
    while ((error=sigwait(&set,&sig)) == EINTR)
      printf("sigwait restarted\n");
    if (error)
    {
      fprintf(stderr,"Got error %d from sigwait\n",error);
      if (err_count++ > 5)
	exit(1);				/* Too many errors in test */
      continue;
    }
#ifdef USE_ONE_SIGNAL_HAND
    if (sig != THR_SERVER_ALARM)
#endif
      printf("Main thread: Got signal %d\n",sig);
    switch (sig) {
    case SIGINT:
    case SIGQUIT:
    case SIGKILL:
    case SIGTERM:
    case SIGHUP:
      printf("Aborting nicely\n");
      end_thr_alarm();
      break;
    case SIGTSTP:
      printf("Aborting\n");
      exit(1);
      break;
#ifdef USE_ONE_SIGNAL_HAND
     case THR_SERVER_ALARM:
       process_alarm(sig);
      break;
#endif
    }
  }
}


int main(int argc __attribute__((unused)),char **argv __attribute__((unused)))
{
  pthread_t tid;
  pthread_attr_t thr_attr;
  int i,*param;
  sigset_t set;
  DBUG_ENTER("main");
  DBUG_PROCESS(argv[0]);
  if (argc > 1 && argv[1][0] == '-' && argv[1][1] == '#')
    DBUG_PUSH(argv[1]+2);

  /* Start a alarm handling thread */
  sigfillset(&set);
  VOID(sigdelset(&set,THR_CLIENT_ALARM));
  VOID(pthread_sigmask(SIG_SETMASK,&set,NULL));

  pthread_attr_init(&thr_attr);
  pthread_attr_setscope(&thr_attr,PTHREAD_SCOPE_SYSTEM);
  pthread_attr_setdetachstate(&thr_attr,PTHREAD_CREATE_DETACHED);
  pthread_attr_setstacksize(&thr_attr,65536L);

  VOID(pthread_mutex_lock(&LOCK_signal_init));	/* linux doesn't have sema */
  pthread_create(&tid,&thr_attr,signal_hand,NULL);
  VOID(pthread_mutex_lock(&LOCK_signal_init));	/* wait for signal handler */
  VOID(pthread_mutex_unlock(&LOCK_signal_init)); /* don't use this anymore */

#ifdef HAVE_THR_SETCONCURRENCY
  VOID(thr_setconcurrency(3));
#endif
  pthread_attr_setscope(&thr_attr,PTHREAD_SCOPE_PROCESS);
  printf("Main thread: %lx\n",pthread_self());
  for (i=0 ; i < 2 ; i++)
  {
    param=(int*) malloc(sizeof(int));
    *param= i;
    pthread_mutex_lock(&LOCK_thread_count);
    if (pthread_create(&tid,&thr_attr,test_thread,(void*) param))
    {
      printf("Can't create thread %d\n",i);
      exit(1);
    }
    thread_count++;
    pthread_mutex_unlock(&LOCK_thread_count);
  }

  pthread_attr_destroy(&thr_attr);
  pthread_mutex_lock(&LOCK_thread_count);
  while (thread_count)
  {
    VOID(pthread_cond_wait(&COND_thread_count,&LOCK_thread_count));
    if (thread_count == 1)
    {
      printf("Calling end_thr_alarm. This should cancel the last thread\n");
      end_thr_alarm();
    }
  }
  pthread_mutex_unlock(&LOCK_thread_count);
  puts("end");
  DBUG_RETURN(0);
}
#endif
