/*
 *  inputbox.c -- implements the input box
 *
 *  AUTHOR: Savio Lam (lam836@cs.cuhk.hk)
 *
 *  hacked/modified 10 Jan. 2000 by Thomas Spahni (thomas@spahni.ch)
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "dialog.h"

unsigned char dialog_input_result[MAX_LEN + 1];

/*
 * scrolling takes place when the cursor approaches 1/n -th of
 * input field width. Set n here:
 */

#define SCROLLPOINT 5


/*
 * Check for national characters
 * allowable range for typed characters is isprint()
 * This gets your national characters through even if your LOCALE
 * is not defined.
 */

#ifndef LOCALE
#define	NATIONAL_CHARACTERS \
("")
int isnatchar (int c)
{
    if ( (void) strchr(NATIONAL_CHARACTERS,c)) {
	return c;
    } else {
	return FALSE;
    }
}
#else
    int isnatchar (int c) { return FALSE; }
#endif
                
/*
 * Display a dialog box for inputing a string
 */
         
int
dialog_inputbox (const char *title, const char *prompt, int height, int width,
		 const char *init)
{
    int i, x, y, box_y, box_x, box_width, scrlpt, scrlamt;
    int shadowheight;
    int input_x = 0, scroll = 0, key = 0, button = -1;
    int crpos_x = 0, partlyscroll = 0;
    unsigned char *instr = dialog_input_result;
    WINDOW *dialog;

    /* center dialog box on screen */
    x = (COLS - width) / 2;
    y = 2 + ((LINES - 2 - height) / 2);

#ifdef HAVE_NCURSES
    if (use_shadow) {
	shadowheight = height;
	/* avoid to draw a shadow below the screen for BIG boxes */
	if ((y + height) > LINES - 1) {
	    shadowheight = LINES - 1 - y;
	}
	draw_shadow (stdscr, y, x, shadowheight, width);
    }
#endif
    dialog = newwin (height, width, y, x);
    mouse_setbase (x, y);
    keypad (dialog, TRUE);

    draw_box (dialog, 0, 0, height, width, dialog_attr, border_attr);
    wattrset (dialog, border_attr);
    wmove (dialog, height - 3, 0);
    waddch (dialog, ACS_LTEE);
    for (i = 0; i < width - 2; i++)
	waddch (dialog, ACS_HLINE);
    wattrset (dialog, dialog_attr);
    waddch (dialog, ACS_RTEE);
    wmove (dialog, height - 2, 1);
    for (i = 0; i < width - 2; i++)
	waddch (dialog, ' ');

    if (title != NULL) {
	wattrset (dialog, title_attr);
	wmove (dialog, 0, (width - strlen (title)) / 2 - 1);
	waddch (dialog, ' ');
	waddstr (dialog, title);
	waddch (dialog, ' ');
    }
    wattrset (dialog, dialog_attr);
    print_autowrap (dialog, prompt, width - 2, 1, 3);

    /* Draw the input field box */
    box_width = width - 6;
    getyx (dialog, y, x);
    box_y = y + 2;
    box_x = (width - box_width) / 2;
    mouse_mkregion (y + 1, box_x - 1, 3, box_width + 2, 'i');
    draw_box (dialog, y + 1, box_x - 1, 3, box_width + 2,
	      border_attr, dialog_attr);

    x = width / 2 - 11;
    y = height - 2;
    print_button (dialog, "Cancel", y, x + 14, FALSE);
    print_button (dialog, "  OK  ", y, x, TRUE);

    /* Set up the initial value */
    wmove (dialog, box_y, box_x);
    wattrset (dialog, inputbox_attr);
    if (!init)
	instr[0] = '\0';
    else
	strcpy (instr, init);
    input_x = strlen (instr);
    if (input_x >= box_width) {
	scroll = input_x - box_width + 1;
	input_x = box_width - 1;
	for (i = 0; i < box_width - 1; i++)
	    waddch (dialog, instr[scroll + i]);
    } else
	waddstr (dialog, instr);
    wmove (dialog, box_y, box_x + input_x);
    crpos_x = input_x;

    scrlpt  = 1 + box_width / SCROLLPOINT ;
    scrlamt = box_width - (2 * scrlpt);

/* some comments:
 * scroll    is the number of characters to the left of the window;
 * input_x   is an offset to the end of text in the current window;
 * crpos_x   is the cursor position. It will be less than
 *           input_x as soon as KEY_LEFT is used;
 */

    wrefresh (dialog);
    while (key != ESC) {
	key = mouse_wgetch (dialog);

	if (button == -1) {	/* Input box selected */
	    switch (key) {
	    case TAB:
	    case KEY_UP:
	    case KEY_DOWN:
	    case M_EVENT + 'i':
	    case M_EVENT + 'o':
	    case M_EVENT + 'c':
		break;
/*
 * hack by tsp: allow a cursor being moved
 * to the left; scroll when cursor reaches
 * window edge minus 1/SCROLLPOINT width.
 */
	    case KEY_LEFT:
		if (crpos_x || scroll) {
		    wattrset (dialog, inputbox_attr);
		    if ((crpos_x == scrlpt) && scroll) {
		        /* near the left edge */
			if (scroll  <= scrlamt) {
			    crpos_x  = crpos_x + scroll - 1;
			    input_x += scroll;
			    scroll   = 0;
			}
			else {
			    scroll   = scroll - scrlamt;
			    crpos_x  = crpos_x + scrlamt - 1;
			    input_x += scrlamt;
			}
			wmove (dialog, box_y, box_x);
			for (i=0; i<box_width; i++)
			    waddch (dialog, instr[scroll + i] ?
				    instr[scroll + i] : ' ');
			wmove (dialog, box_y, crpos_x + box_x);
		    }
		    else {
		        wmove (dialog, box_y, --crpos_x + box_x);
		    }
                    wrefresh (dialog);
		}
                continue;

	    case KEY_RIGHT:
		if ( crpos_x < input_x ) {
		    /* cursor not at right end of the string, can move */
		    wattrset (dialog, inputbox_attr);
		    if (crpos_x > (box_width-scrlpt)) {
			/* near the right edge, should scroll */
			if ((MAX_LEN-scroll-box_width + 1) >= scrlamt) {
			    /* not near end of buffer */
			    scroll  += scrlamt;
			    crpos_x  = crpos_x-scrlamt+1;
			    input_x -= scrlamt;
			} else {
			    partlyscroll = MAX_LEN-scroll-box_width + 2;
			    scroll  += partlyscroll;
			    crpos_x  = crpos_x - partlyscroll + 1;
			    input_x -= partlyscroll;
			}
			wmove (dialog, box_y, box_x);
			for (i=0; i<box_width; i++)
			    waddch (dialog, instr[scroll + i] ?
				instr[scroll + i] : ' ');
			wmove (dialog, box_y, crpos_x + box_x);
		    }
		    else { /* go one step right ; no scrolling needed */
			wmove (dialog, box_y, ++crpos_x + box_x);
		    }
		}
		wrefresh (dialog);
		continue;

	    case KEY_BACKSPACE:
		if (crpos_x || scroll) {
		    /* there is something to the left of the cursor */
		    wattrset (dialog, inputbox_attr);
		    if ((crpos_x == scrlpt) && scroll) {
		        /* near the left edge and can scroll */
			if (scroll <= scrlamt) {
			    /* not much left; scoll all the way */
			    crpos_x  = crpos_x + scroll - 1;
			    input_x += scroll;
			    scroll   = 0;
			}
			else {
			    /* scoll by a fraction of the window */
			    scroll  = scroll - scrlamt;
			    crpos_x = crpos_x + scrlamt - 1;
			    input_x += scrlamt;
			}
		    }
		    else {
			/* no need to scroll */
			crpos_x--;
		    }
		/* scolling done; delete this character */
		/* in the buffer and adjust the display */
		for (i=crpos_x; i<input_x; i++)
		    instr[scroll+i] = instr[scroll+i+1];
		input_x--;
		wmove (dialog, box_y, box_x);
		for (i=0; i<box_width; i++)
		    waddch (dialog, instr[scroll + i] ?
			    instr[scroll + i] : ' ');
		wmove (dialog, box_y, box_x + crpos_x);
		wrefresh (dialog);
		}
		continue;

	    /* DELETE KEY */
	    case KEY_DC:
	    case 127:
	    case CTRL_D:
		if (crpos_x < input_x) {
		    /* there is something to the right of the cursor */
		    /* or under the cursor */
		    wattrset (dialog, inputbox_attr);
		    /* delete this character                */
		    /* in the buffer and adjust the display */
		    for (i=crpos_x; i<input_x; i++)
			instr[scroll+i] = instr[scroll+i+1];
		    input_x--;
		    wmove (dialog, box_y, box_x + crpos_x);
		    for (i=crpos_x; i<box_width; i++)
			waddch (dialog, instr[scroll + i] ?
			    instr[scroll + i] : ' ');
		    wmove (dialog, box_y, box_x + crpos_x);
		    wrefresh (dialog);
		}
		continue;

	    /* Yank the whole line */
	    case CTRL_Y:
	    case KEY_DL:
		wmove (dialog, box_y, box_x);
		wattrset (dialog, inputbox_attr);
		for (i=0; i<box_width; i++)
		    waddch (dialog, ' ');
		for (i=0; i<=MAX_LEN; i++)
		    instr[i] = '\0';
		wmove (dialog, box_y, box_x);
		scroll  = 0;
		crpos_x = 0;
		input_x = 0;
	    continue;

	    case CTRL_K:
		/* delete to end of line */
		for (i=crpos_x; i<=input_x; i++)
		    instr[scroll + i] = '\0';
		wmove (dialog, box_y, box_x + crpos_x);
		wattrset (dialog, inputbox_attr);
		for (i=crpos_x; i<box_width; i++)
		    waddch (dialog, ' ');
		wmove (dialog, box_y, box_x + crpos_x);
		input_x = crpos_x;
	    continue;

	    case KEY_HOME:
		/* go to the start of the line */
		input_x = input_x + scroll;
		scroll  = 0;
		crpos_x = 0;
		wattrset (dialog, inputbox_attr);
		wmove (dialog, box_y, box_x);
		for (i=0; i<(box_width); i++)
		    waddch (dialog, instr[i] ? instr[i] : ' ');
		wmove (dialog, box_y, box_x);
	    continue;

	    case KEY_END:
		/* go to the end of the line */
		input_x = input_x + scroll;
		scroll =  input_x - box_width + 1;
		if (scroll < 0)
		    scroll = 0;
		input_x = input_x - scroll;
		crpos_x = input_x;
		wattrset (dialog, inputbox_attr);
		wmove (dialog, box_y, box_x);
		for (i=0; i<(box_width); i++)
		    waddch (dialog, instr[scroll + i] ?
			instr[scroll + i] : ' ');
		wmove (dialog, box_y, crpos_x + box_x);
	    continue;

	    default:
		if ((key < 0x100 && isprint(key)) || isnatchar(key)) {
		    if (scroll + input_x < MAX_LEN) {
		        /* buffer not yet full */
			wattrset (dialog, inputbox_attr);
			instr[scroll + input_x + 1] = '\0';
                        for (i=input_x; i>crpos_x; i--)
                           instr[scroll+i] = instr[(scroll+i)-1];
			instr[scroll + crpos_x] = key;
			if (crpos_x == box_width - 1) {
			    /* at the end of the window scroll left */
			    /* to make room for the insertion       */
			    scroll++;
			    wmove (dialog, box_y, box_x);
			    for (i=0; i<(box_width); i++)
				waddch (dialog, instr[scroll + i] ?
				    instr[scroll + i] : ' ');
			    wmove (dialog, box_y, crpos_x + box_x);
			}
			else {
			    /* we are scrolling the right part */
			    /* of the buffer to the right      */
			    wmove (dialog, box_y, box_x);
			    for (i=0; i<(box_width); i++)
				waddch (dialog, instr[scroll + i] ?
				    instr[scroll + i] : ' ');
			    crpos_x++;
			    input_x++;
			    wmove (dialog, box_y, crpos_x + box_x);
			}
			wrefresh (dialog);
		    } else {
			beep ();	/* Alarm user about overflow */
			flash ();
		    }
		    continue;
		}
	    }
	}
	switch (key) {
	case 'O':
	case 'o':
	    delwin (dialog);
	    return 0;
	case 'C':
	case 'c':
	    delwin (dialog);
	    return 1;
	case M_EVENT + 'i':	/* mouse enter events */
	case M_EVENT + 'o':	/* use the code for 'UP' */
	case M_EVENT + 'c':
	    button = (key == M_EVENT + 'o') - (key == M_EVENT + 'c');
	case KEY_UP:
	case KEY_LEFT:
	    switch (button) {
	    case -1:
		button = 1;	/* Indicates "Cancel" button is selected */
		print_button (dialog, "  OK  ", y, x, FALSE);
		print_button (dialog, "Cancel", y, x + 14, TRUE);
		wrefresh (dialog);
		break;
	    case 0:
		button = -1;	/* Indicates input box is selected */
		print_button (dialog, "Cancel", y, x + 14, FALSE);
		print_button (dialog, "  OK  ", y, x, TRUE);
		wmove (dialog, box_y, box_x + input_x);
		wrefresh (dialog);
		break;
	    case 1:
		button = 0;	/* Indicates "OK" button is selected */
		print_button (dialog, "Cancel", y, x + 14, FALSE);
		print_button (dialog, "  OK  ", y, x, TRUE);
		wrefresh (dialog);
		break;
	    }
	    break;
	case TAB:
	case KEY_DOWN:
	case KEY_RIGHT:
	    switch (button) {
	    case -1:
		button = 0;	/* Indicates "OK" button is selected */
		print_button (dialog, "Cancel", y, x + 14, FALSE);
		print_button (dialog, "  OK  ", y, x, TRUE);
		wrefresh (dialog);
		break;
	    case 0:
		button = 1;	/* Indicates "Cancel" button is selected */
		print_button (dialog, "  OK  ", y, x, FALSE);
		print_button (dialog, "Cancel", y, x + 14, TRUE);
		wrefresh (dialog);
		break;
	    case 1:
		button = -1;	/* Indicates input box is selected */
		print_button (dialog, "Cancel", y, x + 14, FALSE);
		print_button (dialog, "  OK  ", y, x, TRUE);
		wmove (dialog, box_y, box_x + input_x);
		wrefresh (dialog);
		break;
	    }
	    break;
	case ' ':
	case '\n':
	    delwin (dialog);
	    return (button == -1 ? 0 : button);
	case ESC:
	    break;
	}
    }

    delwin (dialog);
    return -1;			/* ESC pressed */
}
