/******************************************************************************G
	lndb_calls.c

	All the calls to the kernel/ are done here, thus providing the
	upper interface to the clients. Thread safe implementation.

******************************************************************************/

#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>
#include <time.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include "lndb_calls.h"
#include "sync.h"
#include "ret.h"

#define MAX_GROUP_NAME 512

#define PRINT(format, args...) \
	if (fprintf (cpar->reply, format, ## args) < 0)\
		return ECLIENT; else

#define PRINTCLEAN(cleanup, format, args...) \
	if (fprintf (cpar->reply, format, ## args) < 0)\
		{ cleanup; return ECLIENT; } else

#define PUTS(str) \
	if (fputs (str, cpar->reply) == EOF \
	|| fputc ('\n', cpar->reply) == EOF) \
		return ECLIENT; else

#define PUTSR(str) \
	if (fputs (str, reply) == EOF \
	|| fputc ('\n', reply) == EOF) \
		return; else

/*
 * Rationale.
 *  Functions at this level are supposed to be very high level/interace level.
 *  Like `if (func()) return EERROR; else PRINT(...)'
 * The only true jobs are done in the static functions.
 * The kernel knows nothing about streams and clients and thus here only what
 *  needs sending information to a stream is done.
 *
 * The multithreaded syncronization is performed at even higher level.
 * An lndb_call is wants either `Permission to read' or `Permission to write'
 *  even if some part of the call is just readonly.
 */
static inline int id_ok (unsigned int i)
{
	return i < ElTop && element [i];
}

/*
 * Get a group by name. Also performs checks for syntax.
 */
#define MAX_ACCEPT_GN 40
static Group *grp_by_name (Group *cwd, char *name)
{
	char *p, *arrv [MAX_ACCEPT_GN];
	Group *g;
	short int i = 0;

	for (p = name; *p; p++)
		if ((!isalnum (*p) && *p != '.') || *p == ' ') return NULL;

	if (*name == '.')
	{
		g = cwd;
		name++;
	}
	else g = &root;

	for (;;)
	{
		for (; *name == '.'; name++)
			if (i > 0) i--;
			else if (!(g = g->parent)) return NULL;
		if (!*name) break;
		arrv [i++] = name;
		if (!(p = strchr (name,  '.'))) break;
		*p = 0;
		name = p + 1;
	}

	if (i == 0)
		return g;

	arrv [i] = 0;
	return find_group (g, arrv);
}

/*
 * Print the fully qualified group name.
 * We need to reverse the fields; this could be done with recursion but
 * due to thread safe implementation we would need to pass client parameters
 * at each call. Therefore recursion is unrolled.
 */
static int fqgn (Group *g, client_params *cpar)
{
	Group *gp;
	char **gn;
	short int i;

	for (i = 0, gp = g; gp; gp = gp->parent) i++;
	gn = (char**) alloca (i * sizeof (char*));
	for (i = 0; g; g = g->parent)
		gn [i++] = g->node_name;
	if ((i -= 2) >= 0)
	{
		PRINT ("%s", gn [i]);
		while (i--)
			PRINT (".%s", gn [i]);
	}
	return 0;
}

int LNDB_links (unsigned int i, client_params *cpar)
{
	Link *l;

	if (!id_ok (i))
		return EVOIDELEM;

	if ((l = element [i]->link) == NULL)
		return R_NO_LINKS;

	for (; l; l = l->next)
	{
		if (fqgn (l->e->group, cpar))
			return ECLIENT;
		if (l->lid)
			PRINT (" : [%u] / %i\n", l->e->id, l->lid);
		else
			PRINT (" : [%u]\n", l->e->id);
	}

	return R_EOD;
}

int LNDB_sh (unsigned int i, client_params *cpar)
{
	if (!id_ok (i))
		return EVOIDELEM;

	PRINT ("CL: %u\n", element [i]->len);
	data_to_stream (element [i], cpar->reply);
	PUTS ("");

	return R_EOD;
}

int LNDB_find (char *c, client_params *cpar)
{
	unsigned int *ip, i;

	if (c == NULL) return ENOARGS;

	if ((ip = find_cistr (c)) == NULL)
		return R_NOMATCH;

	for (i = 0; ip [i] != FREE_ID; i++)
		PRINTCLEAN (free (ip), " %u\n", ip [i]);

	free (ip);
	return R_EOD;
}

static void dir_r (Group *g, FILE *reply)
{
	PUTSR (g->node_name);
	if (g->less) dir_r (g->less, reply);
	if (g->more) dir_r (g->more, reply);
}

int LNDB_dir (char *c, client_params *cpar)
{
	Group *g;

	if ((g = (c) ? grp_by_name (cpar->cwd, c) : cpar->cwd) == NULL)
		return ENODIR;

	if (g->child)
		dir_r (g->child, cpar->reply);

	return R_EOD;
}

int LNDB_cd (char *c, client_params *cpar)
{
	Group *g;

	if (c == NULL)
	{
		cpar->cwd = &root;
		return R_OKey;
	}

	if ((g = grp_by_name (cpar->cwd, c)) == NULL)
		return ENODIR;

	cpar->cwd = g;
	return R_OKey;
}

int LNDB_pwd (client_params *cpar)
{
	fqgn (cpar->cwd, cpar);
	PUTS ("");
	return R_OKey;
}

int LNDB_locate (char *opts[], char *ptr, unsigned int len,
		 char *pmt, client_params *cpar)
{
	unsigned int *ip, i;

	if (pmt == NULL)
		return ENOARGS;

	if ((ip = locate (opts, ptr, len, pmt)) == NULL)
		return R_NOMATCH;

	if (ip == BAD_PMT)
		return EBADTYPE;

	for (i = 0; ip [i] != FREE_ID; i++)
		PRINTCLEAN (free (ip), " %u\n", ip [i]);

	free (ip);
	return R_EOD;
}

int LNDB_lenof (unsigned int i, client_params *cpar)
{
	if (!id_ok (i))
		return EVOIDELEM;

	PRINT ("%u\n", element [i]->len);
	return R_OKey;
}

int LNDB_groupof (unsigned int i, client_params *cpar)
{
	if (!id_ok (i))
		return EVOIDELEM;

	fqgn (element [i]->group, cpar);
	PUTS ("");
	return R_EOD;
}

int LNDB_ls (char *c, client_params *cpar)
{
	Group *g;
	Element *e;

	if ((g = (c) ? grp_by_name (cpar->cwd, c) : cpar->cwd) == NULL)
		return ENODIR;

	for (e = g->e; e; e = e->ngrp)
		PRINT ("%u\n", e->id);

	return R_EOD;
}

int LNDB_nls (char *c, client_params *cpar)
{
	Group *g;

	if ((g = (c) ? grp_by_name (cpar->cwd, c) : cpar->cwd) == NULL)
		return ENODIR;

	PRINT ("%u\n", g->cnt);

	return R_OKey;
}

int LNDB_link (unsigned int i1, unsigned int i2, short int lid,
	       client_params *cpar)
{
	if (i1 == i2)
		return EBADLINK;

	if (!id_ok (i1) || !id_ok (i2))
		return EVOIDELEM;

	if (find_link (element [i1], element [i2]))
		return EEXISTS;

	link_elems (i1, i2, lid);

	return R_OKey;
}

int LNDB_unlink (unsigned int i1, unsigned int i2, client_params *cpar)
{
	if (!id_ok (i1) || !id_ok (i2))
		return EVOIDELEM;

	if (!find_link (element [i1], element [i2]))
		return R_NOT_FOUND;

	break_link (i1, i2);

	return R_OKey;
}

int LNDB_new (char *c, unsigned int i, client_params *cpar)
{
	unsigned int id;

	if ((id = eexists (c, i)) != FREE_ID)
	{
		PRINT ("%u\n", id);
		return EEXISTS;
	}

	new_element (cpar->cwd, c, i);
	PRINT ("%u\n", id);

	return R_OKey;
}

int LNDB_nlinks (unsigned int e, client_params *cpar)
{
	if (!id_ok (e))
		return EVOIDELEM;

	PRINT ("%u\n", count_links (e));
	return R_OKey;
}

int LNDB_alt (unsigned int e, char *c, unsigned int i, client_params *cpar)
{
	unsigned int id;

	if (!id_ok (e))
		return EVOIDELEM;

	if ((id = eexists (c, i)) != FREE_ID)
	{
		PRINT ("%u\n", id);
		return EEXISTS;
	}

	change_dataof (element [e], c, i);

	return R_OKey;
}

int LNDB_idof (char *c, unsigned int i, client_params *cpar)
{
	unsigned int id;

	if ((id = eexists (c, i)) == FREE_ID)
		return R_NOT_FOUND;

	PRINT ("%u\n", id);
	return R_OKey;
}

int LNDB_typesof (unsigned int i, client_params *cpar)
{
	short int r;
	Element *e;
	char *c;

	if (!id_ok (i))
		return EVOIDELEM;

	e = element [i], r = 0;
	while ((c = show_pmts (e, &r)))
		{ PUTS (c); }

	return R_EOD;
}

int LNDB_del (unsigned int i, client_params *cpar)
{
	extern unsigned int denywrite;

	if (i < denywrite)
		return EDENYWRITE;

	if (!id_ok (i))
		return EVOIDELEM;

	del_elem (i);

	return R_OKey;
}

int LNDB_chgrp (unsigned int i, char *c, client_params *cpar)
{
	Group *g;

	if (!c)
		return ENOARGS;

	if (!id_ok (i))
		return EVOIDELEM;

	if ((g = grp_by_name (cpar->cwd, c)) == NULL)
		return ENODIR;

	if (element [i]->group != g)
		chgrp (element [i], g);

	return R_OKey;
}

static int valid_gname (const char *c)
{
	while (*c)
		if (!isalnum (*c++))
			return 0;

	return 1;
}

int LNDB_mkgrp (char *c, client_params *cpar)
{
	char *p, *arrv [2];
	Group *par;

	if (!c)
		return ENOARGS;

	if ((p = strrchr (c, '.')))
	{
		char ch = *++p;
		*p = 0;
		if ((par = grp_by_name (cpar->cwd, c)) == NULL)
			return ENODIR;
		*p = ch;
	}
	else
	{
		p = c;
		par = &root;
	}

	if (!*p || !valid_gname (p))
		return EBADNAME;

	arrv [0] = p, arrv [1] = NULL;
	if (find_group (par, arrv))
		return EEXISTS;

	add_group (par, p);

	return R_OKey;
}

int LNDB_rengrp (char *og, char *ng, client_params *cpar)
{
	Group *g, *np;
	char *p, *arrv [2];

	if (!ng)
		return ENOARGS;

	if ((g = grp_by_name (cpar->cwd, og)) == NULL)
		return ENODIR;

	if ((p = strrchr (ng, '.')))
	{
		char ch = *++p;
		*p = 0;
		if ((np = grp_by_name (cpar->cwd, ng)) == NULL)
			return ENODIR;
		*p = ch;
	}
	else
	{
		p = ng;
		np = &root;
	}

	// rengrp is special. For example: ``rengrp a.b c.'' will move the
	// group a.b to c.b
	if (*p && !valid_gname (p))
			return EBADNAME;

	arrv [0] = (*p) ? p : g->node_name;
	arrv [1] = NULL;
	if (find_group (np, arrv))
		return EEXISTS;

	return (ch_parent (g, np, (*p) ? p : NULL)) ? R_OKey : EINVDIR;
}

int LNDB_rmgrp (char *c, client_params *cpar)
{
#ifdef MULTITHREADED
	struct sembuf sem_operation = { 0, 0, 0 };
#endif
	Group *g;
	client_params *i;
	int bl = 0;

	if (!c)
		return ENOARGS;

	if ((g = grp_by_name (cpar->cwd, c)) == NULL)
		return ENODIR;

	if (g == &root || g == cpar->cwd)
		return EINVDIR;

	SYNC_CLI
	for (i = first_client; i; i = i->next)
		if (i->cwd == g) {
			bl = 1;
			break;
		}
	SYNC_CF

	if (bl)
		return EDIRUSED;

	if (g->e || g->child)
		return ENOEMPTY;

	del_group (g);

	return R_OKey;
}

int LNDB_set (unsigned int i, const char *c, client_params *cpar)
{
	short int t;

	if (!c)
		return ENOARGS;

	if (!id_ok (i))
		return EVOIDELEM;

	if ((t = pmt_byname (c)) == -1)
		return EBADTYPE;

	return (register_type (element [i], t)) ? R_OKey : EREGFAIL;
}

int LNDB_unset (unsigned int i, const char *c, client_params *cpar)
{
	short int t;

	if (!c)
		return ENOARGS;

	if (!id_ok (i))
		return EVOIDELEM;

	if ((t = pmt_byname (c)) == -1)
		return EBADTYPE;

	return (unregister_type (element [i], t)) ? R_OKey : ENOTREG;
}

int LNDB_stats (client_params *cpar)
{
	extern time_t start_time;
	extern unsigned int clients, tot_clients, denywrite, read_only;
	extern unsigned int pmt_rusage (short int);
	unsigned int dsum, esum, lsum, i;
	Link *l;
	char *c;
	short int t;
	time_t now = time (NULL) - start_time;

	for (dsum = esum = lsum = i = 0; i < ElTop; i++)
		if (element [i])
		{
			Element *e = element [i];
			if (ELOADED(e) && e->len < BIG_ELEMENT)
				esum += e->len;
			else dsum += e->len;
			for (l = e->link; l; l = l->next) lsum++;
		}

	PUTS ("Lndbase " VERSION " \n");
	PRINT ("%ukB in %u data elements (%ukB on disk)\n", 1 + (esum + Elements * sizeof (Element)) / 1024, Elements, 1 + dsum / 1024);
	PRINT ("%ukB in %u links (%.2f links per element)\n", 1 + (lsum * sizeof (Link)) / 1024, lsum, lsum / (2.0 * Elements));
	PRINT ("kernel uptime: %lu:%umin\n", now / 3600, (unsigned int) (now % 3600) / 60);
	PRINT ("%u connected clients, %u total clients served\n", clients, tot_clients);
	if (read_only)
	{ PUTS ("* Readonly *"); }
	if (denywrite != 0)
	{ PRINT ("Static Elements: %u\n", denywrite); }

	PUTS ("Available PMTs:");
	for (t = 0; (c = show_pmts (NULL, &t));)
	{
		PRINT (" %s (%ukB)\n", c, pmt_rusage (t - 1) / 1024);
	}
	return R_EOD;
}

int LNDB_save (client_params *cpar)
{
	pid_t p;

	if ((p = fork ()) == -1)
		return EFAIL;

	if (p == 0)
	{
		save_all ();
		exit (0);
	}

	// ***** Should we waitpid() until forked image has finished?
	// ***** Propably yes. But what pthreads does with SIGCHLD ?

	return R_OKey;
}

int LNDB_vlink (unsigned int i1, unsigned int i2, client_params *cpar)
{
	Link *l;

	if (!id_ok (i1) || !id_ok (i2))
		return EVOIDELEM;

	if (!(l = find_link (element [i1], element [i2])))
		return R_NOT_LINKED;

	PRINT ("%i\n", l->lid);

	return R_OKey;
}

int LNDB_glinks (unsigned int i, char *s, client_params *cpar)
{
	Group *g;
	Link *l;

	if (!id_ok (i))
		return EVOIDELEM;

	if ((l = element [i]->link) == NULL)
		return R_NO_LINKS;

	if ((g = grp_by_name (cpar->cwd, s)) == NULL)
		return ENODIR;

	for (; l; l = l->next)
		if (l->e->group == g) {
			if (l->lid)
				PRINT (" %u / %i\n", l->e->id, l->lid);
			else
				PRINT (" %u\n", l->e->id);
		}

	return R_EOD;
}
