/******************************************************************************

	fs.c

	File operations. Load & Save the database.

******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "global.h"
#include "polytypes.h"

extern char *datafile, *groupfile, *linkfile;
extern int NPTYPES;
extern PolymorphicType *polytype [];

void *mmaped;

static FILE *out;
static unsigned int suint;
static Group **int2grp;

/*
 * At this level malloc failures fatally terminate.
 */
static void *xmalloc (size_t si)
{
	void *p;

	if (!(p = malloc (si)))
	{
		fprintf (stderr, "Virtual memory exhausted\n");
		exit (1);
	}
	return p;
}

/*****************************************************************************
		Saving the data. It MUST be called:
	save_groups,      save_elements,       save_links

	Elements MUST be defraged. No NULL ids since links are saved as
	id-id pairs.
*****************************************************************************/
/*
 * 				Saving groups.
 * Preorder method for recursive tree display is used so that the trees
 * will have exactly the same form when loaded.
 * In any other case they would result in excesive balancing.
 *
 * Preorder leaves the tree the same. An even better approach might be
 * InOrder and mixing to produce a perfectly balanced tree. It would take
 * more time and memory.
 */
static void count_groups (Group *g)
{
	++suint;
	if (g->child) count_groups (g->child);
	if (g->less) count_groups (g->less);
	if (g->more) count_groups (g->more);
}

static void groups_out (Group *g, unsigned int dpth)
{
	char spaces [dpth + 1];

	spaces [dpth] = 0;
	memset (spaces, ' ', dpth);
	fprintf (out, "%s%s\n", spaces, g->node_name);
	*int2grp++ = g;
	if (g->child) groups_out (g->child, dpth + 1);
	if (g->less) groups_out (g->less, dpth);
	if (g->more) groups_out (g->more, dpth);
}

static void save_groups ()
{
#define GRP_TMP "Groups.tmp"
	Group **s;

	out = fopen (GRP_TMP, "w");
	suint = 0;
	if (root.child) count_groups (root.child);
	fprintf (out, "%u\n", suint);

	// We also keep the numbers of the groups so that
	// we can store numbers in the datafile later in save_elements
	s = int2grp = (Group**) malloc (sizeof (Group*) * suint);
	if (root.child) groups_out (root.child, 0);
	int2grp = s;

	fclose (out);
	unlink (groupfile);
	link (GRP_TMP, groupfile);
	unlink (GRP_TMP);
}

/*
 *			Saving Elements
 * The array int2grp has the binding of integer ids to groups.
 * suint is the number of groups. In order to have fast translation we must
 * sort this.
 */
#define NULLPRINT(format, args...) \
	fprintf (out, format, ## args),\
	fputc (0, out)

typedef struct {
	Group *g;
	unsigned int num;
} g2i;

/*
 * qsort. Not the libc qsort, if there's one real reason why we use qsort
 * it's because it's fast. No need to give the comparison function of this.
 */
static void q_sort (g2i *arr, unsigned int len)
{
	Group *sample = arr [0].g;
	g2i tmp;
	int i = 1, j = len - 1;

	if (i == j) goto bbreak;
	for (;;)
	{
		while (arr [i].g < sample)
			if (++i >= j) goto bbreak;
		while (arr [j].g > sample)
			if (--j < i) goto bbreak;
		tmp = arr [i], arr [i] = arr [j], arr [j] = tmp;
	}
bbreak:
	if (arr [j].g > sample) --j;

	tmp = arr [0], arr [0] = arr [j], arr [j] = tmp;

	if (j > 1) q_sort (arr, j);
	if (len - j - 1 > 1) q_sort (arr + j + 1, len - j - 1);
}

static void write_grpid (const Group *g, g2i grp2int [])
{
	unsigned int lo, hi, mid;
	Group *g2;

	if (g == &root)
	{
		fwrite (&suint, sizeof (suint), 1, out);
		return;
	}
	for (lo = 0, hi = suint - 1;;)
	{
		mid = (lo + hi) / 2;
		if (g == (g2 = grp2int [mid].g)) break;
		if (g > g2) lo = mid + 1;
		else hi = mid - 1;
#ifdef BUG_TRAP
		assert (lo <= hi);
#endif
	}
	fwrite (&grp2int [mid].num, sizeof (grp2int [0].num), 1, out);
}

static void write_pmt (Element *e)
{
	short int i;

	for (i = 0; i <= NPTYPES; i++)
		if (TSTBIT(e, i))
			fwrite (&i, sizeof i, 1, out);
	i = -1;
	fwrite (&i, sizeof i, 1, out);
}

static void save_elements ()
{
	extern char *pmtname (short int);
	unsigned int i;
	g2i *grp2int;
	Element *e;

	// Not to worry. We have another hard link in tmp.
	// It will be really discrarded when the number of hard links
	// reaches 0.
	unlink (datafile);
	out = fopen (datafile, "w");

	fwrite (&Elements, sizeof Elements, 1, out);
	NULLPRINT("%u", NPTYPES);
	for (i = 0; i < NPTYPES; i++)
		NULLPRINT("%s", polytype [i]->code);

	grp2int = (g2i*) alloca (sizeof (g2i) * suint);
	for (i = 0; i < suint; i++)
		grp2int [i] = (g2i) { int2grp [i], i };
	free (int2grp);
	if (suint > 1) q_sort (grp2int, suint);

	for (i = 0; i < Elements; i++)
	{
#ifdef BUG_TRAP
		assert (element [i]);
#endif
		e = element [i];
		fwrite (&e->len, sizeof e->len, 1, out);
		write_grpid (e->group, grp2int);
		data_to_stream (e, out);
		write_pmt (e);
	}
	fclose (out);
}

static void save_links ()
{
#define LINK_TMP "links.tmp"
	unsigned int i;
	Link *l;
	char k;

	out = fopen (LINK_TMP, "w");
	for (i = 0; i < Elements; i++)
		for (k = 0, l = element [i]->link; l; l = l->next)
		if (i < l->e->id)
		{
			if (k == 0) fprintf (out, "%u %u", i, l->e->id);
			else fprintf (out, "\" %u", l->e->id);
			if (l->lid) fprintf (out, " %u\n", l->lid);
			else fputc ('\n', out);
			k = 1;
		}

	fclose (out);
	unlink (linkfile);
	link (LINK_TMP, linkfile);
	unlink (LINK_TMP);
}

/*
 * I hope we forked before coming here...
 */
void save_all ()
{
	extern void defrag ();

	defrag ();
	save_groups ();
	save_elements ();
	save_links ();
	puts ("Database saved.");
}
/*****************************************************************************
		Loading the data. It MUST be called:
	load_groups,      load_elements,       load_links

*****************************************************************************/

/*
 *		 Loading polymorphic types & Elements
 *	No need to fopen datafile. It must have been mmaped at init ()
 */

/*
 * Polymorphic types may be added or removed from the kernel. The datafile
 * may even be from another lndbase. Thus PMTs must be binded to our
 * list of known PMTs.
 */

static int load_pmts (unsigned int thres)
{
	extern short int pmt_byname (const char*);
	static int load_elements (char*, unsigned int, short int*);
	char *p = mmaped + sizeof Elements;
	unsigned int i, j;
	short int *binder;

	i = strtoul (p, NULL, 10);
	binder = (short int*) alloca (sizeof (short int) * i);
	for (j = 0; j < i; j++)
	{
		while (*p++);
		if ((binder [j] = pmt_byname (p)) == -1)
			printf ("Not existing Polymorphic type : %s\n", p);
	}

	while (*p++);
	return load_elements (p, thres, binder);
}

extern Element *newelement (Group*, const char*, unsigned int);

static int load_elements (char *ptr, unsigned int thres, short int *binder)
{
	extern int register_type (Element*, short int);
	unsigned int g, l;
	short int s;
	Element *e;

	for (;;) {
		l = *((unsigned int*) ptr)++;
		if (!l) break;
		g = *((unsigned int*) ptr)++;
		e = newelement (int2grp [g], ptr, l);
		if (l < thres)
		{
			e->data = (char*) xmalloc (l);
			memcpy (e->data, ptr, l);
			ISLOADED(e);
		}
		else
			e->data = ptr;
		ptr += l;
		// Now load PMT information.
		for (;;) {
			s = *((short int*) ptr)++;
			if (s == -1) break;
			if (binder [s] != -1)
				if (!register_type (e, binder [s]))
					fprintf (stderr, "Failed registering"
						 " element %i to PMT %i\n",
						 e->id, binder [s]);
		}
	}
	free (int2grp);

	return Elements;
}
/*
 *				Loading Groups.
 * By saving the groups the way we used to, it should be a piece of cake
 * to load them now. Special checks can even be performed, at a cost but
 * weel be sure for correct operation later.
 */
static int load_groups ()
{
#define MAX_DEPTH 60
	extern Group *init_add_group (Group*, const char*);
	char inln [200];
	Group *parent [MAX_DEPTH];
	unsigned int i, j;
	char *p;

	parent [0] = &root;
	if (!(out = fopen (groupfile, "r")))
	{
		perror ("Can't open groupfile");
		exit (1);
	}
	fgets (inln, 200, out);
	suint = strtoul (inln, NULL, 10);
	int2grp = (Group**) malloc (sizeof (Group*) * (suint + 1));

	for (i = 0; i < suint; i++)
	{
		if (!fgets (inln, 200, out))
		{
			fprintf (stderr, "The group file ended too soon:(\n");
			exit (1);
		}
		inln [strlen (inln) - 1] = 0;
		for (p = inln, j = 0; *p == ' '; p++, j++);
		int2grp [i] = parent [j+1] = init_add_group (parent [j], p);
	}
	int2grp [suint] = &root;

	return suint;
}

/*
 * Loading the links
 *  The links are stored as pairs of IDs. Possible displacement then.
 */
static int load_links (unsigned int d)
{
	char t [80], *p, *q;
	unsigned int i = 0, pl = 0;

	if (!(out = fopen (linkfile, "r")))
	{
		perror ("linkfile");
		exit (1);
	}

	while (fgets (t, 80, out))
	{
		int l1, l2;
		if (*t == '"') {
			l1 = pl;
			p = t + 2;
		} else pl = l1 = strtoul (t, &p, 10);
		l2 = strtoul (p, &q, 10) + d;
		link_elems (d + l1, l2, *q == '\n' ? 0 : strtoul (q, NULL, 10));
		++i;
	}

	fclose (out);

	return i;
}

/*
 * If we want to reserve initial IDs:
 *  All have the same name and they're renamed later to avoid
 *  conflicts.
 */
static const char RESERVED_TMP [] = "";

void load_all (unsigned int th, unsigned int d, char *pat)
{
#define NN 100
	extern void true_data (unsigned int, char*, unsigned int);
	char tmp [NN], *c;
	Element *e;
	unsigned int i, j;

	for (i = 0; i < d; i++)
		e = newelement (&root, RESERVED_TMP, sizeof RESERVED_TMP);

	puts ("Loading...");
	printf (" %u Groups\n", load_groups ());
	printf (" %u Elements\n", load_pmts (th));
	printf (" %u links\n", load_links (d));

	if (pat == NULL)
		pat = RESERVED_PAT;

	for (i = 0; i < d; i++)
	{
		j = snprintf (tmp, NN, pat, i);
		while (eexists (tmp, j) != FREE_ID)
			if (j == NN) {
				fprintf (stderr, "Too many reserved garbage\n");
				exit (1);
			} else tmp [j++] = '*';
		c = (char *) malloc (j);
		strncpy (c, tmp, j);
		true_data (i, c, j);
	}
}
