/*****************************************************************************
	Its all stolen from `The Proxy'
	The proxy knows what content to expect after each command
	(with 4 basic reply types) and this is the best way to talk
	to a server... expect what the server protocol defines.

	Another approach with a timeout would make this much smaller
	but it was very messy.
	WeelC
*****************************************************************************/
#include <sys/stat.h>
#include <sys/poll.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>
#include <netdb.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <ctype.h>
#include "common.h"

#ifndef __GLIBC__
 typedef int socklen_t;
#endif

#define DEBUGPRINT fprintf (stderr, 
#define MAX_ARG 4
#define MAX_LINE 100
#define MAX_OPTION_LINE 200
#define CLIENT_TIMEOUT 100000

FILE *lndb_send, *lndb_get;

class CmdChecker
{
   protected:
	char **cmds;
	int CheckCmd (char*);
};

class Message : protected CmdChecker
{
	int GetLines (unsigned int, unsigned int);
	int CheckLine (char*);   
	void FreeContent ();
	FILE *f;
   public:
	Message (FILE *);
	char *argv [MAX_ARG], orig [MAX_LINE];
	unsigned int argc, clen;
	char *content;
	int argize ();
	void Next ();
	unsigned int Atoi (unsigned int);
	~Message ();
};

class BaseReply
{
   public:
	BaseReply ();
	char *StatusLine;
	int StatusCode;
	int CheckReplyStatus (char*);
	int Status ();
	virtual void Send (FILE *);
	virtual ~BaseReply ();
};

class Reply : public BaseReply
{
   protected:
	int CheckLine (char*);
   public:
	Reply ();
	virtual void Get ();
	virtual void Send (FILE *);
	~Reply ();
};

class LineReply : public Reply
{
   public:
	LineReply ();
	char *DataLine;
	virtual void Get ();
	virtual void Send (FILE *);
	~LineReply ();
};

class MultiLineReply : public Reply
{
   public:
	MultiLineReply ();
	char **DataLine;
	int Lines;
	virtual void Get ();
	virtual void Send (FILE *);
	~MultiLineReply ();
};

class shReply : public Reply
{
   public:
	shReply ();
	char *Data;
	unsigned int ContentLength;
	virtual void Get ();
	virtual void Send (FILE *);
	~shReply ();
};

class StdProxy : public CmdChecker
{
	Reply *SendMessage (Message *);
   protected:
	int ArgCmd (char *);
   public:
	StdProxy ();
	FILE *client;
	BaseReply *Do (Message *);
	~StdProxy ();
};

class TermClient { };
class ClientEOF : public TermClient { };
class ServerEOF : public TermClient { };
class BadRequest : public TermClient { };
class BadReply : public TermClient { };
class ClientTimeout : public TermClient { };
class OutofMemory : public TermClient { };
class SemaphoreError : public TermClient { };

//************************************************************************//
//
// fread with timeout. A MUST for fdopened sockets
//
static int fread_timeout (char *t, unsigned int cl, FILE *inf, time_t msec)
{
	unsigned int offset = 0, i;
	pollfd pfd = { fileno (inf), POLLIN + POLLHUP, 0 };

	fcntl (fileno (inf), F_SETFL, O_NONBLOCK);
	do {
		i = fread (t + offset, 1, cl - offset, inf);
		if ((offset += i) == cl)
		{
			fcntl (fileno (inf), F_SETFL,
				fcntl (fileno (inf), F_GETFD) & ~O_NONBLOCK);
			return 0;
		}
	} while (poll (&pfd, 1, msec) > 0
		 && !(pfd.revents & POLLHUP));

	return 1;
}

//
// And the fgets version of timeout.
//
static int fgets_timeout (char *ptr, unsigned int len, FILE *inf, time_t msec)
{
	pollfd pfd = { fileno (inf), POLLIN + POLLHUP, 0 };

	fcntl (fileno (inf), F_SETFL, O_NONBLOCK);

	if (!fgets (ptr, len, inf))
	{
		if (poll (&pfd, 1, msec) <= 0 || (pfd.revents & POLLHUP))
			return 1;

		fgets (ptr, len, inf);
	}
	fcntl (fileno (inf), F_SETFL,
		fcntl (fileno (inf), F_GETFD) & ~O_NONBLOCK);

	return 0;
}

//****************************** CmdChecker ********************************
int CmdChecker::CheckCmd (char *c)
{
	char **p;

	for (p = cmds; *p; p++)
		if (!strcmp (c, *p))
			return p - cmds;

	return -1;
}

// **************************** COMMANDS ************************************

static char *ContentCommands [] = 
	{ "newcl", "altcl", "idofcl", "locate", NULL };

//Used by StdProxy to know what to expect
static char *AllCommands []	= {
 /* 0 */	"sh",
 /* 1 */	"new", "newcl", "alt", "altcl", "groupof", "idof", "idofcl",
			"lenof", "pwd", "quit",
 /* 11 */	"cd", "chgrp", "del", "link", "mkgrp", "rengrp", "rmgrp",
			"save", "shutdown", "sue", "unlink", "unset",
 /* 23 */	"dir", "find", "locate", "ls", "stats", "typesof", "links",
		NULL
};

// ***************************** Message **********************************
Message::Message (FILE *c)
{
	f = c;
	clen = argc = 0;
	content = NULL;
	cmds = ContentCommands;
}

unsigned int Message::Atoi (unsigned int i)
{
	if (i >= argc)
		return 0;
	char *c = strchr (argv [i], ' ');
	if (c) *c = 0;
	int j = atoi (argv [i]);
	if (c) *c = ' ';
	return j;
}

int Message::CheckLine (char *c)
{
	int i = strlen (c);

	if (i == 0 || c [i - 1] != '\n')
		throw BadRequest();

	c [i - 1] = 0;

	return i;
}

int Message::GetLines (unsigned int lines, unsigned int pure)
{
	unsigned int tot = 0, i;
	char tmp [MAX_OPTION_LINE], *optv [lines];

	for (unsigned int j = 0, i = 0; i < lines; i++, tot += j)
	{
		if (fgets_timeout (tmp, MAX_OPTION_LINE, f, CLIENT_TIMEOUT))
			throw ClientEOF();
		optv [i] = (char*) alloca (j = CheckLine (tmp));
		strcpy (optv [i], tmp);
	}

	if (!(content = new char [clen = tot + pure]))
		throw OutofMemory();

	for (i = 0, *content = 0; i < lines; i++)
		strcat (content, optv [i]);

	return tot;
}

void Message::FreeContent ()
{
	delete content;
	content = NULL;
}

int Message::argize ()
{
	char *c;

	for (argc = 0, c = orig; *c && argc < MAX_ARG;)
	{
		argv [argc++] = c;
		while (!isspace (*c) && (*c)) c++;
		while (isspace (*c)) c++;
	}

	return argc;
}

void Message::Next ()
{
	char *c;

	FreeContent ();
	if (fgets_timeout (orig, MAX_LINE, f, CLIENT_TIMEOUT))
		throw ClientEOF();

	CheckLine (orig);

	if (argize () == 1)
		return;

	char *p = strchr (orig, ' ');
	if (p) *p = 0;
	c = (char*) alloca (strlen (orig) + 1);
	strcpy (c, orig);
	if (p) *p = ' ';

	switch (int i = CheckCmd (c)) {
	case -1:
		return;
	default:
		if ((content = new char
		  [clen = (i == 1) ? Atoi (2) + 1 : Atoi(1) + 1]) == NULL)
			throw OutofMemory();
		if (fread_timeout (content, clen, f, CLIENT_TIMEOUT))
			throw ClientTimeout();
		break;
	case 3:
		if (fread_timeout (content + GetLines (Atoi (1), Atoi (2) + 1),
				Atoi (2) + 1, f, CLIENT_TIMEOUT))
			throw ClientTimeout();
	}
}

Message::~Message ()
{
	FreeContent ();
}

// ****************************** BaseReply *********************************

BaseReply::BaseReply ()
{
	StatusLine = NULL;
}

int BaseReply::CheckReplyStatus (char *c)
{
	if (*c != '-' && *c != '*')
		return 0;

	if (c [1] != ' ' || !isdigit (c [2]))
		return 0;

	StatusLine = new char [strlen (c) + 1];
	strcpy (StatusLine, c);

	if (!(c = strchr (StatusLine + 2, ' ')))
		throw BadReply();

	*c = 0;
	StatusCode = atoi (StatusLine + 2);
	*c = ' ';

	return 1;
}

int BaseReply::Status ()
{
	return StatusLine [0] == '*';
}

void BaseReply::Send (FILE *f)
{
	fputs (StatusLine, f);
	fputs ("\n", f);
	fflush (f);
}

BaseReply::~BaseReply ()
{
	if (StatusLine) delete StatusLine;
}

// ******************************* Reply ************************************

Reply::Reply () : BaseReply ()
{ }

int Reply::CheckLine (char *c)
{
	int i = strlen (c);

	if (i == 0 || c [i - 1] != '\n')
		throw BadReply();

	c [i - 1] = 0;

	return i;
}

void Reply::Get ()
{
	char tmp [MAX_LINE];

	if (fgets_timeout (tmp, MAX_LINE, lndb_get, 10 * CLIENT_TIMEOUT))
		throw ServerEOF();

	CheckLine (tmp);
	if (!CheckReplyStatus (tmp))
		throw BadReply();
}

void Reply::Send (FILE *f)
{
	BaseReply::Send (f);
}

Reply::~Reply ()
{ }

// **************************** LineReply **********************************

LineReply::LineReply () : Reply ()
{
	DataLine = NULL;
}

void LineReply::Get ()
{
	char tmp [MAX_LINE];
	int i;

	if (fgets_timeout (tmp, MAX_LINE, lndb_get, 10 * CLIENT_TIMEOUT))
		throw ServerEOF();

	i = CheckLine (tmp);
	if (CheckReplyStatus (tmp))
		return;

	DataLine = new char [i];
	strcpy (DataLine, tmp);

	Reply::Get ();
}

void LineReply::Send (FILE *f)
{
	if (DataLine)
	{
		fputs (DataLine, f);
		fputs ("\n", f);
	}
	BaseReply::Send (f);
}

LineReply::~LineReply ()
{
	if (DataLine) delete DataLine;
}

// ***************************** MultiLineReply *****************************

MultiLineReply::MultiLineReply () : Reply ()
{
	DataLine = NULL;
	Lines = 0;
}

void MultiLineReply::Get ()
{
	char tmp [MAX_LINE];
	int i;
	struct llist {
		char *str;
		llist *next;
	} *first = NULL, *curr = NULL /*avoid waring*/;

	for (Lines = 0;; Lines++)
	{
		if (fgets_timeout (tmp, MAX_LINE, lndb_get, 10 * CLIENT_TIMEOUT))
			throw ServerEOF();
		i = CheckLine (tmp);
		if (CheckReplyStatus (tmp))
			break;
		llist *nanuk = (llist*) alloca (sizeof (llist));
		if (!first)
			first = nanuk;
		else
			curr->next = nanuk;
		curr = nanuk;
		nanuk->str = new char [i];
		strcpy (nanuk->str, tmp);
	}

	if (Lines)
		DataLine = new char* [Lines];
	for (i = 0, curr = first; i < Lines; i++, curr = curr->next)
		DataLine [i] = curr->str;
}

void MultiLineReply::Send (FILE *f)
{
	int i;

	for (i = 0; i < Lines; i++)
	{
		fputs (DataLine [i], f);
		fputs ("\n", f);
	}
	BaseReply::Send (f);
}

MultiLineReply::~MultiLineReply ()
{
	int i;

	for (i = 0; i < Lines; i++)
		delete DataLine [i];

	if (DataLine) delete DataLine;
}

// ****************************** shReply **********************************
//

shReply::shReply () : Reply ()
{
	Data = NULL;
	ContentLength = 0;
}

void shReply::Get ()
{
	char tmp [MAX_LINE];

	if (fgets_timeout (tmp, MAX_LINE, lndb_get, 10 * CLIENT_TIMEOUT))
		throw ServerEOF();

	CheckLine (tmp);
	if (CheckReplyStatus (tmp))
		return;

	if (strncmp (tmp, "CL: ", 4))
		throw BadReply();

	Data = new char [ContentLength = atoi (tmp + 4)];

	if (fread_timeout (Data, ContentLength, lndb_get, 10 * CLIENT_TIMEOUT))
			throw ServerEOF();

	fgetc (lndb_get);

	Reply::Get ();
}

void shReply::Send (FILE *f)
{
	if (Status ())
	{
		fprintf (f, "CL: %u\n", ContentLength);
		fwrite (Data, 1, ContentLength, f);
		fputc ('\n', f);
	}
	BaseReply::Send (f);
}

shReply::~shReply ()
{
	if (Data) delete Data;
}

// ************************* StdProxy **************************************

StdProxy::StdProxy ()
{
	cmds = AllCommands;
}

Reply *StdProxy::SendMessage (Message *M)
{
	fputs (M->orig, lndb_send);
	fputs ("\n", lndb_send);
	fflush (lndb_send);
	if (M->content)
	{
		Reply *R = new Reply;
		R->Get ();
		if (!R->Status ())
			return R;
		delete R;
		if (fwrite (M->content, 1, M->clen, lndb_send) != M->clen)
			throw ServerEOF();
		fflush (lndb_send);
	}
	return NULL;
}

int StdProxy::ArgCmd (char *c)
{
	int i;
	char *p = strchr (c, ' ');

	if (p) *p = 0;

	i = CheckCmd (c);

	if (p) *p = ' ';

	return i;
}

BaseReply *StdProxy::Do (Message *M)
{
	int i = ArgCmd (M->orig);
	Reply *R;

	if ((R = SendMessage (M)))
		return R;

	if (i == 0)
		R = new shReply;
	else if (i < 11)
		R = new LineReply;
	else if (i < 23)
		R = new Reply;
	else
		R = new MultiLineReply;

	R->Get ();

	return R;
}

StdProxy::~StdProxy ()
{ }
//**************************************************************************//
#define LNSH_SOCK ".lnsh_socket"

char *lndb_home;
char *lnsh_sock = LNSH_SOCK;
int client;
pid_t lndb_pid, sh_pid;

void check_env ()
{
	char tmp [50];

	if (getenv (LNSH_ENV))
	{
		fprintf (stderr, "Environment variable:$"LNSH_ENV
			 " already defined.\n Another lnsh is "
			 "propably running\n");
		exit (1);
	}

	getcwd (tmp, 50);
	strcat (strcat (tmp, "/"), lnsh_sock);
	setenv (LNSH_ENV, tmp, 0);
	sprintf (tmp, "%u", getpid ());
	setenv (LNSH_PID, tmp, 0);
}

void setup_socket ()
{
	sockaddr_un addr;

	client = socket (AF_UNIX, SOCK_STREAM, 0);

	addr.sun_family = AF_UNIX;
	strcpy (addr.sun_path, lnsh_sock);
	unlink (lnsh_sock);

	if (bind (client, (struct sockaddr*)&addr, sizeof addr) == -1)
	{
		perror ("bind");
		exit (1);
	}
}

void connect_single (int argc, char **argv)
{
	int fs1 [2], fs2 [2];
	char **passv;

	passv = (char**) alloca ((argc + 3) * sizeof (char*));
	passv [0] = "lndbase";
	passv [1] = "-s";
	memcpy (passv + 2, argv, argc * sizeof (char*));
	passv [argc + 2] = NULL;

	if (pipe (fs1) == -1 || pipe (fs2) == -1)
	{
		perror ("pipe");
		exit (1);
	}

	switch (lndb_pid = fork ()) {
	case 0:
		// child.
		// writting to fs1 [1] will go to child's stdin.
		// child's stdout comes into fs2 [0]
		dup2 (fs1 [0], 0);
		close (fs1 [0]);
		close (fs1 [1]);
		dup2 (fs2 [1], 1);
		close (fs2 [0]);
		close (fs2 [1]);
//		execlp ("lndbase", "lndbase", "-s", "-C", NULL);
//		execlp ("lndbase", "lndbase", "-s", NULL);
//		execlp ("lndbase", "lndbase", "-s", "-d", "/home/stan/DATA/", NULL);
		execvp ("lndbase", passv);
		perror ("execlp");
		exit (1);
	case -1:
		perror ("fork");
		exit (1);
	default:
		// parent.
		sleep (2);
		close (fs1 [0]);
		close (fs2 [1]);
		if ((lndb_send = fdopen (fs1 [1], "w")) == NULL)
		{
			perror ("fdopen");
			exit (1);
		}
		lndb_get = fdopen (fs2 [0], "r");
		puts ("Connected");
	}
}

void forkoff ()
{
	switch (sh_pid = fork ()) {
	default:
		return;
	case -1:
		perror ("fork");
		exit (1);
	case 0:
		execlp (SHELL, SHELL, 0);
		perror ("exelp bash");
		exit (1);
	}
}

static void connect_tcp (char *host)
{
	int sock;
	sockaddr_in addr;
	hostent *ho;
	char *p;

	if ((p = strchr (host, ':')) == NULL)
	{
		fprintf (stderr, "Bad argument to TCP resource.\n 'host:port' argument expected\n");
		exit (1);
	}

	*p = 0;
	if (!(ho = gethostbyname (host)))
	{
		fprintf (stderr, "Cant find host %s\n", host);
		exit (1);
	}

	sock = socket (AF_INET, SOCK_STREAM, 0);
	addr.sin_family = AF_INET;
	addr.sin_port = htons (strtoul (p + 1, NULL, 10));
	addr.sin_addr.s_addr = (*(struct in_addr*)*ho->h_addr_list).s_addr;

	if (connect (sock, (sockaddr*)&addr, sizeof addr) == -1)
	{
		perror ("Cannot connect");
		exit (1);
	}

	lndb_get = fdopen (sock, "r");
	lndb_send = fdopen (sock, "w");
}

static void connect_unix (char *c)
{
	int sock;
	sockaddr_un addr;

	sock = socket (AF_UNIX, SOCK_STREAM, 0);
	addr.sun_family = AF_UNIX;
	sprintf (addr.sun_path, c);

	if (connect (sock, (struct sockaddr*)&addr, sizeof addr) == -1)
	{
		perror ("");
		exit (1);
	}

	lndb_get = fdopen (sock, "r");
	lndb_send = fdopen (sock, "w");
}

void disconnect ()
{
	fprintf (stderr, "Connection to Lndbase lost. Shutting down\n");
	unlink (lnsh_sock);
	unsetenv (LNSH_ENV);
	unsetenv (LNSH_PID);
}

#define BUFF_LEN 2048
void get_welcome ()
{
	MultiLineReply MLR;

	MLR.Get ();
	MLR.Send (stdout);
}

StdProxy TheProxy;

void proxy (FILE *f)
{
	Message M (f);
	BaseReply *R;

	try {
		M.Next ();
		R = TheProxy.Do (&M);
		R->Send (f);
	} catch (TermClient) {
		fprintf (stderr, "Exception!!!!\n");
	}

	delete R;
	fclose (f);
}

void child_done (int x)
{
	fflush (lndb_send);
	fclose (lndb_send);
	signal (SIGCHLD, SIG_IGN);
	sleep (2);
	exit (0);
}

int main (int argc, char *argv[])
{
	int retfd;
	socklen_t i;
	sockaddr_un claddr;

	check_env ();
	setup_socket ();

	if (argc > 1)
	{
		if (!strcmp (argv [1], "-t"))
			if (argc == 3) connect_tcp (argv [2]);
			else {
				fprintf (stderr, "Expecting host:port\n");
				exit (1);
			}
		else if (!strcmp (argv [1], "-u"))
			if (argc == 3) connect_unix (argv [2]);
			else {
				fprintf (stderr, "Expecting path\n");
				exit (1);
			}
		else connect_single (argc - 1, argv + 1);
	}
	else connect_single (argc - 1, argv + 1);
	get_welcome ();

	forkoff ();
	signal (SIGCHLD, child_done);

	if (listen (client, 3) == -1)
		perror ("listen");

	for (;;)
	if ((retfd = accept (client, (struct sockaddr*)&claddr, &i)) == -1)
		perror ("accept");
	else
		proxy (fdopen (retfd, "r+"));

	return 0;
}
