/******************************************************************************

	The macro Library

	All abstracted coded epil entries.

******************************************************************************/
/**************************************************************************
 Copyright (C) 2000 Stelios Xantkakis
**************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/mman.h>
#include <sys/time.h>
#include <sys/resource.h>

#include "projectlog.h"

#include "epil.h"
#include "lnshell.h"
#include "priv.h"

#include "dbstree.h"
#include "sysutil.h"

//*****************************************************************************

static char nocurs [] = "Current S list is not set";

static void check_curs ()
{
	if (!CurrentList)
		throw epilException (nocurs);
}

/******************************************************************************

	System commands

	We want to be able to execute system commands from our macro code.
	This will allow us to intergate our application with the system,
	for example we can call an image viewer to view internal image data.

	But this is also dangerous. A bad person could create a trojan
	macro to call dangerous / malicious system commands. The \system
	macro executes command allowed by the user only.

	Configuring which commands are allowed is an external process not
	available to the macrocode (only `:sys_reload').
	Moreover the executed command is not performed through /bin/sh -c
	but directly to execve and this is -oh- so much safer.

	Sample usage: { display `-size' `300x150' `cokatoo.jpg' } \system

	system uses the altmode {} to count its arguments.

	\system forks into the background. It would be rather interesting
	to generate shell arguments from the output of commands and do
	pipes with macros. Redirect arguments to system commands, etc.

******************************************************************************/

char *fsyscmd;

static dbsTree sysTree;

class sysCmd : dbsNodeStr
{
   public:
	sysCmd () : dbsNodeStr (&sysTree)	  { Name = StrDup (Query); }
	~sysCmd ()		   { sysTree.dbsRemove (this); delete Name; }
using	dbsNodeStr::Name;
};

static int sys_check (char *cmd)
{
	DBS_STRQUERY = cmd;
	return sysTree.dbsFind () == NULL;
}

static void prnt (dbsNode *d)
{
	Logprintf (" %s\n", ((sysCmd*)d)->Name);
}

static void coded_sys_show ()
{
	Logputs ("Allowed system commands:\n");
	sysTree.foreach (prnt);
}

static char sysargs [] = "\\system resumes arguments from altmode!";
static char nosys [] = "command not allowed by \\system:";

static void coded_system ()
{
	if (ep_pargs == 0) throw epilException (sysargs);

	int i;
	char **argv = (char**) alloca ((ep_pargs + 1) * sizeof (char*));

	for (i = 0; i < ep_pargs; i++) {
		TMP_POP_ALLOC(argv[i]);
	}

	ep_pargs = 0;

	argv [i] = NULL;

	if (**argv == 0 || sys_check (argv [0]))
		throw epilException (nosys, argv [0]);

	int pid = fork ();

	if (pid == -1) return;

	if (pid == 0) {
		chdir ("exports");
		execvp (argv [0], argv);
		exit (127);
	}
}

static void dlt (dbsNode *d)
{
	delete ((sysCmd*)d);
}

static void priv_sys_reload ()
{
#define BFF 40
	FILE *f;

	if (!(f = fopen (fsyscmd, "r"))) {
		Logprintf ("Can't open file %s\n", fsyscmd);
		return;
	}

	sysTree.deltree (dlt);
	sysTree.root = NULL, sysTree.nnodes = 0;

	char bf [BFF + 1];
	int i, c;

	DBS_STRQUERY = bf;

	for (;;) {
		for (i = 0; i < BFF; i++) {
			c = fgetc (f);
			if (c == EOF || isspace (c)) break;
			bf [i] = c;
		}
		if (i < BFF) {
			bf [i] = 0;
			if (!sysTree.dbsFind ()) new sysCmd;
		} else while (!isspace (c = fgetc (f)))
			if (c == EOF) break;
		while (isspace (c = fgetc (f)));
		if (c == EOF) break;
		ungetc (c, f);
	}

	fclose (f);
}

/******************************************************************************

	Coded epil macro \relate

	relate is the algorithm to trace a path in the web of links that
	connects two elements of the web. Since multiple paths may exist
	between two elements, its also possible to exclude passing from
	certain nodes.

	connect can be an expensive macro. Mostly in terms of network load.

******************************************************************************/

#define MAX_STEPS 64

class Rlt {
	E *looking_for;
	int depth;
	S *SViz [MAX_STEPS];
	S *xcld;

	int want (E*);
	int do_relate ();
	void traceback ();
	void cleanup ();

   public:
	Rlt ();
};

int Rlt::want (E *e)
{
	int i;

	if (In (xcld, e)) return 0;

	for (i = 0; i <= depth; i++)
		if (In (SViz [i], e)) return 0;

	return 1;
}

int Rlt::do_relate ()
{
	int found = 0;

	if (NElements (SViz [depth]) == 0)
		return 0;

	E *e, *ee;

	SViz [depth + 1] = EmptyS ();

	SLooper L1 (SViz [depth]);
	while ((e = L1.Next ())) {
		S *l = Links (e);

		if (NElements (l) > 0) {

			SLooper L2 (l);
			while ((ee = L2.Next ())) {
				if ((found = ee == looking_for))
					break;
				if (want (ee))
					Add (SViz [depth + 1], ee);
			}
		}

		Kill (l);

		if (found) {
			Kill (SViz [depth + 1]);
			return 1;
		}
	}

	++depth;
	return do_relate ();
}

void Rlt::traceback ()
{
	E *e, *ee;
	int i;
	S *out = EmptyS ();

	Add (out, e = looking_for);

	for (i = depth; i >= 0; i--) {

		S *l = Links (e);

		{
			// The Looper must destruct before calling Kill()
			SLooper L (l);
			while ((ee = L.Next ()))
				if (In (SViz [i], ee)) {
					Add (out, e = ee);
					break;
				}
		}

		Kill (l);
		Kill (SViz [i]);
	}

	makeCurrent (out);
}

void Rlt::cleanup ()
{
	int i;

	EmptyS ();
	for (i = 0; i <= depth; i++)
		Kill (SViz [i]);
}

static const char cErr [] = "\\relate expects a list with two elements\n";

Rlt::Rlt ()
{
	xcld = CurrentList;

	SViz [0] = EmptyS ();

	Add (SViz [0], EIndex (xcld, 0));
	looking_for = EIndex (xcld, 1);

	depth = 0;

	if ((ep_boolean = do_relate ())) traceback ();
	else cleanup ();
}

static void coded_relate ()
{
	check_curs ();

	if (NElements (CurrentList) < 2) throw epilException (cErr);

	Rlt R;

	// we don't want the newbies to freak with the unused variable wanring
	if (&R);
}

/******************************************************************************

	Terrible, terrible, terrible.

	This implements an alloca() for C++ objects.
	We do alloca (sizeof object), memcpy a similar object to copy the
	virtual table, and use an explict contructor.

	Not exactly the proper way...
	
	\gpack is happier now

******************************************************************************/

class gtmp : public dbsNodeStr
{
   public:
	int cnt;
	bool seen;
	gtmp (dbsTree *d)	: dbsNodeStr (d)
		{ cnt = 1; seen = false; }
	void ctor (dbsTree*);
};

void gtmp::ctor (dbsTree *d)
{
	addself (d);
	Name = DBS_STRQUERY;
}

static unsigned int narg ()
{
	char *c;
	TMP_POP_ALLOC(c);
	return strtoul (c, NULL, 10);
}

static void coded_gpack ()
{
	check_curs ();

	int lim = narg ();

	if (lim <= 1) {
		ep_boolean = false;
		return;
	}

	dbsTree D;

	D.root = NULL;
	DBS_STRQUERY = " . ";
	D.dbsFind ();
	gtmp VirtualTable (&D);
	// remove the ghost node from the tree.
	D.dbsRemove ((dbsNode*)&VirtualTable);

	int nu = 0, cs = 0;
	SLooper SL (CurrentList);
	E *e;
	gtmp *gp;

	while ((e = SL.Next ())) {
		DBS_STRQUERY = GroupOf (e);
		if ((gp = (gtmp*) D.dbsFind ())) {
			if (++gp->cnt == lim)
				++nu;
		} else {	// madness
			gp = (gtmp*) alloca (sizeof (gtmp));
			memcpy (gp, &VirtualTable, sizeof (gtmp));
			gp->ctor (&D);
			cs += strlen (DBS_STRQUERY) + 8;
		}
	}

	if (!nu) {
		ep_boolean = false;
		return;
	}

	S *Rs = EmptyS ();
	char *arg = (char*) alloca (cs + 4);
	arg [0] = '{', arg [1] = 0;
	cs = 1;

	SL.Reset ();
	while ((e = SL.Next ())) {
		DBS_STRQUERY = GroupOf (e);
		gp = (gtmp*) D.dbsFind ();
		if (gp->cnt < lim) Add (Rs, e);
		else if (!gp->seen) {
			cs += sprintf (arg + cs, "%u `%s' ",
				       gp->cnt, ((dbsNodeStr*)gp)->Name);
			gp->seen = true;
		}
	}

	arg [cs] = '}', arg [cs + 1] = 0;
	ep_push (arg);
	ep_boolean = true;
	// please note that we don't have to free the nodes
}

/******************************************************************************

	Integer arithmetic calculator

	VALUE OPERATOR REGISTER \calc
	REGISTER \calcout

******************************************************************************/

static int calcRegister ['z' - 'a' + 1];

static char cop [] = "\\calc, undefined operator : ";

static void coded_calc ()
{
#define VINDEX(X) ((X <= 'Z') ? (X - 'A') : (X - 'a'))
	char *v, *o, *n;
	TMP_POP_ALLOC(v)
	TMP_POP_ALLOC(o)
	TMP_POP_ALLOC(n)

	int vi, nn;

	vi = (isalpha (*v)) ? VINDEX(*v) : 0;

	nn = (isdigit (*n)) ? strtol (n, NULL, 10) : ((isalpha (*n)) ?
		calcRegister [VINDEX(*n)] : 0);

	switch (*o) {
	case '+':
		calcRegister [vi] += nn;
		break;
	case '-':
		calcRegister [vi] -= nn;
		break;
	case '*':
		calcRegister [vi] *= nn;
		break;
	case '/':
		calcRegister [vi] /= nn;
		break;
	case '%':
		calcRegister [vi] %= nn;
		break;
	case '>':
		calcRegister [vi] = nn;
		break;
	case '|':
		calcRegister [vi] |= nn;
		break;
	case '&':
		calcRegister [vi] &= nn;
		break;
	default:
		throw epilException (cop, o);
	}
}

static char noreg [] = "\\calcout, not a register : ";

static void coded_calcout ()
{
	char *v;
	TMP_POP_ALLOC(v)

	if (!isalpha (*v)) throw epilException (noreg, v);

	char t [20];
	sprintf (t, "%i", calcRegister [VINDEX (*v)]);

	ep_push (t);
}

/******************************************************************************

	\fspush, \fspop, runfile

******************************************************************************/

int runfile (char *fn)
{
#ifdef _POSIX_MAPPED_FILES_LATER_JOE
	// Is the mmaped file a big null-terminated string ?
	// It appears to work, but maybe for small files we can
	// go with alloca().
	struct munmap_alwz {
		char *program;
		int sz;
		munmap_alwz ()	{ program = (char*) MAP_FAILED; sz = 0; }
		~munmap_alwz ()
		{ if (program != MAP_FAILED) munmap (program, sz); }
	} M;

	if ((M.sz = filesize (fn)) == -1) return -1;

	M.program =
		mmap (0, M.sz, PROT_READ, MAP_PRIVATE, open (fn, O_RDONLY), 0);
	if (M.program == MAP_FAILED) return -1;

	exec_unit (M.program);
#else
	char *program;
	int sz;

	if ((sz = filesize (fn)) == -1) return -1;
	program = (char*) alloca (sz + 1);
	if (readfile (fn, program, sz) == -1) return -1;
	program [sz] = 0;

	exec_unit (program);
#endif

	return 0;
}

static char nofile [] = "No such file: ";

static void coded_runfile ()
{
	char *c;
	TMP_POP_ALLOC(c)

	if (runfile (c))
		throw epilException (nofile, c);
}

static char itsbin [] = "binary file!";

static void coded_fspop ()
{
	char *c;
	TMP_POP_ALLOC(c)

	int i;
	if ((i = exportsize (c)) == -1)
		throw epilException (nofile, c);

	// FIXME: for large files use malloc

	char *p = (char*) alloca (i + 1);
	p [i] = 0;
	if (readexport (c, p, i) == -1)
		throw epilException (nofile, strerror (errno));

	for (int j = 0; j < i; j++)
		if (ISBIN((unsigned char) p [j]))
		if (ISBIN(p [j])) throw epilException (itsbin);

	ep_push (p, i);
}

static char fileerr [] = "Error writing to file: ";

static void coded_fspush ()
{
	char *f, *c;
	TMP_POP_ALLOC(f);
	TMP_POP_ALLOC(c);

	if (writefile (f, c, strlen (c)) == -1)
		throw epilException (fileerr, strerror (errno));
}

static void coded_iffs ()
{
	char *c;
	TMP_POP_ALLOC(c)

	ep_boolean = exportsize (c) != -1;
}

/******************************************************************************

	\rusage

******************************************************************************/

#define T2M(X) (X.tv_sec * 1000 + X.tv_usec / 1000)

static void coded_rusage ()
{
	rusage usage;

	getrusage (RUSAGE_SELF, &usage);

	Logprintf ("CPU time ms: utime %u, stime %u\n",
		   T2M(usage.ru_utime), T2M(usage.ru_stime));

	/* Not measured yet.
	Logprintf ("maxrss %lu, ixrss %lu, idrss %lu, isrss %lu\n",
		   usage.ru_maxrss, usage.ru_ixrss,
		   usage.ru_idrss, usage.ru_isrss);
	*/
}

//**********************************************************************
// Instance
//**********************************************************************

void bgDone (int)
{
	int status;
	waitpid (WAIT_ANY, &status, WNOHANG);
}

void library ()
{
	signal (SIGCHLD, bgDone); //*** Anti-zombie. \system

	priv_sys_reload ();
	priv_register_coded ("sys_reload", priv_sys_reload);
	CODED(sys_show);
	CODED(system);
	CODED(relate);
	CODED(gpack);
	CODED(calc);
	CODED(calcout);
	CODED(runfile);
	CODED(fspop);
	CODED(fspush);
	CODED(iffs);
	CODED(rusage);
}
