/*
 * Copyright (C) 2000, 2001 by Dmitry Morozhnikov, 
 * under the terms of the GNU General Public License, version 2.
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <time.h>
#include <getopt.h>
#include <sysexits.h>
#include <errno.h>
#include <locale.h>
#include "datfile2.h"
#include "datview.h"

dat_header_t dat_header;
dat_field_t *dat_field;
dat_field_int_t *datfint;

void usage(char *p)
{
	fprintf(stderr,
			"Usage: %s [options] <dat_file>\n"
			"Options:\n"
			"-h, --help                Print this help and exit\n"
			"-V, --version             Print version and exit\n"
			"-i, --info                Display dat file information [default]\n"
			"-e, --description         Display fields description\n"
			"-b, --browse              Browse dat file\n"
			"-o, --output=<file>       Output to file <file>\n"
			"\n"
			"-P, --strip-prefix        Strip prefix from field names\n"
			"-D, --browse-desc         Output field names in one line\n"
			"\n"
			"--quote-strings           Do quote string fields and field names\n"
			"--quote-datetime          Do quote date and time fields\n"
			"                            (ignored with '--unix-datetime')\n"
			"--quote-char=<char>       Quotation character [default = \"]\n"
			"--sql-quote               Quote in ISO SQL2 manner\n"
			"                            (quote by ' and replace\n"
			"                             all symbols ' with '',\n"
			"                             this override --quote-char parameter)\n"
			"--no-strip-spaces         Do NOT strip leading spaces in strings\n"
			"-d, --delimiter=<char>    Use <char> as delimiter [default = ,]\n"
			"--no-strip-zeros          Do NOT strip leading zeros in decimals\n"
			"--always-show-dot         Always add dot to decimals\n"
			"--ignore-sign             Ignore sign of decimals\n"
			"\n"
			"--group=<mode>            Mode for group type\n"
			"                            'n'   -- not use\n"
			"                            'u|y' -- use grouping\n"
			"                            'b'   -- work in both manner [default]\n"
			"--show-del=<mode>         Mode for deleted records\n"
			"                            'n'   -- not show[default]\n"
			"                            'd|y' -- show ONLY deleted records\n"
			"                            'b'   -- show all\n"
			"--show-lock=<mode>        Mode for locked records\n"
			"                            'n'   -- not show[default]\n"
			"                            'l|y' -- show ONLY locked records\n"
			"                            'b'   -- show all\n"
			"\n"
			"--as-date=<fld>[,<fld>]   Show this fields as date\n"
			"                            (numbers or names)\n"
			"--as-time=<fld>[,<fld>]   Show this fields as time\n"
			"                            (numbers or names)\n"
			"--unix-datetime           Show date and time in unix notation\n"
			"                            (seconds since epoch)\n"
			"--force-datetime          Force output zero date and time\n"
			"                            if it wrong\n"
			"--date-fmt=<fmt>          Output date in specified format\n"
			"                            'l'   -- in local conversion [default]\n"
			"                            'i'   -- in ISO-8601 conversion\n"
			"                            's'   -- in SQL convertion\n"
			"\n"
			"--with-memo[=<memo_file>] Use memo file if presented\n"
			"                            (use <memo_file> as memo file when defined,\n"
			"                             otherwise <dat_file> with changed suffix\n"
			"                             will be used)\n"
			"--cut-memo                Cut memo into strings\n"
			"                            as defined in .dat file header\n",
			p);
};

void version()
{
	fprintf(stderr, "%s\n", DATVIEW_VERSION);
};

datview_opts_t dopts;
char *dateflds = "";
char *timeflds = "";

int main(int argc, char *argv[])
{
	char c = '\0';
	int indexptr = 0;
	int doinfo, dodesc, dobrowse;
	char *datfile;
	char *memofile;
	int fd, memofd, rc;
	int p = 0;
	
#define QUOTE_STRINGS	256
#define QUOTE_CHAR		257
#define NO_STRIP_SPACES	258
#define GROUP			259
#define DELETED			260
#define LOCKED			261
#define NO_STRIP_ZEROS	262
#define SHOW_DOT		263
#define AS_DATE			264
#define AS_TIME			265
#define UNIX_DT			266
#define FORCE_DATETIME  267
#define QUOTE_DATETIME	268
#define DATE_FMT		269
#define IGNORE_SIGN		270
#define WITH_MEMO       271
#define CUT_MEMO        272
#define SQL_QUOTE       273

	struct option opts[] =
	{
		{"help", no_argument, NULL, 'h'},
		{"version", no_argument, NULL, 'V'},
		{"info", no_argument, NULL, 'i'},
		{"description", no_argument, NULL, 'e'},
		{"strip-prefix", no_argument, NULL, 'P'},
		{"browse-desc", no_argument, NULL, 'D'},
		{"quote-strings", no_argument, &p, QUOTE_STRINGS},
		{"quote-char", required_argument, &p, QUOTE_CHAR},
		{"no-strip-spaces", no_argument, &p, NO_STRIP_SPACES},
		{"delimiter", required_argument, NULL, 'd'},
		{"group", required_argument, &p, GROUP},
		{"browse", no_argument, NULL, 'b'},
		{"show-del", required_argument, &p, DELETED},
		{"show-lock", required_argument, &p, LOCKED},
		{"no-strip-zeros", no_argument, &p, NO_STRIP_ZEROS},
		{"always-show-dot", no_argument, &p, SHOW_DOT},
		{"as-date", required_argument, &p, AS_DATE},
		{"as-time", required_argument, &p, AS_TIME},
		{"unix-datetime", no_argument, &p, UNIX_DT},
		{"force-datetime", no_argument, &p, FORCE_DATETIME},
		{"quote-datetime", no_argument, &p, QUOTE_DATETIME},
		{"date-fmt", required_argument, &p, DATE_FMT},
		{"output", required_argument, NULL, 'o'},
		{"ignore-sign", no_argument, &p, IGNORE_SIGN},
		{"with-memo", optional_argument, &p, WITH_MEMO},
		{"cut-memo", no_argument, &p, CUT_MEMO},
		{"sql-quote", no_argument, &p, SQL_QUOTE},
		{NULL, 0, NULL, 0}
	};
	setlocale(LC_COLLATE, "");
	setlocale(LC_TIME, "");
	doinfo = dodesc = dobrowse = 0;
	memset(&dopts, '\0', sizeof(datview_opts_t));
	dopts.quotechar = '"';
	dopts.separator = ',';
	dopts.group = 'b';
	dopts.deleted = 'n';
	dopts.locked = 'n';
	memofile = NULL;
	memofd = -1;
	opterr = 1;
	while((c = getopt_long(argc, argv, "hViePDd:bo:", opts, &indexptr)) != -1)
	{
		switch(c)
		{
		  case 'h':
			usage(argv[0]);
			exit(0);
			break;
		  case 'V':
			version();
			exit(0);
			break;
		  case 'i':
			doinfo = 1;
			break;
		  case 'e':
			dodesc = 1;
			break;
		  case 'P':
			dopts.strippref = 1;
			break;
		  case 'D':
			dopts.browsedesc = 1;
			break;
		  case 'd':
			dopts.separator = *optarg;
			break;
		  case 'b':
			dobrowse = 1;
			break;
		  case 'o':
			close(1);
			if(open(optarg, O_WRONLY | O_CREAT | O_TRUNC) != 1)
			{
				fprintf(stderr, "Can`t open \"%s\": %s\n",
						optarg, strerror(errno));
				exit(EX_OSERR);
			};
			break;
		  case '\0':
			switch(p)
			{
			  case QUOTE_STRINGS:
				dopts.quotestr = 1;
				break;
			  case QUOTE_CHAR:
				dopts.quotechar = *optarg;
				break;
			  case NO_STRIP_SPACES:
				dopts.nostripsp = 1;
				break;
			  case GROUP:
				if(*optarg != '\0' && *(optarg + 1) != '\0')
				{
					fprintf(stderr, "Too long argument: %s\n", optarg);
					exit(EX_USAGE);
				};
				dopts.group = *optarg;
				if(dopts.group == 'y')
				{
					dopts.group = 'u';
				};
				if(dopts.group != 'n' && dopts.group != 'u' &&
				   dopts.group != 'b')
				{
					fprintf(stderr, "Unknown group mode: '%c'\n",
							dopts.group);
					exit(EX_USAGE);
				};
				break;
			  case DELETED:
				if(*optarg != '\0' && *(optarg + 1) != '\0')
				{
					fprintf(stderr, "Too long argument: \"%s\"\n", optarg);
					exit(EX_USAGE);
				};
				dopts.deleted = *optarg;
				if(dopts.deleted == 'y')
				{
					dopts.deleted = 'd';
				};
				if(dopts.deleted != 'n' && dopts.deleted != 'd' &&
				   dopts.deleted != 'b')
				{
					fprintf(stderr,
							"Unknown mode for deleted recs: '%c'\n",
							dopts.deleted);
					exit(EX_USAGE);
				};
				break;
			  case LOCKED:
				if(*optarg != '\0' && *(optarg + 1) != '\0')
				{
					fprintf(stderr, "Too long argument: \"%s\"\n", optarg);
					exit(EX_USAGE);
				};
				dopts.locked = *optarg;
				if(dopts.locked == 'y')
				{
					dopts.locked = 'l';
				};
				if(dopts.locked != 'n' && dopts.locked != 'l' &&
				   dopts.locked != 'b')
				{
					fprintf(stderr,
							"Unknown mode for locked recs: '%c'\n",
							dopts.locked);
					exit(EX_USAGE);
				};
				break;
			  case NO_STRIP_ZEROS:
				dopts.nostripzdec = 1;
				break;
			  case SHOW_DOT:
				dopts.showdot = 1;
				break;
			  case AS_DATE:
				dateflds = optarg;
				break;
			  case AS_TIME:
				timeflds = optarg;
				break;
			  case UNIX_DT:
				dopts.unixdt = 1;
				break;
			  case FORCE_DATETIME:
				dopts.forcedt = 1;
				break;
			  case QUOTE_DATETIME:
				dopts.quotedt = 1;
				break;
			  case DATE_FMT:
				if(*optarg != '\0' && *(optarg + 1) != '\0')
				{
					fprintf(stderr, "Too long argument: \"%s\"\n", optarg);
					exit(EX_USAGE);
				};
				dopts.datefmt = *optarg;
				if(dopts.datefmt != 'l' && dopts.datefmt != 'i' &&
				   dopts.datefmt != 's')
				{
					fprintf(stderr, "Unknown format for date: '%c'\n",
							dopts.datefmt);
					exit(EX_USAGE);
				};
				break;
			  case IGNORE_SIGN:
				dopts.ignsign = 1;
				break;
			  case WITH_MEMO:
				dopts.withmemo = 1;
				if(optarg && *optarg != '\0')
				{
					memofile = optarg;
				};
				break;
			  case CUT_MEMO:
				dopts.cutmemo = 1;
				break;
			  case SQL_QUOTE:
				dopts.sqlquote = 1;
				break;
			};
			break;
		  default:
			fprintf(stderr, "Unknown option: '%c'\n", c);
			exit(EX_USAGE);
			break;
		};
	};
	if(argc - optind != 1)
	{
		usage(argv[0]);
		exit(EX_USAGE);
	};
	datfile = argv[optind];
	fd = open(datfile, O_RDONLY);
	if(fd == -1)
	{
		fprintf(stderr, "Can`t open \"%s\": %s\n", datfile, strerror(errno));
		exit(EX_NOINPUT);
	};
	rc = read(fd, &dat_header, sizeof(dat_header_t));
	if(rc != sizeof(dat_header_t))
	{
		fprintf(stderr, "Can`t read header of \"%s\": %s\n",
				datfile, strerror(errno));
		exit(EX_DATAERR);
	};
	if(dopts.withmemo == 1)
	{
		if(memofile == NULL)
		{
			char *dotptr;
			
			dotptr = rindex(datfile, '.');
			
			if(dotptr != NULL)
			{
				/* fuck!! my mind not work today */
				memofile = (char *) malloc(dotptr - datfile + 5);
				if(memofile == NULL)
				{
					fprintf(stderr, "Can`t malloc: %s\n", strerror(errno));
					exit(EX_OSERR);
				};
				memcpy(memofile, datfile, dotptr - datfile);
			    memofile[dotptr - datfile] = '\0';
				strcat(memofile, ".MEM");
				memofd = open(memofile, O_RDONLY);
				if(memofd == -1)
				{
					memofile[dotptr - datfile] = '\0';
					strcat(memofile, ".mem");
					memofd = open(memofile, O_RDONLY);
					/* well, memofd == -1 is now a flag for absent memo file */
				};
			};
		}
		else
		{
			memofd = open(memofile, O_RDONLY);
		};
	};
	if(doinfo == 1 || (dodesc == 0 && dobrowse == 0))
	{
		info(&dat_header, &dopts);
		if(dopts.withmemo == 1)
		{
			if(memofd == -1)
			{
				printf("memo file absent\n");
			}
			else
			{
				printf("memo file \"%s\" present\n", memofile);
				memo_info(memofd, &dopts);
			};
		};
	};
	if(dodesc == 1 || dobrowse == 1)
	{
		dat_field = (dat_field_t *) malloc
		  (sizeof(dat_field_t) * dat_header.numflds);
		if(dat_field == NULL)
		{
			fprintf(stderr, "Can`t malloc: %s\n", strerror(errno));
			exit(EX_OSERR);
		};
		datfint = (dat_field_int_t *) malloc
		  (sizeof(dat_field_int_t) * dat_header.numflds);
		if(datfint == NULL)
		{
			fprintf(stderr, "Can`t malloc: %s\n", strerror(errno));
			exit(EX_OSERR);
		};
		memset(datfint, '\0', sizeof(dat_field_int_t) * dat_header.numflds);
		rc = lseek(fd, sizeof(dat_header_t), SEEK_SET);
		if(rc == -1)
		{
			fprintf(stderr, "Can`t lseek: %s\n", strerror(errno));
			exit(EX_DATAERR);
		};
		rc = read(fd, dat_field, sizeof(dat_field_t) * dat_header.numflds);
		if(rc != sizeof(dat_field_t) * dat_header.numflds)
		{
			fprintf(stderr, "Can`t read field: %s\n", strerror(errno));
			exit(EX_DATAERR);
		};
		if(*dateflds != '\0')
		{
			char *p, *str;
			int i;
			p = dateflds;
			while((str = strsep(&p, ",")) != NULL)
			{
				if(isdigits(str) == 0)
				{
					char t[17];
					t[16] = '\0';
					for(i = 0 ; i < dat_header.numflds ; i++)
					{
						memcpy(t, dat_field[i].fldname, 16);
						stripspaces(t);
						if((strncasecmp(str, t,
										16) == 0) ||
						   (strncasecmp(str, t + 4,
										12) == 0))
						{
							datfint[i].isdate = 1;
							goto end1;
						};
					};
					fprintf(stderr, "Wrong field name: \"%s\"\n", str);
					exit(EX_USAGE);
				}
				else
				{
					i = strtol(str, NULL, 10);
					if(i < 1 || i > dat_header.numflds)
					{
						fprintf(stderr, "Wrong field number: %i\n", i);
						exit(EX_USAGE);
					};
					datfint[i - 1].isdate = 1;
				};
				end1:
			};
		};
		if(*timeflds != '\0')
		{
			char *p, *str;
			int i;
			p = timeflds;
			while((str = strsep(&p, ",")) != NULL)
			{
				if(isdigits(str) == 0)
				{
					char t[17];
					t[16] = '\0';
					for(i = 0 ; i < dat_header.numflds ; i++)
					{
						memcpy(t, dat_field[i].fldname, 16);
						stripspaces(t);
						if((strncasecmp(str, t,
										16) == 0) ||
						   (strncasecmp(str, t + 4,
										12) == 0))
						{
							datfint[i].istime = 1;
							goto end2;
						};
					};
					fprintf(stderr, "Wrong field name: \"%s\"\n", str);
					exit(EX_USAGE);
				}
				else
				{
					i = strtol(str, NULL, 10);
					if(i < 1 || i > dat_header.numflds)
					{
						fprintf(stderr, "Wrong field number: %i\n", i);
						exit(EX_USAGE);
					};
					datfint[i - 1].istime = 1;
				};
				end2:
			};
		};
	};
	if(dodesc == 1)
	{
		desc(&dat_header, dat_field, fd, &dopts);
	};
	if(dobrowse == 1)
	{
		browse(&dat_header, dat_field, &dopts, fd, memofd, datfint);
	};
	return 0;
};
