/*
 * eval.c - Process parse tree to produce C code 
 */

/*
 * Modified and extended for AWK-to-C Translator
 * Leonard Theivendra (05/20/96)
 */

/* 
 * Copyright (C) 1986, 1988, 1989, 1991-1995 the Free Software Foundation, Inc.
 * 
 * This file is part of GAWK, the GNU implementation of the
 * AWK Progamming Language.
 * 
 * GAWK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * GAWK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GAWK; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "awk.h"

static NODE *func_call P((NODE *name, NODE *arg_list));

static int loop_tag_valid = 0;	/* nonzero when loop_tag valid */
static int func_tag_valid = 0;
extern int exiting, exit_val;



/* ----------- Additional variables to original eval.c ----------- */

/* Flag indicating state of translator */
flags_type translator_state;

/* Current block ID# */
#define UNCOND_BLOCK  1    /* Special flags to indicate unconditional block */
long blockid = 2;
long blockidnum = 2;

/* Flag for indicating that a field variable has been assigned a non-string */
flags_type FVAR_ASSIGNED = 0;
flags_type FVAR_ASSIGNED2 = 0;
/* If CONVFMT is assigned to, then string optimizations are unsafe */
flags_type CONVFMT_ASSIGNED = 0;

/* Macros to update a variable's type status when it becomes number or
   string current 
   vvv = tree or tree->lnode, etc   
 */

#define UPDATE_NUMCURR(vvv) \
  if ( !(etype_parent & TSTATE_PREWALK) && \
       !(translator_state & (TSTATE_PREWALK|TSTATE_OPTCHECK)) && \
       (vvv)->type == Node_var )  { \
    if ( !((vvv)->flags & NUM_CURRENT_UNCOND) )  { \
      if ( blockid == UNCOND_BLOCK ) \
        (vvv)->flags |= NUM_CURRENT_UNCOND; \
      else  { \
        (vvv)->flags |= NUM_CURRENT; \
        (vvv)->blockID = blockid; \
      } \
    } \
  }   

#define UPDATE_STRCURR(vvv) \
  if ( !(etype_parent & TSTATE_PREWALK) && \
       !(translator_state & (TSTATE_PREWALK|TSTATE_OPTCHECK)) && \
       (vvv)->type == Node_var )  { \
    if ( !((vvv)->flags & STR_CURRENT_UNCOND) )  { \
      if ( blockid == UNCOND_BLOCK ) \
        (vvv)->flags |= STR_CURRENT_UNCOND; \
      else  { \
        (vvv)->flags |= STR_CURRENT; \
        (vvv)->blockID = blockid; \
      } \
    } \
  }



/* --------------------------------------------------------------------------
   interpret2:  New parse tree intepretation function.  Translates the
                parse tree to C code.

   Parameters:  tree            = Pointer to root of tree
                stmt_parent     = Flag used by a parent statement to
                                  influence child statement's behaviour
                emit_ccode_flag = Flag to signify if C code is to be printed or not
                state           = State of the translator (used for optimization) 
   -------------------------------------------------------------------------- */

int
interpret2(tree, stmt_parent, emit_ccode_flag, state)
register NODE *volatile tree;
flags_type stmt_parent;
short emit_ccode_flag;
flags_type state;

{

	register NODE *volatile t = NULL;	/* temporary */
	NODE **volatile lhs;	/* lhs == Left Hand Side for assigns, etc */
	int volatile traverse = 1;	/* True => loop thru tree (Node_rule_list) */
        
        flags_type etype_parent = 0;

        translator_state = state;
        
	/* avoid false source indications */
	source = NULL;
	sourceline = 0;

	if (tree == NULL)
		return 1;
	sourceline = tree->source_line;
	source = tree->source_file;

	switch (tree->type) {

	case Node_rule_node:
		traverse = 0;   /* False => one for-loop iteration only */
		/* FALL THROUGH */

	case Node_rule_list:
		for (t = tree; t != NULL; t = t->rnode) {
			if (traverse)
				tree = t->lnode;

			sourceline = tree->source_line;
			source = tree->source_file;
                      
                        if ( !(translator_state & (TSTATE_BEGIN|TSTATE_END)) )
                          emit_ccode(("\n\n/* -------------------- Rule/Actions -------------------- */\n\n")); 
                        if (tree->lnode != NULL)  { 
                          /* This is a conditional block => new blockid */
                          if ( !(translator_state & TSTATE_PREWALK) )
                            blockid = ++blockidnum;
                          emit_ccode((" if ("));
                          tree_eval2(tree->lnode, EXPR_BOOL); 
                          emit_ccode((")  {"));
                        }
                        else
                          /* Unconditional block => special block ID# */
                          blockid = UNCOND_BLOCK;
    
                        (void) interpret2(tree->rnode, STMT_NULL, emit_ccode_flag, state);
   
                        if (tree->lnode != NULL)  {
                          emit_ccode(("}\n"));
                          emit_ccode(("else {}\n")); 
                        }

 			if (!traverse)          /* case Node_rule_node */
				break;          /* don't loop */
		}
		break;

	case Node_statement_list:
                /* Block ID#'s for begin and end blocks */
                if ( (translator_state & (TSTATE_BEGIN|TSTATE_END)) &&
                     !(translator_state & TSTATE_PREWALK) )
                  blockid = ++blockidnum;
		for (t = tree; t != NULL; t = t->rnode)
			(void) interpret2(t->lnode, STMT_NULL, emit_ccode_flag, state);
		break;

	case Node_K_if:
                {
                long oldblockid = blockid;

                emit_ccode((" if ("));
                /* Instruct expression tree to conform to boolean operation */
                tree_eval2(tree->lnode, EXPR_BOOL);  /* Print out evaluation expr */
                emit_ccode((")  {\n"));
 
                /* Print out if() statement(s) */
                /* Following statements are conditional => new block ID */
                if ( !(translator_state & TSTATE_PREWALK) ) 
                  blockid = ++blockidnum;

                (void) interpret2(tree->rnode->lnode, STMT_NULL, emit_ccode_flag, state);
                emit_ccode((" }\n"));

                /* If there is a else branch, emit it's statements */
                if ( !(translator_state & TSTATE_PREWALK) )
                  blockid = ++blockidnum;
                if (tree->rnode->rnode != NULL)  {
                  emit_ccode(("else  {\n"));
                  (void) interpret2(tree->rnode->rnode, STMT_NULL, emit_ccode_flag, state);
                  emit_ccode(("}\n"));
                }

                blockid = oldblockid;
		break;
                }

	case Node_K_while:
                {
                long oldblockid = blockid;
                flags_type oldstate = state;

                /* Pre-walk the loop body to flag any variable assignments
                   that might hinder optimizations
                   i.e: while ( ... i > 5 ... )  { ... i = "abc" ... } */
                if ( !(state & TSTATE_PREWALK) )  {
                  blockid = ++blockidnum;
                  (void) interpret2(tree->rnode, STMT_NULL, EMIT_CCODE_NO, 
                                    state|TSTATE_OPTCHECK);
                }
                blockidnum = blockid;
 
                if ( !(state & TSTATE_PREWALK) )
                  translator_state = oldstate|TSTATE_OPTSAFE;
                blockid = oldblockid;
                emit_ccode((" while ("));
                tree_eval2(tree->lnode, EXPR_BOOL);

                emit_ccode((")  {\n"));
                blockid = blockidnum;
                (void) interpret2(tree->rnode, STMT_NULL, emit_ccode_flag, 
                                  oldstate);
                emit_ccode(("}\n"));

                translator_state = oldstate; 
                blockid = oldblockid;
		break;
                }

	case Node_K_do:
                emit_ccode((" do {"));
                (void) interpret2(tree->rnode, STMT_NULL, emit_ccode_flag, state);
                emit_ccode(("} while ("));
                tree_eval2(tree->lnode, EXPR_BOOL);
                emit_ccode((");\n"));
                break;

	case Node_K_for:
                {
                long oldblockid = blockid;
                flags_type oldstate = state;

                emit_ccode((" for ("));
                (void) interpret2(tree->forloop->init, STMT_NULL, emit_ccode_flag, state);

                /* Pre-walk the loop body to flag any variable assignments
                   that might hinder optimizations
                   i.e: for ( i=1; i < 5 && j>3; i++ )  { ... j = "abc" ... } */
                if ( !(state & TSTATE_PREWALK) )  {
                  blockid = ++blockidnum;
                  (void) interpret2(tree->lnode, STMT_NULL, EMIT_CCODE_NO,
                                    state|TSTATE_OPTCHECK);
                }
                blockidnum = blockid;

                if ( !(state & TSTATE_PREWALK) )
                  translator_state = oldstate|TSTATE_OPTSAFE;
                blockid = oldblockid;
	        tree_eval2(tree->forloop->cond, EXPR_BOOL);
                emit_ccode((";\n "));
                (void) interpret2(tree->forloop->incr, STMT_FOR, emit_ccode_flag,
                                  translator_state);

	        emit_ccode((")  {\n"));
                blockid = blockidnum;
	        (void) interpret2(tree->lnode, STMT_NULL, emit_ccode_flag, 
                                  oldstate);
	        emit_ccode(("}\n"));
    
                translator_state = oldstate; 
                blockid = oldblockid;
		break;
                }

	case Node_K_arrayfor:
                /* !!!!!!!! Note variable modification & optim tracking */
                emit_ccode(("/* >>> Node_K_arrayfor <<< */\n"));
		break;

	case Node_K_break:
                {	
		/* Old AT&T nawk treats break outside of loops like
                 * next. New ones catch it at parse time. Allow it if
		 * do_unix is on, and complain if lint.
	         */
		static int warned = 0;

		if (do_lint && ! warned) {
			warning("use of `break' outside of loop is not portable");
			warned = 1;
		}
		if (! do_unix)
			fatal("use of `break' outside of loop is not allowed");

		emit_ccode((" break;\n"));
		break;
                }

	case Node_K_continue:
                {
		/*
		 * Old AT&T nawk treats continue outside of loops like
		 * next. New ones catch it at parse time. Allow it if
		 * do_unix is on, and complain if lint.
		 */
		static int warned = 0;

		if (do_lint && ! warned) {
			warning("use of `continue' outside of loop is not portable");
			warned = 1;
		}
		if (! do_unix)
			fatal("use of `continue' outside of loop is not allowed");
			
		emit_ccode((" continue;\n"));
		break;
                }

	case Node_K_print:
                  tree = tree->lnode;

                  while (tree) {
                    emit_ccode(("do_print2("));
                    tree_eval2(tree->lnode, EXPR_NEED_NODE);
                    tree = tree->rnode;
                    if (tree)  
                      emit_ccode((", 1")); 
                    else  
                      emit_ccode((", 0"));
                    emit_ccode((", (struct redirect *) NULL, stdout, OFS);\n"));
                  }
                  emit_ccode(("do_print2_ORS((struct redirect *) NULL, stdout, ORS);\n"));
                  break;


	case Node_K_printf:
                emit_ccode(("/* >>> Node_K_printf <<< */\n"));
                break;

	case Node_K_delete:
                if ( tree->rnode != NULL )
                  emit_ccode(("/* >>> Expression lists not supported <<< */\n"));
                /*
                if ( tree->rnode != NULL )  {
                  emit_ccode(("do_delete(__%s, ", tree->lnode->vname));
                  tree_eval2(tree->rnode, EXPR_STRING_VAL);
                  emit_ccode((");\n"));
                }
                */ 
                else
                  emit_ccode(("assoc_clear(__%s->lnode);\n", tree->lnode->vname));
                break;

	case Node_K_next:
	        emit_ccode((" next;\n"));
		break;

	case Node_K_nextfile:
		do_nextfile();
		break;

	case Node_K_exit:
		/*
		 * In A,K,&W, p. 49, it says that an exit statement "...
		 * causes the program to behave as if the end of input had
		 * occurred; no more input is read, and the END actions, if
		 * any are executed." This implies that the rest of the rules
		 * are not done. So we immediately break out of the main loop.
		 */
		exiting = 1;
		emit_ccode((" break;\n"));
		break;

	case Node_K_return:
		emit_ccode((" return;\n"));
		break;

	default:
		/*
		 * Appears to be an expression statement.  Throw away the
		 * value. 
		 */
		if (do_lint && tree->type == Node_var)
			warning("statement has no effect");
		tree_eval2(tree, EXPR_STMT);
                if ( !(stmt_parent & STMT_FOR) )
                  emit_ccode((";\n"));
		break;
	}
	return 1;

}



/* --------------------------------------------------------------------------
   r_tree_eval2:  New expression evaluation function.  Significantly
                  modified to emit C code and analyze/enforce 
                  number/string duality in expression types
   
   Parameters:    tree            = Pointer to sub-tree
                  emit_ccode_flag = Flag to signify if C code is to be printed or not
                  etype_parent    = Used by parent node which calls for an  
                                    sub-expression evaluation to instruct child
                                    to return a particular type (of itself) that
                                    parent desires

   Returns:       The type of the expression's result (independent of the 
                  context the sub-expression might be in)

   ** Note:       These expression types being passed into/out of tree_eval
                  are flags to signify number/string variables or values,
                  number or string operations, boolean operations, etc
                  (note awk.h for complete list)
   -------------------------------------------------------------------------- */

flags_type r_tree_eval2(tree, emit_ccode_flag, etype_parent) 
register NODE *tree;
/* New parameters for tree_eval from original eval.c */
short emit_ccode_flag;		/* Flag indicating wether to output C-code or not */
flags_type etype_parent;	/* Indicate wether parent operation results in
                                   a numeric or string expression */
{ 

	register NODE *r, *t1, *t2;	/* return value & temporary subtrees */
	register NODE **lhs;
	register int di;
	AWKNUM x, x1, x2;
	long lx;

#if 0
#ifdef DEBUG
	if (tree == NULL)
		return Nnull_string;
	if (tree->type == Node_val) {
		if ((char)tree->stref <= 0) cant_happen();
		return tree;
	}
	if (tree->type == Node_var) {
		if ((char)tree->var_value->stref <= 0) cant_happen();
		return tree->var_value;
	}
#endif
#endif

	if (tree->type == Node_param_list) {
                emit_ccode((" 0/* >>> Expr: Node_param_list <<< */ "));
		return;
	}

        /* Since eval_condition() not used, account for NULL trees <=> true */
        if ( tree == NULL )   
          emit_ccode((" 1 "));  

	switch (tree->type) {


        /* Unary operations */

        /* Variable accessing */
	case Node_var:
        case Node_field_spec:
        case Node_subscript:
        case Node_NR:
        case Node_FNR:
        case Node_NF:
        case Node_FIELDWIDTHS:
        case Node_FS:
        case Node_RS:
        case Node_IGNORECASE:
        case Node_OFMT:
        case Node_CONVFMT:
        case Node_ORS:
        case Node_OFS:
           
                {
               
                flags_type ret_etype = EXPR_VAL_NODE;
                short num_optimize = 0;
                short str_optimize = 0;

                /* Based on parent expression preferences, and variable's 
                   status, emit optimal code to access variable */

                /* Number optimizations: 

                   1) If the variable is only assigned number expressions
                      in it's whole lifetime, then it is always number current.
                      If this is the case and the variable is never USED as
                      a non-number, then emit code to use variable as a 
                      native C number variable and not a NODE structure.
                   
                   2) If variable's number value is current, then read it's
                      number field without forcing it.
                      (it has to have become current in the current block or
                       from a non-conditional block).
   
                   3) Numeric special variables like NF, NR, FNR automatically
                      become number current on each input record => as long
                      as they haven't been assigned a non-number uptil now,
                      they are number current. 

                   Return NUMBER_VAL type to parent so that parent may do any
                   optimization of it's own. 
                */ 

                /* First attempt native C number variable optimization */
                if ( !(translator_state & TSTATE_PREWALK) &&
                     !(etype_parent & (EXPR_NEED_NODE|EXPR_STRING_VAL)) &&
                     tree->type == Node_var )  {
                  if ( !(tree->flags4 & USED_AS_NONNUM) && 
                    /* If parent is an assignment, and this variable is
                       non-initialized in AWK program, don't optimize as
                       number (similar to check below) */
                       !((etype_parent & PARENT_IS_ASGN) &&
                          tree->flags2 == 0) )  {
                    emit_ccode((" num__%s ", tree->vname));
                    ret_etype = EXPR_NUMBER_VAL;
                    goto var_end;
                  }
                }

                if ( (etype_parent & (EXPR_NUMBER_VAL|
                                      EXPR_INDIFF|
                                      EXPR_VAL_NODE|
                                      EXPR_BOOL)) &&
                     !(translator_state & TSTATE_PREWALK)
                   )  {
                  if ( tree->type == Node_var )  {                           
                    if ( (((tree->flags & (NUM_CURRENT_UNCOND|
                                           NUM_ASSIGNED_UNCOND)
                           ) ||
                           ((tree->flags & (NUM_CURRENT|NUM_ASSIGNED)) &&
                            blockid == tree->blockID  
                           )
                          ) && 
                          /* Check against while(), for() pitfalls:
                             i.e: while ( i>5 ) { .. i = "fdsf" .. } */
                          ( !(translator_state & TSTATE_OPTSAFE) ||
                            !(tree->flags3 & (STR_ASSIGNED|NONREGVAR_ASSIGNED))
                          )
                         ) ||
                         (
                          /* Don't assign Nnull_string as a number to 0 */
                          !(etype_parent & EXPR_VAL_NODE) &&
                          !(translator_state & TSTATE_END) &&
                          (tree->flags2 == NUM_ASSIGNED ||
                           tree->flags2 == 0)
                         )
                       )
                      num_optimize = 1;
                    /* Don't number optimize for the case where parent is
                       an assignment, boolean, or comparison operation and
                       the variable was initially assigned a string and has now
                       become number current */
                    if ( (etype_parent & (EXPR_VAL_NODE|EXPR_BOOL|EXPR_INDIFF))
                         && (tree->flags2 & (STR_ASSIGNED|NONREGVAR_ASSIGNED))
                       )
                      /* But don't count out variable which has been
                         assigned a number in current block or unconditional
                         block */
                      if ( !((tree->flags & NUM_ASSIGNED_UNCOND) || 
                             ((tree->flags & NUM_ASSIGNED) && 
                              tree->blockID == blockid ) 
                            )
                         )
                        num_optimize = 0;
                     
                  }                  
                  if ((tree->type == Node_NF || tree->type == Node_NR ||
                       tree->type == Node_FNR) &&
                       tree->flags != NUMSPVAR_ASSIGNED
                     )
                    num_optimize = 1; 
                } 

                if ( num_optimize )  {
                  emit_ccode((" peek_number("));
                  ret_etype |= EXPR_NUMBER_VAL;
                  goto process_var; 
                }

                /* String optimization: if variable's string value is
                   current, then read it's string field without forcing it.
                   (it has to have become current in the current block or
                    from a non-conditional block).  Field variables which
                   haven't been assigned a non-string uptil now are also
                   string current.  If CONVFMT has been set uptil now, then
                   don't bypass force_string(). 
                   Return STRING_VAL type to parent so that parent may do any
                   optimization of it's own. */
                if ( (etype_parent & EXPR_STRING_VAL) &&
                     !(translator_state & TSTATE_PREWALK) &&                     
                     !CONVFMT_ASSIGNED )  {
                  if ( tree->type == Node_var )  {
                    if ( (((tree->flags & (STR_CURRENT_UNCOND|
                                           STR_ASSIGNED_UNCOND)
                           ) ||
                           ((tree->flags & (STR_CURRENT|STR_ASSIGNED)) &&
                            blockid == tree->blockID  
                           )
                          ) &&                                        
                          (                          
                           !(translator_state & TSTATE_OPTSAFE) ||
                           !(tree->flags3 & (NUM_ASSIGNED|NONREGVAR_ASSIGNED))                           
                          )
                         ) ||
                         ( !(translator_state & TSTATE_END) &&
                           (tree->flags2 == STR_ASSIGNED ||
                            tree->flags2 == 0)
                         ) 
                       )
                      str_optimize = 1;
                  }                                                                                         
                  if ( tree->type == Node_field_spec && !FVAR_ASSIGNED )
                    str_optimize = 1;                    
                }
                
                if ( str_optimize )  {
                  emit_ccode((" ("));
                  ret_etype |= EXPR_STRING_VAL; 
                  goto process_var;
                }

                /* If sub-expression only contains variable, then
                   NULL string || 0.0  <=> FALSE <=> 0.0 in C,
                   non-NULL string || non-0.0 <=> TRUE <=> non-0.0 in C */
                if ( etype_parent & EXPR_BOOL )
                  emit_ccode((" bool_var("));

                /* Parent prefers a number value */
                else if ( etype_parent & EXPR_NUMBER_VAL )  { 
                  emit_ccode((" force_number(")); 
                  /* Update variable's status (has now become number current) */
                  UPDATE_NUMCURR(tree);                  
                }

                /* Parent prefers a string value node */
                else if ( etype_parent & EXPR_STRING_VAL )  {
                  emit_ccode((" force_string2("));
                  /* Update variable's status (has now become string current) */
                  UPDATE_STRCURR(tree);
                }

                else
                  emit_ccode((" ("));

                /* Update variable status on optimization prewalk if a
                   variable is used as a non-number
                   (use flags4 field to avoid conflicts w/ other flags) */
                if ( (translator_state & TSTATE_PREWALK) &&
                     tree->type == Node_var )  {
                  if ( etype_parent & (EXPR_STRING_VAL|EXPR_NEED_NODE) &&
                       !(etype_parent & (PARENT_IS_ASGN|PARENT_IS_COMP)) )  {
                    tree->flags4 |= USED_AS_NONNUM;
                  }
                }
              
                process_var:
                switch (tree->type)  {

                  case Node_var:
                    emit_ccode(("access_var(__%s)", tree->vname));
                    break;

                  case Node_field_spec:
                    emit_ccode(("access_fvar((int) (")); 
                    /* Force the field spec expression to be number-biased
                       by sending number as parent type */
                    tree_eval2(tree->lnode, EXPR_NUMBER_VAL);
                    emit_ccode((")) "));
                    break;
                 
                  case Node_subscript:
                    emit_ccode(("access_avar(__%s, ", tree->lnode->vname));
                    tree_eval2(tree->rnode, EXPR_STRING_VAL);
                    emit_ccode((")"));
                    break;

                  case Node_NR:
                    emit_ccode(("access_spvar(Node_NR)"));
                    break;
                  case Node_FNR:
                    emit_ccode(("access_spvar(Node_FNR)"));
                    break;
                  case Node_NF:
                    emit_ccode(("access_spvar(Node_NF)"));
                    break;
                  case Node_FIELDWIDTHS:
                    emit_ccode(("access_spvar(Node_FIELDWIDTHS)"));
                    break;
                  case Node_FS:
                    emit_ccode(("access_spvar(Node_FS)"));
                    break;
                  case Node_RS:
                    emit_ccode(("access_spvar(Node_RS)"));
                    break;
                  case Node_IGNORECASE:
                    emit_ccode(("access_spvar(Node_IGNORECASE)"));
                    break;
                  case Node_OFMT:
                    emit_ccode(("access_spvar(Node_OFMT)"));
                    break;
                  case Node_CONVFMT:
                    emit_ccode(("access_spvar(Node_CONVFMT)"));
                    break;
                  case Node_ORS:
                    emit_ccode(("access_spvar(Node_ORS)"));
                    break;
                  case Node_OFS:
                    emit_ccode(("access_spvar(Node_OFS)"));
                    break;
 
                }

                emit_ccode((") "));

                var_end:
		return ret_etype;

                }


        case Node_val:
                /* In processing constants/values registered by the parser,
                   first check if registered as number or string, then, based
                   on the type of expression which we're currently in, convert
                   from string->number, or number->string, or refer to it's node
                   declaration.
                   Also handle special case of sole string as a boolean 
                   (different from how C treats them) */

                /* ----- Initial type is number ----- */
                if ( tree->flags & (NUM|NUMBER) )  {
                  /* parent wants string node or number node */
                  if ( etype_parent & (EXPR_NEED_NODE|EXPR_STRING_VAL) )  {
                    if ( etype_parent & EXPR_STRING_VAL ) 
                      emit_ccode((" force_string2("));
                    emit_ccode((" constnode%d", tree->constID));
                    if ( etype_parent & EXPR_STRING_VAL )
                      emit_ccode((") "));
                  }
                  else 
                    emit_ccode((" %.10f ", tree->numbr));
                  return EXPR_NUMBER_VAL|EXPR_VAL_NODE;
                }

                /* ----- Initial type is string ----- */
                else {
                  if ( tree->flags & (STR|STRING) )  {
                    /* parent is a boolean expression or string being  
                       used as a conditional */
                    if (etype_parent & EXPR_BOOL )  {
                      if ( tree->stlen )
                        emit_ccode((" 1 "));
                      else
                        emit_ccode((" 0 "));
                    }
                    else if ( etype_parent & EXPR_NUMBER_VAL )
                      emit_ccode((" %.10f ", force_number(tree)));
                    else
                      /* default representation of strings is value node */  
                      emit_ccode((" constnode%d ", tree->constID));
                  }
                  return EXPR_STRING_VAL|EXPR_VAL_NODE;
                }


        /* For all following binary operations, bracketize both the left and
           right sub-expressions and recursively call tree_eval2() for each
           one.  Unary operations appropriately call tree_eval2() once */

        /* For number producing operations, based on wether a string
           or number node required, convert appropriately */

        /* Boolean operations: All result in 1.0 or 0.0 value, so return 
           EXPR_NUMBER_VAL flag */

	case Node_and:
                if ( etype_parent & (EXPR_STRING_VAL|EXPR_NEED_NODE) ) 
                  emit_ccode((" (tmp_node1->numbr="));
	        emit_ccode((" (")); 
	        tree_eval2(tree->lnode, EXPR_BOOL);
	        emit_ccode((") && ("));
	        tree_eval2(tree->rnode, EXPR_BOOL);
	        emit_ccode((") "));
                if (etype_parent & EXPR_STRING_VAL)
                  emit_ccode((", force_string2(tmp_node1))"));
                else if (etype_parent & EXPR_NEED_NODE) 
                  emit_ccode((", tmp_node1) "));
	        return EXPR_NUMBER_VAL;
		
	case Node_or:
                if ( etype_parent & (EXPR_STRING_VAL|EXPR_NEED_NODE) )
                  emit_ccode((" (tmp_node1->numbr="));
	        emit_ccode((" ("));
	        tree_eval2(tree->lnode, EXPR_BOOL);
	        emit_ccode((") || ("));
	        tree_eval2(tree->rnode, EXPR_BOOL);
	        emit_ccode((") "));
                if (etype_parent & EXPR_STRING_VAL)
                  emit_ccode((", force_string2(tmp_node1))"));
                else if (etype_parent & EXPR_NEED_NODE)
                  emit_ccode((", tmp_node1) "));
	        return EXPR_NUMBER_VAL;
		
	case Node_not: 
                if ( etype_parent & (EXPR_STRING_VAL|EXPR_NEED_NODE) )
                  emit_ccode((" (tmp_node1->numbr="));
	        emit_ccode((" !("));
	        tree_eval2(tree->lnode, EXPR_BOOL);
	        emit_ccode((") "));
                if (etype_parent & EXPR_STRING_VAL)
                  emit_ccode((", force_string2(tmp_node1))"));
                else if (etype_parent & EXPR_NEED_NODE)
                  emit_ccode((", tmp_node1) ")); 
	        return EXPR_NUMBER_VAL;
	

		/* Builtins */
	case Node_builtin:
                /* !!!!!!!! Note variable modification & optim tracking */
                emit_ccode((" 0/* >>> Expr: Node_builtin <<< */ "));
                /* Based on what builtin function parser node points to,
                   emit approp. code and call any new/modified/unchanged
                   functions in builtin.c */
		return;

	case Node_K_getline:
		/*return (do_getline(tree));*/

	case Node_in_array:
                if ( etype_parent & (EXPR_STRING_VAL|EXPR_NEED_NODE) )
                  emit_ccode((" (tmp_node1->numbr="));

                emit_ccode((" in_array2(__%s->lnode, ", tree->lnode->vname));
                /* Request child expr to return itself as a string value node */
                tree_eval2(tree->rnode, EXPR_STRING_VAL);
                emit_ccode((") "));
                 
                if (etype_parent & EXPR_STRING_VAL)
                  emit_ccode((", force_string2(tmp_node1))"));
                else if (etype_parent & EXPR_NEED_NODE)
                  emit_ccode((", tmp_node1) "));

		return EXPR_NUMBER_VAL;

	case Node_func_call:
                emit_ccode((" 0/* >>> Expr: Node_func_call <<< */ "));
		return;


        /* Calling array without subscript */
	case Node_var_array:
		fatal("attempt to use array `%s' in a scalar context", tree->vname);

	case Node_unary_minus:
                if ( etype_parent & (EXPR_STRING_VAL|EXPR_NEED_NODE) )
                  emit_ccode((" (tmp_node1->numbr="));
	        emit_ccode((" -("));
	        tree_eval2(tree->subnode, EXPR_NUMBER_VAL);
	        emit_ccode((") "));
                if (etype_parent & EXPR_STRING_VAL)
                  emit_ccode((", force_string2(tmp_node1))"));
                else if (etype_parent & EXPR_NEED_NODE)
                  emit_ccode((", tmp_node1) "));
	        return EXPR_NUMBER_VAL;
		
	case Node_cond_exp:
                {
                long oldblockid = blockid;

	        emit_ccode(("("));
	        tree_eval2(tree->lnode, EXPR_BOOL);
	        emit_ccode((") ? ("));
                
                if ( !(translator_state & TSTATE_PREWALK) ) 
                  blockid = ++blockidnum;
	        tree_eval2(tree->rnode->lnode, EXPR_NEED_NODE);
	        emit_ccode((") : ("));
	        tree_eval2(tree->rnode->rnode, EXPR_NEED_NODE);
	        emit_ccode((") "));
                
                blockid = oldblockid;
                return EXPR_VAL_NODE;
                }
	       
        /* Regular expression matching */ 
	case Node_match:
	case Node_nomatch:
        case Node_regex:      /* Match against $0 */
                {
                  /* If parent prefers string or val node, convert */
                  if ( etype_parent & (EXPR_STRING_VAL|EXPR_NEED_NODE) )
                    emit_ccode((" (tmp_node1->numbr="));

                  if ( tree->type == Node_regex )  {
                    /* To bind to runtime-created regexp node, look at ID
                       stored in syntax tree */                    
                    emit_ccode((" match_op(regnode%d, NULL, ", tree->re_nodeID));
                    /* Now, emit node pointer to reg expression */
                    emit_ccode((" regnode%d->re_exp)", tree->re_nodeID));      
                  }
                  
                  /* Matching two operands */
                  else  {
                    int rgnd_id = tree->rnode->re_nodeID;
              
                    /* If ID is zero, then a reg_exp node has to be created
                       at runtime */ 
                    if ( rgnd_id == 0 )  {
                      emit_ccode((" 0/* >>> Non-const regexp <<< */ "));
                    }

                    else  {

                      /* Though the regexp node is to be intialized (at runtime)
                         as a sole regexp match vs. $0, change it's type based
                         on what the syntax tree says.  Output this as a
                         comma expression in C */
                      if ( tree->type == Node_match )
                        emit_ccode((" (regnode%d->type = Node_match, ", rgnd_id));
                      else
                        emit_ccode((" (regnode%d->type = Node_nomatch, ", rgnd_id));

                      emit_ccode((" match_op(regnode%d, ", rgnd_id));

                      /* Evaluate expression to be compared and ask child to
                         return string value node */
                      tree_eval2(tree->lnode, EXPR_STRING_VAL);
                      emit_ccode((", regnode%d->re_exp)) ", rgnd_id));
                    }
                  }

                  if (etype_parent & EXPR_STRING_VAL)
                    emit_ccode((", force_string2(tmp_node1))"));
                  else if (etype_parent & EXPR_NEED_NODE)
                    emit_ccode((", tmp_node1) ")); 

                  /* Since a comparison, result is 1.0 or 0.0 */ 
                  return EXPR_NUMBER_VAL;
                }


	case Node_func:
		fatal("function `%s' called with space between name and (,\n%s",
			tree->lnode->param,
			"or used in other expression context");


        /* Variable assignments, and assign/operate expressions */

	case Node_assign:
        case Node_preincrement:
        case Node_predecrement:
        case Node_postincrement:
        case Node_postdecrement:
        case Node_assign_exp:
        case Node_assign_times:
        case Node_assign_quotient:
        case Node_assign_mod:
        case Node_assign_plus:
        case Node_assign_minus:
		{ 
             
                flags_type expr_hi_etype = 0;
                flags_type ret_etype = EXPR_VAL_NODE;
                short assign_as_num = 0;
                short postop_wantnode = 0;
                short postop_wantnum = 0;
      
                /* "Look-ahead" and examine expression to be assigned and
                   emit optimal code to access assignment expression
                   and assign it to the left variable.
                   (in cases where right expression can be both a number
                    or a value node, assign as number (if it's safe =>
                    child like Node_var handler would return EXPR_NUMBER_VAL
                    if it's safe, knowing that parent is Node_assign)
                   since assign_var*_num() is faster than assign_var*_var()
                */
      
                if ( tree->type == Node_assign )
                  /* Unlike other pre-walks of expressions, pass EXPR_VAL_NODE
                     instead of EXPR_INDIFF for child to behave properly in
                     assignment special cases */
                  expr_hi_etype = predict_etype(tree->rnode, EXPR_VAL_NODE|
                                                             TSTATE_PREWALK|
                                                             PARENT_IS_ASGN); 
                else
                  expr_hi_etype = EXPR_NUMBER_VAL;

                if ( expr_hi_etype & EXPR_NUMBER_VAL )
                  assign_as_num = 1;


                /* Obvious errors: division/mod by zero */
                if ( (tree->type == Node_assign_quotient ||
                      tree->type == Node_assign_mod) &&
                     tree->rnode->type == Node_val &&
                     ( (tree->rnode->flags & NUM && tree->rnode->numbr == 0)
                       ||
                       (tree->rnode->flags & STR && tree->rnode->stlen == 0)
                     )
                   )
                  fatal("division by zero attempted");


                /* Firstly, attempt to do native number variable operation
                   using native C operators (=, ++, *=, etc).  This
                   is only done if left variable is regular, and it is NEVER
                   used/assigned a non-number */
                if ( tree->lnode->type == Node_var &&
                     !(translator_state & TSTATE_PREWALK) )  {
                  if ( !(tree->lnode->flags4 & USED_AS_NONNUM) )  {
                  
                    if ( tree->type == Node_preincrement )
                      emit_ccode((" ++"));
                    else if ( tree->type == Node_predecrement )
                      emit_ccode((" --"));
                    else if ( tree->type == Node_assign_exp )
                      emit_ccode((" pow((double) "));
                    else if ( tree->type == Node_assign_mod )
                      emit_ccode((" fmod((double) "));
                      
                    emit_ccode((" num__%s",tree->lnode->vname));
                    
                    if ( tree->type == Node_postincrement )
                      emit_ccode(("++"));
                    if ( tree->type == Node_postdecrement )
                      emit_ccode(("--"));
                   
                    switch ( tree->type )  {
                      case Node_assign:
                        emit_ccode((" = "));
                        break;
                      case Node_assign_times:
                        emit_ccode((" *= "));
                        break;
                      case Node_assign_plus:
                        emit_ccode((" += "));
                        break;
                      case Node_assign_minus:
                        emit_ccode((" -= "));
                        break;
                      case Node_assign_quotient:
                        emit_ccode((" /= "));
                        break;
                      case Node_assign_exp:
                      case Node_assign_mod:
                        emit_ccode((", (double) ("));
                        break;
                    }
                     
                    if ( !(tree->type == Node_preincrement ||
                           tree->type == Node_predecrement ||
                           tree->type == Node_postincrement ||
                           tree->type == Node_postdecrement) )
                      tree_eval2(tree->rnode, EXPR_NUMBER_VAL);
                      
                    if ( tree->type == Node_assign_exp ||
                         tree->type == Node_assign_mod )
                      emit_ccode((")) "));
                    else
                      emit_ccode((" "));
                   
                    ret_etype = EXPR_NUMBER_VAL;
                    goto assign_end;
                  } 
                }


                /* Assigning a number expression to variable 
                   (or assign/operate expressions) */
                if ( assign_as_num )  { 

                  ret_etype |= EXPR_NUMBER_VAL;

                  /* First, handle post-increment/decrement */
                  if ( (tree->type == Node_postincrement ||
                       tree->type == Node_postdecrement) &&
                       !(etype_parent & EXPR_STMT)
                     )  {
                    if ( etype_parent & (EXPR_STRING_VAL|EXPR_NEED_NODE) )  { 
                      emit_ccode(("(unref(tmp_node2), tmp_node2=make_number("));
                      tree_eval2(tree->lnode, EXPR_NUMBER_VAL);
                      emit_ccode(("), "));
                      postop_wantnode = 1;
                    }
                    else  {
                      emit_ccode(("(tmp_numbr=")); 
                      tree_eval2(tree->lnode, EXPR_NUMBER_VAL);
                      emit_ccode((", "));
                      postop_wantnum = 1;
                    } 
                  }

                  /* Now, based on parent type, pick up string, number
                     or value node version of assignment result */

                  else if ( etype_parent & EXPR_STRING_VAL )  
                    emit_ccode((" force_string2"));
 
                  /* Assignment operation's result is a number unless
                     assigment is the whole statement, parent wants a
                     node */ 
                  else  {
                    if ( !(etype_parent & (EXPR_STMT|EXPR_NEED_NODE)) ) 
                      emit_ccode((" peek_number"));
                    else
                      ret_etype |= EXPR_VAL_NODE;
                  }
                  if ( tree->lnode->type == Node_var )
                    emit_ccode(("(assign_var_num("));
                  else
                    emit_ccode(("(assign_var2_num("));
                }


                /* Right expression is a value node (number or string):
                   Assigned var just gets the unconverted value node
                   of that expression.  
                   Then, based on parent, convert result appropriately.
                   If parent requires no conversion, then return pointer
                   to value node */
                else  {
                                
                  /* Parent is a number requiring operation */
                  if ( etype_parent & EXPR_NUMBER_VAL )  {
                    if ( expr_hi_etype & EXPR_NUMBER_VAL ) 
                      emit_ccode((" peek_number("));
                    else
                      emit_ccode((" force_number2("));
                  }
                  /* Parent is a boolean operation */
                  else if ( etype_parent & EXPR_BOOL )
                    emit_ccode((" bool_var("));
                  /* Parent is a string operation (force if right expr is
                     not a string) */ 
                  else if ( (etype_parent & EXPR_STRING_VAL) &&
                            !(expr_hi_etype & EXPR_STRING_VAL) )
                    emit_ccode((" force_string2("));
                  else
                    emit_ccode((" ("));
                 
                  if ( tree->lnode->type == Node_var )
                    emit_ccode(("assign_var_var("));
                  else
                    emit_ccode(("assign_var2_var(")); 

                  ret_etype |= EXPR_VAL_NODE;
                }

                /* Now, based on the type of variable, call appropriate version
                   of assign_var to actually perform assignment.  This
                   part prints out the parameters for assign_var_{var,num}() */

                switch ( tree->lnode->type )  {

                  case Node_var:
                    emit_ccode(("addr_var(__%s), ", tree->lnode->vname)); 
                    break;

                  case Node_field_spec:
                    emit_ccode(("Node_field_spec, NULL, (int) ("));
                    /* Force the field spec expression to be number-biased 
                       by sending number as parent type */
                    tree_eval2(tree->lnode->lnode, EXPR_NUMBER_VAL);
                    emit_ccode(("), NULL, "));
                    break;
               
                  case Node_subscript:
                    emit_ccode(("Node_subscript, __%s, 0, ", tree->lnode->lnode->vname));
                    tree_eval2(tree->lnode->rnode, EXPR_STRING_VAL);
                    emit_ccode((", "));
                    break;

                  case Node_NR:
                     emit_ccode(("Node_NR, NULL, 0, NULL, "));
                     break;
                  case Node_FNR:
                     emit_ccode(("Node_FNR, NULL, 0, NULL, "));
                     break;
                  case Node_NF:
                     emit_ccode(("Node_NF, NULL, 0, NULL, "));
                     break;
                  case Node_FIELDWIDTHS:
                     emit_ccode(("Node_FIELDWIDTHS, NULL, 0, NULL, "));
                     break;
                  case Node_FS:
                     emit_ccode(("Node_FS, NULL, 0, NULL, "));
                     break;
                  case Node_RS:
                     emit_ccode(("Node_RS, NULL, 0, NULL, "));
                     break;
                  case Node_IGNORECASE:
                     emit_ccode(("Node_IGNORECASE, NULL, 0, NULL, "));
                     break;
                  case Node_OFMT:
                     emit_ccode(("Node_OFMT, NULL, 0, NULL, "));
                     break;
                  case Node_CONVFMT:
                     emit_ccode(("Node_CONVFMT, NULL, 0, NULL, "));
                     break;
                  case Node_ORS:
                     emit_ccode(("Node_ORS, NULL, 0, NULL, "));
                     break;
                  case Node_OFS:
                     emit_ccode(("Node_OFS, NULL, 0, NULL, "));
                     break;

                }  

                /* Print out expression to be assigned based on type
                   of assignment */
                /* EXPR_VAL_NODE to all operations except
                   assignments themselves indicates that parent
                   wants default version/type of child expr */
                switch ( tree->type )  {

                  case Node_assign:
                    if ( assign_as_num ) 
                      tree_eval2(tree->rnode, EXPR_NUMBER_VAL);
                    else
                      tree_eval2(tree->rnode, EXPR_NEED_NODE|PARENT_IS_ASGN);
                    break;

                  /* For assign/operate expressions, emit code to access number
                     part of variable and apply arithmetic operation to it */

                  case Node_preincrement:
                  case Node_postincrement:
                    tree_eval2(tree->lnode, EXPR_NUMBER_VAL);
                    emit_ccode((" + 1"));
                    break;

                  case Node_predecrement:
                  case Node_postdecrement:
                    tree_eval2(tree->lnode, EXPR_NUMBER_VAL);
                    emit_ccode((" - 1"));
                    break;

                  case Node_assign_exp:
                    emit_ccode(("pow((double) ("));
                    tree_eval2(tree->lnode, EXPR_NUMBER_VAL);
                    emit_ccode(("), (double) ("));
                    tree_eval2(tree->rnode, EXPR_NUMBER_VAL);
                    emit_ccode(("))"));
                    break;

                  case Node_assign_times:
                  case Node_assign_plus:
                  case Node_assign_minus:
                  case Node_assign_quotient:
                    tree_eval2(tree->lnode, EXPR_NUMBER_VAL);
                    switch (tree->type)  {
                      case Node_assign_times:
                        emit_ccode((" * "));
                        break;
                      case Node_assign_plus:
                        emit_ccode((" + "));
                        break;
                      case Node_assign_minus:
                        emit_ccode((" - "));
                        break;
                      case Node_assign_quotient:                        
                        emit_ccode((" / "));                        
                        break;
                    }
                    tree_eval2(tree->rnode, EXPR_NUMBER_VAL);
                    break;
                  
                  case Node_assign_mod:
                    emit_ccode(("fmod((double) ("));
                    tree_eval2(tree->lnode, EXPR_NUMBER_VAL);
                    emit_ccode(("), (double) ("));
                    tree_eval2(tree->rnode, EXPR_NUMBER_VAL);
                    emit_ccode(("))"));
                    break;
                }

                /* Print out approp. # of closing variables */
                emit_ccode((")) "));

                /* Handle post-increment/decrement */
                if ( postop_wantnum )
                  emit_ccode((", tmp_numbr) "));
                else if ( postop_wantnode )  {
                  if ( etype_parent & EXPR_STRING_VAL )
                    emit_ccode((", force_string2(tmp_node2)) "));
                  else
                    emit_ccode((", tmp_node2) "));
                }

                /* Update variable's type status in parse tree
                   (shouldn't do it on an expression type prediction walk) */
                if ( !(etype_parent & TSTATE_PREWALK) )  {
                
                  /* Non-regular variables being assigned to */
                  if ( (tree->lnode->type == Node_NF ||
                        tree->lnode->type == Node_NR ||
                        tree->lnode->type == Node_FNR) &&
                        !(expr_hi_etype & EXPR_NUMBER_VAL) )  {
                    /* A numeric special variable assigned a non-number */
                    if ( translator_state & TSTATE_PREWALK )
                      /* use flags2 for prewalk tracking */
                      tree->lnode->flags2 = NUMSPVAR_ASSIGNED;
                    else
                      tree->lnode->flags = NUMSPVAR_ASSIGNED;
                  }
                  else if ( tree->lnode->type == Node_field_spec )  {
                    /* A field variable assigned a non-string */
                    if ( !(expr_hi_etype & EXPR_STRING_VAL) )  {
                      if ( translator_state & TSTATE_PREWALK )
                        FVAR_ASSIGNED2 = 1;
                      else
                        FVAR_ASSIGNED = 1;
                    }
                  }
                  else if ( tree->lnode->type == Node_CONVFMT )  {
                    if ( !(translator_state & (TSTATE_PREWALK|TSTATE_OPTCHECK)) )
                      CONVFMT_ASSIGNED = 1;
                  } 
                                                                                         
                  /* Regular variables being assigned to */
                  else if ( tree->lnode->type == Node_var )  {
                  
                    /* Assigning a number */
                    if ( expr_hi_etype & EXPR_NUMBER_VAL )  { 
                      if ( (translator_state & TSTATE_PREWALK) &&
                           !(translator_state & TSTATE_END) )
                        tree->lnode->flags2 |= NUM_ASSIGNED;
                      else if ( translator_state & TSTATE_OPTCHECK )
                        tree->lnode->flags3 |= NUM_ASSIGNED;
                      else  {
                        /* All status flags are assigned '=' and not '|='
                           to remove any string status flags */
                        if ( blockid == UNCOND_BLOCK || 
                             (tree->lnode->flags & NUM_ASSIGNED_UNCOND) )
                          /* If already assigned a number unconditionally,
                             propogate that status regardless of current block */
                          tree->lnode->flags = NUM_ASSIGNED_UNCOND;
                        else {
                          tree->lnode->blockID = blockid;
                          if ( tree->lnode->flags & NUM_CURRENT_UNCOND )
                            tree->lnode->flags = NUM_ASSIGNED|NUM_CURRENT_UNCOND;
                          else
                            tree->lnode->flags = NUM_ASSIGNED; 
                        }
                      }                    
                    }
                     
                    /*  Assigning a string value node */
                    else if ( expr_hi_etype & EXPR_STRING_VAL )  {
                      if ( translator_state & TSTATE_PREWALK )  {
                        if ( !(translator_state & TSTATE_END) )
                          tree->lnode->flags2 |= STR_ASSIGNED;
                        tree->lnode->flags4 |= USED_AS_NONNUM; 
                      }
                      else if ( translator_state & TSTATE_OPTCHECK )
                        tree->lnode->flags3 |= STR_ASSIGNED;
                      else  {
                        /* All status flags are assigned '=' and not '|='
                           to remove any number status flags */
                        if ( blockid == UNCOND_BLOCK ||
                             (tree->lnode->flags & STR_ASSIGNED_UNCOND) )
                           /* If already assigned a string unconditionally,
                              propogate that status regardless of current block */
                           tree->lnode->flags = STR_ASSIGNED_UNCOND;
                        else {
                          tree->lnode->blockID = blockid;
                          if ( tree->lnode->flags & STR_CURRENT_UNCOND )
                            tree->lnode->flags = STR_ASSIGNED|STR_CURRENT_UNCOND;
                          else
                            tree->lnode->flags = STR_ASSIGNED;
                        }      
                      }
                    }
                    
                    /* Assigning a regular variable => propogate it's status*/
                    else if ( tree->rnode->type == Node_var )  {
                      if ( translator_state & TSTATE_PREWALK )  {
                        /* At this point, should propogate all possible
                           states of right variable */
                        if ( !(translator_state & TSTATE_END) )
                          tree->lnode->flags2 |= tree->rnode->flags2;
                        tree->lnode->flags4 |= tree->rnode->flags4;
                      }
                      else if ( translator_state & TSTATE_OPTCHECK )
                        tree->lnode->flags3 |= tree->rnode->flags2;
                      else  {
                        tree->lnode->flags = tree->rnode->flags;
                        tree->lnode->blockID = tree->rnode->blockID;
                      }
                    } 
                     
                    /* Assigning a non-regular variable */  
                    else  {
                      if ( translator_state & TSTATE_PREWALK )  {
                        /* If no field variable has been assigned a non-string
                           so far, then => (field_Var <=> string) */
                        if ( tree->rnode->type == Node_field_spec &&
                             !FVAR_ASSIGNED2 )  {
                          if ( !(translator_state & TSTATE_END) )
                            tree->lnode->flags2 |= STR_ASSIGNED;
                          tree->lnode->flags4 |= USED_AS_NONNUM;
                        }
                        else if ( (tree->rnode->type == Node_NF ||
                                   tree->rnode->type == Node_NR ||
                                   tree->rnode->type == Node_FNR) &&
                                  !(tree->rnode->flags2 & NUMSPVAR_ASSIGNED))  {
                          if ( !(translator_state & TSTATE_END) )
                            tree->lnode->flags2 |= NUM_ASSIGNED;
                        }
                        else  {
                          if ( !(translator_state & TSTATE_END) )  
                            tree->lnode->flags2 |= NONREGVAR_ASSIGNED;
                          tree->lnode->flags4 |= USED_AS_NONNUM;
                        }
                      }
                      else if ( translator_state & TSTATE_OPTCHECK )
                        tree->lnode->flags3 |= NONREGVAR_ASSIGNED;
                      else
                        tree->lnode->flags = NONREGVAR_ASSIGNED;
                    };
                    
                    /* reset flags used for safe optim w.r.t loops */
                    if ( !(translator_state & TSTATE_OPTCHECK ) )
                      tree->lnode->flags3 = 0;
                  }    
                }
               
                /* Also update variable status if parent was a number or
                   string operation */
                if ( !(etype_parent & TSTATE_PREWALK) )  {
                  if ( etype_parent & EXPR_NUMBER_VAL )  { 
                    UPDATE_NUMCURR(tree->lnode);
                  }
                  else if ( (etype_parent & EXPR_STRING_VAL) &&
                            !(tree->type == Node_postincrement ||
                              tree->type == Node_postdecrement) )  { 
                    UPDATE_STRCURR(tree->lnode);
                  }
                }
 
                /* Set flags4 if this assignment is an operand to a parent
                   wanting a non-number
                   (used for native variable optimization) */
                if ( (translator_state & TSTATE_PREWALK) &&
                     tree->lnode->type == Node_var )  {
                  if ( etype_parent & (EXPR_STRING_VAL|EXPR_NEED_NODE) )
                    tree->lnode->flags4 |= USED_AS_NONNUM;
                }
                
                assign_end:
                return ret_etype;
                }



	case Node_concat:
                emit_ccode((" 0/* >>> Node_concat <<< */ "));
                return EXPR_VAL_NODE|EXPR_STRING_VAL;


        
        /* BINARY COMPARISON OPERATIONS :  These operations behave differently
           based on the CONTEXT of the expression (that is, the types of the
           left and right operands, the types/values of the variables and 
           fields, determine if the following operations occur under a NUMERIC
           or STRING context - refer to 'The AWK Language: pg 44-45'
           
           First, do a "look-ahead" expression walk of the left and right 
           sub-trees to determine types involved in their expressions, then
           proceed accordingly.
        */ 

        case Node_geq:
        case Node_leq:
        case Node_greater:
        case Node_less:
        case Node_notequal:
        case Node_equal: 
                { 
                  short compare_as_num = 0;        /* Method of comparison: 
                                                      Numeric or not */

                  /* Type flags for left/right expressions */
                  flags_type left_hi_type;
                  flags_type right_hi_type;

                  /* "Look-ahead" walk of expression sub-trees */
                  left_hi_type = predict_etype(tree->lnode, EXPR_INDIFF|
                                                            TSTATE_PREWALK|
                                                            PARENT_IS_COMP);
                  right_hi_type = predict_etype(tree->rnode, EXPR_INDIFF|
                                                             TSTATE_PREWALK|
                                                             PARENT_IS_COMP);
  
                  /* Case when numeric comparison is used:
                     Both left and right expressions has a numeric operation
                     operation at the highest level or is a variable which
                     was assigned a number */
                  if ( (left_hi_type & EXPR_NUMBER_VAL) && 
                       (right_hi_type & EXPR_NUMBER_VAL) )
                    compare_as_num = 1; 

                  /* Since any comparison results in a number value,
                     convert appropriately if parent desires so */
                  if ( etype_parent & (EXPR_STRING_VAL|EXPR_NEED_NODE) )
                    emit_ccode((" (tmp_node1->numbr="));

                  /* Numeric comparison */
                  if ( compare_as_num )  {
                     emit_ccode((" ("));
                     tree_eval2(tree->lnode, EXPR_NUMBER_VAL);
                     switch ( tree->type )  {
                       case Node_geq:
                         emit_ccode((") >= ("));
                         break;
                       case Node_leq:
                         emit_ccode((") <= ("));
                         break;
                       case Node_greater:
                         emit_ccode((") > ("));
                         break;
                       case Node_less:
                         emit_ccode((") < ("));
                         break;
                       case Node_notequal:
                         emit_ccode((") != ("));
                         break;
                       case Node_equal:
                         emit_ccode((") == ("));
                         break;
                     }
                     tree_eval2(tree->rnode, EXPR_NUMBER_VAL);
                     emit_ccode((") "));
                  }


                  /* Not a sure numeric comparison => Convert all 
                     operands to nodes as fit and call runtime cmp_nodes()
                     function */
                  else  {
                    emit_ccode((" cmp_nodes("));
                    tree_eval2(tree->lnode, EXPR_NEED_NODE|PARENT_IS_COMP);
                    emit_ccode((", "));
                    tree_eval2(tree->rnode, EXPR_NEED_NODE|PARENT_IS_COMP);

                    /* Print out comparison type */
                    switch ( tree->type )  {
                        case Node_geq:
                           emit_ccode((") >= 0 "));
                           break;
                        case Node_leq:
                           emit_ccode((") <= 0 "));
                           break;
                         case Node_greater:
                           emit_ccode((") > 0 "));
                           break;
                         case Node_less:
                           emit_ccode((") < 0 "));
                           break;
                         case Node_notequal:
                           emit_ccode((") != 0 "));
                           break;
                         case Node_equal:
                           emit_ccode((") == 0 "));
                           break;
                    }
 
                  } 

                  if (etype_parent & EXPR_STRING_VAL)
                    emit_ccode((", force_string2(tmp_node1))"));
                  else if (etype_parent & EXPR_NEED_NODE)
                    emit_ccode((", tmp_node1) "));

                  /* The result of any comparison is 1.0 or 0.0 => number */
                  return EXPR_NUMBER_VAL;
            
                }


	case Node_exp:
        case Node_times:
        case Node_quotient:
        case Node_mod:
        case Node_plus:
        case Node_minus:
                /* Check for any division by zero constants (unlike
                   interpreter, can't check all div by zero cases) */
                if ( (tree->type == Node_quotient || tree->type == Node_mod) &&
                     tree->rnode->type == Node_val &&
                     ( (tree->rnode->flags & NUM && tree->rnode->numbr == 0)
                       ||
                       (tree->rnode->flags & STR && tree->rnode->stlen == 0)
                     )
                   )  
                  fatal("division by zero attempted");

                if ( etype_parent & (EXPR_STRING_VAL|EXPR_NEED_NODE) )
                    emit_ccode((" (tmp_node1->numbr="));

                if ( tree->type == Node_exp )
                  emit_ccode((" pow((double) ("));
                else if ( tree->type == Node_mod )
                  emit_ccode((" fmod((double) ("));
                else
                  emit_ccode((" ("));
                
                tree_eval2(tree->lnode, EXPR_NUMBER_VAL);
 
                switch (tree->type)  {
                  case Node_exp:
                  case Node_mod:
                    emit_ccode((") , (double) ("));
                    break;
                  case Node_times:
                    emit_ccode((") * ("));
                    break;
                  case Node_quotient:
                    emit_ccode((") / ("));
                    break;
                  case Node_plus:
                    emit_ccode((") + ("));
                    break;
                  case Node_minus:
                    emit_ccode((") - ("));
                    break; 
                }

                tree_eval2(tree->rnode, EXPR_NUMBER_VAL);

                if ( tree->type == Node_exp || tree->type == Node_mod )
                  emit_ccode((")) "));
                else
                  emit_ccode((") "));

                if (etype_parent & EXPR_STRING_VAL)
                  emit_ccode((", force_string2(tmp_node1))"));
                else if (etype_parent & EXPR_NEED_NODE)
                  emit_ccode((", tmp_node1) "));
 
                return EXPR_NUMBER_VAL;


	default:
		fatal("illegal type (%d) in tree_eval", tree->type);
	}
	return 0; 

}



static NODE *
func_call(name, arg_list)
NODE *name;		/* name is a Node_val giving function name */
NODE *arg_list;		/* Node_expression_list of calling args. */
{
#if 0
	register NODE *arg, *argp, *r;
	NODE *n, *f;
	jmp_buf volatile func_tag_stack;
	jmp_buf volatile loop_tag_stack;
	int volatile save_loop_tag_valid = 0;
	NODE **volatile save_stack, *save_ret_node;
	NODE **volatile local_stack = NULL, **sp;
	int count;
	extern NODE *ret_node;

	/*
	 * retrieve function definition node
	 */
	f = lookup(name->stptr);
	if (!f || f->type != Node_func)
		fatal("function `%s' not defined", name->stptr);
#ifdef FUNC_TRACE
	fprintf(stderr, "function %s called\n", name->stptr);
#endif
	count = f->lnode->param_cnt;
	if (count)
		emalloc(local_stack, NODE **, count*sizeof(NODE *), "func_call");
	sp = local_stack;

	/*
	 * for each calling arg. add NODE * on stack
	 */
	for (argp = arg_list; count && argp != NULL; argp = argp->rnode) {
		arg = argp->lnode;
		getnode(r);
		r->type = Node_var;
		/*
		 * call by reference for arrays; see below also
		 */
		if (arg->type == Node_param_list)
			arg = stack_ptr[arg->param_cnt];
		if (arg->type == Node_var_array)
			*r = *arg;
		else {
			n = (NODE *) tree_eval(arg);
			r->lnode = dupnode(n);
			r->rnode = (NODE *) NULL;
			free_temp(n);
  		}
		*sp++ = r;
		count--;
	}
	if (argp != NULL)	/* left over calling args. */
		warning(
		    "function `%s' called with more arguments than declared",
		    name->stptr);
	/*
	 * add remaining params. on stack with null value
	 */
	while (count-- > 0) {
		getnode(r);
		r->type = Node_var;
		r->lnode = Nnull_string;
		r->rnode = (NODE *) NULL;
		*sp++ = r;
	}

	/*
	 * Execute function body, saving context, as a return statement
	 * will longjmp back here.
	 *
	 * Have to save and restore the loop_tag stuff so that a return
	 * inside a loop in a function body doesn't scrog any loops going
	 * on in the main program.  We save the necessary info in variables
	 * local to this function so that function nesting works OK.
	 * We also only bother to save the loop stuff if we're in a loop
	 * when the function is called.
	 */
	if (loop_tag_valid) {
		int junk = 0;

		save_loop_tag_valid = (volatile int) loop_tag_valid;
		PUSH_BINDING(loop_tag_stack, loop_tag, junk);
		loop_tag_valid = 0;
	}
	save_stack = stack_ptr;
	stack_ptr = local_stack;
	PUSH_BINDING(func_tag_stack, func_tag, func_tag_valid);
	save_ret_node = ret_node;
	ret_node = Nnull_string;	/* default return value */
	if (setjmp(func_tag) == 0)
		(void) interpret(f->rnode);

	r = ret_node;
	ret_node = (NODE *) save_ret_node;
	RESTORE_BINDING(func_tag_stack, func_tag, func_tag_valid);
	stack_ptr = (NODE **) save_stack;

	/*
	 * here, we pop each parameter and check whether
	 * it was an array.  If so, and if the arg. passed in was
	 * a simple variable, then the value should be copied back.
	 * This achieves "call-by-reference" for arrays.
	 */
	sp = local_stack;
	count = f->lnode->param_cnt;
	for (argp = arg_list; count > 0 && argp != NULL; argp = argp->rnode) {
		arg = argp->lnode;
		if (arg->type == Node_param_list)
			arg = stack_ptr[arg->param_cnt];
		n = *sp++;
		if ((arg->type == Node_var || arg->type == Node_var_array)
		    && n->type == Node_var_array) {
			/* should we free arg->var_value ? */
			arg->var_array = n->var_array;
			arg->type = Node_var_array;
			arg->array_size = n->array_size;
			arg->table_size = n->table_size;
			arg->flags = n->flags;
		}
		/* n->lnode overlays the array size, don't unref it if array */
		if (n->type != Node_var_array)
			unref(n->lnode);
		freenode(n);
		count--;
	}
	while (count-- > 0) {
		n = *sp++;
		/* if n is an (local) array, all the elements should be freed */
		if (n->type == Node_var_array)
			assoc_clear(n);
		unref(n->lnode);
		freenode(n);
	}
	if (local_stack)
		free((char *) local_stack);

	/* Restore the loop_tag stuff if necessary. */
	if (save_loop_tag_valid) {
		int junk = 0;

		loop_tag_valid = (int) save_loop_tag_valid;
		RESTORE_BINDING(loop_tag_stack, loop_tag, junk);
	}

	if (!(r->flags & PERM))
		r->flags |= TEMP;
	return r;
#endif
}

