/*
 * This program comes with absolutely no guarantees and may
 * be used for whatever purpose,
 *
 *      Frederik H. Andersen - 1996
 *      Dansk Data Elektronik A/S
 *
 *      fha@dde.dk
 *
 */

/*
 * This program performs the decoding of transfer encoded text type
 * mime messages. The message in its entirety is read from stdin.
 * The decoded message is written to stdout; hence, this program
 * behaves as a filter which may be placed wherever convenient.
 * I particular like it in front of my slocal command in my .forward
 * file.
 *
 * It is assumed that the message has reached its point of final
 * delivery and at that point 8-bit text types can be handled
 * natively. Hence, the need for transfer-encodings is not present
 * any more.
 *
 * Only some cases are handled:
 *      - encoded header fields are decoded from QP or B encoding.
 *        The charset is assumed to be iso-8859-1
 *      - part or subparts of content-type text only are decoded
 *      - all other content-types are passed transparently
 *     
 */

/* extensive changes by Ričardas Čepas <rch@richard.eu.org>
 *        All charsets are converted to UTF-8
 */

#define _POSIX_C_SOURCE 2

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>

/* Some defines. Should have gone into a file by itself.
 */
#define MIMED_VERSION "mimedecode version 1.8"

#define lower(c)        ( isupper(c) ? tolower(c) : (c) )

#include "2UTF.h"

/* content types: */
#define UNDEF		0
#define TEXT		1
#define MULT    	2
#define MULT_DIGEST    	3
#define MULT_PGP    	4
#define MESG    	5

/* transfer encodings: */
#define QP	1
#define B64	2
#define BIT7	3
#define BIT8	4
#define BINARY	5

#define UN 255

#define LWSP " \t\n"

unsigned char b64_map[256] =
{
  UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN,	/* 0x00 - 0x0f, NUL - SI */
  UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN,	/* 0x10 - 0x1f, DLE - US */
  UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, 62, UN, UN, UN, 63,	/* 0x20 - 0x2f, SP  - / */
  52, 53, 54, 55, 56, 57, 58, 59, 60, 61, UN, UN, UN, 0x3d, UN, UN,	/* 0x30 - 0x3f, 0   - ? */
  UN, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14,		/* 0x40 - 0x4f, @   - O */
  15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, UN, UN, UN, UN, UN,	/* 0x50 - 0x5f, P   - _ */
  UN, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,	/* 0x60 - 0x6f, `   - o */
  41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, UN, UN, UN, UN, UN,	/* 0x70 - 0x7f, p   - DEL */

  UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN,	/* 0x80 - 0x8f */
  UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN,	/* 0x90 - 0x9f */
  UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN,	/* 0xA0 - 0xAf */
  UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN,	/* 0xB0 - 0xBf */
  UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN,	/* 0xC0 - 0xCf */
  UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN,	/* 0xD0 - 0xDf */
  UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN,	/* 0xE0 - 0xEf */
  UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN, UN,	/* 0xF0 - 0xFf */
};

char *especials = "()<>@,;:\\\"/[]?.=";

int reserve = 100;

/*******************************************************************/
int
parse_message (boundary)
/*******************************************************************/
     char *boundary;
{
  int c, saved_content_type;
  struct mime_header_type mhead =
  {0, 0, "\0", "\0", "\0"};
  int ret;

  if (Debug)
    if (boundary)
      fprintf (stderr, " - Entry parse_message with boundary=\"%s\" \n", boundary);
    else
      fprintf (stderr, " - Entry parse_message without boundary \n");

  /* parse header */
  if (!(ret = parse_header (&mhead)))
    {
      if (Debug)
	fprintf (stderr, "parse of message header failed: %d\n", ret);
      return (FALSE);
    }

  if (mhead.content_type == MULT || mhead.content_type == MULT_DIGEST)
    {
      if (Debug >= 2)
	{
	  fprintf (stderr, "message is multipart\n");
	  fprintf (stderr, "search header boundary line: %s\n", mhead.boundary);
	}

      /* search for the boundary line before parsing further */
      if ((ret = Seek_boundary (stdin, mhead.boundary, PUSH_BOUNDARY)) == TRUE)
	/* When the message is of type multipart we have to handle each
	 * part as an individual message.
	 * We use the boundary to identify the beginning of each part
	 */
	do
	  /* Check if part's own headers are missing */
	  if ((c = getc (stdin)) == '\n')
	    {
	      putc (c, charset_p->pipe);
	      if (mhead.content_type == MULT_DIGEST)
		/* default is message/rfc */
		ret = parse_message (mhead.boundary);
	      else
		{		/* default is text/plain; charset=us-ascii */
		  saved_content_type = mhead.content_type;
		  mhead.content_type = TEXT;
		  if (charset_p->USASCII_is_subset == NO)
		    Validate_charset ("us-ascii", 0);
		  ret = parse_body (&mhead, mhead.boundary);
		  mhead.content_type = saved_content_type;
		}
	    }
	  else
	    {
	      ungetc (c, stdin);
	      ret = parse_message (mhead.boundary);
	    }
	while (ret == TRUE);
      if (boundary)		/* && ret == END_BOUNDARY) */
	/* skip stuff after the last part of inner multipart message */
	ret = Seek_boundary (ret < OUTER_BOUNDARY ? stdin : NULL, boundary, 0);

    }
  else if (mhead.content_type == MESG)
    {
      /* When message is type message we have to handle the body as
       * an individual message.
       */
      ret = parse_message (boundary);
    }
  else
    /* plain message */
    {
      ret = parse_body (&mhead, boundary);
    }
  if (Debug)
    if (boundary)
      fprintf (stderr, " - parse_message with boundary=\"%s\" returned %i - \n", boundary, ret);
    else
      fprintf (stderr, " - parse_message without boundary returned %i - \n", ret);
  return (ret);
}



/*******************************************************************/
int
parse_body (mime_header_p, boundary)
/*******************************************************************/
/*     int type, encoding; */
     struct mime_header_type *mime_header_p;
     char *boundary;
{
  int c;
  int ret = FALSE;		/* ??? */

  /*struct charset_type *old_charset_p; */

  if (Debug)
    fprintf (stderr, " -- Entry parse_body -- \n");
  if (Debug >= 2)
    fprintf (stderr, " -- type: %d, enc: %d\n", mime_header_p->content_type, mime_header_p->transfer_encoding);

  if ((mime_header_p->content_type == TEXT) && \
      ((mime_header_p->transfer_encoding == QP) \
       ||(mime_header_p->transfer_encoding == B64) \
       ||(mime_header_p->transfer_encoding == BIT8) \
       ||(mime_header_p->transfer_encoding == BIT7) \
      ))
    {
      if (boundary)
	{
	  if (Debug >= 2)
	    fprintf (stderr, " -- with boundary: %s\n -- decoding -- \n", boundary);
	  if (mime_header_p->transfer_encoding == QP)
	    ret = decode_quoted_printable (stdin, charset_p->pipe, boundary);
	  else if (mime_header_p->transfer_encoding == B64)
	    ret = decode_base64 (stdin, charset_p->pipe, boundary);

	  if (ret == FALSE)
	    ret = Seek_boundary (stdin, boundary, 0);
	}
      else
	/* no boundary */
	{
	  if (Debug >= 2)
	    fprintf (stderr, " -- decoding -- \n");
	  if (mime_header_p->transfer_encoding == QP)
	    ret = decode_quoted_printable (stdin, charset_p->pipe, 0);
	  else if (mime_header_p->transfer_encoding == B64)
	    ret = decode_base64 (stdin, charset_p->pipe, 0);
	  else			/*if (mime_header_p->transfer_encoding == BIT8 || mime_header_p->transfer_encoding == BIT7) */
	    ret = 2, Pipe_to_UTF8 (stdin);
	}
    }
  else if (boundary)
    {
      if (Debug >= 2)
	fprintf (stderr, " -- with boundary: %s\n", boundary);
      if (Debug >= 2)
	fprintf (stderr, " -- skipping -- \n");
      /*  old_charset_p = charset_p;
         charset_p = &unknown_charset; */
      Validate_charset ("'unknown'", 0);
      ret = Seek_boundary (stdin, boundary, 0);
      Validate_charset (NULL, 0);
/*      charset_p = old_charset_p; */
    }
  else
    {
      if (Debug >= 2)
	fprintf (stderr, " -- skipping -- \n");
      Validate_charset ("'unknown'", 0);
      ret = 2;
      while ((c = getc (stdin)) != EOF)
	putc (c, charset_p->pipe);
    }
  return (ret);
}

char *
Add_to_buffer (struct buffer_type *buffer_p, char *string)
{
/*  size_t old_size = buffer_p->end - buffer_p->start + 1;

   if (Debug >= 9)
   fprintf (stderr, "2UTF: realloc %i += %i \n", old_size, strlen (string));
   buffer_p->start = xrealloc (buffer_p->start, (old_size + strlen (string)));
   *//* buffer_p->start changes *//*
     return (buffer_p->end = Stpcpy (buffer_p->start + old_size - 1, string));
   */
  size_t length;

  length = strlen (string);
  if (buffer_p->next + length - buffer_p->end > reserve)
    {
      fprintf (stderr, "2UTF: %s %li\n", buffer_overflow, (long)(buffer_p->next + length - buffer_p->end - reserve));
      length = buffer_p->end - buffer_p->next - reserve;
    }
  memcpy (buffer_p->next, string, length);
  return (buffer_p->next += length);
}

/*******************************************************************/
int
parse_header (mhp)
/*******************************************************************/
     struct mime_header_type *mhp;
{
  unsigned char *cp;
  unsigned char *nlbp;
  int c;
  int ct_charset_unknown = FALSE;	/* Warning about unknown charset in ct */
  size_t ct_header_charset_name_begin = 0, ct_header_charset_name_end = 0;
  int ct_is_changing = FALSE;	/* Content charset parameter needs change */
  int ct_read = FALSE;		/* Content-type field read */
  int ct_written = FALSE;	/* Content-type written */
  int cte_read = FALSE;		/* Content-transfer-encoding read */
  int cte_written = FALSE;	/* Content-transfer-encoding written */
  int headers_8bit = FALSE;	/* non-MIME-encrypted headers */
  int quoted;
  struct buffer_type headers_buffer =
  {NULL, NULL, NULL, NULL};

#define REALLOC_CHUNK 4000
  line.buf = headers_buffer.next = headers_buffer.start = headers_buffer.header_start = xrealloc (line.buf, line.length += reserve);
  headers_buffer.end = headers_buffer.start + line.length - reserve;
/*    = xmalloc (REALLOC_CHUNK * 2 + reserve);
   headers_buffer.end = headers_buffer.start + REALLOC_CHUNK * 2;
 */
  if (Debug)
    fprintf (stderr, " -- Entry parse_header - pass 1 -- \n");

  mhp->boundary[0] = '\001';
  mhp->content_type = UNDEF;
  mhp->transfer_encoding = BIT8;
  mhp->headers_charset[0] = '\0';
  mhp->content_charset[0] = '\0';

  /* first pass */
  while (TRUE)
    {
      if ((c = getc (stdin)) == EOF)
	{
	  Validate_charset ("us-ascii", 0);
	  fwrite (headers_buffer.start, 1, headers_buffer.next - headers_buffer.start, charset_p->pipe);
	  return (FALSE);
	}
    again:
      *headers_buffer.next++ = c;
      while (headers_buffer.next >= headers_buffer.end)
	{			/* realloc more memory */
	  size_t old_size = headers_buffer.end - headers_buffer.start;
	  size_t next_offset = headers_buffer.next - headers_buffer.start;
	  size_t header_start_offset = headers_buffer.header_start - headers_buffer.start;

	  if (Debug >= 9)
	    fprintf (stderr, "2UTF: realloc %li += %i \n", (long)old_size + reserve, REALLOC_CHUNK);
	  line.buf = headers_buffer.start = xrealloc (headers_buffer.start, old_size + reserve + REALLOC_CHUNK);
	  line.length += REALLOC_CHUNK;
	  headers_buffer.end = headers_buffer.start + old_size + REALLOC_CHUNK;
	  headers_buffer.next = headers_buffer.start + next_offset;
	  headers_buffer.header_start = headers_buffer.start + header_start_offset;
	  if (line.length != old_size + reserve + REALLOC_CHUNK && Debug)
	    fprintf (stderr, "2UTF: memory allocation error: \n  %li != %li \n", (long)line.length, (long)old_size + reserve + REALLOC_CHUNK);
	}
      if ((c > 0x7F || (c != '\n' && c != '\t' && iscntrl (c))) &&headers_8bit == FALSE)
	{
	  headers_8bit = TRUE;
	  if (Debug >= 2)
	    fprintf (stderr, " -- 8 bit bytes or control codes in headers detected \n");
	}
      if (c == '\n')		/* end of line reached */
	{
	  if ((c = getc (stdin)) == EOF)
	    {
	      Add_to_buffer (&headers_buffer, "\n");
	      Validate_charset ("us-ascii", 0);
	      fwrite (headers_buffer.start, 1, headers_buffer.next - headers_buffer.start, charset_p->pipe);
	      break;
	    }
	  else if (c == ' ' || c == '\t')
	    /* if next line begins with space 
	     * it is continuation of the same header
	     */
	    goto again;
	  else
	    ungetc (c, stdin);
	  *headers_buffer.next = '\0';	/* zero terminate the line buf */
/*        headers_buffer.header_start = headers_buffer.next; */

	  if (Debug >= 4)
	    fprintf (stderr, "headers_buffer: %s\n", headers_buffer.header_start);

	  if (!Strcase_has_prefix (headers_buffer.header_start, "content-type:"))
	    {
	      ct_read = TRUE;
	      cp = headers_buffer.header_start + strlen ("Content-Type:");
	      cp += strspn (cp, LWSP);

	      /* we are only doing decoding of text types */
	      if (Strcase_has_prefix (cp, "text/") == 0)
		{
		  if (Debug >= 3)
		    fprintf (stderr, "Content IS text\n");

		  mhp->content_type = TEXT;

		  if ((cp = strchr (cp, ';')))
		    while ((cp = strchr (cp, ';')))
		      if (Strcase_has_prefix (cp += strspn (cp, "; \t\n"), "charset=") == 0)
			{
			  ct_header_charset_name_begin = (cp += 8) - headers_buffer.header_start;
			  cp += strspn (cp, "'\"");
			  if ((cp[0] == 'x' || cp[0] == 'X') &&cp[1] == '-')
			    cp += 2;
			  ct_header_charset_name_end = cp + strcspn (cp, ";\n\r") - headers_buffer.header_start;
			  strncpy (mhp->content_charset, cp, MIN (79, strcspn (cp, "*'\"; \t\n\r")));
			  mhp->content_charset[ MIN (79, strcspn (cp, "*'\"; \t\n\r"))] = '\0';
			  if (Debug)
			    fprintf (stderr, " -- Content charset: %s\n", mhp->content_charset);

			  break;
			}
		}
	      else if (Strcase_has_prefix (cp, "multipart/") == 0)
		{
		  if (Strcase_has_prefix (cp += strlen ("multipart/"), "digest") == 0)
		    mhp->content_type = MULT_DIGEST;
		  else if (Strcase_has_prefix (cp, "pgp") == 0 ||Strcase_has_prefix (cp, "signed") == 0)
		    mhp->content_type = MULT_PGP;
		  else
		    mhp->content_type = MULT;

		  /* search for the boundary parameter */
		  while ((cp = strchr (cp, ';')))
		    if (Strcase_has_prefix (cp += 1 + strspn (1 + cp, LWSP), "boundary=") == 0)
		      /* skip ; and remove white space */
		      {
			cp += strlen ("boundary=");
			if (*cp == '"')
			  {
			    quoted = TRUE;
			    cp++;
			  }
			else
			  quoted = FALSE;
			*(cp = Stpncpy (mhp->boundary, cp, 78)) = '"';
			*++cp = '\0';
			*(cp = mhp->boundary + strcspn (mhp->boundary, quoted ? "\"\n" : "\"" LWSP)) = '\0';
			/* remove trailing white space */
			cp--;
			while ((char *) cp >= mhp->boundary &&(*cp == ' ' || *cp == '\t'))
			  *cp-- = '\0';

			if ((char *) cp <= mhp->boundary ||mhp->boundary[0] == '\0')
			  {
			    mhp->boundary[0] = '\0';
			    mhp->content_type = UNDEF;
			  }
			break;
		      }
		  if (cp == NULL)
		    {
		      if (Debug)
			fprintf (stderr, "boundary parameter missing ! \n");
		      mhp->content_type = UNDEF;
		      mhp->boundary[0] = '\0';
		    }
		}
	      else if (Strcase_has_prefix (cp, "message/") == 0)
		{
		  mhp->content_type = MESG;
		}
	      nlbp = headers_buffer.header_start;
	    }
	  else if (!Strcase_has_prefix (headers_buffer.header_start, "content-transfer-encoding:"))
	    {
	      cte_read = TRUE;

	      /* Remember the encoding in mhp! */
	      cp = headers_buffer.header_start + strlen ("content-transfer-encoding:");
	      cp += strspn (cp, LWSP);	/* remove white space */
	      {
		if (!Strcase_has_prefix (cp, "quoted-printable"))
		  {
		    if (Debug >= 3)
		      fprintf (stderr, "Transfer-encoding IS QP\n");
		    mhp->transfer_encoding = QP;
		  }
		else if (!Strcase_has_prefix (cp, "base64"))
		  {
		    if (Debug >= 3)
		      fprintf (stderr, "Transfer-encoding IS B64\n");
		    mhp->transfer_encoding = B64;
		  }
		else if (!Strcase_has_prefix (cp, "7bit"))
		  {
		    if (Debug >= 3)
		      fprintf (stderr, "Transfer-encoding IS 7bit\n");
		    mhp->transfer_encoding = BIT7;
		  }
		else if (!Strcase_has_prefix (cp, "8bit"))
		  {
		    if (Debug >= 3)
		      fprintf (stderr, "Transfer-encoding IS 8bit\n");
		    mhp->transfer_encoding = BIT8;
		  }
		else if (!Strcase_has_prefix (cp, "binary"))
		  {
		    if (Debug >= 3)
		      fprintf (stderr, "Transfer-encoding IS BINARY\n");
		    mhp->transfer_encoding = BINARY;
		  }
		else
		  {
		    if (Debug >= 3)
		      fprintf (stderr, "Transfer-encoding IS UNKNOWN\n");
		    mhp->transfer_encoding = UNDEF;
		  }
	      }
	      nlbp = NULL;
	    }
	  else
	    {
	      if ((nlbp = decode_header_line (QUERY_CHARSET_NAME, &headers_buffer)))
		{
		  if ((nlbp[0] == 'x' || nlbp[0] == 'X') &&nlbp[1] == '-')
		    nlbp += 2;
		  strncpy (mhp->headers_charset, nlbp, MIN (strcspn (nlbp, "*"), 79));
		  mhp->headers_charset[ MIN (strcspn (nlbp, "*"), 79)] = '\0';
		  if (Debug)
		    fprintf (stderr, " -- Headers's charset: %s\n", mhp->headers_charset);
		}
	      nlbp = headers_buffer.header_start;
	    }

	  if (Debug >= 7)
	    fprintf (stderr, "Adding to buffer: %s\n", nlbp);

/* Generally, we will output _all_ lines as they are read (and
 * decoded).  The exception is the Content-transfer-encoding (cte)
 * and Content-Type header lines:
 * We will only output cte when the Content-type has been read or
 * when the header/body boundary have been reached.
 */
	  if (nlbp)
	    headers_buffer.header_start = headers_buffer.next;
	  else
	    headers_buffer.next = headers_buffer.header_start;

	  /* Have we reached the header/body boundary? */
	  if ((c = getc (stdin)) == '\n')
	    {
	      /* yes we have! */

	      if (Debug >= 2)
		fprintf (stderr, "body reached\n");

	      if (cte_read && !cte_written)
		{
		  if (!ct_read)
		    {
		      /* Content-type field is missing!! 
		       * Assume text/plain!
		       */
		      if (Debug >= 3)
			fprintf (stderr, "Content-type field is missing!!\n");
		      Add_to_buffer (&headers_buffer, "X-2UTF-Content-type: missing\n");
		      headers_buffer.header_start = headers_buffer.next;

		      ct_read = TRUE;
		      mhp->content_type = TEXT;
		    }

		  write_cte (&headers_buffer, mhp->transfer_encoding, mhp->content_type);
		  cte_written = TRUE;
		}
	      /* should we output the '\n' here? */
	      Add_to_buffer (&headers_buffer, "\n");
	      break;
	    }

	  /* 
	   * We have _not_ reached the header/body boundary
	   */
	  if (ct_read && cte_read & !cte_written)
	    {
	      write_cte (&headers_buffer, mhp->transfer_encoding, mhp->content_type);
	      cte_written = TRUE;
	    }
	  goto again;
	}
    }
  if (!ct_read)
    mhp->content_type = TEXT;
  /* second pass */

  if (Debug)
    fprintf (stderr, " -- parse_header - pass 2 -- \n");

  if (Debug >= 4)
    fprintf (stderr, "headers charset: %s\ncontent charset: %s\n", mhp->headers_charset, mhp->content_charset);

/* 1) test if charset is known before changing Content-Type header
 * 2) set charset for non-MIME-encrypted headers - 
 * - assuming the same as Content's one
 */
  if (mhp->content_charset[0] != '\0' && Strcasecmp (mhp->content_charset, "us-ascii") != 0)
    {
      if (Validate_charset (mhp->content_charset, IS))
	{
	  if (charset_p->type != UTF8)
	    ct_is_changing = TRUE;
	}
      else
	ct_charset_unknown = TRUE;
    }
  else if (mhp->headers_charset[0] != '\0')
    {
      Validate_charset (mhp->headers_charset, IS);
    }
  else
    Validate_charset ("us-ascii", 0);
/*
   if (ct_read && mhp->content_type == TEXT && ct_header_charset_name_end)
   if (charset_p->type == KNOWN || charset_p->type == BUF_PIPE || charset_p->type == NON_BUF_PIPE)
   ct_is_changing = TRUE;
   else if (charset_p->type == UNKNOWN && mhp->content_charset[0] != '\0' && Strcasecmp (mhp->content_charset, "us-ascii") != 0)
   ct_charset_unknown = TRUE;
 */
  headers_buffer.end = headers_buffer.next;	/* no more Add_to_buffer() */
  headers_buffer.header_start = headers_buffer.next = headers_buffer.start;
  for (; headers_buffer.next <= headers_buffer.end; c = *headers_buffer.next++)
    {
      /*again2: *//* *headers_buffer.next++ = c; */
      if (c == '\n' && !(*headers_buffer.next == ' ' || *headers_buffer.next == '\t'))
	/* end of (folded?) header line reached */
	{
	  unsigned char saved_char;
	  struct buffer_type new_ct;

	  saved_char = *headers_buffer.next;
	  *headers_buffer.next = '\0';	/* zero terminate the line buf */

	  if (Debug >= 4)
	    fprintf (stderr, "header: %s\n", headers_buffer.header_start);

	  if (!ct_written && Strcase_has_prefix (headers_buffer.header_start, "Content-Type:") == 0)
	    {
	      ct_written = TRUE;
	      if (ct_is_changing)
		{
		  EMPTY_BUFFERS(charset_p);
		  fputs ("X-2UTF-Old-", charset_p->pipe);
		  decode_header_line (DECODE, &headers_buffer);
		  /* changing charset name to UTF-8 */
		  if (ct_header_charset_name_begin > ct_header_charset_name_end)
		    Error (internal_err);
		  new_ct.start = new_ct.header_start = xmalloc ( strlen (headers_buffer.header_start) + strlen ("UTF-8"));
		  strcpy (Stpcpy (Stpncpy (new_ct.start, \
					   headers_buffer.header_start, \
					   ct_header_charset_name_begin), \
				  "UTF-8"),
		  headers_buffer.header_start + ct_header_charset_name_end);
                  new_ct.next = new_ct.header_start + strlen (new_ct.header_start);

		  decode_header_line (DECODE, &new_ct);
		  free (new_ct.start);
		}
	      else
		{
		  decode_header_line (DECODE, &headers_buffer);
		  if (ct_charset_unknown) {
		    EMPTY_BUFFERS(charset_p);
		    fprintf (charset_p->pipe, "X-2UTF-Warning: Unknown charset in Content-Type header: \"%s\"\n", mhp->content_charset);
		  }
		}
              if (mhp->boundary[0] == '\0') {
		EMPTY_BUFFERS(charset_p);
                fputs ("X-2UTF-Warning: missing boundary parameter\n", charset_p->pipe);
	      }

	    }
	  else
	    decode_header_line (DECODE, &headers_buffer);
	  /* Have we reached the header/body boundary? */
	  if (saved_char == '\n')
	    {
	      /* yes we have! */

	      if (Debug >= 2)
		fprintf (stderr, "body reached\n");

	      /* should we output the '\n' here? */
	      PUTC_IN_UTF8 ('\n');
	      goto Return;
	    }
	  *headers_buffer.next = saved_char;
	  headers_buffer.header_start = headers_buffer.next;
	}
    }
/* set charset for message body */
Return:
  if (mhp->content_charset[0] != '\0'	/* && Strcasecmp (mhp->content_charset, "us-ascii") != 0 */ )
    Validate_charset (mhp->content_charset, 0);
  else if (mhp->headers_charset[0] != '\0')
    Validate_charset (mhp->headers_charset, 0);
  else
    Validate_charset ("us-ascii", 0);
  return (TRUE);
}



/*
 * General format of an encoded header field (`[ ]' denotes optional fields):
 *
 * header_field: <text>=?<charset>[*<language>]?<encoding>?<encoded_chars>?=<text>
 *
 * `X-' charset prefix denotes non-standard charset for private use.
 *
 */

#define HUNT	0
#define START	1
#define CHARS	2
#define ENCD	3
#define Q_FIELD	4
#define DEC1	5
#define DEC2	6
#define B_FIELD	7
#define END	8


#define binhex(c) ( (c > '9') ? c-'A'+10 : c-'0')


/* max MIME charset parameter is 70 bytes */
unsigned char rendered_charset_name[80];

/*******************************************************************/
char *
decode_header_line (action, headers_buffer_p)
/*******************************************************************/
     enum action_type action;
     struct buffer_type *headers_buffer_p;
{
  int state = HUNT;
  int c, c1 = 0, c2, b1 = 0, b2, index, tmp;	/* =0 ??? */
  unsigned char *buf = headers_buffer_p->header_start;

  size_t charset_name_length = 0;
  unsigned char charset_name[80];
  unsigned char *charp = charset_name;
  const int max_encoding_length = 1;
  unsigned char encoding[max_encoding_length + 1];
  unsigned char *encp = encoding;
  unsigned char space_between_enc_words[80]="";

  rendered_charset_name[0] = '\0';
  while (buf < headers_buffer_p->next)
    {
      c = *buf++;
      if (Debug > 7)
	{
	  print_state (state);
	  fprintf (stderr, ", c=`%c\'\n", (char) c);
	}

      if (state == END)
	{
	  if (action == DECODE)
	    {
	      if (encode && charset_p->type == ICONV)
		Put_to_iconv_buf(-3, charset_p); /* reencode output to MIME word and empty buffer */
	      /* string may expand while converting */
	      Validate_charset (NULL, 0);	/* pop previous charset */
	    }
	}

      if (state == HUNT)
	{
	  if (c == ' ' || c == '\t' || c == '(')
	    {
	      state = START;
	      /* *retp++ = c; */
	      if (action == DECODE)
		PUTC_IN_UTF8 (c);
	      continue;
	    }
	}

      switch (c)
	{
	case '=':
	  {
	    switch (state)
	      {
	      case START:
		if ((tmp = *buf++) == '?')
		  {
		    state = CHARS;
		    charp = charset_name;
		  }
		else
		  {
		    if (action == DECODE)
		      {
			PUTC_IN_UTF8 (c);
			PUTC_IN_UTF8 (tmp);
		      }
		    state = HUNT;
		  }
		continue;

	      case CHARS:
		goto it_is_not_charset_name;

	      case Q_FIELD:
		state = DEC1;
		continue;

	      case DEC1:
		/* *retp++ = '='; */
		PUTC_IN_UTF8 ('=');
		continue;

	      case B_FIELD:
		/* padding */
		continue;

	      case END:
		if ((index=strspn (buf, LWSP)) && buf[index] == '=' && buf[index+1] == '?')
		{ /* space between adjacent encoded-words should be skipped */
		  memcpy (space_between_enc_words, buf, MIN (79, index));
		  space_between_enc_words[MIN (79, index)]='\0';
		  state = CHARS;
		  charp = charset_name;
		  buf += index + 2;
		}
		else
		{
		  space_between_enc_words[0]='\0';
		  state = HUNT;
		}
		continue;
	      }
	  }
	  break;

	case '?':
	  {
	    switch (state)
	      {
	      case START:
		/* *retp++ = c; */
		if (action == DECODE)
		  PUTC_IN_UTF8 (c);
		state = HUNT;
		continue;

	      case CHARS:
		/* we do not ignore the charset specification.
		 * discard X- prefix for non-standard charsets.
		 */
		*charp = '\0';
		charset_name_length = charp - charset_name;
		strcpy (rendered_charset_name, charset_name);
		if ((rendered_charset_name[0] == 'x' || rendered_charset_name[0] == 'X') && rendered_charset_name[1] == '-')
		  memmove (rendered_charset_name, rendered_charset_name + 2, 78);
		/* ignore language suffix after `*' */
		if (strchr (rendered_charset_name, '*'))
		  *strchr (rendered_charset_name, '*') = '\0';

		if (action == QUERY_CHARSET_NAME)
		  {
		    return (rendered_charset_name);
		  }
		if (Debug >= 4)
		  {
		    fprintf (stderr, "charset: %s\n", charset_name);
		    fprintf (stderr, "rendered charset: %s\n", rendered_charset_name);
		  }

		state = ENCD;
		continue;

	      case ENCD:
		*encp = '\0';
		encp = encoding;
		if (Debug > 7)
		  fprintf (stderr, "encoding: %s\n", encoding);
		if (encode && Str_is_UTF8(rendered_charset_name) || !(Validate_charset (rendered_charset_name, 0) && (*encoding == 'q' || *encoding == 'Q' ||*encoding == 'b' || *encoding == 'B')))
		  {
		    if (Debug > 3)
		      fprintf (stderr, "skipping charset '%s', encoding '%c'\n", rendered_charset_name, encoding[0]);
		    Output_consumed_chars (space_between_enc_words, charset_name, charset_name_length, encoding);
		    PUTC_IN_UTF8 ('?');
		    Validate_charset (NULL, 0);		/* pop previous charset */
		    charp = charset_name;
		    encp = encoding;
		    state = HUNT;
		    continue;
		  }
		if (encode) {
		  int i = -1;
		  while (space_between_enc_words[++i]) /* don't skip space if we will have encoded words again */
		    PUTC_IN_UTF8 (space_between_enc_words[i]);
		  EMPTY_BUFFERS(charset_p);
		}

		if ((*encoding == 'q') || (*encoding == 'Q'))
		  {
		    state = Q_FIELD;
		  }
		else if ((*encoding == 'b') || (*encoding == 'B'))
		  {
		    int c1, c2, c3, c4;

		    state = B_FIELD;

		    while ((buf < headers_buffer_p->next) && (*buf != '\?'))
		      {
			/* find first valid c1 */
			while ((buf < headers_buffer_p->next) && (*buf != '\?') && ((c1 = b64_map[(unsigned int) *buf]) == UN))
			  buf++;
			c1 = *buf;

			if (buf < headers_buffer_p->next)
			  buf++;

			/* find second valid c2 */
			while ((buf < headers_buffer_p->next) && (*buf != '\?') && ((c2 = b64_map[(unsigned int) *buf]) == UN))
			  buf++;
			c2 = *buf;

			if (buf < headers_buffer_p->next)
			  buf++;

			/* find third valid c3 */
			while ((buf < headers_buffer_p->next) && (*buf != '\?') && ((c3 = b64_map[(unsigned int) *buf]) == UN))
			  buf++;
			c3 = *buf;

			if (buf < headers_buffer_p->next)
			  buf++;

			/* find fourth valid c4 */
			while ((buf < headers_buffer_p->next) && (*buf != '\?') && ((c4 = b64_map[(unsigned int) *buf]) == UN))
			  buf++;
			c4 = *buf;

			if (buf < headers_buffer_p->next)
			  buf++;

			if ((c1 == '=') || (c2 == '=') || (c1 == '\0') || (c2 == '\0') || (c1 == '\?') || (c2 == '\?'))
			  {
			    break;
			  }

			/* *retp++ = ((unsigned int) b64_map[c1] << 2) | (((unsigned int) b64_map[c2] & 0x30) >> 4); */
			PUTC_IN_UTF8 (((unsigned int) b64_map[c1] << 2) |(((unsigned int) b64_map[c2] & 0x30) >> 4));

			if ((c3 == '=') || (c3 == '\0') || (c3 == '?'))
			  {
			    break;
			  }

			/* *retp++ = (((unsigned int) b64_map[c2] & 0xF) << 4) | (((unsigned int) b64_map[c3] & 0x3C) >> 2); */
			PUTC_IN_UTF8 ((((unsigned int) b64_map[c2] & 0xF) << 4) |(((unsigned int) b64_map[c3] & 0x3C) >> 2));
			if ((c4 == '=') || (c4 == '\0') || (c4 == '\?'))
			  {
			    break;
			  }

			/* *retp++ = (((unsigned int) b64_map[c3] & 0x3) << 6) | ((unsigned int) b64_map[c4]); */
			PUTC_IN_UTF8 ((((unsigned int) b64_map[c3] & 0x3) << 6) |((unsigned int) b64_map[c4]));
		      }
		      if (encode && charset_p->type == ICONV)
			Put_to_iconv_buf(-3, charset_p); /* reencode output to MIME word and empty buffer */

		    /* *retp = '\0'; */
		  }
		continue;

	      case B_FIELD:
	      case Q_FIELD:
		state = END;
		continue;

	      case DEC1:
		/* output the consumed char */
		/* *retp++ = '='; */
		PUTC_IN_UTF8 ('=');
		state = END;
		continue;

	      case DEC2:
		/* output the consumed chars */
		/* *retp++ = '=';
		   *retp++ = c1; */
		PUTC_IN_UTF8 ('=');
		PUTC_IN_UTF8 (c1);
		state = END;
		continue;
	      }
	  }
	  break;

	default:
	  {
	    if (state == START)
	      {
		/* *retp++ = c; */
		if (action == DECODE)
		  PUTC_IN_UTF8 (c);
		if (!(c == ' ' || c == '\t' || c == '('))
		  state = HUNT;
		continue;
	      }

	    if (state == CHARS)
	      {
		if (charp - charset_name < 79)
		  *charp++ = c;
		else
		  {
		  it_is_not_charset_name:
		    if (action == DECODE)
		      {
			*charp = '\0';
			charset_name_length = charp - charset_name;
			Output_consumed_chars (space_between_enc_words, charset_name, charset_name_length, NULL);
		      }
		    charp = charset_name;
		    state = HUNT;
		  }
		continue;
	      }

	    if (state == ENCD)
	      {
		if (encp - encoding < max_encoding_length)
		  *encp++ = c;
		else
		  {
		    if (action == DECODE)
		      {
			*encp = '\0';
			Output_consumed_chars (space_between_enc_words, charset_name, charset_name_length, encoding);
		      }
		    charp = charset_name;
		    encp = encoding;
		    state = HUNT;
		  }
		continue;
	      }

	    if ((state == Q_FIELD) && (c == '_'))
	      {
		/* *retp++ = ' '; */
		PUTC_IN_UTF8 (' ');
		continue;
	      }

	    if (state == DEC1)
	      {
		if (isxdigit (c))
		  {
		    c1 = c;
		    c = toupper (c1);
		    b1 = binhex (c);
		    state = DEC2;
		    continue;
		  }
		else
		  {
		    /* *retp++ = '='; */
		    PUTC_IN_UTF8 ('=');
		    state = Q_FIELD;
		  }
		break;
	      }

	    if (state == DEC2)
	      {
		if (isxdigit (c))
		  {
		    c2 = toupper (c);
		    b2 = binhex (c2);
		    c = b1 << 4 | b2;
		    /* *retp++ = c; */
		    PUTC_IN_UTF8 (c);
		    state = Q_FIELD;
		    continue;
		  }
		else
		  {
		    /* *retp++ = c1; */
		    PUTC_IN_UTF8 (c1);
		    state = Q_FIELD;
		  }
		break;
	      }

	    if (state == END)
	      {
		fprintf (stderr, "END reached");
		state = HUNT;
	      }
	  }
	  break;
	}
      /* *retp++ = c; */
      if (action == DECODE)
	PUTC_IN_UTF8 (c);
    }

  if (state == CHARS)
    {
      *charp = '\0';
      charset_name_length = charp - charset_name;
      Output_consumed_chars (space_between_enc_words, charset_name, charset_name_length, NULL);
    }
  else if (state == ENCD)
    {
      *encp = '\0';
      encp = encoding;

      if (action == DECODE)
	{
	  *encp = '\0';
	  Output_consumed_chars (space_between_enc_words, charset_name, charset_name_length, encoding);
	}
    }
  /* *retp++ = c; *//* remember the '\0' terminator */
  if (action == QUERY_CHARSET_NAME)
    return (NULL);
  /*FLUSH_OUTPUT(charset_p);*/
  return (rendered_charset_name);
}



/*******************************************************************/
void
print_state (state)
/*******************************************************************/
     int state;
{
  switch (state)
    {
    case HUNT:
      fprintf (stderr, " state HUNT");
      break;

    case START:
      fprintf (stderr, " state START");
      break;

    case CHARS:
      fprintf (stderr, " state CHARS");
      break;

    case ENCD:
      fprintf (stderr, " state ENCD");
      break;

    case Q_FIELD:
      fprintf (stderr, " state Q_FIELD");
      break;

    case B_FIELD:
      fprintf (stderr, " state B_FIELD");
      break;

    case DEC1:
      fprintf (stderr, " state DEC1");
      break;

    case DEC2:
      fprintf (stderr, " state DEC2");
      break;

    case END:
      fprintf (stderr, " state END");
      break;

    default:
      fprintf (stderr, " unknown state; %d", state);
      break;
    }
}



/******************************************************************* 
int
casncmp (s1, s2, n)
 ******************************************************************* 
     register const char *s1, *s2;
     register n;
{
  if (s1 == s2)
    return (0);
  while ((--n >= 0) && (lower (*s1) == lower (*s2)))
    {
      s2++;
      if (*s1++ == '\0')
	return (0);
    }
  return ((n < 0) ? 0 : (lower (*s1) - lower (*s2)));
}
*/


/*******************************************************************/
int
decode_quoted_printable (infp, outfp, boundary)
/*******************************************************************/
     FILE *infp;
     FILE *outfp;
     char *boundary;
{
  unsigned char *lbp = line.buf;
  int c, length, ret;

  if (Debug)
    fprintf (stderr, " --- Entry decode_quoted_print --- \n");
  if (boundary && (Debug >= 2))
    fprintf (stderr, " --- with boundary: %s\n", boundary);

  while ((length = Getline (&line.buf, &line.length, infp)) != -1)
/*  while ((c = getc (infp)) != EOF) */
    {
      lbp = line.buf;

      if (boundary && (line.buf[0] == '-') && (line.buf[1] == '-'))
	{
	  if ((ret = Str_is_boundary (line.buf+2, boundary, FALSE)))
	    {
	      if (ret < OUTER_BOUNDARY) {
		EMPTY_BUFFERS(charset_p);
		fputs (line.buf, outfp);
	      }
	      return (ret);
	    }
	}

      while (lbp - line.buf < length)
	{
	  if ((c = *lbp++) == '=')
	    {			/* c == '=' */
	      int c1 = *lbp++;
	      int c2;

	      if (!(lbp - line.buf < length))
		/*        if (c1 == '\0') */
		break;

	      c2 = *lbp++;
	      /*        if (c2 == '\0')
	         break; */


	      /* check for soft CRLF */
	      if (c1 == '\r')
		{
		  /* this is an error? : c2 = getc(infp); */
		  if (c2 != '\n')	/* not a CRLF */
		    lbp--;	/* put back the char after the =<CR> */
		  continue;
		}

	      /* check for soft newline */
	      if (c1 == '\n')
		{
		  lbp--;	/* put back the char after the newline */
		  continue;
		}

	      /* check for == -> = */
	      if (c1 == '=')
		{
		  putc (c1, outfp);
		  lbp--;	/* put back the char after the == */
		  continue;
		}

	      /* make sure it's =XX */
	      if (!isxdigit (c1) || !isxdigit (c2))
		continue;

	      /* we have two hex digits, so decode them */
	      if (isdigit (c1))
		c1 -= '0';
	      else if (islower (c1))
		{
		  c1 -= 'a';
		  c1 += 10;
		}
	      else
		{
		  c1 -= 'A';
		  c1 += 10;
		}
	      if (isdigit (c2))
		c2 -= '0';
	      else if (islower (c2))
		{
		  c2 -= 'a';
		  c2 += 10;
		}
	      else
		{
		  c2 -= 'A';
		  c2 += 10;
		}
	      PUTC_IN_UTF8 (((c1 << 4) | c2));
	    }
	  else
	    PUTC_IN_UTF8 (c);
	    /* putc (c, outfp); */
	}			/* end while */
      lbp = line.buf;
    }
  /* FLUSH_OUTPUT(charset_p); */
  return (boundary ? FALSE : END_BOUNDARY);
}





/*******************************************************************/
int
nextgetc (infp)
/*******************************************************************/
     FILE *infp;
{
  int c;

  while ((c = getc (infp)) != EOF)
    {
      if (c == '-')
	{
	  c = getc (infp);
	  if (c == '-')
	    {
	      /* Two '-'s in a row - must be a boundary! */
	      if (Debug >= 2)
		fprintf (stderr, " ----- boundary line prefix found\n");
	      return (127);
	    }
	}

      if (b64_map[c] != UN)
	break;
    }
  return c;
}



/*******************************************************************/
int
decode_base64 (infp, outfp, boundary)
/*******************************************************************/
     FILE *infp;
     FILE *outfp;
     char *boundary;
{
  int c1, c2, c3, c4;
  int ret;
  ssize_t length;

  c1 = c2 = c3 = c4 = 0;

  if (Debug)
    fprintf (stderr, " --- Entry decode_base64 --- \n");

  for (;;)
    {
      ret = 0;

      if (((c1 = nextgetc (infp)) == 127)
	  || ((c2 = nextgetc (infp)) == 127)
	  || ((c3 = nextgetc (infp)) == 127)
	  || ((c4 = nextgetc (infp)) == 127))
	{
	  if (Debug >= 2)
	    fprintf (stderr, " --- boundary line prefix found: %d\n", ret);
	  EMPTY_BUFFERS(charset_p);
	  putc ('\n', outfp);/*7*/
	  putc ('-', outfp);
	  putc ('-', outfp);
	  if (!boundary || -1 == (length = Getline (&line.buf, &line.length, infp)))
	    return (FALSE);
	  if ((ret = Str_is_boundary (line.buf, boundary, FALSE)))
	    {
	      if (ret < OUTER_BOUNDARY)
		fprintf (outfp, "%s", line.buf);
	      return (ret);
	    }
	  return (Seek_boundary (infp, boundary, FALSE));
	}

      if ((c1 == '=') || (c2 == '=') || (c1 == EOF) || (c2 == EOF))
	{
	  break;
	}

      PUTC_IN_UTF8 ((((unsigned int) b64_map[c1] << 2) | (((unsigned int) b64_map[c2] & 0x30) >> 4)));
      if ((c3 == '=') || (c3 == EOF))
	{
	  break;
	}

      PUTC_IN_UTF8 (((((unsigned int) b64_map[c2] & 0XF) << 4) | (((unsigned int) b64_map[c3] & 0x3C) >> 2)));

      if ((c4 == '=') || (c4 == EOF))
	{
	  break;
	}

      PUTC_IN_UTF8 (((((unsigned int) b64_map[c3] & 0x03) << 6) | (unsigned int) b64_map[c4]));
    }
  if (Debug >= 2)
    fprintf (stderr, " --- end of base64 \n");
  EMPTY_BUFFERS(charset_p);
  putc ('\n', outfp);/*7*/
  if (boundary)
    return (Seek_boundary (infp, boundary, 0));
  else
    while ((getc (infp)) != EOF);
/*      putc (c, outfp); */
  return (END_BOUNDARY);
}



/*******************************************************************/
void
write_cte (headers_buffer_p, transfer_encoding, content_type)
/*******************************************************************/
     struct buffer_type *headers_buffer_p;
     int transfer_encoding, content_type;
{
  if (content_type == TEXT)
    {
      switch (transfer_encoding)
	{
	case QP:
	  Add_to_buffer (headers_buffer_p, "X-2UTF-Content-transfer-encoding: fixed from quoted-printable\n");
	  Add_to_buffer (headers_buffer_p, "Content-Transfer-Encoding: 8bit\n");
	  break;

	case B64:
	  Add_to_buffer (headers_buffer_p, "X-2UTF-Content-transfer-encoding: fixed from base64\n");
	  Add_to_buffer (headers_buffer_p, "Content-Transfer-Encoding: 8bit\n");
	  break;

	case BIT7:
	  Add_to_buffer (headers_buffer_p, "Content-Transfer-Encoding: 7bit\n");
	  break;

	case BIT8:
	  Add_to_buffer (headers_buffer_p, "Content-Transfer-Encoding: 8bit\n");
	  break;

	case BINARY:
	  Add_to_buffer (headers_buffer_p, "Content-Transfer-Encoding: binary\n");
	  break;

	default:
	  if (Debug >= 3)
	    fprintf (stderr, "Transfer-encoding is UNKNOWN\n");
	  break;
	}
    }
  else
    /* ! TEXT */
    {
      /* remember to output the Content-transfer-encoding
       * header field in case type is NOT text.
       */
      switch (transfer_encoding)
	{
	case QP:
	  Add_to_buffer (headers_buffer_p, "Content-Transfer-Encoding: quoted-printable\n");
	  break;

	case B64:
	  Add_to_buffer (headers_buffer_p, "Content-Transfer-Encoding: base64\n");
	  break;

	case BIT7:
	  Add_to_buffer (headers_buffer_p, "Content-Transfer-Encoding: 7bit\n");
	  break;

	case BIT8:
	  Add_to_buffer (headers_buffer_p, "Content-Transfer-Encoding: 8bit\n");
	  break;
	}
    }
  headers_buffer_p->header_start = headers_buffer_p->next;
}
