/*
** $Id: splitter.c,v 1.10 1997/01/04 00:23:15 bje Exp $
**
** Description:
** Reads standard input and feeds the stream to one or more programs'
** standard input.  This behaves like an signal splitter (hence the name).
**
** Usage:
** splitter <command1> <command2> ... <commandN>
**
** Note: each argument to splitter(1L) is treated as an individual program
**       to execute.  If you need to pass arguments to any of the programs
**       in the command list, use your shell to quote the commands (e.g.
**       "cmd -arg1 -arg2" in Bourne shell).
**
** Author: Ben Elliston <bje@air.net.au>
**
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/wait.h>

static const char rcsid[] = "$Id: splitter.c,v 1.10 1997/01/04 00:23:15 bje Exp $";
static const int bufsize  = 512;

static int i;				/* general purpose counter */
static unsigned numProgs;		/* number of commands given */

/*
 * Each node of the kidlist associates a process ID with the writing end
 * of the pipe to that process.
 */

struct kid
{
  pid_t pid;
  int fd;
} *kidlist;

/*
 * Check arguments for validity (currently very simple)
 *
 * Return 0 to the caller if the command line is OK or a negative
 * result otherwise.
 */

int
check_args(int argc, char *argv[])
{
  if (argc < 2) {
    return(-1);
  }
  return(0);
}

/*
 * Initialise the kidlist.
 */

void
init_kidlist(void)
{
  kidlist = (struct kid *) malloc(numProgs * sizeof(struct kid));
  if (kidlist == NULL)
  {
    (void) fprintf(stderr, "splitter(malloc): failed\n");
    exit(1);
  }
  (void) memset(kidlist, 0, numProgs * sizeof(struct kid));
}

/*
 * Create child processes for each command in the command line and open
 * a pipe to the process.  The PID and pipe descriptor are held in the kidlist
 * for later reference.
 */ 

void
open_pipes(char *cmds[])
{
  for (i = 0; i < numProgs; ++i) {
    int pd[2];
    if (pipe(pd) < 0) {
      perror("splitter(pipe)");
      exit(1);
    }

    if ((kidlist[i].pid = fork()) == 0) { /* child */
      (void) close(pd[1]);
      (void) dup2(pd[0], STDIN_FILENO);
      exit(system(cmds[i]));
    }
    else { /* open a pipe to the child */
      (void) close(pd[0]);
      kidlist[i].fd = pd[1];
    }
  }
}

void
feed_data(void)
{
  int nread, nwrite;
  char buf[bufsize];

  while ((nread = read(STDIN_FILENO, buf, sizeof(buf))) > 0)
  {
    for (i = 0; i < numProgs; ++i) {
      nwrite = write(kidlist[i].fd, buf, nread);
      if (nwrite < nread) {
        perror("splitter(write)");
        exit(1);
      }
    }
  }
  if (nread < 0) {
    perror("splitter(read)");
    exit(1);
  }
}

/*
 * Close all the pipes to the child processes.  For standard shell utilities
 * running in the child process, this should have the effect of making them
 * exit (we hope!)
 */

void
close_pipes(void)
{
  for (i = 0; i < numProgs; ++i)
  {
    (void) close(kidlist[i].fd);
  }
}

/*
 * Sit around and wait for all the children to exit.  Badly behaved child
 * programs (which would normally make the shell hang if run from a shell
 * prompt) will also make this function hang.  Be careful out there!
 */

void
collect_kiddies(void)
{
  while (numProgs > 0) {
    (void) wait(NULL);
    --numProgs;
  }
}

int
main(int argc, char *argv[])
{
  if (check_args(argc, argv) < 0)
  {
    fprintf(stderr, "Usage: %s commands ..\n", argv[0]);
    return(1);
  }

  /* 
   * drop executable name from command line
   */
  numProgs = --argc;
  argv = &argv[1];

  init_kidlist();
  open_pipes(argv);
  feed_data();
  close_pipes();
  collect_kiddies();

  return(0);
}
