/*
	dirutil.cc	Directory Utility Functions
	Copyright (c) 2000,2001 Kriang Lerdsuwanakij
	email:		lerdsuwa@users.sourceforge.net

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "dirutil.h"
#include "cstrlib.h"
#include "gtstream.h"
#include "error.h"
#include "strmisc.h"

#if HAVE_UNISTD_H
# include <unistd.h>
# include <sys/types.h>
#endif

/*************************************************************************
	String Expansion Functions
*************************************************************************/


						// Expand ~ to full dir. path
string	ExpandDirectory(const string &src, int quote)
{
	string	outStr;
	if (!src.size())			// String empty
		return outStr;			// Nothing to do
		
	if (src[0] == '~') {
		struct passwd	*p;
		
					// "~" or "~/..."
		if (src.size() == 1 || src[1] == '/') {
					// Get passwd entry for current user
			p = getpwuid(getuid());
			if (p == NULL) {
				char *env = getenv("HOME");
				if (env == NULL) {
					// Give up
					throw ErrorGenericSyntax(_("error obtaining the user"
								   " home directory"));
				}
	
				outStr = env;
			}
			else {
			
					// Store home directory
				outStr = p->pw_dir;
			}

					// No sysadmin on earth should create user 
					// with strange home directory name but
					// this code is here just in case.
			if (quote)
				outStr = QuoteGlobChars(outStr);

					// Remaining doesn't quoting since we
					// allow globing in SkipDir
			outStr.append(src, 1, string::npos);
		}
		else {
			string::size_type	s = src.find('/');
			if (s == string::npos) {
					// Skip the `~'
				string	temp(src, 1);
				if (quote)
					temp = UnquoteShellChars(temp);

				p = k_getpwnam(temp);
				if (p == NULL) {
					throw ErrorGenericSyntax(_("error obtaining home"
								   " directory for user `%$\'"),
								 temp);
				}
				
				outStr = p->pw_dir;
			}
			else {
			
				string	temp(src, 1, s-1);
				if (quote)
					temp = UnquoteShellChars(temp);

				p = k_getpwnam(temp);
				if (p == NULL) {
					throw ErrorGenericSyntax(_("error obtaining home"
								   " directory for user `%$\'"),
								 temp);
				}
				
				outStr = p->pw_dir;
			}
					// No sysadmin on earth should create user 
					// with strange home directory name but
					// this code is here just in case.
			if (quote)
				outStr = QuoteGlobChars(outStr);

					// Append remaining of directory name.
					// Remaining doesn't quoting since we
					// allow globing in SkipDir
			if (s != string::npos)
				outStr.append(src, s+1, string::npos);
		}
	}
	else
		outStr = src;

	if (outStr[0] != '/' && outStr[0] != '*' && outStr[0] != '?' 
	    && outStr[0] != '[' && outStr[0] != '\\') {
		throw ErrorGenericSyntax(_("bad directory `%$\'"), outStr);
	}
	
	string::size_type	ptr;
	while((ptr = outStr.find("//")) != string::npos) {
						// Remove the first `/'
#ifdef _CXX_STRING_ERASE
		outStr.erase(ptr, 1);
#else
		// Same as outStr.erase(ptr, 1);
		outStr.replace(ptr, 1, static_cast<string::size_type>(0), 0);
#endif
	}
	
	while((ptr = outStr.find("/./")) != string::npos ||
	      (ptr = StrLast(outStr, "/.")) != string::npos) {
		if (ptr+2 == outStr.size()) {	// Dir ends with dir/.
		
						// Delete ptr+1 .. ptr+1
						// Turn it to    dir/
						// [Note: dir can be empty.
						//  The trailing / will be
						//  handled later.]
#ifdef _CXX_STRING_ERASE
			outStr.erase(ptr+1, 1);
#else
			// Same as outStr.erase(ptr+1, 1);
			outStr.replace(ptr+1, 1, static_cast<string::size_type>(0), 0);
#endif
		}
		else {
						// Delete ptr .. ptr+2
#ifdef _CXX_STRING_ERASE
			outStr.erase(ptr, 2);
#else
			// Same as outStr.erase(ptr, 2);
			outStr.replace(ptr, 2, static_cast<string::size_type>(0), 0);
#endif
		}
	}
	while((ptr = outStr.find("/../")) != string::npos ||
	      (ptr = StrLast(outStr, "/..")) != string::npos) {
		string::size_type	ptr2 = ptr;
		if (ptr2 != 0) {		// Have a parent directory
			do {
				ptr2--;
			}
			while(outStr[ptr2] != '/' && ptr2 != 0);
		}

		if (ptr+3 == outStr.size()) {	// Dir ends with dir1/dir2/..
						// Turn it to    dir1/
						// [Note: dir can be empty]
#ifdef _CXX_STRING_ERASE
			outStr.erase(ptr2+1, string::npos);
#else
			// Same as outStr.erase(ptr2+1, string::npos);
			outStr.replace(ptr2+1, string::npos,
					static_cast<string::size_type>(0), 0);
#endif
		}
		else {				// dir1/dir2/../dir3
			ptr += 4;
						// Remove dir2/../
#ifdef _CXX_STRING_ERASE
			outStr.erase(ptr2+1, ptr-(ptr2+1));
#else
			// Same as outStr.erase(ptr2+1, ptr-(ptr2+1));
			outStr.replace(ptr2+1, ptr-(ptr2+1),
					static_cast<string::size_type>(0), 0);
#endif
		}
	}

	string::size_type	len = outStr.size();	// Remove trailing `/'
	if (len > 1 && outStr[len-1] == '/') {
#ifdef _CXX_STRING_ERASE
		outStr.erase(len-1, 1);
#else
			// Same as outStr.erase(len-1, 1);
		outStr.replace(len-1, 1, static_cast<string::size_type>(0), 0);
#endif
	}
		
	return outStr;
}

