/*
	miscobj.cc	Various NCurses C++ Classes
	Copyright (c) 1996-8,2000,2001 Kriang Lerdsuwanakij
	email:		lerdsuwa@users.sourceforge.net

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "miscobj.h"
#include "error.h"

//
// Remove conflict macros introduced by <curses.h>
//
#undef ERR
#undef OK

#include <signal.h>
#include CXX__HEADER_cstdio
#include <sys/stat.h>

#include <term.h>
#include <termios.h>
#include <sys/ioctl.h>

#if HAVE_UNISTD_H
# include <unistd.h>
# include <sys/types.h>
#endif

#ifdef HAVE_WCHAR_H
# define __USE_XOPEN
# include CXX__HEADER_cwchar
#endif

#include "strmisc.h"

/*************************************************************************
	ACS fallback support
*************************************************************************/

bool	isACSFallBack = false;

chtype	ACSFallBackSelect(chtype acs, chtype ascii, chtype unicode)
{
	if (isACSFallBack)
		return ascii;
	else if (IsUTF8Mode())
		return unicode;
	else
		return acs;
}

chtype	ACSFallBack(chtype acs)
{
	if (acs == ACS_ULCORNER)
		return ACSFallBackSelect(acs, '+', 0x250C);
	if (acs == ACS_LLCORNER)
		return ACSFallBackSelect(acs, '+', 0x2514);
	if (acs == ACS_URCORNER)
		return ACSFallBackSelect(acs, '+', 0x2510);
	if (acs == ACS_LRCORNER)
		return ACSFallBackSelect(acs, '+', 0x2518);

	if (acs == ACS_RTEE)
		return ACSFallBackSelect(acs, '+', 0x2524);
	if (acs == ACS_LTEE)
		return ACSFallBackSelect(acs, '+', 0x251C);
	if (acs == ACS_BTEE)
		return ACSFallBackSelect(acs, '+', 0x2534);
	if (acs == ACS_TTEE)
		return ACSFallBackSelect(acs, '+', 0x252C);

	if (acs == ACS_HLINE)
		return ACSFallBackSelect(acs, '-', 0x2500);
	if (acs == ACS_VLINE)
		return ACSFallBackSelect(acs, '|', 0x2502);
	if (acs == ACS_PLUS)
		return ACSFallBackSelect(acs, '+', 0x253C);

	if (acs == ACS_LARROW)
		return ACSFallBackSelect(acs, '<', 0x2190);
	if (acs == ACS_RARROW)
		return ACSFallBackSelect(acs, '>', 0x2192);
	if (acs == ACS_DARROW)
		return ACSFallBackSelect(acs, 'v', 0x2193);
	if (acs == ACS_UARROW)
		return ACSFallBackSelect(acs, '^', 0x2191);

	if (acs == ACS_DEGREE)
		return ACSFallBackSelect(acs, '\'', 0x00B0);
	if (acs == ACS_PLMINUS)
		return ACSFallBackSelect(acs, '#', 0x00B1);
	if (acs == ACS_LEQUAL)
		return ACSFallBackSelect(acs, '<', 0x2264);
	if (acs == ACS_GEQUAL)
		return ACSFallBackSelect(acs, '>', 0x2265);
	if (acs == ACS_NEQUAL)
		return ACSFallBackSelect(acs, '!', 0x2262);

	if (acs == ACS_BULLET)
		return ACSFallBackSelect(acs, 'o', 0x2022);
	if (acs == ACS_DIAMOND)
		return ACSFallBackSelect(acs, '+', 0x25C6);

	if (acs == ACS_CKBOARD)		// ncurses uses ':'
		return ACSFallBackSelect(acs, '#', 0x2592);
	if (acs == ACS_BOARD)
		return ACSFallBackSelect(acs, '#', 0x2591);
	if (acs == ACS_LANTERN)
		return ACSFallBackSelect(acs, '#', 0x256C);
	if (acs == ACS_BLOCK)
		return ACSFallBackSelect(acs, '#', 0x2588);

	if (acs == ACS_PI)
		return ACSFallBackSelect(acs, '*', 0x00B6);
	if (acs == ACS_STERLING)
		return ACSFallBackSelect(acs, 'f', 0x00A3);

	if (acs == ACS_S1)
		return ACSFallBackSelect(acs, '-', 0x2594);
	if (acs == ACS_S9)
		return ACSFallBackSelect(acs, '-', 0x2581);
	if (acs == ACS_S3)
		return ACSFallBackSelect(acs, '-', 0x2501);
	if (acs == ACS_S7)
		return ACSFallBackSelect(acs, '-', 0x2501);

	return acs;
}

/*************************************************************************
	Unicode output support
*************************************************************************/

// Return width
int	k_waddch_int(WINDOW *win, const char *&s, int max_width)
{
#ifdef USE_UTF8_MODE
	static mbstate_t state;

	if (IsUTF8Mode()) {
		wchar_t wc;
		memset(&state, 0, sizeof(mbstate_t));
		int ret = mbrtowc(&wc, s, MB_CUR_MAX, &state);
		if (ret <= 0)
			throw ErrorBadSequence();
						// Exceed ncurses limit
		if (wc - (wc & 0xFFFF))
			throw ErrorBadSequence();

		int	width = wcwidth(wc);
		if (max_width < 0 || (max_width >= 0 && width <= max_width)) {
			waddch(win, static_cast<chtype>(wc));
			s += ret;
		}
		return width;
	}
	else
#endif
	     	if (max_width) {
			if (isprint(*s) || *s == ' ') {
				waddch(win, *s);
			}
			else {
				waddch(win, '?');
			}
			s++;
		}
	return 1;
}

// Return number of bytes consumed
int	k_waddch(WINDOW *win, const char *s, int max_width)
{
	const char *ss = s;
	k_waddch_int(win, ss, max_width);
	return ss-s;
}

// Return number of bytes consumed
int	k_waddstr(WINDOW *win, const char *s)
{
	const char *ss = s;
	while (*ss != 0) {
		k_waddch_int(win, ss);
	}
	return ss-s;
}

// Return number of bytes consumed
int	k_waddnstr(WINDOW *win, const char *s, int max_width)
{
	const char *ss = s;
	while (*ss != 0) {
		const char *save_ss = ss;
		int width = k_waddch_int(win, ss, max_width);
		if (ss == save_ss)	// Not enough space to print
			break;
		max_width -= width;
	}
	return ss-s;
}

/*************************************************************************
	Cursor state class
*************************************************************************/

void	CursorState::Hide()
{
	if (isSave)
		curs_set(0);
	else {				// Original state not saved yet
		oldCursor = curs_set(0);
		isSave = 1;
	}
}

void	CursorState::Show()
{
	if (isSave)
		curs_set(1);
	else {				// Original state not saved yet
		oldCursor = curs_set(1);
		isSave = 1;
	}
}

void	CursorState::Restore()
{
	if (isSave)
		curs_set(oldCursor);	// Restore original cursor state
}

/*************************************************************************
	ncurses screen setup and clean up class
*************************************************************************/

notifyFuncType	NCurses::notifySize = NULL;	// For window size notification
int	NCurses::isInitScr = 0;			// For ncurses initscr
int	NCurses::isInitScrMore = 0;		// Terminal modes, attributes
int	NCurses::isRedirect = 1;		// Start with redirect

void	NCurses::Init(int exitIfRedir)
{
	if (isInitScr == 0) {
		CheckTerminal(exitIfRedir);

		if (! IsRedirect()) {
			initscr();

			signal(SIGINT, EventSIGINT);
#ifdef HAVE_RESIZETERM
			signal(SIGWINCH, EventSIGWINCH);
#endif
#ifdef NCURSES_MOUSE_VERSION
			mmask_t oldMouseMask;
				// No REPORT_MOUSE_POSITION yet since 
				// ncurses does not support it
			mousemask(ALL_MOUSE_EVENTS, &oldMouseMask);
#endif
		}
		isInitScr = 1;
	}
}

void	NCurses::InitMore(WINDOW *win)
{
	if (IsRedirect())
		return;

	if (win)
		keypad(win, TRUE);

	if (isInitScrMore == 0) {
		isInitScrMore = 1;
		nonl();
		cbreak();
		noecho();
		typeahead(-1);
		if (has_colors() == TRUE) {
			start_color();
		}
	}
	
	if (win)
		intrflush(win, FALSE);
}

void	NCurses::End()
{
	if (isInitScr) {
		SetNotify(NULL);	// No more size change notification
		
		isInitScr = 0;

		if (!IsRedirect()) {
				// Opposite of ncurses start_color()
			putp(tigetstr("sgr0"));		// Exit attribute mode
			endwin();
		}
	}
}

void	NCurses::CheckTerminal(int exitIfRedir)
{
				// Check if stdin and stdout are tty
	if (!isatty(fileno(stdout)) || !isatty(fileno(stdin))) {
		isRedirect = 1;
		if (exitIfRedir) {
			/* Pipe/Redirection detected for stdout */
			throw ErrorIORedirect();
		}
	}
	else
		isRedirect = 0;
}

void	NCurses::SetNotify(notifyFuncType func)
{
	notifySize = func;
}

void	NCurses::EventSIGINT(int /*sig*/)
{
	if (isInitScr) {
		if (! IsRedirect()) {
			clear();
			refresh();
		}

		// Don't throw exception here yet.  C library may not 
		// support it.
		NCurses::End();
	}
	exit(RET_OK);
}

void	NCurses::EventSIGWINCH(int /*sig*/)
{
	if (IsRedirect())
		return;
		
#ifdef HAVE_RESIZETERM
	// These stuffs is adapted from ncurses view.c test program
	struct winsize	size;
	if (notifySize && ioctl(fileno(stdout), TIOCGWINSZ, &size) ==0) {
		resizeterm(size.ws_row, size.ws_col);
		wrefresh(curscr); 		// For Linux
		(*notifySize)();
	}
	signal(SIGWINCH, EventSIGWINCH);	// Reinstall signal
#endif
}
