/*
	kcdmain.cc	Kriang's Change Directory
	Copyright (c) 1996,1997,1998,1999,2000,2001 Kriang Lerdsuwanakij
	email:		lerdsuwa@users.sourceforge.net

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "config.h"

#include CXX__HEADER_iostream
#include CXX__HEADER_iomanip

#include CXX__HEADER_algorithm

#ifdef HAVE_LOCALE_H
# include CXX__HEADER_clocale
#endif

#include "cxxlib.h"

#include "error.h"
#include "conffile.h"
#include "scandir.h"
#include "gentree.h"
#include "strmisc.h"
#include "cstrlib.h"
#include "dirutil.h"

// setupcurses.h is included in miscobj.h

// Get version number and release date
#include "version.cc"
char	progName[] = "kcd";

string	saveFile;		// For home dir + ".kcd.save.gz"
string	dirFile;		// For home dir + ".kcd.newdir"
string	confLocalFile;		// For home dir + ".kcd.conf"

int	forceFullScan = 0;
int	forceQuiet = -1;		// Set to 0 or 1 by -rQ and -rq


/*************************************************************************
	Display program version, help.
*************************************************************************/

void	Version()
{
	cout << progName << ' ' << version << '\n';
}

void	PrintBanner()
{
	struct tm rel_time;
	rel_time.tm_mday = rel_day;
	rel_time.tm_mon = rel_month;
	rel_time.tm_year = rel_year;

	cout << progName << ' ' << version
	     << _(" - Kriang's Change Directory\n");
	gtout(cout, _("(c) 1996-9,2000,2001 Kriang Lerdsuwanakij    "
		      "Release date: %$\n\n"))
	     << my_strftime(_("%b %e, %Y"), &rel_time);

#ifdef DEBUG_EXPAND_DIR
	cout << ExpandDirectory("~zsh/bin") << '\n'
	     << ExpandDirectory("~zsh") << '\n'
	     << ExpandDirectory("~") << '\n'
	     << ExpandDirectory("~/bin") << '\n'
	     << ExpandDirectory("////bin///") << '\n'
	     << ExpandDirectory("~/../bin") << '\n'
	     << ExpandDirectory("/././...//") << '\n'
	     << ExpandDirectory("/../usr1") << '\n'
	     << ExpandDirectory("/usr1/../usr2/../../usr3") << '\n';
#endif
}

void	Help()
{
	PrintBanner();
	gtout(cout, _("Usage:   %$ [DIR]\n"
		      "  or:    %$ [OPTION]\n")) << progName << progName;
	cout << _("Available options:\n");
	cout << _("     --config       Display current configuration\n"
		  "     --configattr   Display current screen attribute configuration\n"
		  "     --features     Display features included during compilation\n"
		  "     -h, --help     Display this help\n"
		  "     --helpinst     Display help about shell alias/function installation\n");
	cout << _("     -r             Rescan directory tree\n"
		  "     -rf            Rescan directory tree using full rescan mode\n"
		  "     -rp DIR        Rescan part of directory tree beginning at DIR\n"
		  "     -rq            Rescan directory tree, do not display progress\n"
		  "     -rQ            Rescan directory tree, display progress\n"
		  "     -v, --version  Display version number\n\n");
}

void	HelpInst()
{
	PrintBanner();
	gtout(cout, _("Usage:   eval `%$ [OPTION]`\n")) << progName;
	cout << _("Available options:\n");
	cout << _("     -ia            Print command for shell alias installation\n"
		  "                    (Works with bash, pdksh, zsh)\n");
	cout << _("     -ic            Print command for shell alias installation\n"
		  "                    (Works with tcsh)\n");
	cout << _("     -if            Print command for shell function installation\n"
		  "                    (Works with ash)\n");
	cout << _("     -ias DIR,      Same as -ia,-ic or -if but use the specified\n"
		  "        -ics DIR,   directory\n"
		  "        -ifs DIR\n");
	cout << _("     -iau,-icu,-ifu Same as -ia,-ic or -if but use programs in user's\n"
		  "                    directory\n\n");
}

void	Features()
{
	PrintBanner();
	
	cout << _("     xterm/rxvt resize   : ")
#ifdef	HAVE_RESIZETERM
	     << _("Yes")
#else
	     << _("No")
#endif
	     << "\n";

	cout << _("     mouse support       : ")
#ifdef	NCURSES_MOUSE_VERSION
	     << _("Yes")
#else
	     << _("No")
#endif
	     << "\n";

	cout << _("     default scroll bar  : ")
#ifdef	KCD_NC_GOOD_SCROLL
	     << _("Yes")
#else
	     << _("No")
#endif
	     << "\n";
	
	cout << _("     UTF-8 support       : ")
#ifdef	USE_UTF8_MODE
	     << _("Yes")
#else
	     << _("No")
#endif
	     << "\n";
	
	cout << '\n';
}

/*************************************************************************
	Initialize ~/.kcd.newdir to contain "."
*************************************************************************/

void	InitNewDir()
{
	FILE	*file;

	file = k_fopen(dirFile, "wb");
	if (file == NULL) {		// Cannot create or truncate
					// ~/.kcd.newdir
		throw ErrorGenericFile(_("cannot create file %$"), dirFile);
	}
	fputs(".", file);		// Make sure cd "`cat ~/.kcd.newdir`"
					// will not cause problem if we are
					// not going to change dir., i.e.,
					// when displaying help, etc.
	fclose(file);
}

/*************************************************************************
	Main
*************************************************************************/

void	mainReal(int argc, char *argv[])
{
#ifdef HAVE_SETENV
	setenv("IFS", " \t\n", 1);	// Security precaution
	unsetenv("NLSPATH");
#else
	putenv("IFS= \t\n");		// Security precaution
	putenv("NLSPATH=");
#endif

#ifdef HAVE_LOCALE_H
	setlocale(LC_ALL, "");		// FIXME: Switch to C++ locale object
					// when available

	bindtextdomain(PACKAGE, KCD_LOCALEDIR);
	textdomain(PACKAGE);
#endif

					// Save current working dir
	saveCwd = k_getcwd();
					// Initialize ~/.kcd.save.gz,
					// ~/.kcd.newdir and
					// ~/.kcd.conf filename
	string	home_dir = ExpandDirectory("~");
	if (home_dir[home_dir.size()-1] != '/')	// Not end with "/"
		home_dir += '/';	// Need to add a "/" after home dir

	if (getenv("KCD_DIRTREE") && strlen(getenv("KCD_DIRTREE"))) {
		saveFile = getenv("KCD_DIRTREE");
	}
	else {
		saveFile = home_dir;
		saveFile += ".kcd.save.gz";	// Add appropriate filename
	}

	if (getenv("KCD_TEMP") && strlen(getenv("KCD_TEMP"))) {
		dirFile = getenv("KCD_TEMP");
	}
	else {
		dirFile = home_dir;
		dirFile += ".kcd.newdir";	// Add appropriate filename
	}

	if (getenv("KCD_CONFIG") && strlen(getenv("KCD_CONFIG"))) {
		confLocalFile = getenv("KCD_CONFIG");
		NeedSecureUserConfig();		// Turn on security check
	}
	else {
		confLocalFile = home_dir;
		confLocalFile += ".kcd.conf";	// Add appropriate filename
	}

	InitNewDir();			// Initialize ~/.kcd.newdir

	if (argc == 1) { 		// No parameters
		CD("");
	}
	else {
		int	argcCur = 1;
		int	argcLeft = argc-1;

		int	rescan = 0;
		vector<string>	partial_dir;

		while (argcLeft) {
			if (strcmp(argv[argcCur], "-h") == 0) {
				Help();
				return;
			}
			else if (strcmp(argv[argcCur], "--help") == 0) {
				Help();
				return;
			}

			else if (strcmp(argv[argcCur], "--helpinst") == 0) {
				HelpInst();
				return;
			}

			else if (strcmp(argv[argcCur], "-v") == 0) {
				Version();
				return;
			}
			else if (strcmp(argv[argcCur], "--version") == 0) {
				Version();
				return;
			}
			else if (strcmp(argv[argcCur], "--features") == 0) {
				Features();
				return;
			}
			else if (strcmp(argv[argcCur], "--config") == 0) {
				LoadAllConfig(false);
				PrintConfig();
				return;
			}
			else if (strcmp(argv[argcCur], "--configattr") == 0) {
				LoadAllConfig(false);
				PrintAttrConfig();
				return;
			}
			else if (strcmp(argv[argcCur], "-ia") == 0) {
				if (argcLeft == 1)
					cout << "alias kcd=\'. /usr/bin/kcdscr\'\n";
				else
					throw ErrorGenericCommandLine(_("argument not allowed for `-ia\'"));
				return;
			}
			else if (strcmp(argv[argcCur], "-ic") == 0) {
				if (argcLeft == 1)
					cout << "alias kcd \'source /usr/bin/kcdscr\'\n";
				else
					throw ErrorGenericCommandLine(_("argument not allowed for `-ic\'"));
				return;
			}
			else if (strcmp(argv[argcCur], "-if") == 0) {
				if (argcLeft == 1)
					cout << "kcd () { . /usr/bin/kcdscr \"$@\" ; }\n";
				else
					throw ErrorGenericCommandLine(_("argument not allowed for `-if\'"));
				return;
			}
			else if (strcmp(argv[argcCur], "-iau") == 0) {
				if (argcLeft == 1)
					cout << "alias kcd=\'. $HOME/kcd-bin/kcdscr\'\n";
				else
					throw ErrorGenericCommandLine(_("argument not allowed for `-iau\'"));
				return;
			}
			else if (strcmp(argv[argcCur], "-icu") == 0) {
				if (argcLeft == 1)
					cout << "alias kcd \'source $home/kcd-bin/kcdscr\'\n";
				else
					throw ErrorGenericCommandLine(_("argument not allowed for `-icu\'"));
				return;
			}
			else if (strcmp(argv[argcCur], "-ifu") == 0) {
				if (argcLeft == 1)
					cout << "kcd () { . $HOME/kcd-bin/kcdscr \"$@\" ; }\n";
				else
					throw ErrorGenericCommandLine(_("argument not allowed for `-ifu\'"));
				return;
			}
			else if (strcmp(argv[argcCur], "-ias") == 0) {
				if (argcLeft == 2)
					cout << "alias kcd=\'. " << argv[argcCur+1] << "/kcdscr\'\n";
				else
					throw ErrorGenericCommandLine(_("one argument required for `-ias\'"));
				return;
			}
			else if (strcmp(argv[argcCur], "-ics") == 0) {
				if (argcLeft == 2)
					cout << "alias kcd \'source " << argv[argcCur+1] << "/kcdscr\'\n";
				else
					throw ErrorGenericCommandLine(_("one argument required for `-ics\'"));
				return;
			}
			else if (strcmp(argv[argcCur], "-ifs") == 0) {
				if (argcLeft == 2)
					cout << "kcd () { . " << argv[argcCur+1] << "/kcdscr \"$@\" ; }\n";
				else
					throw ErrorGenericCommandLine(_("one argument required for `-ifs\'"));
				return;
			}
						// Rescan dir.
			else if (strncmp(argv[argcCur], "-r", 2) == 0) {
				rescan = 1;

				bool	use_partial_dir = false;

				for (char *p = argv[argcCur]+2; *p; ++p) {
					switch (*p) {
						case 'q':
							forceQuiet = 1;
							break;
						case 'Q':
							forceQuiet = 0;
							break;
						case 'f':
							forceFullScan = 1;
							break;
						case 'p':
							if (argcLeft < 2) {
								throw ErrorGenericCommandLine(_("argument required for `-rp\'"));
							}
							else if (use_partial_dir) {
								throw ErrorGenericCommandLine(_("`-rpp\' not allowed"));
							}
							else if (strlen(argv[argcCur+1]) == 0) {
								throw ErrorGenericCommandLine(_("argument for `-rp\' cannot be empty"));
							}
							else if (saveCwd.size() == 0 && partial_dir.size()) {
								throw ErrorGenericCommandLine(_("multiple `-rp\' not allowed when current directory cannot be determined"));
							}
							partial_dir.push_back(argv[argcCur+1]);
							use_partial_dir = true;
							break;
						default:
							throw ErrorGenericCommandLine(_("unknown specifier for `-r\' option"));
					}
				}
				if (use_partial_dir) {	// Consume argument of -rp
					argcCur++;
					argcLeft--;
				}
			}
			else if (argv[argcCur][0] == '-') {	// Unrecognized option
				throw ErrorGenericCommandLine(_("unknown option"));
			}
			else {				// Not an option, must be a directory name
				if (argcLeft == 1 && rescan == 0)
					CD(argv[argcCur]);
				else			// Too many arguments
					throw ErrorGenericCommandLine(_("too many arguments"));
				return;
			}

			argcCur++;
			argcLeft--;
		}

		if (rescan) {
			if (partial_dir.size())
				PartialScanDir(partial_dir, forceFullScan);
			else
				ScanDir(forceFullScan);
		}
		else {					// Consumed all option but
							// no action specified yet
			CD("");
		}
	}
}

int	main(int argc, char *argv[])
{
	cxxlib();

	try {
		// Work around for gcc EH optimization bugs
		mainReal(argc, argv);
	}
	catch(BAD_ALLOC &) {
		cout << flush;			// Avoid out-of-order display
		cerr << progName << _(": not enough memory\n");
		return RET_MEMERR;
	}
	catch(ErrorBase &e) {
		cout << flush;			// Avoid out-of-order display
		cerr << progName << ": " << e.what() << '\n';
		return e.ret();
	}
	catch(...) {
		cout << flush;			// Avoid out-of-order display
		cerr << progName << _(": unknown exception\n");
		return RET_UNKERR;
	}
}
