;; -*- lisp -*-

(in-package :it.bese.ucw)

;;;; ** STANDARD-SESSION-FRAME

(defmethod make-new-action ((frame standard-session-frame) (lambda function))
  (insert-with-new-key (frame.actions frame) +action-id-length+ lambda))

(defmethod find-action ((f standard-session-frame) (action-id string))
  (gethash action-id (frame.actions f)))

(defmethod find-action ((f standard-session-frame) (action-id list))
  (find-action f (car action-id)))

(defmethod find-action ((f standard-session-frame) (action-id null))
  nil)

(defmethod (setf find-action) (lambda (f standard-session-frame) (action-id string))
  (setf (gethash action-id (frame.actions f))
	lambda))

(defun register-action (action-lambda)
  (make-new-action (context.current-frame *context*) action-lambda))

(defun make-new-callback (lambda &key (frame (context.current-frame *context*))
                                 (name nil))
  "Registers a new callback.

LAMBDA when the request arrives LAMBDA will be passed the
value (a string) associated with the input. If NAME is not
provided, or NIL, a random name will be generated.

Returns NAME, or the random name if NAME was NIL."
  (let ((effective-name (or name
                            (new-random-key (frame.callbacks frame)
                                            +action-id-length+))))
    (make-new-callback-using-class frame lambda effective-name)
    effective-name))

(defmethod make-new-callback-using-class ((f standard-session-frame)
                                          lambda &optional (name nil))
  "Returns the name for a query param which, when passed in a
  action generated by F will call LAMBDA passing it the value of
  the param."
  (let ((effective-name (or name
                            (new-random-key (frame.callbacks f)
                                            +action-id-length+))))
    (setf (gethash effective-name (frame.callbacks f)) lambda)
    effective-name))

(defmethod call-callback ((f standard-session-frame) (callback-id string) value)
  (aif (gethash callback-id (frame.callbacks f))
       (with-action-error-handler ()
         (funcall it value))
       (ucw.rerl.warn "No callback found with id ~S in frame ~S (value is ~S)."
                      callback-id f value)))

(defun register-callback (callback)
  (make-new-callback callback))

(defmethod call-callbacks ((frame standard-session-frame) (request request))
  "Execute all the callback lambda in CONTEXT's request.

Simply goes through the request's params and, for every param
which isn't +action-parameter-name+, +frame-parameter-name+ or
+session-parameter-name+, looks up and call the associated lambda
in the current frame."
  (map-parameters request
                  (lambda (param value)
                    (unless (member param (list +action-parameter-name+
                                                +frame-parameter-name+
                                                +session-parameter-name+)
                                    :test #'string=)
                      (call-callback frame param value)))))

(defmethod make-next-frame ((f standard-session-frame) new-id)
  (make-instance 'standard-session-frame
                 :backtracks (clone-backtracks (frame.backtracks f))
                 :window-component (frame.window-component f)
                 :id new-id))

(defmethod make-next-frame ((f null) new-id)
  (make-instance 'standard-session-frame :id new-id))

;; Copyright (c) 2003-2005 Edward Marco Baringer
;; All rights reserved. 
;; 
;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions are
;; met:
;; 
;;  - Redistributions of source code must retain the above copyright
;;    notice, this list of conditions and the following disclaimer.
;; 
;;  - Redistributions in binary form must reproduce the above copyright
;;    notice, this list of conditions and the following disclaimer in the
;;    documentation and/or other materials provided with the distribution.
;; 
;;  - Neither the name of Edward Marco Baringer, nor BESE, nor the names
;;    of its contributors may be used to endorse or promote products
;;    derived from this software without specific prior written permission.
;; 
;; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
;; "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
;; LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
;; A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT
;; OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
;; SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
;; LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
;; DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
;; THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
;; (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
;; OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
