(in-package :it.bese.ucw)

(enable-bracket-reader)

(defvar *ajax-component-being-rendered*)

(defun currently-ajax-rendered-component ()
  (when (boundp '*ajax-component-being-rendered*)
    *ajax-component-being-rendered*))

(defun ajax-rendering-in-progress-p ()
  (boundp '*ajax-component-being-rendered*))

(defcomponent ajax-component-mixin (html-element standard-component)
  ((dom-id :initform (js:gen-js-name-string :prefix "ajax")) ; override the initform of the inherited slot
   (has-ever-been-rendered :initform nil :accessor has-ever-been-rendered-p)
   (forbid-ajax-rendering :initform nil :accessor forbid-ajax-rendering-p :initarg :forbid-ajax-rendering-p
                          :documentation "This predicate my forbid AJAX rendering from this component and instruct the renderer to look further on the parent chain. The primary use of this is that sometimes (mostly due to browser rendring bugs) it's better to render bigger chunks of the page."))
  (:documentation "This is a marker class that marks a point in the component
hierarchy from where a partial (AJAX) render may be started. The component
must render exactly one top-level DOM node and it must have an ID attribute.
The client side js will look up the DOM node identified by ID and replace it
with the freshly rendered one.

Please note that this component in itself is not suitable for ajax
DOM node replacements because it does not render any wrapper nodes.
See WIDGET-COMPONENT for an ajax component that works on its own."))

(defmethod allow-ajax-rendering-restart ((component ajax-component-mixin))
  (and (not (forbid-ajax-rendering-p component))
       (has-ever-been-rendered-p component)))

(defparameter %ajax-stub-rendering-in-progress% nil
  "Marks that we are going to render only a stub, so bail out in render :wrapping ajax-component-mixin.")

(defgeneric render-ajax-stub (ajax-component-mixin)
  (:method :around ((self ajax-component-mixin))
           (let ((%ajax-stub-rendering-in-progress% t))
             (call-next-method)))
  (:method ((self ajax-component-mixin))
           (render self))
  (:documentation "Start rendering and stop at ajax-component-mixin boundaries. Only render a stub at those points (usually a <:div with an id) that can be later lazily replaced with an AJAX request."))

(defmethod render :after ((self ajax-component-mixin))
  (setf (has-ever-been-rendered-p self) t))

(defmethod render :wrapping ((self ajax-component-mixin))
  (unless %ajax-stub-rendering-in-progress%
    (call-next-method)))

(defgeneric ajax-render (component)
  (:documentation "This method is called when we are rendering parts of the component hierarchy with AJAX.
By default it simply calls render after marking this fact on the ajax-component-mixin.")
  (:method :around ((self ajax-component-mixin))
           (let ((*ajax-component-being-rendered* self))
             (call-next-method)))
  (:method ((self ajax-component-mixin))
           (render self)))

(defmacro within-xhtml-tag (tag-name &body body)
  "Execute BODY and wrap its yaclml output in a TAG-NAME xml node
with \"http://www.w3.org/1999/xhtml\" xml namespace."
  `{with-xml-syntax
     <(progn ,tag-name) :xmlns #.+xhtml-namespace-uri+
                        (@ "xmlns:dojo" #.+dojo-namespace-uri+)
       ,@body>})

(defmacro within-dom-replacements-tag (&body body)
  "Execute BODY and wrap its yaclml output in a dom-replacements xml node
with \"http://www.w3.org/1999/xhtml\" xml namespace. Client side js
iterates the elements of this node and replaces their counterparts
in the DOM tree with them."
  `(within-xhtml-tag "dom-replacements"
     ,@body))

(defun render-nearest-ajax-component (component &key (wrap-in-dom-replacements t))
  (ucw.rerl.ajax.debug "render-nearest-ajax-component from ~S" component)
  (let ((ajax-component (iter (for current :first component :then (parent current))
                              (while current)
                              (ucw.rerl.ajax.dribble "Checking ~S" current)
                              (when (and (typep current 'ajax-component-mixin)
                                         (allow-ajax-rendering-restart current))
                                (return current))
                              (while (slot-boundp current 'parent))
                              (finally (return nil)))))
    (ucw.rerl.ajax.debug "render-nearest-ajax-component ended up at ~S" ajax-component)
    (unless ajax-component
      (error "No suitable ajax-component-mixin was found while walking the parent slots of ~A, unable to render AJAX answer" component))
    (if wrap-in-dom-replacements
        (within-dom-replacements-tag
          (ajax-render ajax-component))
        (ajax-render ajax-component))))

(define-condition visible-dirty-component-remained (error)
  ((component :initarg :component :accessor component-of))
  (:report (lambda (c stream)
             (format stream "A visible dirty component ~A remained in session ~A after calling ajax-render-dirty-components. This would lead to a constant ajax rerendering in the poller. Make sure you either render all connected components or detach them!"
                     (component-of c) (session-of (component-of c))))))

(defmethod handle-toplevel-condition :around (application
                                              (error visible-dirty-component-remained)
                                              (action ajax-action))
  (when (debug-on-error application)
    (invoke-slime-debugger-if-possible error))
  ;; when we are not debugging, just remove dirtyness and continue normal operation
  (continue))

(defun ajax-render-dirty-components ()
  (within-dom-replacements-tag
    (iterate-dirty-components
     (lambda (component)
       (ucw.rerl.ajax.debug "ajax-render-dirty-components at component ~S" component)
       (render-nearest-ajax-component component :wrap-in-dom-replacements nil)))
    (iterate-dirty-components
     (lambda (c)
       (when (visiblep c)
         (restart-case
              (error 'visible-dirty-component-remained :component c)
           (continue ()
             :report "Remove dirtyness and leave me alone..."
             (setf (dirtyp c) nil))))))))


(defmacro js-server-callback ((&rest args &key
                                     (invocation-isolated nil invocation-isolated-provided-p)
                                     &allow-other-keys) &body body)
  "This macro can be used to define unnamed server callbacks in parenscript bodies."
  (let ((action-args nil))
    (when invocation-isolated-provided-p
      (push* action-args invocation-isolated :invocation-isolated))
    (remf-keywords args :invocation-isolated)
    ``(ucw.io.execute-ajax-action
       (create :url
        ,(action-href
          (register-ajax-action (,@action-args)
            ,@body))
        ,,@args))))

