/*
  diskwrite.c  --  write to file at given offset
  Copyright (C) 1999 Steffen Solyga <solyga@absinth.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 *	$Id: diskwrite.c,v 1.2 1999/11/22 02:17:59 solyga Exp $
 */

#include	"diskwrite.h"


int
display_help( char* pn ) {
  fprintf( HELP_CHANNEL, "%s v%s (%s): ", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( HELP_CHANNEL, "Write to file at given offset.\n" );
  fprintf( HELP_CHANNEL, "Flowers & bug reports to %s.\n", MY_EMAIL_ADDRESS );
  fprintf( HELP_CHANNEL, "Usage: %s [options] file\n", pn );
  fprintf( HELP_CHANNEL, "switches:\n" );
  fprintf( HELP_CHANNEL, "  -c\t create non-existing output file\n" );
  fprintf( HELP_CHANNEL, "  -h\t write this info to %s and exit sucessfully\n", HELP_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "  -v\t raise verbosity level on %s (max %d)\n", VERBOSE_CHANNEL==stdout?"stdout":"stderr", VERBOSE_LEVEL_MAX );
  fprintf( HELP_CHANNEL, "  -V\t print version and compilation info to %s and exit sucessfully\n", VERSION_CHANNEL==stdout?"stdout":"stderr" );
  fprintf( HELP_CHANNEL, "controllers:\n" );
  fprintf( HELP_CHANNEL, "  -f\t FILE\t input file (default stdin)\n" );
  fprintf( HELP_CHANNEL, "  -n\t NUMBER\t number of bytes to use from input (default all)\n" );
  fprintf( HELP_CHANNEL, "  -o\t OFFSET\t file offset to start writing at (default 0)\n" );
  return( 0 );
}


int
display_version( char* pn ) {
  fprintf( VERSION_CHANNEL, "%s v%s (%s)\n", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( VERSION_CHANNEL, "compilation settings:\n" );
  fprintf( VERSION_CHANNEL, "  BUFFER_SIZE  : %d\n", BUFFER_SIZE );
  fprintf( VERSION_CHANNEL, "  HAVE_LONGLONG: " );
#ifdef	HAVE_LONGLONG
  fprintf( VERSION_CHANNEL, "yes\n" );
#else
  fprintf( VERSION_CHANNEL, "no\n" );
#endif
  return( 0 );
}


char*
basename( char* name ) {
/*
 * strip directory from name
 * returns pointer to stripped name
 * hint: basename("/usr/bin/") == ""
 *   basename(1) would return "bin" !!
 */
  char* p= name;
  while( *p != '\0' ) p++;
  while( p > name ) {
    if( *(p-1) == '/' ) break;
    else p--;
  }
  return( p );
}


char*
str_dsc( char* str1, char* str2) {
/*
returns pointer to first char of str1 not contained in str2
started 1997-02-09
*/
  char *p1, *p2;
  for ( p1=str1; ; p1++ ) {
    if ( *p1 == '\0' ) return( p1 );
    for ( p2=str2; ; p2++ ) {
      if ( *p2 == '\0' ) return( p1 );
      if ( *p2 == *p1 ) break;
    }
  }
}


OFF_T
getoptarg( char opt, char* arg, OFF_T min, OFF_T max, char* pn, int* flag ) {
  OFF_T val;
  char* p;
  *flag= 1;
  val= STRTOL( arg, &p, 0 );
  if( p == arg  ||  *str_dsc(p," ") != '\0' ) {
    fprintf( ERROR_CHANNEL,
             "%s: Argument of option %c (`%s') must be a number.\n",
             pn, opt, arg );
    return( val );
  }
  if( val < min  ||  val > max ) { /* out of range */
    fprintf( ERROR_CHANNEL,
#ifdef	HAVE_LONGLONG
             "%s: Argument value of option %c (%lld) must be in [%lld,%lld].\n",
#else
             "%s: Argument value of option %c (%ld) must be in [%ld,%ld].\n",
#endif
             pn, opt, val, min, max );
    return( val );
  }
  *flag= 0;
  return( val );
}


ssize_t
my_read( int fd, void* buf, size_t count ) {
/*
 * (attempts to) read exactly count bytes from fd into buf
 * returns number of bytes read or -1 on error
 * retval < count indicates EOF
 * started 1998-01-01
 */
  unsigned char* p= buf;
  ssize_t nbr;
  ssize_t tnbr= 0;
  size_t rem= count;
  do {
    if( (nbr= read( fd, p+tnbr, rem )) == -1 ) return( -1 );
    tnbr+= nbr;
    rem-= nbr;
  } while( nbr>0 && rem>0 );
  return( tnbr );
}


ssize_t
my_write( int fd, void *buf, size_t count ) {
/*
 * (attempts to) write exactly count bytes from buf to fd
 * returns number of bytes written or -1 on error
 * retval < count indicates nothing special (see write(2))
 * started 1998-01-01
 */
  unsigned char* p= buf;
  ssize_t nbw;
  ssize_t tnbw= 0;
  size_t rem= count;
  do {
    if( (nbw= write( fd, p+tnbw, rem )) == -1 ) return( -1 );
    tnbw+= nbw;
    rem-= nbw;
  } while( nbw>0 && rem>0 );
  return( tnbw );
}


int
main( int argc, char** argv ) {
/*
 * main() prg_name
 * started 
 */
  char* fpn= *argv;
  char* pn= basename( fpn );		/* prg_name is Makefile constant */
  int retval= RETVAL_OK;
  int c;
  int verbose= 0;			/* verbosity level */
  int create= 0;
  OFF_T start_offset= 0;		/* offset for writing */
  OFF_T off;
  char* in_fn= NULL;
  int in_fd= STDIN_FILENO;
  char* out_fn= NULL;
  int out_fd= STDOUT_FILENO;
  ssize_t nbr;				/* number of bytes read */
  ssize_t nbw;				/* number of bytes written */
  size_t nbtr;				/* number of bytes to read */
  OFF_T tnbtr= SIZE_MAX;		/* total number of bytes to read */
  OFF_T rem;				/* number of bytes remaining */
  OFF_T tnbr;				/* total number of bytes read */
  OFF_T tnbw;				/* total number of bytes written */
  UCHAR buf[BUFFER_SIZE];


/* process options */
  *argv= pn;				/* give getop() the cut name */
  while( (c=getopt(argc,argv,"cf:hn:o:vV")) != EOF ) {
    int flag;
    switch( c ) {
      case 'c': /* create output file */
        create= 1;
        break;
      case 'f': /* input from file */
        in_fn= optarg;
        break;
      case 'h': /* display help and exit sucessfully */
        display_help( pn );
        retval= RETVAL_OK; goto DIE_NOW;
      case 'n': /* number of bytes to use from input */
        tnbtr= getoptarg( 'n', optarg, 1, SIZE_MAX, pn, &flag );
        if( flag ) {
          retval= RETVAL_ERROR; goto DIE_NOW;
        }
        break;
      case 'o': /* offset to start writing at */
        start_offset= getoptarg( 'o', optarg, 0, SIZE_MAX, pn, &flag );
        if( flag ) {
          retval= RETVAL_ERROR; goto DIE_NOW;
        }
        break;
      case 'v': /* raise verbosity level */
        verbose++;
        break;
      case 'V': /* display version to VERSION_CHANNEL and exit sucessfully */
        display_version( pn );
        retval= RETVAL_OK; goto DIE_NOW;
      case '?': /* refer to -h and exit unsucessfully */
        fprintf( ERROR_CHANNEL, "%s: Try `%s -h' for more information.\n",
                 pn, pn );
        retval= RETVAL_ERROR; goto DIE_NOW;
      default : /* program error */
        fprintf( ERROR_CHANNEL, "%s: Options bug! E-mail me at %s.\n",
                 pn, MY_EMAIL_ADDRESS );
        retval= RETVAL_BUG; goto DIE_NOW;
    }
  }


/* open input */
  if( in_fn != NULL  &&  *in_fn != '-') {
    if( (in_fd=open(in_fn,O_RDONLY)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot open `%s' read-only. %s.\n",
               pn, in_fn, sys_errlist[errno] );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
  }


/* open output */
  if( argc > optind )
    out_fn= argv[optind];
  else {
    fprintf( ERROR_CHANNEL, "%s: Where do you want to write to?\n", pn );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }
  if( out_fn != NULL  &&  *out_fn != '-' ) {
    if( create ) {
      if( (out_fd=open(out_fn,O_CREAT|O_WRONLY,PERMISSIONS)) == -1 ) {
        fprintf( ERROR_CHANNEL, "%s: Cannot create `%s'. %s.\n",
                 pn, out_fn, sys_errlist[errno] );
        retval= RETVAL_ERROR; goto DIE_NOW;
      }
    }
    else {
      if( (out_fd=open(out_fn,O_WRONLY)) == -1 ) {
        fprintf( ERROR_CHANNEL, "%s: Cannot open `%s' write-only. %s.\n",
                 pn, out_fn, sys_errlist[errno] );
        if( errno == ENOENT ) {
          int i;
          fprintf( ERROR_CHANNEL, "%s: Say `%s -c", pn, fpn );
          for( i=1; i<argc; i++ )
            fprintf( ERROR_CHANNEL, " %s", argv[i] );
          fprintf( ERROR_CHANNEL, "' if you want to create `%s'.\n", out_fn );
        }
        retval= RETVAL_ERROR; goto DIE_NOW;
      }
    }
  }


/* lseek output */
  if( start_offset > 0 ) {
    if( (off=LSEEK(out_fd,start_offset,SEEK_SET)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot seek ", pn );
      if( out_fd == STDOUT_FILENO )
        fprintf( ERROR_CHANNEL, "stdout" );
      else
        fprintf( ERROR_CHANNEL, "`%s'", out_fn );
      fprintf( ERROR_CHANNEL, ". %s.\n", sys_errlist[errno] );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
  }
#ifdef	DEBUG
  fprintf( DEBUG_CHANNEL, "DEBUG: errno after lseek() %d (%s)\n",
           errno, sys_errlist[errno] );
#endif


/* do the work */
  tnbr= 0;
  tnbw= 0;
  rem= tnbtr;
  do {
    nbtr= MIN( BUFFER_SIZE, rem );
    if( (nbr=my_read(in_fd,buf,nbtr)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot read from ", pn );
      if( in_fd == STDIN_FILENO )
        fprintf( ERROR_CHANNEL, "stdin" );
      else
        fprintf( ERROR_CHANNEL, "`%s'", in_fn );
      fprintf( ERROR_CHANNEL, ". %s.\n", sys_errlist[errno] );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
    tnbr+= nbr;
    if( (nbw=my_write(out_fd,buf,nbr)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot write to ", pn );
      if( out_fd == STDOUT_FILENO )
        fprintf( ERROR_CHANNEL, "stdout" );
      else
        fprintf( ERROR_CHANNEL, "`%s'", out_fn );
#ifdef	HAVE_LONGLONG
      fprintf( ERROR_CHANNEL, " at offset %lld. %s.\n",
#else
      fprintf( ERROR_CHANNEL, " at offset %ld. %s.\n",
#endif
               off, sys_errlist[errno] );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
    tnbw+= nbw;
    off+= nbw;
    if( nbw < nbr ) {
      fprintf( ERROR_CHANNEL, "%s: Bug detected (nbw<nbr)! E-mail me at %s.\n",
               pn, MY_EMAIL_ADDRESS );
      retval= RETVAL_BUG; goto DIE_NOW;
    }
    rem-= nbr;
  } while( nbr == nbtr  &&  rem > 0 );


/* close output */
  if( out_fd != STDOUT_FILENO  &&  close(out_fd) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Cannot close `%s'. %s.\n",
             pn, out_fn, sys_errlist[errno] );
    retval= RETVAL_ERROR;
  }


/* close input */
  if( in_fd != STDIN_FILENO  &&  close(in_fd) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Cannot close `%s'. %s.\n",
             pn, in_fn, sys_errlist[errno] );
    retval= RETVAL_ERROR;
  }


/* talk a bit */
  if( verbose >= 1 )
#ifdef	HAVE_LONGLONG
    fprintf( VERBOSE_CHANNEL, "%s: %lld/%lld bytes read/written.\n",
#else
    fprintf( VERBOSE_CHANNEL, "%s: %ld/%ld bytes read/written.\n",
#endif
             pn, tnbr, tnbw );


DIE_NOW:
  close( out_fd );
  close( in_fd );
  exit( retval );
}
