/*
  diskdump.c  --  a copy program with simple cutting capability
  Copyright (C) 1998-2002 Steffen Solyga <solyga@absinth.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 *	$Id: diskdump.c,v 1.5 2002/10/16 15:38:53 solyga Exp $
 */

#include	"diskdump.h"


int
display_help( char* pn ) {
  fprintf( HELP_CHANNEL, "%s v%s (%s): ", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( HELP_CHANNEL, "Copy to stdout.\n" );
  fprintf( HELP_CHANNEL, "Flowers & bug reports to %s.\n", MY_EMAIL_ADDRESS );
  fprintf( HELP_CHANNEL, "Usage: %s [options] [file]\n", pn );
  fprintf( HELP_CHANNEL, "switches:\n" );
  fprintf( HELP_CHANNEL, "  -h\t help, write this info to %s and exit sucessfully\n", HELP_CHANNEL==stdout ? "stdout" : "stderr" );
  fprintf( HELP_CHANNEL, "  -v\t increase verbosity level on %s\n", VERBOSE_CHANNEL==stdout ? "stdout" : "stderr" );
  fprintf( HELP_CHANNEL, "  -V\t print version and compilation info to %s and exit sucessfully\n", VERSION_CHANNEL==stdout ? "stdout" : "stderr" );
  fprintf( HELP_CHANNEL, "controllers:\n" );
  fprintf( HELP_CHANNEL, "  -H\t SIZE\t write hash marks to %s (one per SIZE bytes)\n", VERBOSE_CHANNEL==stdout ? "stdout" : "stderr" );
  fprintf( HELP_CHANNEL, "  -n\t NUMBER\t number of bytes to copy\n" );
  fprintf( HELP_CHANNEL, "  -o\t NUMBER\t file offset to start at\n" );
  return( 0 );
}


int
display_version( char* pn ) {
  fprintf( VERSION_CHANNEL, "%s v%s (%s)\n", pn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( VERSION_CHANNEL, "Compilation settings:\n" );
  fprintf( VERSION_CHANNEL, "  HAVE_LSEEK64: " );
#ifdef HAVE_LSEEK64
  fprintf( VERSION_CHANNEL, "defined\n" );
#else
  fprintf( VERSION_CHANNEL, "undefined\n" );
#endif
  fprintf( VERSION_CHANNEL, "  DEBUG       : " );
#ifdef DEBUG
  fprintf( VERSION_CHANNEL, "defined\n" );
#else
  fprintf( VERSION_CHANNEL, "undefined\n" );
#endif
  fprintf( VERSION_CHANNEL, "  SIZE_MIN    : " );
#ifdef HAVE_LSEEK64
  fprintf( VERSION_CHANNEL, "%lld\n", SIZE_MIN );
#else
  fprintf( VERSION_CHANNEL, "%ld\n", SIZE_MIN );
#endif
  fprintf( VERSION_CHANNEL, "  SIZE_MAX    : " );
#ifdef HAVE_LSEEK64
  fprintf( VERSION_CHANNEL, "%lld\n", SIZE_MAX );
#else
  fprintf( VERSION_CHANNEL, "%ld\n", SIZE_MAX );
#endif
  return( 0 );
}


off_t
getoptarg( char	opt, char* arg, off_t vmin, off_t vmax, char* pn ) {
  char* p;
  off_t val;
  val= STRTOL( arg, &p, 0 );
  if( p == arg ) {	/* nothing read */
    fprintf( ERROR_CHANNEL,
             "%s: Argument of option %c (`%s') must be a number.\n",
             pn, opt, arg );
    exit( RETVAL_ERROR );
  }
  if( val<vmin || val > vmax ) {	/* out of range */
#ifdef	HAVE_LSEEK64
    fprintf( ERROR_CHANNEL, "%s: Argument value of option %c (%lld) must be in [%lld,%lld].\n", pn, opt, val, vmin, vmax );
#else
    fprintf( ERROR_CHANNEL, "%s: Argument value of option %c (%ld) must be in [%ld,%ld].\n", pn, opt, val, vmin, vmax );
#endif
    exit( RETVAL_ERROR );
  }
  return( val );
}


char*
str_dsc( char* str1, char* str2) {
/*
returns pointer to first char of str1 not contained in str2
started 1997-02-09
*/
  char *p1, *p2;
  for ( p1=str1; ; p1++ ) {
    if ( *p1 == '\0' ) return( p1 );
    for ( p2=str2; ; p2++ ) {
      if ( *p2 == '\0' ) return( p1 );
      if ( *p2 == *p1 ) break;
    }
  }
}


char*
basename( char* name ) {
/*
 * strip directory from name
 * returns pointer to stripped name
 * hint: basename("/usr/bin/") == ""
 *   basename(1) would return "bin" !!
 */
  char* p= name;
  while( *p != '\0' ) p++;
  while( p > name ) {
    if( *(p-1) == '/' ) break;
    else p--;
  }
  return( p );
}



int
main( int argc, char** argv ) {
/*
 * main() diskdump
 * started      : Sat Dec 12 00:00:00 CET 1998 @beast
*/
  char* pn= NULL;			/* prg_name is Makefile constant */
  char* fpn= *argv;
  int retval;				/* return value */
  int c;				/* option char */
  int in_fd= STDIN_FILENO;		/* input file descriptor */
  char stdin_fn[]= "stdin";
  char* in_fn= stdin_fn;		/* input file name */
  int out_fd= STDOUT_FILENO;		/* output file descriptor */
  int verbose= 0;			/* verbosity level */
  int until_eof= 1;			/* read until eof ? */
  int hash= 0;				/* hash marks ? */
  ssize_t nbr;				/* number of bytes read */
  size_t nbtr;				/* number of bytes to read */
  ssize_t nbw;				/* number of bytes written */
  unsigned char buf[BUFFER_SIZE];
  /* lseek() specific stuff */
  off_t offset= 0;			/* from option */
  off_t off;				/* real */
  off_t tnbr;				/* total number of bytes read */
  off_t tnbtr= 0;			/* total number of bytes to read */
  off_t tnbw;				/* total number of bytes written */
  off_t hash_size= 0;			/* bytes per one hash mark */
  off_t hash_nbr= 0;			/* bytes read since latest hash mark */

/* process options */
  pn= basename( fpn );
  *argv= pn;				/* give getop() the cut name */
  while( (c=getopt(argc,argv,"hH:n:o:vV")) != EOF )
    switch( c ) {
      case 'h': /* display help to HELP_CHANNEL and exit sucessfully */
        display_help( pn );
        retval= RETVAL_OK; goto DIE_NOW;
      case 'H': /* print hash marks after given number of bytes */
        hash= 1;
        {
          char* p= str_dsc( optarg, DIVIDERS );
          char* pold= p;
          hash_size= strtoq( p, &p, 0 );
          if( p==pold || hash_size==0 ) {
            fprintf( ERROR_CHANNEL, "%s: Bad size format `%s'.\n",
                     *argv, optarg );
            retval= 1; goto DIE_NOW;
          }
          switch( *(p=str_dsc(p,DIVIDERS)) ) {
            case '\0': break;
            case 'G' : hash_size*= 1024;
            case 'M' : hash_size*= 1024;
            case 'k' : hash_size*= 1024; break;
            default  :
              fprintf( ERROR_CHANNEL, "%s: Bad unit `%s'.\n", *argv, p );
              retval= 1; goto DIE_NOW;
          }
        }
        break;
      case 'n': /* get number of bytes to dump */
        tnbtr= getoptarg( c, optarg, SIZE_MIN, SIZE_MAX, pn );
        until_eof= 0;
        break;
      case 'o': /* get offset */
        offset= getoptarg( c, optarg, SIZE_MIN, SIZE_MAX, pn );
        break;
      case 'v': /* switch to verbose mode */
        verbose++;
        break;
      case 'V': /* print version info and exit successfully */
        display_version( pn );
        retval= RETVAL_OK; goto DIE_NOW;
      case '?': /* refer to help and exit unsuccessfully */
        fprintf( ERROR_CHANNEL, "%s: Try `%s -h' for more information.\n",
                 pn, pn );
        retval= RETVAL_ERROR; goto DIE_NOW;
      default : /* program error */
        fprintf( ERROR_CHANNEL, "%s: Options bug! E-mail me at %s.\n",
                 pn, MY_EMAIL_ADDRESS );
        retval= RETVAL_BUG; goto DIE_NOW;
    }
#ifdef	DEBUG
#ifdef	HAVE_LSEEK64
fprintf( DEBUG_CHANNEL, "DEBUG: offset= %lld, bytes= %lld%s\n",
         offset, tnbtr, until_eof?", until EOF":"" );
#else
fprintf( DEBUG_CHANNEL, "DEBUG: offset= %ld, bytes= %ld%s\n",
         offset, tnbtr, until_eof?", until EOF":"" );
#endif
#endif
/* print infos */
  if( verbose ) {
    if( hash )
      fprintf( VERBOSE_CHANNEL, "%s: Writing one hash per %lld bytes.\n",
               pn, hash_size );
  }

/* open input file */
  if( argc > optind ) {
    in_fn= argv[optind];
    if( (in_fd=open(in_fn,O_RDONLY)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot open `%s'. %s.\n",
               pn, in_fn, sys_errlist[errno] );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
  }

/* lseek input file */
  if( in_fd != STDIN_FILENO ) {
    if( (off=lseek(in_fd,offset,SEEK_SET)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot lseek `%s'. %s.\n",
               pn, in_fn, sys_errlist[errno] );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
  }
  else {
    if( offset!=0 ) {
      fprintf( ERROR_CHANNEL, "%s: Offsets not implemented for stdin.\n", pn );
      fprintf( ERROR_CHANNEL, "%s: Use dd if you really want to do this.\n", 
               pn );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
  }

/* dump */
  tnbr= 0; tnbw= 0;
  nbtr= until_eof ? (size_t)BUFFER_SIZE : (size_t)MIN(BUFFER_SIZE,tnbtr);
  while( nbtr > 0 ) {
#ifdef	DEBUG
fprintf( DEBUG_CHANNEL, "DEBUG: %d bytes to read\n", nbtr );
#endif
    if( (nbr=read(in_fd,buf,nbtr)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot read from input file. %s.\n",
               pn, sys_errlist[errno] );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
#ifdef  DEBUG
fprintf( DEBUG_CHANNEL, "DEBUG: %d bytes read\n", nbr );
#endif
    tnbr+= nbr;
    hash_nbr+= nbr;
    if( (nbw=write(out_fd,buf,nbr)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot write to output file. %s.\n",
               pn, sys_errlist[errno] );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
    tnbw+= nbw;
    if( hash ) {
      while( hash_nbr >= hash_size ) {
        fprintf( VERBOSE_CHANNEL, "#" );
        hash_nbr-= hash_size;
      }
    }
    if( nbr < nbtr ) {
      if( in_fd != STDIN_FILENO ) break;
      if( nbr == 0 ) break;		/* CD-ROM: read() returns -1 at EOF */
    }
    nbtr= until_eof ? (size_t)BUFFER_SIZE : (size_t)MIN(BUFFER_SIZE,tnbtr-tnbr);
  }
  if( hash ) fprintf( VERBOSE_CHANNEL, "\n" );

/* close input and output file */
  if( close(in_fd) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Cannot close `%s'. %s.\n",
             pn, in_fn, sys_errlist[errno] );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }
  if( close(out_fd) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Cannot close output. %s.\n", 
             pn, sys_errlist[errno] );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }

DIE_NOW:
  if( verbose )
#ifdef	HAVE_LSEEK64
    fprintf( VERBOSE_CHANNEL, "%s: %lld/%lld bytes read/written.\n",
             pn, tnbr, tnbw );
#else
    fprintf( VERBOSE_CHANNEL, "%s: %ld/%ld bytes read/written.\n", 
             pn, tnbr, tnbw );
#endif
  close( in_fd );
  close( out_fd );
  exit( retval );
}
