/*************************************************************************
** interpcom-3.1    (command interpreter)                                **
** _mapm.c : expression evaluator with double precision numbers          **
**                                                                       **
** Copyright (C) 2003  Jean-Marc Drezet                                  **
**                                                                       **
**  This library is free software; you can redistribute it and/or        **
**  modify it under the terms of the GNU Library General Public          **
**  License as published by the Free Software Foundation; either         **
**  version 2 of the License, or (at your option) any later version.     **
**                                                                       **
**  This library is distributed in the hope that it will be useful,      **
**  but WITHOUT ANY WARRANTY; without even the implied warranty of       **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    **
**  Library General Public License for more details.                     **
**                                                                       **
**  You should have received a copy of the GNU Library General Public    **
**  License along with this library; if not, write to the Free           **
**  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.   **
**                                                                       **
** Please mail any bug reports/fixes/enhancements to me at:              **
**      drezet@math.jussieu.fr                                           **
** or                                                                    **
**      Jean-Marc Drezet                                                 **
**      Institut de Mathematiques                                        **
**      UMR 7586 du CNRS                                                 **
**      173, rue du Chevaleret                                           **
**      75013 Paris                                                      **
**      France                                                           **
**                                                                       **
 *************************************************************************/

#include "interp.h"
extern int _matrix_ev;  /* number of the matrix expr. ev */



/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
int
_matrix_iszero(void *z)
{
    double        **t;

    t = (double **)z;
    if (t[0][0] == 0. && t[1][0] == 0. && t[0][1] == 0. && t[0][0] == 0.)
        return 1;
    else
        return 0;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_Zero(void *z)
{
    double        **t;

    t = (double **)z;
    t[0][0] = 0.;
    t[0][1] = 0.;
    t[1][0] = 0.;
    t[1][1] = 0.;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_clear(void *z)
{
    XFREE(z);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_neg(void *z)
{
    double        **t;

    t = (double **)z;
    t[0][0] = - t[0][0];
    t[0][1] = - t[0][1];
    t[1][0] = - t[1][0];
    t[1][1] = - t[1][1];
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_add(void *res, void *a, void *b)
{
    double        **t,
                  **u,
                  **v;

    t = (double **)res;
    u = (double **)a;
    v = (double **)b;
    t[0][0] = u[0][0] + v[0][0];
    t[0][1] = u[0][1] + v[0][1];
    t[1][0] = u[1][0] + v[1][0];
    t[1][1] = u[1][1] + v[1][1];
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_sub(void *res, void *a, void *b)
{
    double        **t,
                  **u,
                  **v;

    t = (double **)res;
    u = (double **)a;
    v = (double **)b;
    t[0][0] = u[0][0] - v[0][0];
    t[0][1] = u[0][1] - v[0][1];
    t[1][0] = u[1][0] - v[1][0];
    t[1][1] = u[1][1] - v[1][1];
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_mul(void *res, void *a, void *b)
{
    double        **t,
                  **u,
                  **v,
                  **w;

    t = (double **)res;
    u = (double **)a;
    v = (double **)b;
    w = double_alloc2(2, 2);
    w[0][0] = u[0][0]*v[0][0] + u[0][1]*v[1][0];
    w[0][1] = u[0][0]*v[0][1] + u[0][1]*v[1][1];
    w[1][0] = u[1][0]*v[0][0] + u[1][1]*v[1][0];
    w[1][1] = u[1][0]*v[0][1] + u[1][1]*v[1][1];
    t[0][0] = w[0][0];
    t[0][1] = w[0][1];
    t[1][0] = w[1][0];
    t[1][1] = w[1][1];
    XFREE(w);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_pow(void *res, void *a, void *b)
{
    double        **t,
                  **u,
                  **v;

    t = (double **)res;
    u = (double **)a;
    v = (double **)b;
    t[0][0] = 0.;
    t[0][1] = 0.;
    t[1][0] = 0.;
    t[1][1] = 0.;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_div(void *res, void *a, void *b)
{
    double        **t,
                  **v,
                  **w,
                    det;

    t = (double **)res;
    v = (double **)b;
    det = v[0][0] * v[1][1] - v[0][1] * v[1][0];
    if (det == 0.) {
        t[0][0] = 0.;
        t[0][1] = 0.;
        t[1][0] = 0.;
        t[1][1] = 0.;
        return;
    }
    w = double_alloc2(2, 2);
    w[0][0] = v[1][1] / det;
    w[0][1] = -v[0][1] / det;
    w[1][0] = -v[1][0] / det;
    w[1][1] = v[0][0] / det;
    _matrix_mul(res, a, (void *)w);
    XFREE(w);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_set(void *res, char *h)
{
    double        **t,
                    x;

    t = (double **)res;
    x = atof(h);
    t[0][0] = x;
    t[0][1] = 0.;
    t[1][0] = 0.;
    t[1][1] = x;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_Init(void **s)
{
    double         **t;

    t = double_alloc2(2, 2);
    s[0] = (void *) t;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
char *
_matrix_print(void *c)
{
    double        **t;
    char           *h00,
                   *h01,
                   *h10,
                   *h11,
                   *h;
    int             l;

    t = (double **)c;
    h00 = ch_copy_float(t[0][0]);
    h01 = ch_copy_float(t[0][1]);
    h10 = ch_copy_float(t[1][0]);
    h11 = ch_copy_float(t[1][1]);
    l = strlen(h00) + strlen(h10) + strlen(h01) + strlen(h11) + 10;
    h = (char *) malloc((size_t) l * sizeof(char));
    memset(h ,0, l);
    sprintf(h, "[%s,%s,%s,%s]", h00, h01, h10, h11);
    free(h00);
    free(h01);
    free(h10);
    free(h11);
    return h;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void *
_matrix_copy(void *c)
{
    double        **t,
                  **u;

    t = (double **)c;
    u = double_alloc2(2, 2);
    u[0][0] = t[0][0];
    u[0][1] = t[0][1];
    u[1][0] = t[1][0];
    u[1][1] = t[1][1];
    return (void *) u;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
int
_matrix_sign(void *c)
{
    return 0;
}
/*------------------------------------------------------------------*/



EXPREVAL_GEN        _matrix_exprev = {
    _matrix_clear,
    _matrix_Zero,
    _matrix_neg,
    _matrix_iszero,
    _matrix_add,
    _matrix_sub,
    _matrix_mul,
    _matrix_pow,
    _matrix_div,
    _matrix_set,
    _matrix_Init,
    _matrix_print,
    _matrix_copy,
    _matrix_sign,
    0,
    NULL,
    NULL,
    1,
    "2X2 matrix expression evaluator"
};




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_sin(void **res, void **x)
{
    double        **t,
                  **u;

    t = (double **) x[0];
    u = (double **) res[0];
    u[0][0] = sin(t[0][0]);
    u[0][1] = sin(t[0][1]);
    u[1][0] = sin(t[1][0]);
    u[1][1] = sin(t[1][1]);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_cos(void **res, void **x)
{
    double        **t,
                  **u;

    t = (double **) x[0];
    u = (double **) res[0];
    u[0][0] = cos(t[0][0]);
    u[0][1] = cos(t[0][1]);
    u[1][0] = cos(t[1][0]);
    u[1][1] = cos(t[1][1]);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_tan(void **res, void **x)
{
    double        **t,
                  **u;

    t = (double **) x[0];
    u = (double **) res[0];
    u[0][0] = tan(t[0][0]);
    u[0][1] = tan(t[0][1]);
    u[1][0] = tan(t[1][0]);
    u[1][1] = tan(t[1][1]);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_asin(void **res, void **x)
{
    double        **t,
                  **u;

    t = (double **) x[0];
    u = (double **) res[0];
    u[0][0] = asin(t[0][0]);
    u[0][1] = asin(t[0][1]);
    u[1][0] = asin(t[1][0]);
    u[1][1] = asin(t[1][1]);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_acos(void **res, void **x)
{
    double        **t,
                  **u;

    t = (double **) x[0];
    u = (double **) res[0];
    u[0][0] = acos(t[0][0]);
    u[0][1] = acos(t[0][1]);
    u[1][0] = acos(t[1][0]);
    u[1][1] = acos(t[1][1]);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_atan(void **res, void **x)
{
    double        **t,
                  **u;

    t = (double **) x[0];
    u = (double **) res[0];
    u[0][0] = atan(t[0][0]);
    u[0][1] = atan(t[0][1]);
    u[1][0] = atan(t[1][0]);
    u[1][1] = atan(t[1][1]);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_sinh(void **res, void **x)
{
    double        **t,
                  **u;

    t = (double **) x[0];
    u = (double **) res[0];
    u[0][0] = sinh(t[0][0]);
    u[0][1] = sinh(t[0][1]);
    u[1][0] = sinh(t[1][0]);
    u[1][1] = sinh(t[1][1]);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_cosh(void **res, void **x)
{
    double        **t,
                  **u;

    t = (double **) x[0];
    u = (double **) res[0];
    u[0][0] = cosh(t[0][0]);
    u[0][1] = cosh(t[0][1]);
    u[1][0] = cosh(t[1][0]);
    u[1][1] = cosh(t[1][1]);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_tanh(void **res, void **x)
{
    double        **t,
                  **u;

    t = (double **) x[0];
    u = (double **) res[0];
    u[0][0] = tanh(t[0][0]);
    u[0][1] = tanh(t[0][1]);
    u[1][0] = tanh(t[1][0]);
    u[1][1] = tanh(t[1][1]);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_exp(void **res, void **x)
{
    double        **t,
                  **u;

    t = (double **) x[0];
    u = (double **) res[0];
    u[0][0] = exp(t[0][0]);
    u[0][1] = exp(t[0][1]);
    u[1][0] = exp(t[1][0]);
    u[1][1] = exp(t[1][1]);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_log(void **res, void **x)
{
    double        **t,
                  **u;

    t = (double **) x[0];
    u = (double **) res[0];
    u[0][0] = log(t[0][0]);
    u[0][1] = log(t[0][1]);
    u[1][0] = log(t[1][0]);
    u[1][1] = log(t[1][1]);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_log10(void **res, void **x)
{
    double        **t,
                  **u;

    t = (double **) x[0];
    u = (double **) res[0];
    u[0][0] = log10(t[0][0]);
    u[0][1] = log10(t[0][1]);
    u[1][0] = log10(t[1][0]);
    u[1][1] = log10(t[1][1]);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_sqrt(void **res, void **x)
{
    double        **t,
                  **u;

    t = (double **) x[0];
    u = (double **) res[0];
    u[0][0] = sqrt(t[0][0]);
    u[0][1] = sqrt(t[0][1]);
    u[1][0] = sqrt(t[1][0]);
    u[1][1] = sqrt(t[1][1]);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_floor(void **res, void **x)
{
    double        **t,
                  **u;

    t = (double **) x[0];
    u = (double **) res[0];
    u[0][0] = floor(t[0][0]);
    u[0][1] = floor(t[0][1]);
    u[1][0] = floor(t[1][0]);
    u[1][1] = floor(t[1][1]);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_ceil(void **res, void **x)
{
    double        **t,
                  **u;

    t = (double **) x[0];
    u = (double **) res[0];
    u[0][0] = ceil(t[0][0]);
    u[0][1] = ceil(t[0][1]);
    u[1][0] = ceil(t[1][0]);
    u[1][1] = ceil(t[1][1]);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_fabs(void **res, void **x)
{
    double        **t,
                  **u;

    t = (double **) x[0];
    u = (double **) res[0];
    u[0][0] = fabs(t[0][0]);
    u[0][1] = fabs(t[0][1]);
    u[1][0] = fabs(t[1][0]);
    u[1][1] = fabs(t[1][1]);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
void
_matrix_hypot(void **res, void **x)
{
    double        **t,
                  **u,
                  **v;

    t = (double **) x[0];
    u = (double **) res[0];
    v = (double **) x[1];
    u[0][0] = hypot(t[0][0], v[0][0]);
    u[0][1] = hypot(t[0][1], v[0][1]);
    u[1][0] = hypot(t[1][0], v[1][0]);
    u[1][1] = hypot(t[1][1], v[1][1]);
}
/*------------------------------------------------------------------*/




FUNCTIONGen         Funcs_matrix[] =
{
   /* name, function to call */
   { "sin",     1,    _matrix_sin },            /* 0    */
   { "cos",     1,    _matrix_cos },            /* 1    */
   { "tan",     1,    _matrix_tan },            /* 2    */
   { "asin",    1,    _matrix_asin },           /* 3    */
   { "acos",    1,    _matrix_acos },           /* 4    */
   { "atan",    1,    _matrix_atan },           /* 5    */
   { "sinh",    1,    _matrix_sinh },           /* 6    */
   { "cosh",    1,    _matrix_cosh },           /* 7    */
   { "tanh",    1,    _matrix_tanh },           /* 8    */
   { "exp",     1,    _matrix_exp },            /* 9    */
   { "log",     1,    _matrix_log },            /* 10   */
   { "log10",   1,    _matrix_log10 },          /* 11   */
   { "sqrt",    1,    _matrix_sqrt },           /* 12   */
   { "floor",   1,    _matrix_floor },          /* 13   */
   { "ceil",    1,    _matrix_ceil },           /* 14   */
   { "abs",     1,    _matrix_fabs },           /* 15   */
   { "hypot",   2,    _matrix_hypot },          /* 16   */
   { 0 }                                       /* 17   */
};

int                 _matrix_NBFONC0=17;





/*---------------------------------------------------------------------
    Function associated to the command 'set_mat' used to
    set the value of a 2X2 matrix
---------------------------------------------------------------------*/
int
set_mat(int argc, char *argv[])
{
    double        **a;
    flow_data      *flow_interp;

    INIT_FLOW(flow_interp);
    a = (double **) GetValue_Gen_0(_matrix_ev, argv[1], flow_interp);
    if (a != NULL) {
        a[0][0] = convert_float(argv[2], flow_interp);
        a[0][1] = convert_float(argv[3], flow_interp);
        a[1][0] = convert_float(argv[4], flow_interp);
        a[1][1] = convert_float(argv[5], flow_interp);
    }
    return 1;
}
/*-------------------------------------------------------------------*/
