/*************************************************************************
** interpcom-3.1   (command interpreter)                                 **
** complex.c : Complex numbers                                           **
**                                                                       **
** Copyright (C) 2003  Jean-Marc Drezet                                  **
**                                                                       **
**  This library is free software; you can redistribute it and/or        **
**  modify it under the terms of the GNU Library General Public          **
**  License as published by the Free Software Foundation; either         **
**  version 2 of the License, or (at your option) any later version.     **
**                                                                       **
**  This library is distributed in the hope that it will be useful,      **
**  but WITHOUT ANY WARRANTY; without even the implied warranty of       **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    **
**  Library General Public License for more details.                     **
**                                                                       **
**  You should have received a copy of the GNU Library General Public    **
**  License along with this library; if not, write to the Free           **
**  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.   **
**                                                                       **
** Please mail any bug reports/fixes/enhancements to me at:              **
**      drezet@math.jussieu.fr                                           **
** or                                                                    **
**      Jean-Marc Drezet                                                 **
**      Institut de Mathematiques                                        **
**      UMR 7586 du CNRS                                                 **
**      173, rue du Chevaleret                                           **
**      75013 Paris                                                      **
**      France                                                           **
**                                                                       **
 *************************************************************************/

#include "interp.h"



/*--------------------------------------------------------------------
    Addition of two complex numbers in simple precision
--------------------------------------------------------------------*/
fcomplex
Cadd(fcomplex a, fcomplex b)
{
    fcomplex        c;

    c.r = a.r + b.r;
    c.i = a.i + b.i;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Substraction of two complex numbers in simple precision
--------------------------------------------------------------------*/
fcomplex
Csub(fcomplex a, fcomplex b)
{
    fcomplex        c;

    c.r = a.r - b.r;
    c.i = a.i - b.i;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Returns the opposit of a complex number in simple precision
--------------------------------------------------------------------*/
fcomplex
Cnegat(fcomplex a)
{
    fcomplex        c;

    c.r = -a.r;
    c.i = -a.i;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Multiplication of two complex numbers in simple precision
--------------------------------------------------------------------*/
fcomplex
Cmul(fcomplex a, fcomplex b)
{
    fcomplex        c;

    c.r = a.r * b.r - a.i * b.i;
    c.i = a.r * b.i + a.i * b.r;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Definition of a complex number using its real and imaginary parts
--------------------------------------------------------------------*/
fcomplex
Complex(float re, float im)
{
    fcomplex        c;

    c.r = re;
    c.i = im;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Definition of a complex number in polar coordinates using its
    modulus and phase in simple precision
--------------------------------------------------------------------*/
fpolaire
Polaire(float r, float t)
{
    fpolaire        c;

    c.rm = r;
    c.th = t;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Returns the conjugate of a complex number in simple precision
--------------------------------------------------------------------*/
fcomplex
Conjg(fcomplex z)
{
    fcomplex        c;

    c.r = z.r;
    c.i = -z.i;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Division of two complex numbers in simple precision
--------------------------------------------------------------------*/
fcomplex
Cdiv(fcomplex a, fcomplex b)
{
    fcomplex        c;
    float           r,
                    x;

    if (fabs(b.r) >= fabs(b.i)) {
        r = b.i / b.r;
        x = b.r + r * b.i;
        c.r = (a.r + r * a.i) / x;
        c.i = (a.i - r * a.r) / x;
    } else {
        r = b.r / b.i;
        x = b.i + r * b.r;
        c.r = (a.r * r + a.i) / x;
        c.i = (a.i * r - a.r) / x;
    }
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Modulus of a complex number in simple precision
--------------------------------------------------------------------*/
float
Cabs(fcomplex z)
{
    float           x,
                    y,
                    module,
                    t;

    x = (float) fabs(z.r);
    y = (float) fabs(z.i);
    if (x == 0.0)
        module = y;
    else if (y == 0.0)
        module = x;
    else if (x > y) {
        t = y / x;
        module = x * (float) sqrt(1.0 + t * t);
    } else {
        t = x / y;
        module = y * (float) sqrt(1.0 + t * t);
    }
    return module;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Square root of a complex number in simple precision
--------------------------------------------------------------------*/
fcomplex
Csqrt(fcomplex z)
{
    fcomplex        c;
    float           x,
                    y,
                    w,
                    r;

    if ((z.r == 0.0) && (z.i == 0.0)) {
        c.r = 0.0;
        c.i = 0.0;
        return c;
    } else {
        x = (float) fabs(z.r);
        y = (float) fabs(z.i);
        if (x >= y) {
            r = y / x;
            w = (float) sqrt(x) * (float) sqrt(0.5 * (1.0 +
            (float) sqrt(1.0 + r * r)));
        } else {
            r = x / y;
            w = (float) sqrt(y) * (float) sqrt(0.5 * (r + sqrt(1.0 + r * r)));
        }
        if (z.r >= 0.0) {
            c.r = w;
            c.i = z.i / (2.0 * w);
        } else {
            c.i = (z.i >= 0.0) ? w : -w;
            c.r = z.i / (2.0 * c.i);
        }
        return c;
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Multiplication of a real number and a complex number in simple
    precision
--------------------------------------------------------------------*/
fcomplex
RCmul(float x, fcomplex a)
{
    fcomplex        c;

    c.r = x * a.r;
    c.i = x * a.i;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Division of a  complex number by a real number in simple
    precision
--------------------------------------------------------------------*/
fcomplex
RCdiv(fcomplex a, float x)
{
    fcomplex        c;

    c.r = a.r / x;
    c.i = a.i / x;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Returns the inverse of a complex number in simple precision
--------------------------------------------------------------------*/
fcomplex
Cinv(fcomplex b)
{
    fcomplex        c;
    float           r,
                    x;

    if ((float) fabs(b.r) >= (float) fabs(b.i)) {
        r = b.i / b.r;
        x = b.r + r * b.i;
        c.r = 1.0 / x;
        c.i = -r / x;
    } else {
        r = b.r / b.i;
        x = b.i + r * b.r;
        c.r = r / x;
        c.i = -1.0 / x;
    }
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Addition of two complex numbers in double precision
--------------------------------------------------------------------*/
dcomplex
dCadd(dcomplex a, dcomplex b)
{
    dcomplex        c;

    c.r = a.r + b.r;
    c.i = a.i + b.i;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Substraction of two complex numbers in double precision
--------------------------------------------------------------------*/
dcomplex
dCsub(dcomplex a, dcomplex b)
{
    dcomplex        c;

    c.r = a.r - b.r;
    c.i = a.i - b.i;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Returns the opposit of a complex number in double precision
--------------------------------------------------------------------*/
dcomplex
dCnegat(dcomplex a)
{
    dcomplex        c;

    c.r = -a.r;
    c.i = -a.i;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Multiplication of two complex numbers in double precision
--------------------------------------------------------------------*/
dcomplex
dCmul(dcomplex a, dcomplex b)
{
    dcomplex        c;

    c.r = a.r * b.r - a.i * b.i;
    c.i = a.r * b.i + a.i * b.r;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Definition of a complex number in double precision using its
    real and imaginary parts
--------------------------------------------------------------------*/
dcomplex
dComplex(double re, double im)
{
    dcomplex        c;

    c.r = re;
    c.i = im;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Definition of a complex number in polar coordinates using its
    modulus and phase in double precision
--------------------------------------------------------------------*/
dpolaire
dPolaire(double r, double t)
{
    dpolaire        c;

    c.rm = r;
    c.th = t;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Returns the conjugate of a complex number in double precision
--------------------------------------------------------------------*/
dcomplex
dConjg(dcomplex z)
{
    dcomplex        c;
    c.r = z.r;
    c.i = -z.i;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Division of two complex numbers in double precision
--------------------------------------------------------------------*/
dcomplex
dCdiv(dcomplex a, dcomplex b)
{
    dcomplex        c;
    double          r,
                    x;

    if (fabs(b.r) >= fabs(b.i)) {
        r = b.i / b.r;
        x = b.r + r * b.i;
        c.r = (a.r + r * a.i) / x;
        c.i = (a.i - r * a.r) / x;
    } else {
        r = b.r / b.i;
        x = b.i + r * b.r;
        c.r = (a.r * r + a.i) / x;
        c.i = (a.i * r - a.r) / x;
    }
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Modulus of a complex number in double precision
--------------------------------------------------------------------*/
double
dCabs(dcomplex z)
{
    double          x,
                    y,
                    module,
                    t;

    x = fabs(z.r);
    y = fabs(z.i);
    if (x == 0.0)
        module = y;
    else if (y == 0.0)
        module = x;
    else if (x > y) {
        t = y / x;
        module = x * sqrt(1.0 + t * t);
    } else {
        t = x / y;
        module = y * sqrt(1.0 + t * t);
    }
    return module;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Square root of a complex number in double precision
--------------------------------------------------------------------*/
dcomplex
xdCsqrt(dcomplex z)
{
    dcomplex        c;
    double          x,
                    y,
                    w,
                    r;

    if ((z.r == 0.0) && (z.i == 0.0)) {
        c.r = 0.0;
        c.i = 0.0;
        return c;
    } else {
        x = fabs(z.r);
        y = fabs(z.i);
        if (x >= y) {
            r = y / x;
            w = sqrt(x) * sqrt(0.5 * (1.0 + sqrt(1.0 + r * r)));
        } else {
            r = x / y;
            w = sqrt(y) * sqrt(0.5 * (r + sqrt(1.0 + r * r)));
        }
        if (z.r >= 0.0) {
            c.r = w;
            c.i = z.i / (2.0 * w);
        } else {
            c.i = (z.i >= 0.0) ? w : -w;
            c.r = z.i / (2.0 * c.i);
        }
        return c;
    }
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Multiplication of a real number and a complex number in double
    precision
--------------------------------------------------------------------*/
dcomplex
dRCmul(double x, dcomplex a)
{
    dcomplex        c;

    c.r = x * a.r;
    c.i = x * a.i;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Returns the inverse of a complex number in double precision
--------------------------------------------------------------------*/
dcomplex
xdCinv(dcomplex b)
{
    dcomplex        c;
    double          r,
                    x;

    if (fabs(b.r) >= fabs(b.i)) {
        r = b.i / b.r;
        x = b.r + r * b.i;
        c.r = 1.0 / x;
        c.i = -r / x;
    } else {
        r = b.r / b.i;
        x = b.i + r * b.r;
        c.r = r / x;
        c.i = -1.0 / x;
    }
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Division of a complex number by a real number in double precision
--------------------------------------------------------------------*/
dcomplex
dRCdiv(dcomplex a, double x)
{
    dcomplex        c;

    c.r = a.r / x;
    c.i = a.i / x;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Conversion of a complex number in polar coordinates into a complex
    number in usual coordinates, in simple precision
--------------------------------------------------------------------*/
fcomplex
Conv_ptc(fpolaire a)
{
    fcomplex        z;

    z.r = a.rm * (float) cos(a.th);
    z.i = a.rm * (float) sin(a.th);
    return z;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Conversion of a complex number in usual coordinates into a complex
    number in polar coordinates, in simple precision
--------------------------------------------------------------------*/
fpolaire
Conv_ctp(fcomplex z)
{
    fpolaire        a;
    float           x,
                    y,
                    t;

    x = (float) fabs(z.r);
    y = (float) fabs(z.i);
    if (x == 0.0 && y == 0.0) {
        a.rm = 0.;
        a.th = 0.;
        return a;
    }
    if (x < y) {
        t = x / y;
        a.rm = y * (float) sqrt(1.0 + t * t);
    } else {
        t = y / x;
        a.rm = x * (float) sqrt(1.0 + t * t);
    }

    x = z.r / a.rm;
    y = z.i / a.rm;
    if (x == 1.0) {
        a.th = 0.0;
        return a;
    }
    if (x == -1.0) {
        a.th = 3.141593;
        return a;
    }
    a.th = (float) acos(x);
    if (y < 0.0)
        a.th = 6.2831853 - a.th;
    return a;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Conversion of a complex number in polar coordinates into a complex
    number in usual coordinates, in double precision
--------------------------------------------------------------------*/
dcomplex
dConv_ptc(dpolaire a)
{
    dcomplex        z;

    z.r = a.rm * cos(a.th);
    z.i = a.rm * sin(a.th);
    return z;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Conversion of a complex number in usual coordinates into a complex
    number in polar coordinates, in double precision
--------------------------------------------------------------------*/
dpolaire
dConv_ctp(dcomplex z)
{
    dpolaire        a;
    double          x,
                    y,
                    t;

    x = fabs(z.r);
    y = fabs(z.i);
    if (x == 0.0 && y == 0.0) {
        a.rm = 0.;
        a.th = 0.;
        return a;
    }
    if (fabs(x) < fabs(y)) {
        t = x / y;
        a.rm = y * sqrt(1.0 + t * t);
    } else {
        t = y / x;
        a.rm = x * sqrt(1.0 + t * t);
    }

    x = z.r / a.rm;
    y = z.i / a.rm;
    if (x == 1.0) {
        a.th = 0.0;
        return a;
    }
    if (x == -1.0) {
        a.th = 3.141592653589793;
        return a;
    }
    a.th = acos(x);
    if (y < 0.0)
        a.th = 6.2831853071795864 - a.th;
    return a;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Logarithm of a complex number in polar coordinates in simple
    precision. The result is in usual coordinates
--------------------------------------------------------------------*/
fcomplex
Clog_p(fpolaire a)
{
    fcomplex        z;

    z.r = (float) log(a.rm);
    z.i = a.th;
    return z;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Logarithm of a complex number in polar coordinates in simple
    precision. The result is also in polar coordinates
--------------------------------------------------------------------*/
fpolaire
Plog_p(fpolaire a)
{
    return Conv_ctp(Clog_p(a));
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Logarithm of a complex number in usual coordinates in simple
    precision. The result is also in usual coordinates
--------------------------------------------------------------------*/
fcomplex
Clog_c(fcomplex z)
{
    return Clog_p(Conv_ctp(z));
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Sinus of a complex number in simple precision
--------------------------------------------------------------------*/
fcomplex
Csin_c(fcomplex z)
{
    fcomplex        u;

    u.r = (float) sin(z.r) * (float) cosh(z.i);
    u.i = (float) cos(z.r) * (float) sinh(z.i);
    return u;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Cosinus of a complex number in simple precision
--------------------------------------------------------------------*/
fcomplex
Ccos_c(fcomplex z)
{
    fcomplex        u;

    u.r = (float) cos(z.r) * (float) cosh(z.i);
    u.i = - (float) sin(z.r) * (float) sinh(z.i);
    return u;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Exponential of a complex number in usual coordinates in simple
    precision. The result is in polar coordinates
--------------------------------------------------------------------*/
fpolaire
Pexp_c(fcomplex z)
{
    fpolaire        u;
    float           x;

    u.rm = (float) exp(z.r);
    x = (float) floor(z.i / 6.283185);
    u.th = z.i - 6.283185 * x;
    return u;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Exponential of a complex number in usual coordinates in simple
    precision. The result is also in usual coordinates
--------------------------------------------------------------------*/
fcomplex
Cexp_c(fcomplex z)
{
    return Conv_ptc(Pexp_c(z));
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Exponential of a complex number in polar coordinates in simple
    precision. The result is also in polar coordinates
--------------------------------------------------------------------*/
fpolaire
Pexp_p(fpolaire a)
{
    return Pexp_c(Conv_ptc(a));
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Logarithm of a complex number in polar coordinates in double
    precision. The result is in usual coordinates
--------------------------------------------------------------------*/
dcomplex
dClog_p(dpolaire a)
{
    dcomplex        z;

    z.r = log(a.rm);
    z.i = a.th;
    return z;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Logarithm of a complex number in polar coordinates in double
    precision. The result is also in polar coordinates
--------------------------------------------------------------------*/
dpolaire
dPlog_p(dpolaire a)
{
    return dConv_ctp(dClog_p(a));
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Logarithm of a complex number in usual coordinates in double
    precision. The result is also in usual coordinates
--------------------------------------------------------------------*/
dcomplex
dClog_c(dcomplex z)
{
    return dClog_p(dConv_ctp(z));
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Sinus of a complex number in double precision
--------------------------------------------------------------------*/
dcomplex
dCsin_c(dcomplex z)
{
    dcomplex        u;

    u.r = sin(z.r) * cosh(z.i);
    u.i = cos(z.r) * sinh(z.i);
    return u;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Cosinus of a complex number in double precision
--------------------------------------------------------------------*/
dcomplex
dCcos_c(dcomplex z)
{
    dcomplex        u;

    u.r = cos(z.r) * cosh(z.i);
    u.i = -sin(z.r) * sinh(z.i);
    return u;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Exponential of a complex number in usual coordinates in double
    precision. The result is in polar coordinates
--------------------------------------------------------------------*/
dpolaire
dPexp_c(dcomplex z)
{
    dpolaire        u;
    double          x;

    u.rm = exp(z.r);
    x = floor(z.i / 6.283185307179586);
    u.th = z.i - 6.283185307179586 * x;
    return u;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Exponential of a complex number in usual coordinates in double
    precision. The result is also in usual coordinates
--------------------------------------------------------------------*/
dcomplex
dCexp_c(dcomplex z)
{
    return dConv_ptc(dPexp_c(z));
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Exponential of a complex number in polar coordinates in double
    precision. The result is also in polar coordinates
--------------------------------------------------------------------*/
dpolaire
dPexp_p(dpolaire a)
{
    return dPexp_c(dConv_ptc(a));
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Addition of two complex numbers in polar coordinates in simple
    presision
--------------------------------------------------------------------*/
fpolaire
Padd(fpolaire a, fpolaire b)
{
    return Conv_ctp(Cadd(Conv_ptc(a), Conv_ptc(b)));
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Substraction of two complex numbers in polar coordinates in simple
    presision
--------------------------------------------------------------------*/
fpolaire
Psub(fpolaire a, fpolaire b)
{
    return Conv_ctp(Csub(Conv_ptc(a), Conv_ptc(b)));
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Multiplication of two complex numbers in polar coordinates in simple
    presision
--------------------------------------------------------------------*/
fpolaire
Pmul(fpolaire a, fpolaire b)
{
    fpolaire        c;

    c.rm = a.rm * b.rm;
    c.th = a.th + b.th;
    if (c.th > 6.283185)
        c.th -= 6.283185;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Division of two complex numbers in polar coordinates in simple
    presision
--------------------------------------------------------------------*/
fpolaire
Pdiv(fpolaire a, fpolaire b)
{
    fpolaire        c;

    c.rm = a.rm / b.rm;
    c.th = a.th - b.th;
    if (c.th < 0.)
        c.th += 6.283185;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Conjugate of a complex number in polar coordinates in simple
    precision
--------------------------------------------------------------------*/
fpolaire
Pconjg(fpolaire a)
{
    fpolaire        c;

    c.rm = a.rm;
    c.th = -a.th;
    if (c.th < 0.)
        c.th += 6.283185;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Square root of a complex number in polar coordinates in simple
    precision
--------------------------------------------------------------------*/
fpolaire
Psqrt(fpolaire a)
{
    fpolaire        c;

    if (a.rm <= 0.0)
        c.rm = 0.0;
    else
        c.rm = (float) sqrt(a.rm);
    c.th = a.th / 2.;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Real power of a complex number in polar coordinates in simple
    precision
--------------------------------------------------------------------*/
fpolaire
Ppuis(fpolaire a, float x)
{
    fpolaire        c;

    if (a.rm <= 0.0)
        c.rm = 0.0;
    else
        c.rm = (float) pow(a.rm, x);
    c.th = a.th * x;
    c.th = c.th - (float) floor(c.th / 6.283185) * 6.283185;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Real power of a complex number in usual coordinates in simple
    precision
--------------------------------------------------------------------*/
fcomplex
Cpuis(fcomplex a, float x)
{
    return Conv_ptc(Ppuis(Conv_ctp(a), x));
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Multiplication of a real and a complex number in polar coordinates
    in simple precision
--------------------------------------------------------------------*/
fpolaire
RPmul(float x, fpolaire a)
{
    fpolaire        c;

    c.rm = (float) fabs(x * a.rm);
    if (x >= 0)
        c.th = a.th;
    else {
        c.th = a.th + 3.141593;
        if (c.th > 6.283185)
            c.th -= 6.283185;
    }
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Division of a complex number in polar coordinates by a real number
    in simple precision
--------------------------------------------------------------------*/
fpolaire
RPdiv(fpolaire a, float x)
{
    fpolaire        c;

    c.rm = (float) fabs(a.rm / x);
    if (x >= 0)
        c.th = a.th;
    else {
        c.th = a.th + 3.141593;
        if (c.th > 6.283185)
            c.th -= 6.283185;
    }
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Inversion of a complex number in polar coordinates in simple
    precision
--------------------------------------------------------------------*/
fpolaire
Pinv(fpolaire a)
{
    fpolaire        c;

    c.rm = 1. / a.rm;
    c.th = - a.th;
    if (c.th < 0.0)
        c.th += 6.283185;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Addition of two complex numbers in polar coordinates in double
    precision
--------------------------------------------------------------------*/
dpolaire
dPadd(dpolaire a, dpolaire b)
{
    return dConv_ctp(dCadd(dConv_ptc(a), dConv_ptc(b)));
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Substraction of two complex numbers in polar coordinates in double
    precision
--------------------------------------------------------------------*/
dpolaire
dPsub(dpolaire a, dpolaire b)
{
    return dConv_ctp(dCsub(dConv_ptc(a), dConv_ptc(b)));
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Multiplication of two complex numbers in polar coordinates in double
    precision
--------------------------------------------------------------------*/
dpolaire
dPmul(dpolaire a, dpolaire b)
{
    dpolaire        c;

    c.rm = a.rm * b.rm;
    c.th = a.th + b.th;
    if (c.th > 6.283185307179586)
        c.th -= 6.283185307179586;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Division of two complex numbers in polar coordinates in double
    precision
--------------------------------------------------------------------*/
dpolaire
dPdiv(dpolaire a, dpolaire b)
{
    dpolaire        c;

    c.rm = a.rm / b.rm;
    c.th = a.th - b.th;
    if (c.th < 0.)
        c.th += 6.283185307179586;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Conjugate of a complex number in polar coordinates in double
    precision
--------------------------------------------------------------------*/
dpolaire
dPconjg(dpolaire a)
{
    dpolaire        c;

    c.rm = a.rm;
    c.th = -a.th;
    if (c.th < 0.)
        c.th += 6.283185307179586;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Square root of a complex number in polar coordinates in double
    precision
--------------------------------------------------------------------*/
dpolaire
dPsqrt(dpolaire a)
{
    dpolaire        c;

    if (a.rm <= 0.0)
        c.rm = 0.0;
    else
        c.rm = sqrt(a.rm);
    c.th = a.th / 2.;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Real power of a complex number in polar coordinates in double
    precision
--------------------------------------------------------------------*/
dpolaire
dPpuis(dpolaire a, double x)
{
    dpolaire        c;

    if (a.rm <= 0.0)
        c.rm = 0.0;
    else
        c.rm = pow(a.rm, x);
    c.th = a.th * x;
    c.th = c.th - floor(c.th / 6.283185307179586) * 6.283185307179586;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Real power of a complex number in usual coordinates in double
    precision
--------------------------------------------------------------------*/
dcomplex
dCpuis(dcomplex a, double x)
{
    return dConv_ptc(dPpuis(dConv_ctp(a), x));
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Real power of a complex number in usual coordinates in double
    precision
--------------------------------------------------------------------*/
dcomplex
dCpow(dcomplex a, dcomplex b)
{
    dcomplex        z,
                    zz;

    z = dConv_ptc(dPpuis(dConv_ctp(a), b.r));
    zz = dComplex(cos(b.i), sin(b.i));
    return dCmul(zz, z);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Multiplication dof a real number and a complex number in polar
    coordinates in double precision
--------------------------------------------------------------------*/
dpolaire
dRPmul(double x, dpolaire a)
{
    dpolaire        c;

    c.rm = fabs(x * a.rm);
    if (x >= 0)
        c.th = a.th;
    else {
        c.th = a.th + 3.141592653589793;
        if (c.th > 6.283185307179586)
            c.th -= 6.283185307179586;
    }
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Division of a complex number in polar coordinates by a real number
    in double precision
--------------------------------------------------------------------*/
dpolaire
dRPdiv(dpolaire a, double x)
{
    dpolaire        c;

    c.rm = fabs(a.rm / x);
    if (x >= 0)
        c.th = a.th;
    else {
        c.th = a.th + 3.141592653589793;
        if (c.th > 6.283185307179586)
            c.th -= 6.283185307179586;
    }
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Inversion of a complex number in polar coordinates in double
    precision
--------------------------------------------------------------------*/
dpolaire
dPinv(dpolaire a)
{
    dpolaire        c;

    c.rm = 1. / a.rm;
    c.th = - a.th;
    if (c.th < 0.0)
        c.th += 6.283185307179586;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Tangent of a complex number in double precision
--------------------------------------------------------------------*/
dcomplex
dCtan_c(dcomplex z)
{
    dcomplex        u,
                    v,
                    w;

    u = dCsin_c(z);
    v = dCcos_c(z);
    w = dComplex(0., 0.);
    if (v.r == 0. && v.i == 0.)
        return w;
    w = dCdiv(u, v);
    return w;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Hyperbolic cosinus of a complex number in double precision
--------------------------------------------------------------------*/
dcomplex
dCcosh_c(dcomplex z)
{
    dcomplex        u;

    u.r = -z.r;
    u.i = -z.i;
    return dRCdiv(dCadd(dCexp_c(z), dCexp_c(u)), 2.);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Hyperbolic sinus of a complex number in double precision
--------------------------------------------------------------------*/
dcomplex
dCsinh_c(dcomplex z)
{
    dcomplex        u,
                    v;

    u.r = -z.r;
    u.i = -z.i;
    v = dRCdiv(dCsub(dCexp_c(z), dCexp_c(u)), 2.);
    u.r = v.i;
    u.i = -v.r;
    return u;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Hyperbolic tangent of a complex number in double precision
--------------------------------------------------------------------*/
dcomplex
dCtanh_c(dcomplex z)
{
    dcomplex        u,
                    v,
                    w;

    u = dRCmul(2., z);
    v = dCexp_c(u);
    w = dComplex(0., 0.);
    if (v.r == -1. && v.i == 0.)
        return w;
    u.r = v.r - 1.;
    u.i = v.i;
    w.r = -v.i;
    w.i = v.r + 1;
    return dCdiv(u, w);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Tangent of a complex number in simple precision
--------------------------------------------------------------------*/
fcomplex
Ctan_c(fcomplex z)
{
    fcomplex        u,
                    v,
                    w;

    u = Csin_c(z);
    v = Ccos_c(z);
    w = Complex(0., 0.);
    if (v.r == 0. && v.i == 0.)
        return w;
    w = Cdiv(u, v);
    return w;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Hyperbolic cosinus of a complex number in simple precision
--------------------------------------------------------------------*/
fcomplex
Ccosh_c(fcomplex z)
{
    fcomplex        u;

    u.r = -z.r;
    u.i = -z.i;
    return RCdiv(Cadd(Cexp_c(z), Cexp_c(u)), 2.);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Hyperbolic sinus of a complex number in simple precision
--------------------------------------------------------------------*/
fcomplex
Csinh_c(fcomplex z)
{
    fcomplex        u,
                    v;

    u.r = -z.r;
    u.i = -z.i;
    v = RCdiv(Csub(Cexp_c(z), Cexp_c(u)), 2.);
    u.r = v.i;
    u.i = -v.r;
    return u;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Hyperbolic tangent of a complex number in simple precision
--------------------------------------------------------------------*/
fcomplex
Ctanh_c(fcomplex z)
{
    fcomplex        u,
                    v,
                    w;

    u = RCmul(2., z);
    v = Cexp_c(u);
    w = Complex(0., 0.);
    if (v.r == -1. && v.i == 0.)
        return w;
    u.r = v.r - 1.;
    u.i = v.i;
    w.r = -v.i;
    w.i = v.r + 1;
    return Cdiv(u, w);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
--------------------------------------------------------------------*/
dcomplex
CdCabs(dcomplex z)
{
    return dComplex(dCabs(z), 0.);
}
/*------------------------------------------------------------------*/
