/***************************************************************************/
/***************************************************************************/
/*                                                                         */
/*   (c) 1995.  The Regents of the University of California.  All rights   */
/*   reserved.                                                             */
/*                                                                         */
/*   This work was produced at the University of California, Lawrence      */
/*   Livermore National Laboratory (UC LLNL) under contract no.            */
/*   W-7405-ENG-48 (Contract 48) between the U.S. Department of Energy     */
/*   (DOE) and The Regents of the University of California (University)    */
/*   for the operation of UC LLNL.  Copyright is reserved to the           */
/*   University for purposes of controlled dissemination,                  */
/*   commercialization through formal licensing, or other disposition      */
/*   under terms of Contract 48; DOE policies, regulations and orders;     */
/*   and U.S. statutes.  The rights of the Federal Government are          */
/*   reserved under Contract 48 subject to the restrictions agreed upon    */
/*   by the DOE and University.                                            */
/*                                                                         */
/*                                                                         */
/*                              DISCLAIMER                                 */
/*                                                                         */
/*   This software was prepared as an account of work sponsored by an      */
/*   agency of the United States Government.  Neither the United States    */
/*   Government nor the University of California nor any of their          */
/*   employees, makes any warranty, express or implied, or assumes any     */
/*   liability or responsibility for the accuracy, completeness, or        */
/*   usefulness of any information, apparatus, product, or process         */
/*   disclosed, or represents that its specific commercial products,       */
/*   process, or service by trade name, trademark, manufacturer, or        */
/*   otherwise, does not necessarily constitute or imply its               */
/*   endorsement, recommendation, or favoring by the United States         */
/*   Government or the University of California. The views and opinions    */
/*   of the authors expressed herein do not necessarily state or reflect   */
/*   those of the United States Government or the University of            */
/*   California, and shall not be used for advertising or product          */
/*   endorsement purposes.                                                 */
/*                                                                         */
/*   Permission to use, copy, modify and distribute this software and its  */
/*   documentation for any non-commercial purpose, without fee, is         */
/*   hereby granted, provided that the above copyright notice and this     */
/*   permission notice appear in all copies of the software and            */
/*   supporting documentation, and that all UC LLNL identification in      */
/*   the user interface remain unchanged.  The title to copyright LLNL     */
/*   XDIR shall at all times remain with The Regents of the University     */
/*   of California and users agree to preserve same. Users seeking the     */
/*   right to make derivative works with LLNL XDIR for commercial          */
/*   purposes may obtain a license from the Lawrence Livermore National    */
/*   Laboratory's Technology Transfer Office, P.O. Box 808, L-795,         */
/*   Livermore, CA 94550.                                                  */
/*                                                                         */
/***************************************************************************/
/***************************************************************************/

#include <Xm/Xm.h>
#include <Xm/List.h>
#include <Xm/SelectioB.h>
#include <Xm/TextF.h>
#include <Xm/RowColumn.h>
#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/Frame.h>
#include "xdir.h"
#include "list.h"
#include "str.h"
#include "history.h"

static struct {
    struct dirwin_st *dirwin;
    struct entry_link *head;
	int error;
} qc;

static struct {
    Widget w_dialog;
    Widget w_form;
    Widget w_listLabel;
    Widget w_list;
    Widget w_textLabel;
    Widget w_text;
    Widget w_warningLabel;
    Widget w_warningFrame;
    Widget w_warning;
	Widget w_warningMsg1;
	Widget w_warningMsg2;
	Widget w_warningMsg3;
} quote;

static char *quote_help[] = {
	"This dialog enables you to send a \"quoted\" (that is,",
	"low_level) command directly to the FTP server of the associated",
	"remote host.  (This is similar the the capability offered by",
	"FTP's QUOTE command.)\n",
	"\n",
	"WARNING: Exercise extreme caution when using this command",
	"because LLNL XDIR depends on the FTP server being in a known state.  ",
	"In particular, be sure that you only issue commands that",
	"generate one response from the FTP server.  Do not issue",
	"commands that change the state of the FTP server (e.g.,",
	"connection/login sequences, file transfers, directory",
	"listings, etc.).\n",
	"\n",
    "Previously referenced commands are remembered and",
    "displayed in the scrollable list labeled \"Recently",
    "Referenced Commands\".  Double-click on a command in the",
    "list to issue it.  Single-click on a command",
    "in the list to place that command into the text",
    "field labeled \"Low-Level FTP Command\", where it can be edited.\n",
    "\n",
    "The user preference HISTORY SORT ORDER specifies whether previously",
    "used quoted commands are displayed in alphabetical order or",
    "with the most-recently-referenced commands at the top of",
    "the list.  The user preference MAX QUOTED CMD HISTORY",
    "specifies the maximum number of quoted commands to remember",
    "per host.  The quoted command information is preserved across LLNL XDIR",
    "sessions.\n",
    "\n",
	"To enable a command to act on a set of items selected in",
	"the associated host's directory list, use the",
	"placeholder \"%s\" to represent a",
	"selected item.  For example, type \"stage 1 %s\" to send",
	"a STAGE command to the remote host for each selected item.\n",
	"\n",
	"Click on the SEND button to send the command to the FTP",
	"server on the remote host.  A dialog will pop up to enable",
	"the user to monitor the progress of the sending of commands.  ",
	"The dialog contains a button for aborting the operation.\n",
	"\n",
	"Click on the CANCEL button to remove",
    "the dialog without performing the operation.",
    NULL
};

extern struct st_host_info hinfo[];
extern Widget w_dirList[];
extern XtAppContext app;
extern int beep_when_ops_done;
extern int inquire_on_quote;
extern int sort_caches;

void cb_quote_ok();
void cb_quote_single_selection();
void cb_quote_help();
void cb_quote_cancel();
void cb_map_dialog();
int cb_quote_send_commands();
char *entry_to_full_path();
struct sl_struct *create_array_list();
char *cstring_to_text();


/*
 * cb_send_quoted_command - Pop up dialog that prompts for low-level command
 *                          to be sent to remote FTP server.
 */
void
cb_send_quoted_command(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	struct dirwin_st *dirwin = (struct dirwin_st *)client_data;
	int retval;

	/* Start operation */
	if (!start_op(True))
		return;

	/* Sanity check */
	if (dirwin->host == LOCAL)
		fatal_error("Bug in cb_send_quoted_command()");

    /* Clear error flag */
    raise_okflag();

    /* Create quote dialog */
    create_quote_dialog(dirwin);

	/* cd into working directory */
	show_stop_button(dirwin);
	use_busy_cursor();
	retval = remote_cd(dirwin->host, dirwin->dirname, True);
	restore_prev_cursor();
	if (retval < 0) {
	    switch (retval) {
		case -1:
			record_and_alert(
				"Unable to issue command(s).  Trouble changing directory.",
				dirwin->w_shell);
			break;
		case -3:
	        restore_lost_connection(dirwin->host, dirwin);
			break;
		case -6:
			hide_abort_dialog();
			write_log("Command aborted\n");
		}
		hide_stop_button();
		end_op();
		return;
	}
	hide_stop_button();

    update_quote_dialog(dirwin);

    XtManageChild(quote.w_dialog);
	traverse_to_widget(quote.w_text);
}


/*
 * create_quote_dialog - Create dialog that prompts for a quoted command.
 *                       "dirwin" is the directory window to center the
 *                       Quoted Command dialog over.
 */
create_quote_dialog(dirwin)
struct dirwin_st *dirwin;
{
    int i;
    Arg args[2];
    Widget widget;

    /* Create prompt dialog to get quoted command */
    i = 0;
    XtSetArg(args[i], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL); i++;
	XtSetArg(args[i], XmNdefaultPosition, False); i++;
    quote.w_dialog = XmCreatePromptDialog(dirwin->w_shell, "quote", args, i);
    XtAddCallback(quote.w_dialog, XmNokCallback, cb_quote_ok,
		(XtPointer)dirwin);
	XtAddCallback(quote.w_dialog, XmNcancelCallback, cb_quote_cancel,
		(XtPointer)NULL);
	XtAddCallback(quote.w_dialog, XmNhelpCallback, cb_quote_help,
		(XtPointer)NULL);
	XtAddCallback(quote.w_dialog, XmNmapCallback, cb_map_dialog,
		(XtPointer)XtWindow(dirwin->w_shell));

    /* Don't show prompt dialog's selection label and text */
    widget = XmSelectionBoxGetChild(quote.w_dialog, XmDIALOG_SELECTION_LABEL);
    XtUnmanageChild(widget);
    widget = XmSelectionBoxGetChild(quote.w_dialog, XmDIALOG_TEXT);
    XtUnmanageChild(widget);

	/* Add callback for the WM_DELETE_WINDOW protocol */
	add_wm_delete_window_cb(quote.w_dialog, cb_quote_cancel, NULL, False);

	/* Create form for control area */
	quote.w_form = XtVaCreateWidget(
		"form",
		xmFormWidgetClass,
		quote.w_dialog,
		NULL
	);

	/* Create label for scrolled list of quoted expressions */
	quote.w_listLabel = XtVaCreateManagedWidget(
		"listLabel",
		xmLabelWidgetClass,
		quote.w_form,
		XmNalignment,		XmALIGNMENT_BEGINNING,
		XmNtopAttachment,	XmATTACH_FORM,
		XmNleftAttachment,	XmATTACH_FORM,
		XmNrightAttachment,	XmATTACH_FORM,
		NULL
	);

	/* Create scrolled list of quoted expressions */
	i = 0;
	XtSetArg(args[i], XmNselectionPolicy, XmSINGLE_SELECT); i++;
	XtSetArg(args[i], XmNlistSizePolicy, XmCONSTANT); i++;
	quote.w_list = XmCreateScrolledList(quote.w_form, "list", args, i);
	XtVaSetValues(
		XtParent(quote.w_list),
		XmNleftAttachment,	XmATTACH_FORM,
		XmNrightAttachment,	XmATTACH_FORM,
		XmNtopAttachment,	XmATTACH_WIDGET,
		XmNtopWidget,		quote.w_listLabel,
		NULL
	);
	XtManageChild(quote.w_list);

    /* Add single-selection callback */
    XtAddCallback(
        quote.w_list,
        XmNsingleSelectionCallback,
        cb_quote_single_selection,
        (XtPointer)NULL
    );

    /* Add double-click callback to directory list*/
    XtAddCallback(
        quote.w_list,
        XmNdefaultActionCallback,
        cb_quote_ok,
        (XtPointer)dirwin
    );

	/* Create label for quoted command */
	quote.w_textLabel = XtVaCreateManagedWidget(
		"textLabel",
		xmLabelWidgetClass,
		quote.w_form,
		XmNalignment,		XmALIGNMENT_BEGINNING,
		XmNtopAttachment,	XmATTACH_WIDGET,
		XmNtopWidget,		quote.w_list,
		XmNtopOffset,		10,
		XmNleftAttachment,	XmATTACH_FORM,
		XmNrightAttachment,	XmATTACH_FORM,
		NULL
	);

	/* Create textfield for quoted command */
	quote.w_text = XtVaCreateManagedWidget(
		"text",
		xmTextFieldWidgetClass,
		quote.w_form,
		XmNtopAttachment,	XmATTACH_WIDGET,
		XmNtopWidget,		quote.w_textLabel,
		XmNleftAttachment,	XmATTACH_FORM,
		XmNrightAttachment,	XmATTACH_FORM,
		NULL
	);

	/* Create label for warning message */
	quote.w_warningLabel = XtVaCreateManagedWidget(
		"warningLabel",
		xmLabelWidgetClass,
		quote.w_form,
		XmNalignment,		XmALIGNMENT_CENTER,
		XmNtopAttachment,	XmATTACH_WIDGET,
		XmNtopWidget,		quote.w_text,
		XmNtopOffset,		15,
		XmNleftAttachment,	XmATTACH_FORM,
		XmNrightAttachment,	XmATTACH_FORM,
		NULL
	);

	/* Create frame for warning message */
	quote.w_warningFrame = XtVaCreateManagedWidget(
		"warningFrame",
		xmFrameWidgetClass,
		quote.w_form,
		XmNtopAttachment,		XmATTACH_WIDGET,
		XmNtopWidget,			quote.w_warningLabel,
		XmNbottomAttachment,	XmATTACH_FORM,
		XmNbottomOffset,		10,
		XmNleftAttachment,		XmATTACH_FORM,
		XmNleftOffset,			10,
		XmNrightAttachment,		XmATTACH_FORM,
		XmNrightOffset,			10,
		NULL
	);

	/* Create rowcolumn to hold warning message */
    quote.w_warning = XtVaCreateManagedWidget(
        "warning",
        xmRowColumnWidgetClass,
        quote.w_warningFrame,
        NULL
    );

	/* Create label for first line of warning message */
	quote.w_warningMsg1 = XtVaCreateManagedWidget(
		"warningMsg1",
		xmLabelWidgetClass,
		quote.w_warning,
		NULL
	);

	/* Create label for second line of warning message */
	quote.w_warningMsg2 = XtVaCreateManagedWidget(
		"warningMsg2",
		xmLabelWidgetClass,
		quote.w_warning,
		NULL
	);

	/* Create label for third line of warning message */
	quote.w_warningMsg3 = XtVaCreateManagedWidget(
		"warningMsg3",
		xmLabelWidgetClass,
		quote.w_warning,
		NULL
	);

	XtManageChild(quote.w_form);
}


/*
 * cb_quote_ok - This callback actually sends the quoted command.
 */
void
cb_quote_ok(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	struct dirwin_st *dirwin = (struct dirwin_st *)client_data;
	char *raw_command_line;
	struct entry_link *ptr;
	struct entry_link *head;
    int nselected_items;
    int i;
	int j;
	int len;
	int process_selected_items = False;
	int found_percent;
	int found_percent_s;
	char ch;
	struct entry_link *head_selected_entries = NULL;
	struct sl_struct *list;
	char *full_path;
	char buf[2];
	char *command_line;

	/* Get command */
	raw_command_line = XmTextFieldGetString(quote.w_text);

	/* Add command to history */
	add_to_history(QUOTE, hinfo[dirwin->host].hostname, raw_command_line);

	/* Is %s present in command line? */
	found_percent = False;
	found_percent_s = False;
	len = strlen(raw_command_line);
	for (i=0; i<len; i++) {
		ch = raw_command_line[i];
		if (found_percent) {
			if (ch == 's')
				found_percent_s = True;
			found_percent = False;
		} else if (ch == '%')
			found_percent = True;
	}

	/* Should the selection (if any) be processed? */
	if (found_percent_s) {
		nselected_items = 0;
		for (i=0; i<dirwin->nentries; i++)
			if (dirwin->entries[i].state == SELECTED)
				nselected_items++;
		if (nselected_items == 0) {
			XtUnmanageChild(quote.w_dialog);
			XtFree(raw_command_line);
			warn("A \"%s\" placeholder is used, but no items are selected.",
				dirwin->w_shell);
			end_op();
			return;
		}
		XtUnmanageChild(quote.w_dialog);
		if (inquire_on_quote && !verify_selection(dirwin, dirwin,
				"Do you really want to operate on these items?")) {
			XtFree(raw_command_line);
			end_op();
			return;
		}
		process_selected_items = True;
		for (i=0; i<dirwin->nentries; i++)
			if (dirwin->entries[i].state == SELECTED) {
				full_path = entry_to_full_path(&dirwin->entries[i]);
				add_to_linked_list(&head_selected_entries, full_path);
				XtFree(full_path);
			}
		list = create_array_list(&head_selected_entries);
	}

	/* Set up linked list of quoted commands */
	if (process_selected_items) {   /* Replace pattern with selected entries */
	    head = NULL;
	    for (i=nselected_items-1; i>=0; i--) {
			found_percent = False;
			len = strlen(raw_command_line);
			command_line = XtNewString("");
			for (j=0; j<len; j++) {
				ch = raw_command_line[j];
				if (found_percent) {
					if (ch == 's')
						concat(&command_line, list->entries[i]);
					else {
						buf[0] = ch;
						buf[1] = '\0';
						concat(&command_line, buf);
					}
					found_percent = False;
				} else if (ch == '%')
					found_percent = True;
				else {
					buf[0] = ch;
					buf[1] = '\0';
					concat(&command_line, buf);
				}
			}
			if (found_percent)
				concat(&command_line, "%");

	        ptr = XtNew(struct entry_link);
			ptr->entry = command_line;
	        ptr->next = head;
	        head = ptr;
	    }
	} else {   /* Form single cmd */
		head = XtNew(struct entry_link);
		head->entry = XtNewString(raw_command_line);
		head->next = NULL;
	}
	XtFree(raw_command_line);

	/* Set up control block */
	qc.dirwin = dirwin;
	qc.head = head;
	qc.error = False;

	/* Make operation interruptable */
	show_stop_button(dirwin);

    /* This might take some time */
    use_busy_cursor();

	/* Preform quoted command */
	show_mp_monitor(dirwin, "Now Issuing:");
	XtAppAddWorkProc(app, (XtWorkProc)cb_quote_send_commands, NULL);

	/* Get rid of Quoted Command Dialog */
	XtDestroyWidget(XtParent(quote.w_dialog));
}


/*
 * cb_quote_cancel - Callback to handel Quote Command "Cancel" button.
 */
void
cb_quote_cancel(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	/* Get rid of Quoted Command Dialog */
	XtDestroyWidget(XtParent(quote.w_dialog));

	/* Command is done */
	end_op();
}


/*
 * update_quote_dialog - Display last issued "quoted" commands in scrolled
 *                       list in "Quoted Command" dialog.
 */
update_quote_dialog(dirwin)
struct dirwin_st *dirwin;
{
	struct sl_struct *list;
	XmString string;
	int i;

    /* Enter quoted expressions into scrolled list */
	retrieve_history(QUOTE, hinfo[dirwin->host].hostname, &list, sort_caches);
	reset_list(quote.w_list);
    for (i=0; i<list->nentries; i++) {
        string = XmStringCreateSimple(list->entries[i]);
		XmListAddItem(quote.w_list, string, 0);
		XmStringFree(string);
	}
    release_array_list(list);

	/* Clear text and selection */
	XmTextFieldSetString(quote.w_text, "");
}


/*
 * cb_quote_single_selection - This callback is invoked when a remembered
 *                             quoted command is selected.
 */
void
cb_quote_single_selection(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
    XmListCallbackStruct *cbs = (XmListCallbackStruct *)call_data;
    XmStringTable selected_items;
    int nselected_items;
    char *text;

	/* Don't let user deselect item */
    XtVaGetValues(
        quote.w_list,
        XmNselectedItemCount, &nselected_items,
        XmNselectedItems,     &selected_items,
        NULL
    );
    if (nselected_items == 0) {
        XmListSelectPos(quote.w_list, cbs->item_position, False);
        XtVaGetValues(
            quote.w_list,
            XmNselectedItems,     &selected_items,
            NULL
        );
    }

    /* Use selected item to text field */
	text = cstring_to_text(selected_items[0]);
    XmTextFieldSetString(quote.w_text, text);
    XtFree(text);
}


/*
 * cb_quote_send_commands - Work proc that sends commands specified in Quoted
 *                          Command dialog to remote FTP server.
 */
cb_quote_send_commands()
{
    struct entry_link *ptr;
    char msg[MAXPATHLEN+40];

    /* Did user push stop button? */
    if (stop())
        goto abort;

    /* Are we done? */
    if (qc.head == NULL) {
		if (qc.error) 
			record_and_alert("At least one command not issued.",
				qc.dirwin->w_shell);
        goto done;
	}

    /* Let user know where we are */
    update_mp_monitor(qc.head->entry);

    /* Issue quoted command */
	switch (quote_cmd(qc.dirwin->host, qc.head->entry)) {
	case -6:
		goto abort;
    case -3:
        goto lost;
    case -1:
        sprintf(msg, "Unable to issue command: '%s'", qc.head->entry);
        record_warning(msg);
		qc.error = True;
    }
    ptr = qc.head;
    qc.head = qc.head->next;
    XtFree(ptr->entry);
    XtFree((char *)ptr);
    return False;

abort:

	hide_mp_monitor();
	hide_abort_dialog();
	write_log("*** Quoted command(s) aborted.");
	goto final;

done:

	hide_mp_monitor();
	goto final;

lost:

	hide_mp_monitor();
    restore_lost_connection(qc.dirwin->host, qc.dirwin);

final:

	/* Miscellaneous cleanup */
	release_linked_list(&qc.head);
	clear_selected_entries();
	update_dir_controls();
	if (beep_when_ops_done)
		beep();
	restore_prev_cursor();

	/* Signal end of operation */
	hide_stop_button();
	end_op();
	return True;
}


/*
 * cb_quote_help - Callback that display help info for the Quoted Command
 *                 dialog.
 */
void
cb_quote_help(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
    help_dialog(widget, True, "Send Quote Command", quote_help);
}

