
/*

    File: tcpproxy/tcpproxy.c

    Copyright (C) 1999  Wolfgang Zekoll  <wzk@quietsche-entchen.de>
  
    This software is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
  
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
  
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */


 /*
  * Version 1.1.5 - 08MAR00wzk
  *
  *   - Changed the way tcpproxy works with the userid: tcpproxy now doesn't
  *     not longer change it's userid after it has started, if started by
  *     root it will run as root.  When it accepts an incoming connection
  *     is changes it's userid as soon as the port configuration is
  *	determined.  Default values for uid/gid are 65535/65534 = -1/-2.
  *
  *
  * Version 1.1.4 - 11NOV99wzk
  *
  *   - shutdown(sfd, 1) wenn close() auf der Client Seite erkannt wird.
  *
  *   -w Option
  *
  *
  * Version 1.1.2 - 11SEP99wzk
  *
  *   - Die Optionen -c und -z sind neu.
  *
  *
  * Version 1.1.0 - 01SEP99wzk
  *
  * Aenderungen gegenueber den Vorgaengerversionen:
  *
  *   - Unterstuetzung von Server-Programmen.
  *   - Der netcat-Modus wird nicht mehr unterstuetzt.
  *   - Erweiterung des Kommandozeilenoptionen.
  *
  */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include <signal.h>
#include <wait.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <syslog.h>
#include <sys/time.h>

#include "ip-lib.h"
#include "lib.h"

#define	PIDFILE			"/var/run/tcpproxy.pid"
#define	ERROR_LOG		"/tmp/tcpproxy-error.log"
#define	MAX_PORT		80


char	*program =		"";
char	progname[80] =		"";

char	logname[80] =		"";
int	debug =			0;
char	configfile[300] =	"";

int	use_errorlog =		0;
char	errorlog[200] =		ERROR_LOG;

int	standalone =		0;
char	pidfile[200] =		"";

int	listports =		0;


typedef struct _port {
    unsigned int port;
    int		count;
    char	ip[40];
    int		sock;
    } port_t;

int	binds =			0;
port_t	*bindport;


#define	PROXY_DEFAULT		0
#define	PROXY_SERVER		1
#define	PROXY_PROGRAM		2

#define	MAX_ARGS		32


typedef struct _config {
    char	interface[300];
    unsigned int port;
    
    unsigned int uid;
    unsigned int gid;
    char	logname[80];
    
    int		timeout;
    int		cleanenv;
    char	varname[80];
    char	acp[200];

    char	writefile[200];

    int		proxytype;
    union {
	struct {
	    char	name[300];
	    unsigned int port;
	    } server;

	struct {
	    int		exec;
	    
	    char	command[300];
	    char	*argv[MAX_ARGS + 2];
	    } program;
	} u;


    struct {
        char	ip[300];
        unsigned int port;
        } incoming;

    char	client_ip[300];
    char	client[300];
    int		local_client;

    long	bytes_send;
    long	bytes_received;

    struct _config *next;
    } config_t;

config_t *first =	NULL;
config_t *last =	NULL;
int	config_count =	0;


/* void *allocate(int size); */


config_t *new_configuration(config_t *def)
{
	config_t *x;

	x = allocate(sizeof(config_t));
	if (def != NULL)
		memmove(x, def, sizeof(config_t));

	if (first == NULL) {
		first = x;
		last  = x;
		}
	else {
		last->next = x;
		last = x;
		}

	config_count++;
	return (x);
}

config_t *find_configuration(char *interface, unsigned int port)
{
	config_t *x, *global;

	global = NULL;
	x = first;
	while (x != NULL) {
		if (x->port == port) {
			if (*x->interface == 0  ||  strcmp(x->interface, "0.0.0.0") == 0)
				global = x;
			else if (strcmp(x->interface, interface) == 0)
				break;
			}
			
		x = x->next;
		}

	if (x == NULL  &&  global != NULL)
		x = global;
		
	return (x);
}


int _check_interface(config_t *x, int lineno)
{
	if (*x->interface == 0) {
		fprintf (stderr, "%s: line %d: interface not set\n", program, lineno);
		exit (1);
		}
	
	return (0);
}

int _check_proxytype(config_t *x, int lineno)
{
	if (x->proxytype != 0) {
		fprintf (stderr, "%s: line %d: service redefinition\n", program, lineno);
		exit (1);
		}
	
	return (0);
}

int read_configuration(char *filename, config_t *def)
{
	int	lineno;
	unsigned int port;
	char	*p, word[80], line[300];
	FILE	*fp;
	config_t *x;

	/*
	 * Konfiguration lesen
	 */
	 
	if ((fp = fopen(filename, "r")) == NULL) {
		fprintf (stderr, "%s: can't read config file: %s\n", program, filename);
		exit (-1);
		}


	/*
	 * Einstellungen vor dem ersten bind gehen in die
	 * default-Konfiguration.
	 */
	 
	x = def;

	
	lineno = 0;
	port   = 0;
	while (fgets(line, sizeof(line), fp) != NULL) {
		lineno++;

		p = skip_ws(noctrl(line));
		if (*p == 0  ||  *p == '#')
			continue;

		get_word(&p, word, sizeof(word));
		strlwr(word);
		p = skip_ws(p);

		/*
		 * globale Optionen
		 */
		 
		if (strcmp(word, "pidfile") == 0) {
			copy_string(pidfile, p, sizeof(pidfile));
			if (*pidfile == 0)
				strcpy(pidfile, PIDFILE);
			}
		else if (strcmp(word, "standalone") == 0) {
			get_word(&p, word, sizeof(word));
			strlwr(word);
			if (strcmp(word, "yes") == 0  ||  strcmp(word, "on") == 0)
				standalone = 1;
			else if (strcmp(word, "no") == 0  ||  strcmp(word, "off") == 0)
				standalone = 0;
			}
		else if (strcmp(word, "logname") == 0) {
			if (*(p = skip_ws(p)) == 0) {
				fprintf (stderr, "%s: line %d: missing logname\n", program, lineno);
				exit (-1);
				}

			/*
			 * The logname can now be interface specific.  But
			 * notice that for connections which are handled by
			 * a specific proxy program this name might be
			 * changed by the handler program.
			 */

			if (binds == 0)
				copy_string(logname, p, sizeof(logname));
			else
				copy_string(x->logname, p, sizeof(x->logname));
			}


		/*
		 * Globale Vorgaben, bzw. Serviceparameter
		 */

		else if (strcmp(word, "timeout") == 0) {
			x->timeout = atoi(p);
			if (x->timeout < 1)
				x->timeout = 60;
			}
		else if (strcmp(word, "setenv") == 0) {
			get_word(&p, x->varname, sizeof(x->varname));
			if (*x->varname == 0)
				strcpy(x->varname, "PROXY_");
			}
		else if (strcmp(word, "acp") == 0) {
			copy_string(x->acp, p, sizeof(x->acp));
			}

		/*
		 * New port configuration directives for user and groupid:
		 *
		 * Since x points either to the default or to a specific
		 * interface configuration this sets either the global default
		 * is set before the first interface definition or the user
		 * and group id for the current service -- 08MAR00wzk
		 */

		else if (strcmp(word, "uid") == 0) {
			x->uid = atoi(p);
			}
		else if (strcmp(word, "gid") == 0) {
			x->gid = atoi(p);
			}


		/*
		 * Server-Konfiguration
		 */
		 
		else if (strcmp(word, "port") == 0  ||  strcmp(word, "bind") == 0) {
			if (binds >= MAX_PORT) {
				fprintf (stderr, "%s: line %d: too many ports\n", program, lineno);
				exit (-1);
				}
				
			port = getportnum(p);
			
			bindport[binds].port  = port;
			bindport[binds].count = 0;
			binds++;
			}
		else if (strcmp(word, "interface") == 0) {
			if (binds == 0  ||  port == 0) {
				fprintf (stderr, "%s: line %d: no port specification\n", program, lineno);
				exit (-1);
				}
			else if (*(p = skip_ws(p)) == 0) {
				fprintf (stderr, "%s: line %d: no interface\n", program, lineno);
				exit (-1);
				}

			x = new_configuration(def);
			x->port = port;
			get_word(&p, x->interface, sizeof(x->interface));
			if (strcmp(x->interface, "*") == 0  ||  strcmp(x->interface, "any") == 0)
				strcpy(x->interface, "0.0.0.0");

			copy_string(bindport[binds-1].ip, x->interface, sizeof(bindport[binds-1].ip));
			bindport[binds-1].count++;

			get_word(&p, word, sizeof(word));
			strlwr(word);
			if (*word == 0)
				/* nichts */ ;
			else if (strcmp(word, "server") == 0) {
				_check_proxytype(x, lineno);

				x->proxytype = PROXY_SERVER;
				copy_string(x->u.server.name, p, sizeof(x->u.server.name));
				x->u.server.port = get_port(x->u.server.name, x->port);
				}
			else if (strcmp(word, "exec") == 0) {
				_check_proxytype(x, lineno);

				x->proxytype = PROXY_PROGRAM;
				x->u.program.exec = 1;
				copy_string(x->u.program.command, p, sizeof(x->u.program.command));
				}
			else {
				fprintf (stderr, "%s: line %d: syntax error: %s\n", program, lineno, word);
				exit (-1);
				}
			}
		else if (strcmp(word, "server") == 0) {
			_check_interface(x, lineno);
			_check_proxytype(x, lineno);

			copy_string(x->u.server.name, p, sizeof(x->u.server.name));
			x->u.server.port = get_port(x->u.server.name, x->port);
			}
		else if (strcmp(word, "exec") == 0) {
			int	argc;

			_check_interface(x, lineno);
			_check_proxytype(x, lineno);

			x->proxytype = PROXY_PROGRAM;
			x->u.program.exec = 1;
			
			copy_string(x->u.program.command, p, sizeof(x->u.program.command));
			argc = split(x->u.program.command, x->u.program.argv, ' ', MAX_ARGS);
			if (argc >= MAX_ARGS) {
				fprintf (stderr, "%s: line %d: too much command line parameters\n", program, lineno);
				exit (1);
				}

			x->u.program.argv[argc] = NULL;
			}
		else if (strcmp(word, "writefile") == 0  ||  strcmp(word, "debug") == 0) {
			_check_interface(x, lineno);
			get_word(&p, x->writefile, sizeof(x->writefile));
			}

		else {
			fprintf (stderr, "%s: line %d: unkown configuration directive %s\n", program, lineno, word);
			exit (-1);
			}
		}

	fclose (fp);
	return (0);
}


int cleanenv(config_t *x, char **envp)
{
	int	k, n;
	char	*p, **env;
	
	env = envp;
	n = 0;
	while (env[n] != NULL)
		n++;


	env = envp;
	k = 0;
	while (k < n  &&  *env != NULL) {
		char	var[800];

		p = *env;
		get_quoted(&p, '=', var, sizeof(var));

		/*
		 * Ist nicht ungefaehrlich.  Wenn der folgende Aufruf
		 * fehlschlaegt muessten wir unseren env-Pointer
		 * inkrementieren, um nicht in eine Endlosschleife,
		 * die immer auf derselben Variablen stehenbleibt,
		 * zu landen.  Das waere falsch, wenn wir die Variable
		 * erfolgreich loeschen.  Ungluecklicherweise liefert
		 * unsetenv() keinen return-code zurueck.
		 */
			 
		unsetenv(var);
		k++;
		}
	
	if (*env != NULL  ||  k != n)
		syslog(LOG_NOTICE, "cleanenv error");

	return (0);
}

int set_variables(config_t *x)
{
	char	var[200], val[200];
	
	snprintf (var, sizeof(var) - 2, "%sINTERFACE", x->varname);
	setenv(var, x->incoming.ip, 1);
	
	snprintf (val, sizeof(val) - 2, "%u", x->port);
	snprintf (var, sizeof(var) - 2, "%sPORT", x->varname);
	setenv(var, val, 1);

	snprintf (var, sizeof(var) - 2, "%sCLIENT", x->varname);
	setenv(var, x->client_ip, 1);

	snprintf (var, sizeof(var) - 2, "%sCLIENTNAME", x->varname);
	setenv(var, x->client, 1);

	return (0);
}


int run_acp(config_t *x)
{
	int	rc, pid, pfd[2];
	char	line[300];
	
	if (*x->acp == 0)
		return (0);

	rc = 0;
	if (pipe(pfd) != 0) {
		syslog(LOG_NOTICE, "can't pipe: %m");
		exit (1);
		}
	else if ((pid = fork()) < 0) {
		syslog(LOG_NOTICE, "can't fork acp: %m");
		exit (1);
		}
	else if (pid == 0) {
		int	argc;
		char	*argv[32];

		close(0);		/* Das acp kann nicht vom client lesen. */
		dup2(pfd[1], 2);	/* stderr wird vom parent gelesen. */
		close(pfd[0]);
		set_variables(x);
		
		copy_string(line, x->acp, sizeof(line));
		argc = split(line, argv, ' ', 30);
		argv[argc] = NULL;
		execvp(argv[0], argv);

		syslog(LOG_NOTICE, "can't exec acp %s: %m", argv[0]);
		exit (1);
		}
	else {
		int	len;
		char	message[300];

		close(pfd[1]);
		*message = 0;
		if ((len = read(pfd[0], message, sizeof(message) - 2)) < 0)
			len = 0;

		message[len] = 0;
		noctrl(message);
		

		if (waitpid(pid, &rc, 0) < 0) {
			syslog(LOG_NOTICE, "error while waiting for acp: %m");
			exit (1);
			}

		rc = WIFEXITED(rc) != 0? WEXITSTATUS(rc): 1;
		if (*message == 0)
			copy_string(message, rc == 0? "access granted": "access denied", sizeof(message));

		if (*message != 0)
			syslog(LOG_NOTICE, "%s (rc= %d)", message, rc);
		}
		
	return (rc);
}


int multiaccept_loop(config_t *start)
{
	int	i, max;
	fd_set	bound, available;

	FD_ZERO(&bound);
	max = 0;
	for (i=0; i<binds; i++) {
		if (bindport[i].sock < 0) {
			syslog(LOG_NOTICE, "invalid socket number found: %d", bindport[i].sock);
			exit (-1);
			}
			
		FD_SET(bindport[i].sock, &bound);
		if (bindport[i].sock > max)
			max = bindport[i].sock;
		}


	while (1) {
		memmove(&available, &bound, sizeof(fd_set));
		if (select(max + 1, &available, NULL, NULL, NULL) < 0) {
			if (errno == EINTR) {
				syslog(LOG_NOTICE, "interrupted select call");
				continue;
				}

			syslog(LOG_NOTICE, "select error: %m");
			exit (-1);
			}

		for (i=0; i<binds; i++) {
			if (FD_ISSET(bindport[i].sock, &available)) {
				int	connect, pid, len;
				struct sockaddr_in client;

				/*
				 * hier kommt ein accept an
				 */

				len = sizeof(client);
				if ((connect = accept(bindport[i].sock, (struct sockaddr *) &client, &len)) < 0) {
					if (errno == EINTR  ||  errno == ECONNABORTED)
						continue;

					syslog(LOG_NOTICE, "accept error: %m");
					exit (-1);
					}
		
				if ((pid = fork()) < 0) {
					syslog(LOG_NOTICE, "can't fork process: %m");
					exit (-1);
					}
				else if (pid == 0) {
					dup2(connect, 0);
					dup2(connect, 1);
					dup2(connect, 2);

					close (connect);
					close (bindport[i].sock);

					/*
					 * We have to close all listening ports
					 * before returning -- 24MAR00wzk
					 */

					for (i=0; i<binds; i++) {
						if (bindport[i].sock >= 0)
							close (bindport[i].sock);
						}

					return (0);
					}

				close (connect);
				}
			}
		}

	syslog(LOG_NOTICE, "server broke while loop -- terminating");
	exit (-1);

	return (0);
}


int proxy_request(config_t *x, int cfd)
{
	int	sfd, rc, bytes;
	unsigned long started, now;
	char	buffer[4096];
	struct timeval tov;
	fd_set	connection, available;
	FILE	*fp;


	fp = NULL;
	if (*x->writefile != 0) {
		char	filename[200];

		snprintf (filename, sizeof(filename) - 2, "%s.%d.log", x->writefile, getpid());
		if ((fp = fopen(filename, "a")) == NULL)
			syslog(LOG_NOTICE, "can't open writefile: %s\n", filename);
		}

	time((time_t *) &started); 
	cfd = cfd;
	sfd = fileno(ip_open(x->u.server.name, x->u.server.port));
	if (sfd < 0) {
		char	*error;

		error = get_error(ip_lib_error);
		syslog(LOG_NOTICE, "connection error: client= %s, server= %s:%u, error= %s\n",
			x->client, x->u.server.name, x->u.server.port, error);
		exit (-1);
		}
	

	syslog(LOG_NOTICE, "connect: client= %s, server= %s:%u, timeout= %d, uid/gid= %u/%u",
		x->client, x->u.server.name, x->u.server.port, x->timeout, getuid(), getgid());

	FD_ZERO(&connection);
	FD_SET(cfd, &connection);
	FD_SET(sfd, &connection);

	while (1) {
		memmove(&available, &connection, sizeof(fd_set));
		tov.tv_sec  = x->timeout;
		tov.tv_usec = 0;

		rc = select(sfd + 1, &available, (fd_set *) NULL, (fd_set *) NULL, &tov);
		if (rc < 0) {
			syslog(LOG_NOTICE, "select() error: %m\n");
			break;
			}
		else if (rc == 0) {
			syslog(LOG_NOTICE, "connection timed out: client= %s, server= %s:%u",
				x->client, x->u.server.name, x->u.server.port);
			break;
			}

		if (FD_ISSET(cfd, &available)) {
			if ((bytes = read(0, buffer, sizeof(buffer))) <= 0)
				shutdown(sfd, 1);
			else if (write(sfd, buffer, bytes) != bytes)
				break;

			if (fp != NULL) {
				buffer[bytes] = 0;
				fprintf (fp, "--- CLIENT [%d]:\n", getpid());
				fprintf (fp, "%s\n", buffer);
				fprintf (fp, "\n");
				}

			x->bytes_send = x->bytes_send + bytes;
			}
			
		if (FD_ISSET(sfd, &available)) {
			if ((bytes = read(sfd, buffer, sizeof(buffer))) <= 0)
				break;
			else if (write(1, buffer, bytes) != bytes)
				break;

			if (fp != NULL) {
				buffer[bytes] = 0;
				fprintf (fp, "--- SERVER [%d]:\n", getpid());
				fprintf (fp, "%s\n", buffer);
				fprintf (fp, "\n");
				}

			x->bytes_received = x->bytes_received + bytes;
			}
		}

	time((time_t *) &now); 
	syslog(LOG_NOTICE, "disconnect: client= %s, server= %s:%u, received= %ld, send= %ld, duration= %ld",
		x->client, x->u.server.name, x->u.server.port,
		x->bytes_received, x->bytes_send,
		now - started);
		
	return (0);
}

unsigned get_interface_info(int pfd, char *ip, int max)
{
	int	size;
	unsigned int port;
	struct sockaddr_in saddr;

	size = sizeof(saddr);
	if (getsockname(pfd, (struct sockaddr *) &saddr, &size) < 0) {
		syslog(LOG_NOTICE, "can't get interface info: %m");
		exit (-1);
		}

	copy_string(ip, (char *) inet_ntoa(saddr.sin_addr), max);
	port = ntohs(saddr.sin_port);

	return (port);
}

int get_client_info(config_t *x, int pfd)
{
	int	size;
	struct sockaddr_in saddr;
	struct in_addr *addr;
	struct hostent *hostp = NULL;

	*x->client = 0;
	size = sizeof(saddr);
	if (getpeername(pfd, (struct sockaddr *) &saddr, &size) < 0) {
		return (-1);
		}		
		
	copy_string(x->client_ip, (char *) inet_ntoa(saddr.sin_addr), sizeof(x->client_ip));
	addr = &saddr.sin_addr,
	hostp = gethostbyaddr((char *) addr,
			sizeof (saddr.sin_addr.s_addr), AF_INET);

	if (hostp == NULL) {
		strcpy(x->client, x->client_ip);
		return (0);
		}

	strcpy(x->client, hostp->h_name);
	strlwr(x->client);

	return (0);
}

int open_errorlog(char *filename)
{
	int	fno;

	if (use_errorlog == 0)
		return (0);

	if ((fno = open(filename, O_WRONLY | O_CREAT | O_APPEND, 0664)) >= 0)
		dup2(fno, 2);
	else 
		fprintf (stderr, "%s: can't open %s, using stderr\n", program, filename);

	return (0);
}

void missing_arg(int c, char *string)
{
	fprintf (stderr, "%s: missing arg: -%c, %s\n", program, c, string);
	exit (-1);
}

int main(int argc, char *argv[], char *envp[])
{
	int	c, i, k, bound;
	unsigned int port;
	char	*p, option[80], interface[300];
	config_t *x;
	

	if ((p = strrchr(argv[0], '/')) == NULL)
		program = argv[0];
	else {
		copy_string(progname, &p[1], sizeof(progname));
		program = progname;
		}

	x = allocate(sizeof(config_t));
	x->timeout = 60;
	x->uid     = 65535;
	x->gid     = 65534;
	strcpy(x->varname, "PROXY_");

	bindport = allocate(sizeof(port_t) * MAX_PORT);
	openlog(program, LOG_PID, LOG_MAIL);

	bound = 0;
	k = 1;
	while (k < argc  &&  argv[k][0] == '-'  &&  argv[k][1] != 0) {
		copy_string(option, argv[k++], sizeof(option));
		for (i=1; (c = option[i]) != 0; i++) {
			if (c == 'd')
				debug = 1;
			else if (c == 'a') {
				if (k >= argc)
					missing_arg(c, "access control program");

				copy_string(x->acp, argv[k++], sizeof(x->acp));
				}
			else if (c == 'b') {
				char	addr[300];
				
				if (k >= argc)
					missing_arg(c, "bind address and port");

				copy_string(addr, argv[k++], sizeof(addr));
				if ((p = strchr(addr, ':')) == NULL)
					x->port = getportnum(addr);
				else {
					*p++ = 0;
					x->port = getportnum(p);
					copy_string(x->interface, addr, sizeof(x->interface));
					}

				bound = 1;
				standalone = 1;
				}
			else if (c == 'e') {
				if (use_errorlog != 0) {
					if (k >= argc)
						missing_arg(c, "logfile name");

					copy_string(errorlog, argv[k++], sizeof(errorlog));
					}
					
				use_errorlog = 1;
				}
			else if (c == 'f'  ||  c == 'c') {
				if (k >= argc)
					missing_arg(c, "configuration file");

				copy_string(configfile, argv[k++], sizeof(configfile));
				}
			else if (c == 'l') {
				if (k >= argc)
					missing_arg(c, "logname");

				copy_string(logname, argv[k++], sizeof(logname));
				}
			else if (c == 'p') {
				if (*pidfile == 0)
					strcpy(pidfile, PIDFILE);
				else {
					if (k >= argc)
						missing_arg(c, "pidfile");

					copy_string(pidfile, argv[k++], sizeof(pidfile));
					}
				}
			else if (c == 's')
				standalone = 1;
			else if (c == 't') {
				if (k >= argc)
					missing_arg(c, "timeout");

				x->timeout = atoi(argv[k++]);
				if (x->timeout < 1)
					x->timeout = 60;
				}
			else if (c == 'v') {
				if (k >= argc)
					missing_arg(c, "varname prefix");

				copy_string(x->varname, argv[k++], sizeof(x->varname));
				}
			else if (c == 'w') {
				if (k >= argc)
					missing_arg(c, "write file");

				copy_string(x->writefile, argv[k++], sizeof(x->writefile));
				}
			else if (c == 'y')
				x->cleanenv = 1;
			else if (c == 'z') {
				if (listports == 0)
					listports = 1;
				else
					listports = 2;
				}
			else {
				fprintf (stderr, "%s: unknown option: -%c\n", program, c);
				exit (-1);
				}
			}
		}



	if (*configfile != 0)
		read_configuration(configfile, x);
 	else if (bound == 0  &&  k >= argc) {
		copy_string(configfile, "/etc/tcpproxy.conf", sizeof(configfile));
		standalone = 1;

		read_configuration(configfile, x);
		}
	else if (listports == 1)
		/* ist ok, ein Serverprogramm wird nicht benoetigt */ ;
	else if (k >= argc) {
		fprintf (stderr, "%s: missing server\n", program);
		exit (1);
		}
	else {
		if (*argv[k] == '/'  ||  *argv[k] == '.') {
			x->proxytype = PROXY_PROGRAM;

			i = 0;
			while (k < argc)
				x->u.program.argv[i++] = argv[k++];

			x->u.program.argv[i] = NULL;
			}
		else {
			x->proxytype = PROXY_SERVER;

			copy_string(x->u.server.name, argv[k++], sizeof(x->u.server.name));
			x->u.server.port = get_port(x->u.server.name, 0);

			if (k < argc) {
				fprintf (stderr, "%s: extra arguments after server\n", program);
				exit (1);
				}
			}
		}
	

	if (*logname != 0) {
		copy_string(progname, logname, sizeof(progname));
		closelog();
		openlog(logname, LOG_PID, LOG_MAIL);
		}

	if (first == NULL) {
		x = new_configuration(x);
		bindport[binds].port  = x->port;
		bindport[binds].count = 1;
		binds++;
		}


	if (listports == 1) {
		for (i=0; i<binds; i++) {
			if (bindport[i].port != 0)
				printf ("%u\n", bindport[i].port);
			}
			
		exit (0);
		}
	else if (listports == 2) {
		config_t *y;

		y = first;
		while (y != NULL) {
			printf ("%s %u", y->interface, y->port);
			if (y->proxytype == PROXY_PROGRAM) {
				int	i;

				printf ("  exec:");
				for (i=0; y->u.program.argv[i] != NULL; i++)
					printf (" %s", y->u.program.argv[i]);
				}
			else {
				printf ("  proxy:");
				printf (" %s %u", y->u.server.name, y->u.server.port);
				}

			printf ("\n");
			y = y->next;
			}

		exit (0);
		}


	if (standalone != 0) {
		int	i;
		char	*ip;
		
		for (i=0; i<binds; i++) {
			if (bindport[i].port == 0)
				continue;	/* Mogelt sich irgendwie in die Port-Liste -- 20AUG99wzk */
				
			ip = bindport[i].ip;
			if (strcmp(ip, "*") == 0  ||  bindport[i].count > 1)
				ip = "0.0.0.0";
				
			bindport[i].sock = bind_to_port(strcmp(ip, "0.0.0.0") == 0? "": ip, bindport[i].port);
			syslog(LOG_NOTICE, "bound to %s:%u",
				ip, bindport[i].port);
			}
			
		signal(SIGCHLD, SIG_IGN);
		standalone = 1;
		}


	/*
	 * Prozess in den Hintergrund legen.
	 */

	if (debug == 0) {
		if (standalone != 0) {
			int	pid;
		
			if ((pid = fork()) < 0) {
				fprintf (stderr, "%s: can't fork into background\n", program);
				exit (-1);
				}
			else if (pid > 0)
				exit (0);

			syslog(LOG_NOTICE, "ready");
			}
		
		open_errorlog(errorlog);
		}

	/*
	 * pidfile schreiben
	 */

	if (*pidfile != 0) {
		FILE	*fp;

		if ((fp = fopen(pidfile, "w")) == NULL) {
			fprintf (stderr, "%s: can't write pidfile: %s\n", program, pidfile);
			exit (1);
			}

		fprintf (fp, "%d\n", getpid());
		fclose (fp);
		}
		

/*
 * Hier wurde vor Version 1.1.5 die userid gesetzt.  Ab 1.1.5 wird die
 * erst gesetzt wenn ein request angenommen wurde.
 * 
 *	if (getuid() != 0  &&  geteuid() != 0)
 *		;
 *	else if (setgid(gid) != 0  ||  setuid(uid) != 0) {
 *		fprintf (stderr, "%s: can't change uid/gid to %d/%d\n", program, uid, gid);
 *		exit (1);
 *		}
 */

	/*
	 * Server ist einsatzbereit
	 */
	 
	if (standalone != 0)
		multiaccept_loop(first);

	signal(SIGCHLD, SIG_DFL);


	/*
	 * An dieser Stelle liegt auf stdin/stdout eine TCP/IP-Verbindung
	 * vor.  Im standalone-Modus bleibt der Master-Server immer in der
	 * multiaccept-loop.
	 *
	 * Wir fragen zunaechst die Interface-Daten ab.
	 */

	port = get_interface_info(0, interface, sizeof(interface));
	syslog(LOG_NOTICE, "connected via %s:%u", interface, port);


	/*
	 * Konfiguration zum Interface suchen, ggf. die Verbindung
	 * terminieren.
	 */

	x = find_configuration(interface, port);
	if (x == NULL  &&  (x = find_configuration(interface, 0)) == NULL) {
		syslog(LOG_NOTICE, "no configuration found for interface %s:%u", interface, port);
		exit (-1);
		}


	/*
	 * Ok, es existiert eine Konfiguration.  Wir setzen uid und gid und
	 * uebernehmen dann die Interface-Daten.
	 */

	if (*x->logname != 0) {
		closelog();
		openlog(x->logname, LOG_PID, LOG_MAIL);
		}

 	if (getuid() != 0  &&  geteuid() != 0)
		/* nichts */ ;
	else if (setgid(x->gid) != 0  ||  setuid(x->uid) != 0) {
		syslog(LOG_NOTICE, "can't change uid/gid to %u/%u", x->uid, x->gid);
		exit (-1);
		}

	copy_string(x->incoming.ip, interface, sizeof(x->incoming.ip));
	x->incoming.port = port;

	get_client_info(x, 0);
	if (*x->client == 0) {
		syslog(LOG_NOTICE, "can't get client info");
		exit (-1);
		}


	/*
	 * access control Programm ausfuehren
	 */

	if (*x->acp != 0) {
		int	rc;

		rc = run_acp(x);
		if (rc != 0)
			exit (0);
		}


	/*
	 * Server oder Proxy starten
	 */

	if (x->proxytype == PROXY_PROGRAM) {
		syslog(LOG_NOTICE, "starting %s, client= %s, uid/gid= %u/%u", x->u.program.argv[0], x->client, getuid(), getgid());
		
		if (x->cleanenv != 0)
			cleanenv(x, envp);

		set_variables(x);
		execvp(x->u.program.argv[0], x->u.program.argv);
		
		syslog(LOG_NOTICE, "can't execute %s: %m", x->u.program.argv[0]);
		exit (1);
		}
	else {
		/*
		 * proxytype ist entweder PROXY_SERVER oder PROXY_DEFAULT.
		 */

		if (x->u.server.port == 0)
			x->u.server.port = x->incoming.port;

		if (x->u.server.port == 0) {
			syslog(LOG_NOTICE, "no destination port for interface %s:%u", interface, port);
			exit (-1);
			}

		proxy_request(x, 0);
		close(0);
		}

	exit (0);
}


