
/*
 *  Copyright (C) 1999  Wolfgang Zekoll  <wzk@quietsche-entchen.de>
 *
 *  This software is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/fcntl.h>
#include <sys/wait.h>

#include <signal.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <syslog.h>

#include "lib.h"


char	*program =		"";
char	progname[80] =		"";

char	interface[200] =	"";
int	bindport =		0;
int	selectport =		0;
int	bindsock =		-1;

int	timeout =		300;
int	uid =			0;
int	gid =			0;
char	directory[200] =	"";

int	background =		0;
int	forkserver =		0;
int	execprog =		0;
int	standalone =		0;
int	waitchild =		0;
int	maxclients =		0;
int	cleanenv =		0;
int	numericonly =		0;

int	pargc =			0;
char	**pargv =		NULL;



int	numclients =		0;
char	clientip[200] =		"";
char	clientname[200] =	"";

long	bytesin =		0;
long	bytesout =		0;


	/*
	 * Abfragen von Interface- und Client-Informationen.
	 */

unsigned int getportnum(char *name)
{
	unsigned int port;
	struct servent *portdesc;
	
	if (isdigit(*name) != 0)
		port = atol(name);
	else {
		portdesc = getservbyname(name, "tcp");
		if (portdesc == NULL) {
			fprintf (stderr, "%s: service not found: %s\n", program, name);
			exit (-1);
			}

		port = ntohs(portdesc->s_port);
		if (port == 0) {
			fprintf (stderr, "%s: port error: %s\n", program, name);
			exit (-1);
			}
		}
	
	return (port);
}

unsigned int get_interface_info(int pfd, char *ip, int max)
{
	int	size;
	unsigned int port;
	struct sockaddr_in saddr;

	size = sizeof(saddr);
	if (getsockname(pfd, (struct sockaddr *) &saddr, &size) < 0) {
		fprintf (stderr, "%s: can't get interface info\n", program);
		exit (1);
		}

	copy_string(ip, (char *) inet_ntoa(saddr.sin_addr), max);
	port = ntohs(saddr.sin_port);

	return (port);
}

unsigned int get_clientip(int pfd, char *client, int max)
{
	int	size;
	struct sockaddr_in saddr;

	*client = 0;
	size = sizeof(saddr);
	if (getpeername(pfd, (struct sockaddr *) &saddr, &size) < 0 ) {
		fprintf (stderr, "%s: can't get peer name\n", program);
		exit (1);
		}		
		
	copy_string(client, (char *) inet_ntoa(saddr.sin_addr), max - 2);
 	return (0);
}

unsigned int get_clientname(int pfd, char *name, int max)
{
	int	size;
	char	ip[80];
	struct sockaddr_in saddr;
	struct in_addr *addr;
	struct hostent *hostp = NULL;

	*name = 0;
	size = sizeof(saddr);
	if (getpeername(pfd, (struct sockaddr *) &saddr, &size) < 0 )
		return (-1);
		
	copy_string(ip, (char *) inet_ntoa(saddr.sin_addr), sizeof(ip));
	addr = &saddr.sin_addr,
	hostp = gethostbyaddr((char *) addr,
			sizeof (saddr.sin_addr.s_addr), AF_INET);

	hostp = NULL;
	if (hostp == NULL) {
		copy_string(name, ip, max);
		return (0);
		}

	copy_string(name, hostp->h_name, max);
	strlwr(name);

 	return (0);
}


void sigchld_handler(int sig)
{
	int	pid, status;

again:
	pid = wait(&status);
	if (pid < 0) {
		if (errno == EINTR  /* ||  errno == ERESTARTSYS */ ) {
			fprintf (stderr, "%s: interrupted call to wait() in handler\n", program);
			goto again;
			}
		else if (errno == ECHILD) {
			fprintf (stderr, "%s: sigchld_handler called without terminated child\n", program);
			goto end;
			}
		}
		
	numclients--;

end:
	signal(SIGCHLD, sigchld_handler);
	return;
}

int bind_to_port(char *interface, unsigned int port)
{
	struct sockaddr_in saddr;
	int	sock;

	if ((sock = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		fprintf (stderr, "%s: can't create socket\n", program);
		exit (-1);
		}
	else {
		int	opt;

		opt = 1;
		setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt));
		}


	memset(&saddr, 0, sizeof(saddr));
	saddr.sin_family = AF_INET;
	saddr.sin_port   = htons(port);
	
	if (interface == NULL  ||  *interface == 0)
		interface = "0.0.0.0";
	else {
		struct hostent *ifp;

		ifp = gethostbyname(interface);
		if (ifp == NULL) {
			fprintf (stderr, "%s: can't lookup %s\n", program, interface);
			exit (-1);
			}

		memcpy(&saddr.sin_addr, ifp->h_addr, ifp->h_length);
		}


	if (bind(sock, (struct sockaddr *) &saddr, sizeof(saddr))) {
		fprintf (stderr, "%s: can't bind to %s:%u\n", program, interface, port);
		exit (-1);
		}


	if (listen(sock, 5) < 0) {
		fprintf (stderr, "%s: listen error\n", program);
		exit (-1);
		}

	return (sock);
}

int accept_loop(int sock, int forkproc, int waitproc)
{
	int	rc, connect, pid, len;
	struct sockaddr_in client;
	static int errcount = 0;

	while (1) {
		if (maxclients > 0  &&  numclients >= maxclients) {
			fprintf (stderr, "%s: maximum number is reached: %d\n", program, maxclients);
			pause();
			}


		len = sizeof(client);
		if ((connect = accept(sock, (struct sockaddr *) &client, &len)) < 0) {
			if (errno == EINTR  ||  errno == ECONNABORTED)
				continue;

			fprintf (stderr, "%s: accept error, errno= %d\n", program, errno);
			errcount++;
			if (errcount > 10) {
				fprintf (stderr, "%s: too many accept errors, terminating\n", program);
				exit (-1);
				}
			}

		errcount = 0;

		pid = 0;
		if (forkproc != 0) {
			if ((pid = fork()) < 0) {
				fprintf (stderr, "%s: can't fork process\n", program);
				exit (-1);
				}
			}

		if (forkproc == 0  ||  pid == 0) {

			/*
			 * Entweder haben wir keinen neuen Prozess geforkt,
			 * weil es so nicht verlangt wird, oder wir befinden
			 * uns hier im child-Prozess.
			 *
			 * Dieser Prozess wird in jedem Fall mit dem Client
			 * kommunizieren.
			 *
			 * An dieser Stelles brauchen wir den connected und
			 * den listening socket nicht mehr.
			 */

			dup2(connect, 0);
			dup2(connect, 1);

			close (connect);
			close (sock);

			/*
			 * Wir sorgen dafuer, dass SIGCHLD so wie immer
			 * funktioniert, egal was der listening-Server
			 * eingestellt hat, damit etwaige Unterprozesse
			 * das Signal bekommen koennen.
			 */

			signal(SIGCHLD, SIG_DFL);
			
			return (0);
			}

		/*
		 * Falls wir einen neuen Prozess geforkt() haben, dann
		 * kommt nur der parent-Prozess an diese Stelle.  Er
		 * benoetigt den connected socket nicht mehr.
		 */

		close (connect);
		numclients++;

		if (waitproc != 0)
			wait(&rc);
		}

	fprintf (stderr, "%s: server broke while loop\n", program);
	exit (-1);

	return (0);
}



int socket_request(int sin, int sout)
{
	int	rc, bytes;
	unsigned long started, now;
	char	buffer[4096];
	struct timeval tov;
	fd_set	connection, available;


	time((time_t *) &started); 

	FD_ZERO(&connection);
	FD_SET(0, &connection);
	FD_SET(sin, &connection);

	while (1) {
		memmove(&available, &connection, sizeof(fd_set));
		tov.tv_sec  = timeout;
		tov.tv_usec = 0;

		rc = select(sin + 1, &available, (fd_set *) NULL, (fd_set *) NULL, &tov);
		if (rc < 0) {
			fprintf (stderr, "%s: select() error\n", program);
			break;
			}
		else if (rc == 0) {
			/* Timeout */
			break;
			}

		if (FD_ISSET(0, &available)) {
			if ((bytes = read(0, buffer, sizeof(buffer) - 10)) <= 0)
				close(sout);
			else if (write(sout, buffer, bytes) != bytes)
				break;
				
			bytesin = bytesin + bytes;
			}
			
		if (FD_ISSET(sin, &available)) {
			if ((bytes = read(sin, buffer, sizeof(buffer) - 10)) <= 0)
				break;
			else if (write(1, buffer, bytes) != bytes)
				break;
				
			bytesout = bytesout + bytes;
			}
		}

	close (sin);
	close (sout);

	close (0);
	close (1);
	
	time((time_t *) &now); 
	return (0);
}


void missing_arg(int c, char *string)
{
	fprintf (stderr, "%s: missing arg: -%c, %s\n", program, c, string);
	exit (-1);
}

int main(int argc, char *argv[], char *envp[])
{
	int	c, i, k;
	unsigned int port;
	char	*p, option[80];
	

	if ((p = strrchr(argv[0], '/')) == NULL)
		program = argv[0];
	else {
		copy_string(progname, &p[1], sizeof(progname));
		program = progname;
		}

	timeout = 300;

	k = 1;
	while (k < argc  &&  argv[k][0] == '-'  &&  argv[k][1] != 0) {
		copy_string(option, argv[k++], sizeof(option));
		for (i=1; (c = option[i]) != 0; i++) {
			if (c == 'b')
				background = 1;
			else if (c == 'c') {
				if (k >= argc)
					missing_arg(c, "directory");

				copy_string(directory, argv[k++], sizeof(directory));
				}
			else if (c == 'e')
				execprog = 1;
			else if (c == 'l')
				forkserver = 1;
			else if (c == 'm') {
				if (k >= argc)
					missing_arg(c, "maxclients");

				maxclients = atoi(argv[k++]);
				if (maxclients == 0) {
					fprintf (stderr, "%s: number of maxclients is 0\n", program);
					exit (1);
					}
				}
			else if (c == 'n')
				numericonly = 1;
			else if (c == 'p') {
				char	addr[300];
				
				if (k >= argc)
					missing_arg(c, "[interface:]port");

				copy_string(addr, argv[k++], sizeof(addr));
				if ((p = strchr(addr, ':')) == NULL)
					bindport = getportnum(addr);
				else {
					*p++ = 0;
					bindport = getportnum(p);
					copy_string(interface, addr, sizeof(interface));
					}

				if (bindport == 0) {
					selectport = 1;
					background = 1;
					}

				standalone = 1;
				}
			else if (c == 't') {
				if (k >= argc)
					missing_arg(c, "timeout");

				timeout = atoi(argv[k++]);
				if (timeout < 1)
					timeout = 60;
				}
			else if (c == 'u') {
				char	*param;

				if (k >= argc)
					missing_arg(c, "uid[:gid]");

				param = argv[k++];
				uid = strtol(param, &p, 10);
				if (*p == 0)
					/* nichts */ ;
				else if (*p == ':')
					gid = strtol(&p[1], &p, 10);

				if (*p != 0) {
					fprintf (stderr, "%s: bad uid:gid parameter: %s\n", program, param);
					exit (-1);
					}
				}
			else if (c == 'w')
				waitchild = 1;
			else if (c == 'y')
				cleanenv = 1;
			else {
				fprintf (stderr, "%s: unknown option: -%c\n", program, c);
				exit (-1);
				}
			}
		}


	if (k >= argc) {
		fprintf (stderr, "usage: %s [options] <command> [<arg> ...]\n", program);
		exit (-1);
		}
	else {
		pargc = k - argc;
		pargv = &argv[k];
		}


	if (cleanenv != 0)
		envp = NULL;
	else if (cleanenv != 0) {
		char	**env;
		
		env = envp;
		while (*env != NULL) {
			char	var[800];

			p = *env;
			get_quoted(&p, '=', var, sizeof(var));

			/*
			 * Ist nicht ungefaehrlich.  Wenn der folgende Aufruf
			 * fehlschlaegt muessten wir unseren env-Pointer
			 * inkrementieren, um nicht in eine Endlosschleife,
			 * die immer auf derselben Variablen stehenbleibt,
			 * zu landen.  Das waere falsch, wenn wir die Variable
			 * erfolgreich loeschen.  Ungluecklicherweise liefert
			 * unsetenv() keinen return-code zurueck.
			 */
			 
			unsetenv(var);
			}
		}


	if (bindport != 0  ||  selectport != 0) {
		char	*ip;

		ip = interface;
		if (*ip == 0  ||  strcmp(ip, "*") == 0)
			ip = "0.0.0.0";

		signal(SIGCHLD, sigchld_handler);
		bindsock = bind_to_port(strcmp(ip, "0.0.0.0") == 0? "": ip, bindport);

		if (selectport != 0) {
			char	ipnum[200];
			
			bindport = get_interface_info(bindsock, ipnum, sizeof(ipnum));
			printf ("%u\n", bindport);
			}

		standalone = 1;
		}


	if (standalone == 0) {
		
		/*
		 * Im inetd-Modus werden einige Schalter annuliert.
		 */

		background = 0;
		}


	/*
	 * Es folgen ein paar Optionen fuer den root-User.
	 */

	if (*directory != 0) {
		if (chroot(directory) != 0) {
			fprintf (stderr, "%s: can't set root directory: %s\n", program, directory);
			exit (-1);
			}
		else if (chdir("/") != 0) {
			fprintf (stderr, "%s: can't change directory\n", program);
			exit (-1);
			}
		}
		
	if (gid != 0  &&  setregid(gid, gid) != 0) {
		fprintf (stderr, "%s: can't set group id\n", program);
		exit (-1);
		}

	if (uid != 0  &&  setreuid(uid, uid) != 0) {
		fprintf (stderr, "%s: can't set user id\n", program);
		exit (-1);
		}



	if (background != 0) {
		int	pid;
		
		if ((pid = fork()) < 0) {
			fprintf (stderr, "%s: can't fork into background\n", program);
			exit (-1);
			}
		else if (pid > 0)
			exit (0);
		}


	/*
	 * Der Server ist einsatzbereit - wir warten auf eine
	 * einkommende Verbindung.
	 *
	 * Wenn (forkserver != 0) ist, dann kehrt der Hauptserver
	 * niemals zurueck, sondern laeuft in einer Endlosschleife
	 * weiter.
	 */
	 
	if (standalone != 0)
		accept_loop(bindsock, forkserver, waitchild);


	/*
	 * Die Filedeskriptoren 0 und 1 sind nun mit dem Client
	 * verbunden.
	 *
	 * Wir besorgen uns zunaechst mal die Daten ueber das
	 * Interface auf dem der Client hereingekommen ist.
	 */

	port = get_interface_info(0, interface, sizeof(interface));


	/*
	 * Wir versorgen den Server mit ein paar Umgebungsvariablen.
	 */

	setenv("SOCKETD_INTERFACE", interface, 1);

	snprintf (option, sizeof(option) - 2, "%u", port);
	setenv("SOCKETD_PORT", option, 1);

	get_clientip(0, clientip, sizeof(clientip));
	setenv("SOCKETD_CLIENT", clientip, 1);

	if (numericonly != 0)
		setenv("SOCKETD_CLIENT", clientip, 1);
	else {
		get_clientname(0, clientname, sizeof(clientname));
		setenv("SOCKETD_CLIENTNAME", clientname, 1);
		}


	if (execprog != 0) {
		
		/*
		 * Das ist einfach: Wir starten einfach die Server
		 * Anwendung und terminieren gleichzeitig.
		 */

		execvp(pargv[0], pargv);
		fprintf (stderr, "%s: can't exec %s\n", program, pargv[0]);

		exit (1);
		}
	else {
		int	pid, pin[2], pout[2];

		if (pipe(pin) != 0  ||  pipe(pout) != 0) {
			fprintf (stderr, "%s: can't create pipes\n", program);
			exit (1);
			}
		else if ((pid = fork()) < 0) {
			fprintf (stderr, "%s: can't fork\n", program);
			exit (1);
			}
		else if (pid == 0) {
			dup2(pin[0], 0);
			close (pin[1]);

			dup2(pout[1], 1);
			close (pout[0]);

			execvp(pargv[0], pargv);
			fprintf (stderr, "%s: can't exec %s\n", program, pargv[0]);

			exit (1);
			}
		else {
			close (pin[0]);
			close (pout[1]);
			}
			
		socket_request(pout[0], pin[1]);
		}

	close (0);
	close (1);

	exit (0);
}


