
/*
 *  Copyright (C) 1999  Wolfgang Zekoll  <wzk@quietsche-entchen.de>
 *
 *  This software is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <errno.h>
#include <sys/wait.h>


#include "lib.h"


#define	DEBUG(x)		

#define	SORT_NETWORKS		1
#define	SORT_NETMASKS		2
#define	SORT_IPNUMBERS		3
#define	SORT_NAMES		4

#define	ADJUST_STANDARD		11
#define	ADJUST_SMALLEST		12
#define	ADJUST_LARGEST		13

#define	OUTPUT_STANDARD		21
#define	OUTPUT_CONFIG		22
#define	OUTPUT_ROUTES		23

#define	BY_NETWORK		1
#define	BY_NETMASK		2
#define	BY_IPNUM		3
#define	BY_NAMES		4


char	*program =		"";

int	mode =			0;
int	output =		0;


typedef struct _net {
    char	name[10];
    int		interface, virtual;
    int		modified;

    unsigned long number;

    unsigned long adr;
    unsigned long last;

    unsigned long mask;

    struct _net	*prev, *next;
    } net_t;

net_t	*netlist =		NULL;
net_t	*lastnet =		NULL;

net_t	**netv =		NULL;
int	netc =			0;


net_t *alloc_net(void)
{
	net_t	*x;

	if ((x = malloc(sizeof(net_t))) == NULL) {
		fprintf (stderr, "%s: memory allocation error\n", program);
		exit (1);
		}
	
	memset(x, 0, sizeof(net_t));
	return (x);
}

net_t *append_net(net_t *x)
{
	if (netlist == NULL) {
		netlist = x;
		}
	else {
		lastnet->next = x;
		x->prev = lastnet;
		}

	lastnet = x;
	return (x);
}

int create_netv(void)
{
	int	k;
	net_t	*x;

	x = netlist;
	k = 0;
	while (x != NULL) {
		k++;
		x = x->next;
		}

	if (k == 0)
		return (0);

	netv = malloc(k * sizeof(net_t *));
	x = netlist;
	k = 0;

	while (x != NULL) {
		netv[k++] = x;
		x = x->next;
		}

	netc = k;
	return (netc);
}

	/*
	 *  S O R T I E R F U N K T I O N E N
	 */

int _cmp_by_ipnum(const net_t **x, const net_t **y)
{
	int	rc;

	rc = (*x)->number - (*y)->number;
	return (rc);
}

int _cmp_by_network(const net_t **x, const net_t **y)
{
	int	rc;

	rc = (*x)->adr - (*y)->adr;
	if (rc == 0)
		rc = -((*x)->mask - (*y)->mask);

	return (rc);
}

int _cmp_by_netmask(const net_t **x, const net_t **y)
{
	int	rc;

	rc = -((*x)->mask - (*y)->mask);
	if (rc == 0)
		rc = (*x)->adr - (*y)->adr;

	return (rc);
}

int _cmp_by_name(const net_t **x, const net_t **y)
{
	int	rc;

	rc = (*x)->interface - (*y)->interface;
	if (rc == 0)
		rc = (*x)->virtual - (*y)->virtual;

	return (rc);
}

int sort_netv(int type)
{
	int	(*sort)(const net_t **, const net_t **);

	if (netc == 0  ||  netc == 1)
		return (0);

	switch (type) {
	case BY_IPNUM:
		sort = _cmp_by_ipnum;
		break;

	case BY_NETWORK:
		sort = _cmp_by_network;
		break;

	case BY_NETMASK:
		sort = _cmp_by_netmask;
		break;

	case BY_NAMES:
		sort = _cmp_by_name;
		break;

	default:
		sort = _cmp_by_network;
		}
		
	qsort(netv, netc, sizeof(net_t *), sort);
	return (0);
}

	/*
	 *  I N T E R F A C E K O R R E K T U R
	 */

int adjust_netv(int mode)
{
	int	i, k;

	sort_netv(BY_NAMES);
	for (i=0; i<netc; i++) {
		for (k=0; k<i; k++) {
			if (netv[i]->interface != netv[k]->interface)
				/* nichts, anderes Interface */ ;
			else if (netv[k]->adr > netv[i]->number  ||  netv[k]->last < netv[i]->number)
				/* nichts, ip[i] liegt nicht in net[k] */ ;
			else {
				/* Konfigurationsdaten uebernehmen
				 */

				netv[i]->adr  = netv[k]->adr;
				netv[i]->mask = netv[k]->mask;
				netv[i]->last = netv[k]->last;
				netv[i]->modified = 1;
				}
			}
		}

	sort_netv(BY_NAMES);
	return (0);
}

	/*
	 *  E I N G A B E F U N K T I O N E N
	 */

unsigned long atoip(char *string, char **r, int *error)
{
	int	i, k;
	unsigned long num;
	char	*p;

	*error = 0;
	num    = 0;

	p = string;
	for (i=0; i<4  &&  *p != 0; i++) {
		k = (unsigned) strtol(p, &p, 10);
		if (k < 0  ||  k > 255) {
			*error = 1;
			return (1);
			}

		num = (num << 8) + k;
		if (*p == '.') {
			p++;
			if (isdigit(*p) == 0) {
				*error = 1;
				return (1);
				}
			}
		else if (i != 3) {
			*error = 1;
			return (1);
			}
		}

	*r = p;
	return (num);
}

int read_ipnet(net_t *x, char *line, char **r)
{
	int	error;
	char	*p, *s;

	s = line;
	x->adr = x->number = atoip(skip_ws(line), &p, &error);
	if (error != 0) {
		fprintf (stderr, "%s: invalid ip number: %s\n", program, line);
		return (1);
		}
	else if (*p == '/') {
		p++;
		error = 0;
		if (strchr(p, '.') != NULL)
			x->mask = atoip(p, &p, &error);
		else {
			int	n;

			n = strtol(p, &p, 10);
			if (n < 0  ||  n > 32)
				error = 1;

			x->mask = (0xFFFFFFFFUL << (32 - n));
			}

		if (error != 0) {
			fprintf (stderr, "%s: invalid netmask: %s\n", program, line);
			return (1);
			}
			
		x->adr = x->adr & x->mask;
		}
	else
		x->mask = 0xFFFFFFFFUL;

	*r = p;
	if (*p != 0) {
		fprintf (stderr, "%s: extra characters behind specification: %s\n", program, line);
		return (1);
		}

	x->last = x->adr + (~ x->mask);
	return (0);
}

int read_interface(net_t *x, char *line, char **r)
{
	int	c;
	char	*p, *s, name[20];

	p = line;
	get_word(&p, name, sizeof(name));
	if (*name == 0)
		return (1);
	
	copy_string(x->name, name, sizeof(name));
	x->interface = -1;
	x->virtual   = 0;
	
	s = name;
	while ((c = *s) != 0) {
		if (isdigit(c) == 0)
			s++;
		else {
			x->interface = strtoul(s, &s, 10);
			if (*s == ':') {
				s++;
				x->virtual = strtoul(s, &s, 10);
				}
			}
		}

	if (x->interface < 0)
		return (1);

	return (read_ipnet(x, p, r));
}

int read_networks(FILE *fp, int interfaces)
{
	int	rc;
	char	*p, line[300];
	net_t	*x;

	x = alloc_net();
	while (fgets(line, sizeof(line), fp) != NULL) {
		noctrl(line);

		memset(x, 0, sizeof(net_t));
		if (interfaces != 0)
			rc = read_interface(x, line, &p);
		else
			rc = read_ipnet(x, line, &p);

		if (rc == 0) {
			append_net(x);
			x = alloc_net();
			}
		}

	create_netv();
	return (netc);
}


	/*
	 *  A U S G A B E F U N K T I O N E N
	 */

int print_ipnum(unsigned long ip, int crlf)
{
	unsigned char *p;

	if (crlf == 0)
		crlf = '\n';

	p = (unsigned char *) &ip;
	printf ("%u.%u.%u.%u%c", p[3], p[2], p[1], p[0], crlf);

	return (0);
}

int print_ipnet(net_t *x)
{
	print_ipnum(x->number, '/');
	print_ipnum(x->mask, '\n');

	return (0);
}

int print_iprange(net_t *net)
{
	unsigned long start, end, adr;
	unsigned char *p;

	adr = net->adr;
	p = (unsigned char *) &adr;

	if (net->last == net->adr) {
		start = net->adr;		/* x.x.x.x/32 - einzelner Host */
		end   = start + 1;
		}
	else if (net->last == net->adr + 1) {
		start = net->adr;		/* x.x.x.x/31 - 2er Netz */
		end   = start + 2;
		}
	else {
		start = net->adr + 1;
		end   = net->last;
		}
		
	for (adr = start; adr < end; adr++)
		printf ("%u.%u.%u.%u\n", p[3], p[2], p[1], p[0]);

	return (0);
}

int print_networks(int interfaces)
{
	int	i;

	for (i=0; i<netc; i++) {
		if (interfaces)
			printf ("%s ", netv[i]->name);

		print_ipnet(netv[i]);
		}

	return (0);
}


	/*
	 *  M A I N
	 */

void missing_arg(int c, char *string)
{
	fprintf (stderr, "%s: missing arg: -%c, %s\n", program, c, string);
	exit (-1);
}

int main(int argc, char *argv[])
{
	int	c, i, k;
	char	*p, option[80];
	net_t	net;
	
	program = argv[0];
	k = 1;

	while (k < argc  &&  argv[k][0] == '-'  &&  argv[k][1] != 0) {
		copy_string(option, argv[k++], sizeof(option));
		for (i=1; (c = option[i]) != 0; i++) {
			if (c == 'a') {
				if (k >= argc)
					missing_arg(c, "mode");

				c = argv[k++][0];
				mode = ADJUST_STANDARD;
				}
			else if (c == 'c') {
				if (k >= argc)
					missing_arg(c, "ip/mask");

				if (read_ipnet(&net, argv[k++], &p) == 0) {
					print_ipnum(net.number, '\n');
					print_ipnum(net.adr, '\n');
					print_ipnum(net.last, '\n');
					print_ipnum(net.mask, '\n');

					if (k < argc) {
						int	error;
						unsigned long host;
						
						for (i=k; i<argc; i++) {
							host = atoip(skip_ws(argv[i]), &p, &error);
							if (error != 0  ||  *p != 0) {
								fprintf (stderr, "%s: invalid ip nmber: %s\n", program, argv[i]);
								continue;
								}

							if (host > net.adr  &&  host < net.last)
								printf ("+ ");
							else
								printf ("- ");

							print_ipnum(host, '\n');
							}
						}
					}

				exit (0);
				}
			else if (c == 'o') {
				if (k >= argc)
					missing_arg(k, "output mode");

				c = argv[k++][0];
				if (c == 'n')
					output = OUTPUT_STANDARD;
				else if (c == 'c')
					output = OUTPUT_CONFIG;
				else if (c == 'r')
					output = OUTPUT_ROUTES;
				else
					output = OUTPUT_STANDARD;
				}
			else if (c == 's') {
				if (k >= argc)
					missing_arg(k, "sort mode");

				if ((c = argv[k++][0]) == 'n')
					mode = SORT_NETWORKS;
				else if (c == 'm')
					mode = SORT_NETMASKS;
				else if (c == 'i')
					mode = SORT_IPNUMBERS;
				else if (c == 'e')
					mode = SORT_NAMES;
				else
					mode = SORT_IPNUMBERS;
				}
			else {
				fprintf (stderr, "%s: unknown option: -%c\n", program, c);
				exit (-1);
				}
			}
		}


	switch (mode) {
	case SORT_NETWORKS:
		if (read_networks(stdin, 0) > 0) {
			sort_netv(BY_NETWORK);
			print_networks(0);
			}

		break;

	case SORT_NETMASKS:
		if (read_networks(stdin, 0) > 0) {
			sort_netv(BY_NETMASK);
			print_networks(0);
			}

		break;

	case SORT_IPNUMBERS:
		if (read_networks(stdin, 0) > 0) {
			sort_netv(BY_IPNUM);
			for (i=0; i<netc; i++)
				print_ipnum(netv[i]->number, '\n');
			}

		break;

	case SORT_NAMES:
		if (read_networks(stdin, 1) > 0) {
			sort_netv(BY_NAMES);
			print_networks(1);
			}

		break;

	case ADJUST_STANDARD:
	case ADJUST_SMALLEST:
	case ADJUST_LARGEST:
		if (read_networks(stdin, 1) > 0) {
			adjust_netv(mode);

			if (output == OUTPUT_STANDARD)
				print_networks(1);
			else if (output == OUTPUT_CONFIG) {
				for (i=0; i<netc; i++) {
					printf ("%s ", netv[i]->name);
					print_ipnum(netv[i]->number, ' ');
					print_ipnum(netv[i]->adr, ' ');
					print_ipnum(netv[i]->last, ' ');
					print_ipnum(netv[i]->mask, ' ');
					printf ("%c\n", netv[i]->modified? '-': '+');
					}
				}
			else if (output == OUTPUT_ROUTES) {
				int	j;
				char	interface[20];

				for (i=0; i < netc; i++) {
					if (i == 0  ||  netv[k]->interface != netv[i]->interface) {
						copy_string(interface, netv[i]->name, sizeof(interface));
						k = i;
						}

					if (*netv[i]->name == 0)
						continue;

					printf ("%s ", interface);
					print_ipnum(netv[i]->adr, ' ');
					print_ipnum(netv[i]->mask, ' ');
					print_ipnum(netv[i]->last, '\n');

					for (j=i+1; j < netc; j++) {
						if (netv[i]->adr == netv[j]->adr  &&  netv[i]->mask == netv[j]->mask)
							*netv[j]->name = 0;
						}
					}
				}
			}
			
	default:
		if (k < argc) {
			for (i=k; i<argc; i++) {
				if (read_ipnet(&net, argv[i], &p) == 0)
					print_iprange(&net);
				}
			}
		else {
			char	word[80], line[300];

			while (fgets(line, sizeof(line), stdin) != NULL) {
				p = skip_ws(noctrl(line));
				if (*p == 0  ||  *p == '#')
					continue;

				get_word(&p, word, sizeof(word));
				if (read_ipnet(&net, word, &p) == 0)
					print_iprange(&net);
				}
			}

		break;
		}
		
	exit (0);
}


