/*
 * linux_vars.c --
 *
 * This file contains the implementation of Linux specific SNMP variables.
 *
 * Copyright (c) 1996
 *
 * Juergen Schoenwaelder	University of Twente, The Netherlands
 *
 * 
 * Permission to use, copy, modify, and distribute this software and its 
 * documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 * 
 */

#include "mib_module.h"

#ifdef HAVE_LINUX

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>


struct variable linux_variables[] = {
    {LINUXCPU, STRING, RONLY, var_linux, 1, {1}},
    {LINUXBOGO, INTEGER, RONLY, var_linux, 1, {2}}
};

static oid linux_base [] = { 1, 3, 6, 1, 4, 1, 1575, 1, 5, 2 };

void
linux_init()
{
    mib_register(linux_base, 10, (struct variable *)linux_variables,
		 sizeof(linux_variables)/sizeof(*linux_variables),
		 sizeof(*linux_variables));
}

/*
 * Entry for the linux mib:
 */

u_char *
var_linux(vp, name, length, exact, var_len, write_method)
     struct variable *vp;    /* IN - ptr to variable entry that points here */
     oid     *name;	     /* IN/OUT - input name req, output name found */
     int     *length;	     /* IN/OUT - length of input and output oid's */
     int     exact;	     /* IN - TRUE if an exact match was requested. */
     int     *var_len;	     /* OUT - length of var or 0 if function ret. */
     int     (**write_method)();   /* OUT - ptr to func to set var, else 0 */
{
    oid newname[MAX_NAME_LEN];
    int result;
    static char *cpu = NULL;
    static long bogo = 0;

    /* nothing writable provided: */
    *write_method = 0;

    /* default return type: */
    *var_len = sizeof(long);

    bcopy((char *)vp->name, (char *)newname, vp->namelen * sizeof(oid));
    newname[vp->namelen] = 0;
    result = compare(name, *length, newname, (int)vp->namelen + 1);
    if ((exact && (result != 0)) || (!exact && (result >= 0)))
	return NULL;
    bcopy((char *)newname, (char *)name, (vp->namelen + 1) * sizeof(oid));
    *length = vp->namelen + 1;
    
    if (! cpu) {
	char line[256], key[256], val[256];
	FILE *in = fopen("/proc/cpuinfo", "r");
	if (in) {
	    while (fgets(line, sizeof(line), in)) {
		if (2 == sscanf(line, "%s : %s", key, val)) {
		    if (strcmp(key, "cpu") == 0) {
			cpu = strdup(val);
		    } else if (strcmp(key, "bogomips") == 0) {
			bogo = atoi(val);
		    }
		}
	    }
	    fclose(in);
	}
    }

    switch (vp->magic) {
    case LINUXCPU:
	*var_len = strlen(cpu);
	return cpu;
	break;
    case LINUXBOGO:
	return (u_char *) &bogo;
	break;
    }

    return NULL;
}

#endif
