/*
 * hr_vars.c --
 *
 * This file contains the implementation of the Host Resources MIB.
 *
 * Copyright (c) 1995-1997
 *
 * Erik Schoenfelder		TU Braunschweig, Germany
 * Juergen Schoenwaelder	University of Twente, The Netherlands
 * Patrick Weemeeuw		KU Leuven, Belgium
 *
 * 
 * Permission to use, copy, modify, and distribute this software and its 
 * documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in 
 * supporting documentation, and that the name of CMU not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 * ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL
 * THE AUTHORS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR
 * ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
 * WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 * 
 */

#include "mib_module.h"

#ifdef HAVE_HR

#include "hr_vars.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <string.h>
#include <malloc.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/vfs.h>
#include <dirent.h>
#include <linux/tasks.h>
#include <utmp.h>
#include <netinet/in.h>

#include "snmp_util.h"


static struct variable hr_variables[] = {
    {HRSYSTEMUPTIME, TIMETICKS, RONLY, var_hr, 2, {1, 1}},
    {HRSYSTEMDATE, STRING, RONLY, var_hr, 2, {1, 2}},
    {HRSYSTEMINITIALLOADDEVICE, INTEGER, RWRITE, var_hr, 2, {1, 3}},
    {HRSYSTEMINITIALLOADPARAMETERS, STRING, RWRITE, var_hr, 2, {1, 4}},
    {HRSYSTEMNUMUSERS, GAUGE, RONLY, var_hr, 2, {1, 5}},
    {HRSYSTEMPROCESSES, GAUGE, RONLY, var_hr, 2, {1, 6}},
    {HRSYSTEMMAXPROCESSES, INTEGER, RONLY, var_hr, 2, {1, 7}},

    {HRMEMORYSIZE, INTEGER, RONLY, var_hr, 2, {2, 2}},

    {HRSTORAGEINDEX, INTEGER, RONLY, var_hr, 4, {2, 3, 1, 1}},
    {HRSTORAGETYPE, OBJID, RONLY, var_hr, 4, {2, 3, 1, 2}},
    {HRSTORAGEDESCR, STRING, RONLY, var_hr, 4, {2, 3, 1, 3}},
    {HRSTORAGEALLOCUNITS, INTEGER, RONLY, var_hr, 4, {2, 3, 1, 4}},
    {HRSTORAGESIZE, INTEGER, RONLY, var_hr, 4, {2, 3, 1, 5}},
    {HRSTORAGEUSED, INTEGER, RONLY, var_hr, 4, {2, 3, 1, 6}},
    {HRSTORAGEALLOCFAILURES, COUNTER, RONLY, var_hr, 4, {2, 3, 1, 7}},

    {HRDEVICEINDEX, INTEGER, RONLY, var_hr, 4, {3, 2, 1, 1}},
    {HRDEVICETYPE, OBJID, RONLY, var_hr, 4, {3, 2, 1, 2}},
    {HRDEVICEDESCR, STRING, RONLY, var_hr, 4, {3, 2, 1, 3}},
    {HRDEVICEID, OBJID, RONLY, var_hr, 4, {3, 2, 1, 4}},
    {HRDEVICESTATUS, INTEGER, RONLY, var_hr, 4, {3, 2, 1, 5}},
    {HRDEVICEERRORS, COUNTER, RONLY, var_hr, 4, {3, 2, 1, 6}},

    {HRPROCESSORFRWID, OBJID,  RONLY, var_hr, 4, {3, 3, 1, 1 }},
    {HRPROCESSORLOAD, INTEGER, RONLY, var_hr, 4, {3, 3, 1, 2 }},

    {HRFSINDEX, INTEGER, RONLY, var_hr, 4, {3, 8, 1, 1 }},
    {HRFSMOUNTPOINT, STRING, RONLY, var_hr, 4, {3, 8, 1, 2 }},
    {HRFSREMOTEMOUNTPOINT, STRING, RONLY, var_hr, 4, {3, 8, 1, 3 }},
    {HRFSTYPE, OBJID, RONLY, var_hr, 4, {3, 8, 1, 4 }},
    {HRFSACCESS, INTEGER, RONLY, var_hr, 4, {3, 8, 1, 5 }},
    {HRFSBOOTABLE, INTEGER, RONLY, var_hr, 4, {3, 8, 1, 6 }},
    {HRFSSTORAGEINDEX, INTEGER, RONLY, var_hr, 4, {3, 8, 1, 7 }},
    {HRFSLASTFULLBACKUPDATE, STRING, RONLY, var_hr, 4, {3, 8, 1, 8 }},
    {HRFSLASTPARTIALBACKUPDATE, STRING, RONLY, var_hr, 4, {3, 8, 1, 9 }},

    {HRSWOSINDEX, INTEGER, RONLY, var_hr, 2, {4, 1}},

    {HRSWRUNINDEX, INTEGER, RONLY, var_hr, 4, {4, 2, 1, 1}},
    {HRSWRUNNAME, STRING, RONLY, var_hr, 4, {4, 2, 1, 2}},
    {HRSWRUNID, OBJID, RONLY, var_hr, 4, {4, 2, 1, 3}},
    {HRSWRUNPATH, STRING, RONLY, var_hr, 4, {4, 2, 1, 4}},
    {HRSWRUNPARAMETERS, STRING, RONLY, var_hr, 4, {4, 2, 1, 5}},
    {HRSWRUNTYPE, INTEGER, RONLY, var_hr, 4, {4, 2, 1, 6}},
    {HRSWRUNSTATUS, INTEGER, RONLY, var_hr, 4, {4, 2, 1, 7}},

    {HRSWRUNPERFCPU, INTEGER, RONLY, var_hr, 4, {5, 1, 1, 1}},
    {HRSWRUNPERFMEM, INTEGER, RONLY, var_hr, 4, {5, 1, 1, 2}}
};

static oid hr_base [] = { MIB, 25 };



/* initial load device index (points into hrDeviceEntry) */
static unsigned long hr_initial_load_dev = 0;

/* 
 * The code below was contributed by Patrick Weemeeuw
 * <patrick.weemeeuw@kulnet.kuleuven.ac.be> in order to compute
 * hrProcessorLoad more accurately.
 *
 * Well, at least with 12 samples per minute, the hr_processor_load
 * drops to 0 with my 90 mhz box at home. So i would say, don't care
 * about using samples > 2 -- (schoenfr)
 */

#define NROFSAMPLES 6		/* number of samples per minute;
				   minimum 2 */

static int interval = 60/NROFSAMPLES;
static long int idleJiffies[NROFSAMPLES+1];
				/* cyclic array with idle jiffies sampled */
static time_t sampleTimes[NROFSAMPLES+1]; /* time of each sample */
static int sindex;		/* index of the most recent sample */

static void update_HrProcessorLoad();


static long int 
get_IdleJiffies()
{
    FILE *in;
    static long int idle = 0;

    if((in = fopen("/proc/stat", "r"))) {
	(void) fscanf(in, "cpu %*d %*d %*d %ld", &idle);
	fclose(in);
    } else {
	fprintf(stderr, "snmpd: cannot open /proc/stat - please make sure /proc is mounted.\n");
    }
    return idle;
}


void 
hr_init()
{
    int i;
    time_t t = time(0);
    long int ij = get_IdleJiffies();
    
    for (i=0; i <= NROFSAMPLES ;i++) {
	idleJiffies[i] = ij;
	sampleTimes[i] = t;
    }
    sindex = 0;

    /* schedule timer interrupt */
    snmp_alarm (interval, update_HrProcessorLoad);

    /* register: */
    mib_register (hr_base, sizeof(hr_base) / sizeof(oid),
		  hr_variables,
		  sizeof(hr_variables)/sizeof(*hr_variables),
		  sizeof(*hr_variables));

}


static void 
update_HrProcessorLoad()
{
#if 0
    struct timeval tv;
    gettimeofday (&tv, 0);
    fprintf (stderr, "tv: %d.%06d...\n", tv.tv_sec, tv.tv_usec);
#endif
    sindex = (sindex+1) % (NROFSAMPLES+1);
    sampleTimes[sindex] = time((time_t *) 0);
    idleJiffies[sindex] = get_IdleJiffies();
}

static int 
get_HrProcessorLoad()
{
    long int elapsedJiffies, nonIdleJiffies;
    sigset_t currentBlocked, previousBlocked;
    int oldestIndex = (sindex+1) % (NROFSAMPLES+1);
    
    /* prevent signal handler from clobbering the data structures */
    (void) sigemptyset(&currentBlocked);
    (void) sigaddset(&currentBlocked, SIGALRM);
    sigprocmask(SIG_BLOCK, &currentBlocked, &previousBlocked);
    
    /* actual computation */
    elapsedJiffies = (sampleTimes[sindex] - sampleTimes[oldestIndex])*100;
    nonIdleJiffies =
	elapsedJiffies - (idleJiffies[sindex] - idleJiffies[oldestIndex]);

    /* restore signal mask */
    sigprocmask(SIG_SETMASK, &previousBlocked, NULL);
    
    /*
      printf("non idle: %d elapsed: %d\n", nonIdleJiffies, elapsedJiffies);
      */
    
    if (elapsedJiffies > 0)
	return (int) ((100 * nonIdleJiffies) / elapsedJiffies + 0.5);
    else
	return 0;
}


/* 
 * Try to find the boot-device; just a guess -- return root filesystem:
 */

static unsigned int
hr_find_initial_load_dev()
{
    FILE *in = fopen("/etc/mtab", "r");
    char line [512];
    char fs_path [128], mnt_path [128];
    struct stat stbuf;
    
    while (in && fgets(line, sizeof (line), in) 
	   && 2 == sscanf(line, "%s %s", fs_path, mnt_path)) {
	if (! strcmp(mnt_path, "/") && stat(fs_path, &stbuf) >= 0) {
	    fclose (in);
	    return stbuf.st_rdev;
	}
    }
    
    if (in) {
	fclose(in);
    }
    
    /* XXX: wrong, but a legal value */
    return 1;
}

/*
 * returns totalsize (tag == HRSTORAGESIZE) or used (tag == HRSTORAGEUSED)
 * amount in kByte for ram (idx == 0) or swap (idx == 1)
 */

static int
hr_getstor(idx, tag)
    int idx, tag;
{
    char line [1024], s [1024];
    int i, t, u, val;
    int memtotal = 0, memfree = 0, swaptotal = 0, swapfree = 0;
    FILE *in;

    if (idx != 0 && idx != 1) {
      /* should not happen */
      return 9999;
    }

    if (! (in = fopen ("/proc/meminfo", "r"))) {
      perror("snmpd: cannot open /proc/meminfo");
      return 9999;		/* what to do ??? */
    }
	
    for (i = 0; fgets(line, sizeof(line), in); i++) {
      
      /*
       * scan values for v2.1.41 kernels with changed /proc/meminfo layout:
       */
      if (2 == sscanf (line, "%[^:]: %d kB", s, &val)) {
	if (! strcmp (s, "MemTotal")) {
	  memtotal = val;
	} else if (! strcmp (s, "MemFree")) {
	  memfree = val;
	} else if (! strcmp (s, "SwapTotal")) {
	  swaptotal = val;
	} else if (! strcmp (s, "SwapFree")) {
	  swapfree = val;
	}
      }
      
      /*
       * scan for pre 2.1.40 kernels:
       */
      if (3 == sscanf(line, "%s %d %d", s, &t, &u)) {
	if ((idx == 0 && ! strcmp(s, "Mem:"))
	    || (idx == 1 && ! strcmp(s, "Swap:"))) {
	  fclose (in);
	  return (tag == HRSTORAGESIZE ? t : u) / 1024;
	}
      }

    }
    fclose (in);

    if (idx == 0) {
      return tag == HRSTORAGESIZE ? memtotal : memtotal - memfree;
    } else {
      return tag == HRSTORAGESIZE ? swaptotal : swaptotal - swapfree;
    }
}


/*
 * fill process-table for hrswrun/hrswperf table:
 * return list-start.
 */

typedef struct _pslist {
    int pid;
    char stat, *cmd, *cmd_line;
    int type, time, rss;
    struct _pslist *next;
} pslist;

static pslist *
fill_ps_list ()
{
    static pslist *all_ps = 0;
    static time_t tstamp = 0;
    time_t now;

    /* only reread ps-list, of older than 5 seconds: */
    time (&now);
    if (tstamp + 5 > now) {
	return all_ps;
    }
    tstamp = now;

    /* free old list */
    while (all_ps) {
	pslist *p = all_ps;
	all_ps = p->next;
	if (p->cmd) free(p->cmd);
	if (p->cmd_line) free(p->cmd_line);
	free(p);
    }

#ifdef linux
    { 
	DIR *d = opendir("/proc");
	struct dirent *de;

	while (d && (de = readdir(d))) {
	    pslist *nnew, **p;
	    
	    if (de->d_name [0] < '0' && de->d_name [0] > '9') {
		continue;
	    } else {
		char tmp [256], c;
		FILE *in = 0;
		int rss, utime, stime;
		
		nnew = (pslist *) calloc (1, sizeof (pslist));
		if (! nnew) {
		    fprintf (stderr, "warning: out of mem - ignored...\n");
		    break;
		}
		nnew->pid = atoi (de->d_name);
		/* XXX: still its always application: */
		nnew->type = 4;
		
		sprintf(tmp, "/proc/%d/stat", nnew->pid);
		if (! (in = fopen(tmp, "r")) ||
		      5 != fscanf(in, "%*d %s %c %*d %*d %*d %*d %*d %*d %*d %*d %*d %*d %d %d %*d %*d %*d %*d %*d %*d %*d %*d %d", 
				  tmp, &c, &utime, &stime, &rss))
		    nnew->cmd = 0, nnew->stat = '?';
		else {
		    nnew->cmd = strdup (tmp);
		    nnew->stat = c;
		    nnew->time = utime + stime;
		    nnew->rss = rss;
		    
		    fclose (in);
		}
		
		sprintf (tmp, "/proc/%d/cmdline", nnew->pid);
		nnew->cmd_line = 0;
		if ((in = fopen (tmp, "r"))) {
		    int c, i;
		    static len = 0;
		    static char *buf;
		    
		    if (! len && ! (buf = malloc (len = 100))) {
			fclose (in);
			/* ignore and go on: */
			continue;
		    }
		    
		    for (i = 0; buf && (c = fgetc (in)) != EOF; i++) {
			if (i + 1 >= len) {
			    len += 10;
			    if (! (buf = realloc (buf, len)))
				break;
			}
			buf [i] = c ? c : ' ';
		    }
		    if (buf && i > 0) {
			buf [i] = 0;
			nnew->cmd_line = strdup (buf);
		    } else if (nnew->cmd)
			nnew->cmd_line = strdup (nnew->cmd);
		    
		    fclose(in);
		}
	    }
	    
	    /* merge in: */
	    for (p = &all_ps; *p && (*p)->pid < nnew->pid; p = &(*p)->next);
	    nnew->next = *p; 
	    *p = nnew;
	}
      
	if (d) {
	    closedir(d);
	}
    }
#endif
    
    return all_ps;
}


/*
 * retuns static string containing cpu decription;
 * length of string is returned in parameter len.
 */ 
static char *
hr_cpu_desc (len)
int *len;
{
    char line [256];
    static char data [512];
    FILE *in;
    int i;

    if (! (in = fopen("/proc/cpuinfo", "r"))) {
	perror("snmpd: cannot open /proc/cpuinfo");
	*len = 7;
	return "Unknown";
    }
    
    /*
     * return the first 2 lines 
     * (maybe we should select more appropriate info):
     */
    data [0] = *len = 0;
    for (i = 0; fgets(line, sizeof(line), in) && i < 2; i++) {
        char key [256], val [256];
	if (2 == sscanf(line, "%s : %s", key, val)) {
	    sprintf(data + *len, "%s%s: %s", i ? ", " : "", key, val);
	    *len = strlen(data);
	}
    }
    fclose(in);

    return data;
}



u_char *
var_hr(vp, name, length, exact, var_len, write_method)
    struct variable *vp;   /* IN - pointer to var entry that points here */
    oid     *name;	   /* IN/OUT - input name req, output name found */
    int     *length;	   /* IN/OUT - length of input and output oid's */
    int     exact;	   /* IN - TRUE if an exact match was requested. */
    int     *var_len; 	   /* OUT - length of var or 0 if function returned. */
    int     (**write_method)();  /* OUT - pointer to func to set var, else 0 */
{
    oid newname[MAX_NAME_LEN];
    int result;

    /* nothing writable provided: */
    *write_method = 0;

     if (vp->magic <= HRMEMORYSIZE || vp->magic == HRSWOSINDEX) {

	bcopy((char *)vp->name, (char *)newname, vp->namelen * sizeof(oid));
	newname[9] = 0;
	result = compare(name, *length, newname, (int)vp->namelen + 1);
	if ((exact && (result != 0)) || (!exact && (result >= 0)))
	  return NULL;
	bcopy((char *)newname, (char *)name, (vp->namelen + 1) * sizeof(oid));
	*length = vp->namelen + 1;
	
	*var_len = sizeof(long);	/* default length */
	
	/* default value: */
	long_return = 0;
	
	switch (vp->magic){

	  case HRSYSTEMUPTIME:
	    long_return = 0;
	    {
#ifdef linux
		FILE *in;
		double up;
		if ((in = fopen ("/proc/uptime", "r"))) {
		    if (1 == fscanf (in, "%lf", &up)) {
			long_return = (int) (up * 100);
		    }
		    fclose (in);
		}
#endif
	    }
	    break;

	  case HRSYSTEMDATE:
	    { time_t t;
	      struct tm *tt;
	      static char ret [8];
	      time (&t);
	      tt = localtime (&t);
	      * (short *) ret = htons (tt->tm_year);
	      ret [2] = tt->tm_mon + 1;	    ret [3] = tt->tm_mday;
	      ret [4] = tt->tm_hour;	    ret [5] = tt->tm_min;
	      ret [6] = tt->tm_sec;	    ret [7] = 0;
	      *var_len = 8;
	      return ret;
	    }
	    break;

	  case HRSYSTEMINITIALLOADDEVICE:
	    if (hr_initial_load_dev == 0) {
		hr_initial_load_dev = hr_find_initial_load_dev ();
	    }
	    long_return = hr_initial_load_dev;
	    break;

	  case HRSYSTEMINITIALLOADPARAMETERS:
	    {
		static char line[128];
		FILE *in;
		*var_len = 0;
#ifdef linux
		if ((in = fopen ("/proc/cmdline", "r"))) {
		    fgets (line, sizeof (line), in);
		    fclose (in);
		    *var_len = strlen(line);
		    while (*var_len > 0 && isspace(line[*var_len - 1])) {
			line[*var_len - 1] = 0;
			(*var_len)--;
		    }
		}
#endif
		return line;
	    }
	    break;

	  case HRSYSTEMNUMUSERS:
	    {
		struct utmp *ut;
		long_return = 0;
		setutent ();
		while ((ut = getutent ()))
		  long_return += ut->ut_user[0] != 0
				  && strcmp (ut->ut_user, "LOGIN");
		endutent ();
	    }
	    break;

	  case HRSYSTEMPROCESSES:
	    {
#ifdef linux
		DIR *d = opendir ("/proc");
		struct dirent *de;
		int cnt = 0;
		while (d && (de = readdir (d)))
		  cnt += de->d_name [0] >= '0' && de->d_name [0] <= '9';
		if (d) {
		  closedir (d);
		}
		long_return = cnt;
#else
		long_return = 42;
#endif
	    }
	    break;

	  case HRSYSTEMMAXPROCESSES:
#ifdef linux
	    long_return = NR_TASKS;
#else
	    long_return = 99;
#endif
	    break;

	  case HRMEMORYSIZE:
	    { struct stat sbuf;
	      if (stat ("/proc/kcore", &sbuf) < 0)
		{
		    perror ("snmpd: cannot stat /proc/kcore");
		    long_return = 0;
		}
	      else {	/* gives 4k more than avail for 16m - 
			   silently round: */
		long_return = 1024 * (sbuf.st_size / 1024 / 1024);
	      }
  	    }
	    break;

	  case HRSWOSINDEX:
	    /* XXX: fix me: the default is a lie - but what else should 
	       we return ? */
	    long_return = 1;
	    break;
	    
	  default:
	    ERROR("unknown hostresources var");
	    return NULL;
	}
	
	return (u_char *) &long_return;

    } else if (vp->magic <= HRSTORAGEALLOCFAILURES) {

	/*
	 * storage:
	 *
	 * OID:  1.3.6.1.2.1.25.2.3.1.tag.idx
	 */

	int idx, i;
	oid lowest[MAX_NAME_LEN], *op;
	FILE *in = 0;
	char line [128], i_line [128];
	char dev [128];
	struct statfs sbuf;
	struct stat stbuf;
	int stype = 0;		/* storage type: 4 hd, 6 floppy, 1 other */

	static struct hr_stor { 
	    oid type [10];
	    char *descr;
	    int units;
	} stor [] = {
	    { /* idx 0: */ { 1,3,6,1,2,1,25,2,1,2 }, "Mem", 1024 },
	    { /* idx 1: */ { 1,3,6,1,2,1,25,2,1,3 }, "Swap", 1024 },
	};
#define HR_N_STOR	2

	bcopy ((char *) vp->name, (char *) newname, 
	       (int) vp->namelen * sizeof (oid));
	idx = -1;

	for (i = 0; ; i++) 
	  {
	    if (i == HR_N_STOR && ! in) {
		    in = fopen ("/etc/mtab", "r");
	    }
	    if (i >= HR_N_STOR) {
	        if (! in || ! fgets (line, sizeof (line), in))
		  break; 
		if (strncmp (line, "/dev/sd", 7)
		    && strncmp (line, "/dev/sr", 7)
		    && strncmp (line, "/dev/hd", 7)
		    && strncmp (line, "/dev/xd", 7)
		    && strncmp (line, "/dev/fd", 7))
		  continue;
		
		if (1 != sscanf (line, "%s", dev) || stat (dev, &stbuf) < 0)
		      continue;
	    }
	      
	    op = newname + 11;
	    *op++ = i < HR_N_STOR ? i + 1 : stbuf.st_rdev;

	    if (exact) {
		if (compare (newname, 12, name, *length) == 0) {
		    bcopy((char *)newname, (char *)lowest, 12 * sizeof(oid));
		    idx = i;
		    bcopy (line, i_line, sizeof (line));
		    break;  /* no need to search further */
		}
	    } else {
		if (compare (newname, 12, name, *length) > 0 &&
		    ( idx < 0 || compare(newname, 12, lowest, 12) < 0)) {
		    /*
		     * if new one is greater than input and closer to input
		     * than previous lowest, save this one as the "next" one.
		     */
		    bcopy((char *)newname, (char *)lowest, 12 * sizeof(oid));
		    bcopy (line, i_line, sizeof (line));
		    idx = i;
		}
	    }
	}

	if (in) {
	    fclose (in);
	}

	if (idx < 0) {
	    return NULL;
	}

	bcopy ((char *) lowest, (char *) name, 
	       ((int) vp->namelen + 1) * sizeof(oid));
	*length = vp->namelen + 1;

	if (idx >= HR_N_STOR && in)
	  {
	      char mnt [256];
	      if (1 != sscanf (i_line, "%*s %s", mnt)
		  || statfs (mnt, &sbuf) < 0) {
		  sbuf.f_blocks = sbuf.f_bfree = 0;
		  sbuf.f_bsize = 1;
	      }
	      /* hrStorageTypes: 
		 1 == other, 4 == disk, 5 == removeable, 6 == floppy */
	      stype = ! strncmp (i_line, "/dev/sd", 7) ? 4 :
	              ! strncmp (i_line, "/dev/sr", 7) ? 5 :
		      ! strncmp (i_line, "/dev/hd", 7) ? 4 :   /* XXX: cdrom */
		      ! strncmp (i_line, "/dev/xd", 7) ? 4 :
		      ! strncmp (i_line, "/dev/fd", 7) ? 6 : 1;
	  }

	*var_len = sizeof(long);

	switch (vp->magic) {

	  case HRSTORAGEINDEX:
/*	    long_return = idx + 1; */
	    long_return = name [11];
	    break;

	  case HRSTORAGETYPE:
	    { static oid otype [10] = { 1,3,6,1,2,1,25,2,1, };
	      *var_len = sizeof (otype);
	      if (idx < 2)
		return (char *) stor[idx].type;
	      otype [9] = stype;
	      return (char *) otype;
	    }

	  case HRSTORAGEDESCR:
	    { char *s = idx < 2 ? stor[idx].descr : 
		stype == 4 ? "Disk" : 
		stype == 6 ? "Floppy" : 
		stype == 5 ? "CDROM" : "Unknown";
	      *var_len = strlen (s);
	      return s;
	    }

	  case HRSTORAGEALLOCUNITS:
	    long_return = 1024;			/* in KB */
	    break;

	  case HRSTORAGESIZE:
	    long_return = idx < 2 ? hr_getstor (idx, HRSTORAGESIZE) :
		(1.0 * sbuf.f_blocks * sbuf.f_bsize) / 1024;
	    break;

	  case HRSTORAGEUSED:
	    long_return = idx < 2 ? hr_getstor (idx, HRSTORAGEUSED) :
		(1.0 * (sbuf.f_blocks-sbuf.f_bfree) * sbuf.f_bsize) / 1024;
	    break;

	  case HRSTORAGEALLOCFAILURES:
	    long_return = 0;			/* XXX: dummy */
	    break;

	  default:
	    ERROR("unknown hostresources var");
	    return NULL;
	}

	return (u_char *) &long_return;

    } else if (vp->magic <= HRDEVICEERRORS) {

	/*
	 * device:
	 *
	 * OID:  1.3.6.1.2.1.25.3.2.1.tag.idx
	 */

	int idx, i;
	oid lowest[MAX_NAME_LEN], *op;

	static struct hr_dev { 
	    oid type [10];
	    char *descr;
	    int status;
	} dev [] = {
	    { /* idx 0: */ { 1,3,6,1,2,1,25,3,1,3 }, "CPU", 2 }
	};
#define HR_N_DEV	1

	bcopy ((char *) vp->name, (char *) newname, 
	       (int) vp->namelen * sizeof (oid));
	idx = -1;

	for (i = 0; i < HR_N_DEV; i++) 
	  {
	      op = newname + 11;
	      *op++ = i + 1;

	    if (exact) {
		if (compare (newname, 12, name, *length) == 0) {
		    bcopy((char *)newname, (char *)lowest, 12 * sizeof(oid));
		    idx = i;
		    break;  /* no need to search further */
		}
	    } else {
		if (compare (newname, 12, name, *length) > 0 &&
		    ( idx < 0 || compare(newname, 12, lowest, 12) < 0)) {
		    /*
		     * if new one is greater than input and closer to input
		     * than previous lowest, save this one as the "next" one.
		     */
		    bcopy((char *)newname, (char *)lowest, 12 * sizeof(oid));
		    idx = i;
		}
	    }
	}

	if (idx < 0 || idx >= HR_N_DEV)
	  return NULL;

	bcopy ((char *) lowest, (char *) name, 
	       ((int) vp->namelen + 1) * sizeof(oid));

	*length = vp->namelen + 1;
	*var_len = sizeof(long);

	switch (vp->magic) {

	  case HRDEVICEINDEX:
	    long_return = idx + 1;
	    break;

	  case HRDEVICETYPE:
	    *var_len = sizeof (dev[idx].type);
	    return (char *) dev[idx].type;
	    break;

	  case HRDEVICEDESCR:
	    if (idx == 0)
	      return hr_cpu_desc (var_len);
	    else {
		*var_len = strlen (dev[idx].descr);
		return dev[idx].descr;
	    }

	  case HRDEVICEID:
	    { static oid no [2] = { 0, 0 };
	      *var_len = sizeof (no);
	      return (char *) no;
	    }
	    break;

	  case HRDEVICESTATUS:
	    long_return = 2;		/* running */
	    break;

	  case HRDEVICEERRORS:
	    long_return = 0;
	    break;

	  default:
	    ERROR("unknown hostresources var");
	    return NULL;
	}

	return (u_char *) &long_return;

    } else if (vp->magic <= HRPROCESSORLOAD) {

	/*
	 * device:
	 *
	 * OID:  1.3.6.1.2.1.25.3.3.1.tag.idx
	 */

	int idx;
	oid lowest[MAX_NAME_LEN], *op;

	bcopy ((char *) vp->name, (char *) newname, 
	       (int) vp->namelen * sizeof (oid));
	idx = -1;

	/* single processor assumed: */
	op = newname + 11;
	*op++ = 1;

	if (exact) {
	    if (compare (newname, 12, name, *length) == 0) {
		bcopy((char *)newname, (char *)lowest, 12 * sizeof(oid));
		idx = 0;
	    }
	} else {
	    if (compare (newname, 12, name, *length) > 0 &&
		( idx < 0 || compare(newname, 12, lowest, 12) < 0)) {
		/*
		 * if new one is greater than input and closer to input
		 * than previous lowest, save this one as the "next" one.
		 */
		bcopy((char *)newname, (char *)lowest, 12 * sizeof(oid));
		idx = 0;
	    }
	}
	
	if (idx < 0)
	  return NULL;

	bcopy ((char *) lowest, (char *) name, 
	       ((int) vp->namelen + 1) * sizeof(oid));

	*length = vp->namelen + 1;

	if (vp->magic == HRPROCESSORFRWID) {
	    static oid nulloid [2];
	    *var_len = 2 * sizeof(oid);
	    return (char *) nulloid;
	} else {
#ifdef linux
	    *var_len = sizeof(long);
	    long_return = get_HrProcessorLoad();
	    return (char *) &long_return;
#else
	    /* guess a processorload from the avenrun index (over the
               last minute): */
	    { 
		FILE *in = fopen ("/proc/loadavg", "r");
		long_return = 0;		/* fail default */
		if (in) {
		    double d;
		    if (1 == fscanf(in, "%lf", &d))
			long_return = d >= 1.0 ? 100 : d * 100;
		    fclose(in);
		}
		*var_len = sizeof(long);
		return (char *) &long_return;
	    }
#endif
	}

    } else if (vp->magic <= HRFSLASTPARTIALBACKUPDATE) {

	/*
	 * device:
	 *
	 * OID:  1.3.6.1.2.1.25.3.8.1.tag.idx
	 */

	int idx, i;
	oid lowest[MAX_NAME_LEN], *op;
	FILE *in = fopen ("/etc/mtab", "r");
	char line [512];
	char fs_path [128], mnt_path [128];
	static char i_fs_path [128], i_mnt_path [128];
	struct stat stbuf;

	bcopy ((char *) vp->name, (char *) newname, 
	       (int) vp->namelen * sizeof (oid));
	idx = -1;

	for (i = 0; in; i++) 
	  {
	      if (! fgets (line, sizeof (line), in) 
		  || 2 != sscanf (line, "%s %s", fs_path, mnt_path))
		break;

		    if (strncmp (line, "/dev/sd", 7)
			&& strncmp (line, "/dev/sr", 7)
			&& strncmp (line, "/dev/hd", 7)
			&& strncmp (line, "/dev/xd", 7)
			&& strncmp (line, "/dev/fd", 7))
		      continue;
	      if (stat (fs_path, &stbuf) < 0)
		continue;
	      
	      op = newname + 11;
	      *op++ = stbuf.st_rdev;		/* i + 1 */

	      if (exact) {
		  if (compare (newname, 12, name, *length) == 0) {
		      bcopy((char *)newname, (char *)lowest, 12 * sizeof(oid));
		      idx = i;
		      bcopy (fs_path, i_fs_path, 128);
		      bcopy (mnt_path, i_mnt_path, 128);
		      break;  /* no need to search further */
		  }
	      } else {
		  if (compare (newname, 12, name, *length) > 0 &&
		      ( idx < 0 || compare(newname, 12, lowest, 12) < 0)) {
		      /*
		       * if new one is greater than input and closer to input
		       * than previous lowest, save this one as the "next" one.
		       */
		      bcopy((char *)newname, (char *)lowest, 12 * sizeof(oid));
		      idx = i;
		      bcopy (fs_path, i_fs_path, 128);
		      bcopy (mnt_path, i_mnt_path, 128);
		  }
	      }
	  }

	if (in)
	  fclose (in);
	
	if (idx < 0)
	  return NULL;
	
	bcopy ((char *) lowest, (char *) name, 
	       ((int) vp->namelen + 1) * sizeof(oid));
	
	*length = vp->namelen + 1;
	*var_len = sizeof(long);

	switch (vp->magic) {

	  case HRFSINDEX:
	    long_return = name [11];
	    break;

	  case HRFSMOUNTPOINT:
	    *var_len = strlen (i_fs_path);
	    return i_fs_path;
	    break;

	  case HRFSREMOTEMOUNTPOINT:
	    *var_len = strlen (i_mnt_path);
	    return i_mnt_path;
	    break;

	  case HRFSTYPE:
	    { /* hrFSUnknown */
	      static oid fsoid [10] = { 1, 3, 6, 1, 2, 1, 25, 3, 9, 2 };
	      *var_len = sizeof (fsoid);
	      return (char *) fsoid;
	    }
	    break;

	  case HRFSACCESS:
	    long_return = 1;		/* rw */
	    break;

	  case HRFSBOOTABLE:
	    /* XXX: just a guess - maybe bullshit */
	    { int flag = strcmp (i_mnt_path, "/") ? 2 : 1;
	      long_return = flag;
	    }
	    break;

	  case HRFSSTORAGEINDEX:
	    long_return = name [11];		/* same as index */
	    break;

	  case HRFSLASTFULLBACKUPDATE:
	  case HRFSLASTPARTIALBACKUPDATE:
	    { static char *epoch = "\000\000\001\001\000\000\000\000";
	      *var_len = 8;
	      return epoch;
	    }
	    break;

	  default:
	    ERROR("unknown hostresources var");
	    return NULL;
	}

	return (u_char *) &long_return;

    } else if (vp->magic <= HRSWRUNPERFMEM) {

	/*
	 * runtable:
	 *
	 * swrun-OID:   1.3.6.1.2.1.25.4.2.1.tag.idx
	 * swperf-OID:  1.3.6.1.2.1.25.5.1.1.tag.idx
	 */

	oid lowest[MAX_NAME_LEN], *op;
	pslist *ps = 0, *p;
	bcopy ((char *) vp->name, (char *) newname, 
	       (int) vp->namelen * sizeof (oid));

	if (! (p = fill_ps_list ()))
	  return 0;

	for (; p; p = p->next)
	  {
	      op = newname + 11;
	      *op++ = p->pid;
	      
	      if (exact) {
		  if (compare (newname, 12, name, *length) == 0) {
		      bcopy((char *)newname, (char *)lowest, 12 * sizeof(oid));
		      ps = p;
		      break;  /* no need to search further */
		  }
	      } else {
		  if (compare (newname, 12, name, *length) > 0 &&
		      ( ! ps || compare(newname, 12, lowest, 12) < 0)) {
		      /*
		       * if new one is greater than input and closer to input
		       * than previous lowest, save this one as the "next" one.
		       */
		      bcopy((char *)newname, (char *)lowest, 12 * sizeof(oid));
		      ps = p;
		  }
	      }
	  }
	
	if (! ps)
	  return NULL;
	
	bcopy ((char *) lowest, (char *) name, 
	       ((int) vp->namelen + 1) * sizeof(oid));

	*length = vp->namelen + 1;
	*var_len = sizeof(long);
	
	switch (vp->magic) {

	  case HRSWRUNINDEX:
	    /* well, not a unique number for each piece of software, 
	       just simply the pid: */
	    long_return = ps->pid;
	    break;

	  case HRSWRUNNAME:
	    /* XXX: we could use configfile-entries to set descriptions
	       for selected pieces of software; currently nothing: */
	    *var_len = 9; 
	    return "(unknown)";	    
	    break;

	  case HRSWRUNID:
	    { /* unknown product id: */
	      static oid o [2] = { 0, 0 };
	      *var_len = sizeof (o);
	      return (char *) o;
	    }
	    break;

	  case HRSWRUNPATH:
	    /* return at least the basename of the program: */
	    if (ps->cmd)
	      {
		  *var_len = strlen (ps->cmd);
		  if (ps->cmd [0] == '(' && ps->cmd [*var_len - 1] == ')')
		    {
			*var_len -= 2;
			return ps->cmd + 1;
		    }
		  return ps->cmd;
	      }
	    *var_len = 9; 
	    return "(unknown)";	    
	    break;

	  case HRSWRUNPARAMETERS:
	    /* complete commandline currently known (includes argv[0]): */
	    if (ps->cmd_line)
	      {
		  *var_len = strlen (ps->cmd_line);
		  return ps->cmd_line;
	      }
	    *var_len = 9; 
	    return "(unknown)";	    
	    break;

	  case HRSWRUNTYPE:
	    long_return = ps->type;
	    break;

	  case HRSWRUNSTATUS:
	    long_return = ps->stat == 'R' ? 1 : ps->stat == 'D' ? 3 : 2;
	    break;

	  case HRSWRUNPERFCPU:
	    long_return = ps->time;
	    break;

	  case HRSWRUNPERFMEM:
	    long_return = ps->rss << 2;
	    break;

	  default:
	    ERROR("unknown hostresources var");
	    return NULL;
	}

	return (u_char *) &long_return;
    }

    return 0;
}

#endif /* HAVE_HR */
